/*-------------------------------------------------------------------------
 *
 * pl_comp.c		- Compiler part of the PL/pgSQL
 *			  procedural language
 *
 * Portions Copyright (c) 1996-2006, PostgreSQL Global Development Group
 * Portions Copyright (c) 1994, Regents of the University of California
 *
 *
 * IDENTIFICATION
 *	  $PostgreSQL: pgsql/src/pl/plpgpsm/src/pl_comp.c,v 1.108 2006/10/04 00:30:13 momjian Exp $
 *
 *-------------------------------------------------------------------------
 */

#include "plpgpsm.h"

#include <ctype.h>

#include "pl_gram.h"

#include "catalog/namespace.h"
#include "catalog/pg_attrdef.h"
#include "catalog/pg_attribute.h"
#include "catalog/pg_class.h"
#include "catalog/pg_proc.h"
#include "catalog/pg_proc_fn.h"
#include "catalog/pg_type.h"
#include "funcapi.h"
#include "nodes/makefuncs.h"
#include "parser/gramparse.h"
#include "parser/parse_type.h"
#include "tcop/tcopprot.h"
#include "utils/array.h"
#include "utils/builtins.h"
#include "utils/lsyscache.h"
#include "utils/memutils.h"
#include "utils/syscache.h"


/* ----------
 * Our own local and global variables
 * ----------
 */
static int	datums_alloc;
int			plpgpsm_nDatums;
PLpgPSM_datum **plpgpsm_Datums;
static int	datums_last = 0;

int			plpgpsm_error_lineno;
char	   *plpgpsm_error_funcname;
bool		plpgpsm_DumpExecTree = false;
bool		plpgpsm_check_syntax = false;

int plpgpsm_user_cond;

PLpgPSM_function *plpgpsm_curr_compile;

/* A context appropriate for short-term allocs during compilation */
MemoryContext compile_tmp_cxt;

/* ----------
 * Hash table for compiled functions
 * ----------
 */
static HTAB *plpgpsm_HashTable = NULL;

typedef struct plpgpsm_hashent
{
	PLpgPSM_func_hashkey key;
	PLpgPSM_function *function;
} plpgpsm_HashEnt;

#define FUNCS_PER_USER		128 /* initial table size */

/* ----------
 * Lookup table for EXCEPTION condition names
 * ----------
 */
typedef struct
{
	const char *label;
	int			sqlerrstate;
} ExceptionLabelMap;

const ExceptionLabelMap exception_label_map[] = {
#include "plerrcodes.h"
	{NULL, 0}
};


/* ----------
 * static prototypes
 * ----------
 */
static PLpgPSM_function *do_compile(FunctionCallInfo fcinfo,
		   HeapTuple procTup,
		   PLpgPSM_function *function,
		   PLpgPSM_func_hashkey *hashkey,
		   bool forValidator);
static PLpgPSM_row *build_row_from_class(Oid classOid, List **initvars);
static PLpgPSM_row *build_row_from_vars(PLpgPSM_variable **vars, int numvars);
static PLpgPSM_type *build_datatype(HeapTuple typeTup, int32 typmod);
static void compute_function_hashkey(FunctionCallInfo fcinfo,
						 Form_pg_proc procStruct,
						 PLpgPSM_func_hashkey *hashkey,
						 bool forValidator);
static void plpgpsm_resolve_polymorphic_argtypes(int numargs,
									 Oid *argtypes, char *argmodes,
									 Node *call_expr, bool forValidator,
									 const char *proname);
static PLpgPSM_function *plpgpsm_HashTableLookup(PLpgPSM_func_hashkey *func_key);
static void plpgpsm_HashTableInsert(PLpgPSM_function *function,
						PLpgPSM_func_hashkey *func_key);
static void plpgpsm_HashTableDelete(PLpgPSM_function *function);
static void delete_function(PLpgPSM_function *func);


/* ----------
 * plpgpsm_compile		Make an execution tree for a PL/pgSQL function.
 *
 * If forValidator is true, we're only compiling for validation purposes,
 * and so some checks are skipped.
 *
 * Note: it's important for this to fall through quickly if the function
 * has already been compiled.
 * ----------
 */
PLpgPSM_function *
plpgpsm_compile(FunctionCallInfo fcinfo, bool forValidator)
{
	Oid			funcOid = fcinfo->flinfo->fn_oid;
	HeapTuple	procTup;
	Form_pg_proc procStruct;
	PLpgPSM_function *function;
	PLpgPSM_func_hashkey hashkey;
	bool		function_valid = false;
	bool		hashkey_valid = false;

	/*
	 * Lookup the pg_proc tuple by Oid; we'll need it in any case
	 */
	procTup = SearchSysCache(PROCOID,
							 ObjectIdGetDatum(funcOid),
							 0, 0, 0);
	if (!HeapTupleIsValid(procTup))
		elog(ERROR, "cache lookup failed for function %u", funcOid);
	procStruct = (Form_pg_proc) GETSTRUCT(procTup);

	/*
	 * See if there's already a cache entry for the current FmgrInfo. If not,
	 * try to find one in the hash table.
	 */
	function = (PLpgPSM_function *) fcinfo->flinfo->fn_extra;

recheck:
	if (!function)
	{
		/* Compute hashkey using function signature and actual arg types */
		compute_function_hashkey(fcinfo, procStruct, &hashkey, forValidator);
		hashkey_valid = true;

		/* And do the lookup */
		function = plpgpsm_HashTableLookup(&hashkey);
	}

	if (function)
	{
               /* We have a compiled function, but is it still valid? */
                if (function->fn_xmin == HeapTupleHeaderGetXmin(procTup->t_data) &&                  
			ItemPointerEquals(&function->fn_tid, &procTup->t_self))
                        function_valid = true; 
                else                                                                                 
                { 
                        /* 
                         * Nope, so remove it from hashtable and try to drop associated              
                         * storage (if not done already).                                            
                         */
                        delete_function(function);
                        /*                                                                           
                         * If the function isn't in active use then we can overwrite the             
                         * func struct with new data, allowing any other existing fn_extra           
                         * pointers to make use of the new definition on their next use.             
                         * If it is in use then just leave it alone and make a new one.              
                         * (The active invocations will run to completion using the                  
                         * previous definition, and then the cache entry will just be                
                         * leaked; doesn't seem worth adding code to clean it up, given              
                         * what a corner case this is.)                                              
                         *                                                                           
                         * If we found the function struct via fn_extra then it's possible           
                         * a replacement has already been made, so go back and recheck               
                         * the hashtable.                                                            
                         */ 
                        if (function->use_count != 0)
                        {
                                function = NULL;
                                if (!hashkey_valid)
                                        goto recheck;
                        } 
                } 
        }

	/*
	 * If the function wasn't found or was out-of-date, we have to compile it
	 */
	if (!function_valid)
	{
		/*
		 * Calculate hashkey if we didn't already; we'll need it to store the
		 * completed function.
		 */
		if (!hashkey_valid)
			compute_function_hashkey(fcinfo, procStruct, &hashkey,
									 forValidator);

		/*
		 * Do the hard part.
		 */
		function = do_compile(fcinfo, procTup, function,
							&hashkey, forValidator);
	}

	ReleaseSysCache(procTup);

	/*
	 * Save pointer in FmgrInfo to avoid search on subsequent calls
	 */
	fcinfo->flinfo->fn_extra = (void *) function;

	/*
	 * Finally return the compiled function
	 */

	return function;
}

/*
 * This is the slow part of plpgpsm_compile().
 *
 * The passed-in "function" pointer is either NULL or an already-allocated                           
 * function struct to overwrite.                                                                     
 *      
 * While compiling a function, the CurrentMemoryContext is the
 * per-function memory context of the function we are compiling. That
 * means a palloc() will allocate storage with the same lifetime as
 * the function itself.
 *
 * Because palloc()'d storage will not be immediately freed, temporary
 * allocations should either be performed in a short-lived memory
 * context or explicitly pfree'd. Since not all backend functions are
 * careful about pfree'ing their allocations, it is also wise to
 * switch into a short-term context before calling into the
 * backend. An appropriate context for performing short-term
 * allocations is the compile_tmp_cxt.
 * NB: this code is not re-entrant.  We assume that nothing we do here could                         
 * result in the invocation of another plpgsql function.                                             
 */
static PLpgPSM_function *
do_compile(FunctionCallInfo fcinfo,
		   HeapTuple procTup,
		   PLpgPSM_function *function,
		   PLpgPSM_func_hashkey *hashkey,
		   bool forValidator)
{
	Form_pg_proc procStruct = (Form_pg_proc) GETSTRUCT(procTup);
	int			functype = CALLED_AS_TRIGGER(fcinfo) ? T_TRIGGER : T_FUNCTION;
	Datum		prosrcdatum;
	bool		isnull;
	char	   *proc_source;
	HeapTuple	typeTup;
	Form_pg_type typeStruct;
	PLpgPSM_variable *var;
	PLpgPSM_rec *rec;
	int			i;
	ErrorContextCallback plerrcontext;
	int			parse_rc;
	Oid			rettypeid;
	int			numargs;
	int			num_in_args = 0;
	int			num_out_args = 0;
	Oid		   *argtypes;
	char	  **argnames;
	char	   *argmodes;
	int		   *in_arg_varnos = NULL;
	PLpgPSM_variable **out_arg_variables;
	MemoryContext func_cxt;

	/*
	 * Setup the scanner input and error info.	We assume that this function
	 * cannot be invoked recursively, so there's no need to save and restore
	 * the static variables used here.
	 */
	prosrcdatum = SysCacheGetAttr(PROCOID, procTup,
								  Anum_pg_proc_prosrc, &isnull);
	if (isnull)
		elog(ERROR, "null prosrc");
	proc_source = DatumGetCString(DirectFunctionCall1(textout, prosrcdatum));
	plpgpsm_scanner_init(proc_source, functype);

	plpgpsm_error_funcname = format_procedure(fcinfo->flinfo->fn_oid);
	plpgpsm_error_lineno = 0;

	/*
	 * Setup error traceback support for ereport()
	 */
	plerrcontext.callback = plpgpsm_compile_error_callback;
	plerrcontext.arg = forValidator ? proc_source : NULL;
	plerrcontext.previous = error_context_stack;
	error_context_stack = &plerrcontext;

	/*
	 * Initialize the compiler
	 */
	plpgpsm_ns_init();
	plpgpsm_ns_push(NULL);
	plpgpsm_DumpExecTree = false;

	datums_alloc = 128;
	plpgpsm_nDatums = 0;
	/* This is short-lived, so needn't allocate in function's cxt */
	plpgpsm_Datums = palloc(sizeof(PLpgPSM_datum *) * datums_alloc);
	datums_last = 0;

	plpgpsm_user_cond = 0;

	/*
	 * Do extra syntax checks when validating the function definition. We skip
	 * this when actually compiling functions for execution, for performance
	 * reasons.
	 */
	plpgpsm_check_syntax = forValidator;

       /*                                                                                           
         * Create the new function struct, if not done already.  The function                        
         * structs are never thrown away, so keep them in TopMemoryContext.                          
         */                                                                                          
        if (function == NULL)
        {
                function = (PLpgPSM_function *)
                        MemoryContextAllocZero(TopMemoryContext, sizeof(PLpgPSM_function));
        }
        else
        {
                /* re-using a previously existing struct, so clear it out */
                memset(function, 0, sizeof(PLpgPSM_function));
        }
        plpgpsm_curr_compile = function;

       /*                                                                                           
         * All the rest of the compile-time storage (e.g. parse tree) is kept in                     
         * its own memory context, so it can be reclaimed easily.                                    
         */                          
	func_cxt = AllocSetContextCreate(TopMemoryContext,
									 "PL/PgSQL function context",
									 ALLOCSET_DEFAULT_MINSIZE,
									 ALLOCSET_DEFAULT_INITSIZE,
									 ALLOCSET_DEFAULT_MAXSIZE);
	compile_tmp_cxt = MemoryContextSwitchTo(func_cxt);

	function->fn_name = format_procedure(fcinfo->flinfo->fn_oid);
	function->fn_oid = fcinfo->flinfo->fn_oid;
	function->fn_xmin = HeapTupleHeaderGetXmin(procTup->t_data);
	function->fn_tid = procTup->t_self;
	function->fn_functype = functype;
	function->fn_cxt = func_cxt;
	function->out_param_varno = -1;		/* set up for no OUT param */

	switch (functype)
	{
		case T_FUNCTION:

			/*
			 * Fetch info about the procedure's parameters. Allocations aren't
			 * needed permanently, so make them in tmp cxt.
			 *
			 * We also need to resolve any polymorphic input or output
			 * argument types.	In validation mode we won't be able to, so we
			 * arbitrarily assume we are dealing with integers.
			 */
			MemoryContextSwitchTo(compile_tmp_cxt);

			numargs = get_func_arg_info(procTup,
										&argtypes, &argnames, &argmodes);

			plpgpsm_resolve_polymorphic_argtypes(numargs, argtypes, argmodes,
												 fcinfo->flinfo->fn_expr,
												 forValidator,
												 plpgpsm_error_funcname);

			in_arg_varnos = (int *) palloc(numargs * sizeof(int));
			out_arg_variables = (PLpgPSM_variable **) palloc(numargs * sizeof(PLpgPSM_variable *));

			MemoryContextSwitchTo(func_cxt);

			/*
			 * Create the variables for the procedure's parameters.
			 */
			 
			plpgpsm_ns_push(NameStr(procStruct->proname));
			 
			for (i = 0; i < numargs; i++)
			{
				char		buf[32];
				Oid			argtypeid = argtypes[i];
				char		argmode = argmodes ? argmodes[i] : PROARGMODE_IN;
				PLpgPSM_type *argdtype;
				PLpgPSM_variable *argvariable;
				int			argitemtype;

				/* Create $n name for variable */
				snprintf(buf, sizeof(buf), "$%d", i + 1);

				/* Create datatype info */
				argdtype = plpgpsm_build_datatype(argtypeid, -1);

				/* Disallow pseudotype argument */
				/* (note we already replaced polymorphic types) */
				/* (build_variable would do this, but wrong message) */
				if (argdtype->ttype != PLPGPSM_TTYPE_SCALAR &&
					argdtype->ttype != PLPGPSM_TTYPE_ROW)
					ereport(ERROR,
							(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
							 errmsg("plpgpsm functions cannot take type %s",
									format_type_be(argtypeid))));

				/* Build variable and add to datum list */
				argvariable = plpgpsm_build_variable(buf, 0,
													 argdtype, false, NULL);

				if (argvariable->dtype == PLPGPSM_DTYPE_VAR)
				{
					argitemtype = PLPGPSM_NSTYPE_VAR;
				}
				else
				{
					Assert(argvariable->dtype == PLPGPSM_DTYPE_ROW);
					argitemtype = PLPGPSM_NSTYPE_ROW;
				}

				/* Remember arguments in appropriate arrays */
				if (argmode == PROARGMODE_IN || argmode == PROARGMODE_INOUT)
					in_arg_varnos[num_in_args++] = argvariable->dno;
				if (argmode == PROARGMODE_OUT || argmode == PROARGMODE_INOUT)
					out_arg_variables[num_out_args++] = argvariable;

				/* Add to namespace under the $n name */
				plpgpsm_ns_additem(argitemtype, argvariable->dno, buf);

				/* If there's a name for the argument, make an alias */
				if (argnames && argnames[i][0] != '\0')
					plpgpsm_ns_additem(argitemtype, argvariable->dno,
									   argnames[i]);
			}

			/*
			 * If there's just one OUT parameter, out_param_varno points
			 * directly to it.	If there's more than one, build a row that
			 * holds all of them.
			 */
			if (num_out_args == 1)
				function->out_param_varno = out_arg_variables[0]->dno;
			else if (num_out_args > 1)
			{
				PLpgPSM_row *row = build_row_from_vars(out_arg_variables,
													   num_out_args);

				plpgpsm_adddatum((PLpgPSM_datum *) row);
				function->out_param_varno = row->dno;
			}

			/*
			 * Check for a polymorphic returntype. If found, use the actual
			 * returntype type from the caller's FuncExpr node, if we have
			 * one.  (In validation mode we arbitrarily assume we are dealing
			 * with integers.)
			 *
			 * Note: errcode is FEATURE_NOT_SUPPORTED because it should always
			 * work; if it doesn't we're in some context that fails to make
			 * the info available.
			 */
			rettypeid = procStruct->prorettype;
			if (IsPolymorphicType(rettypeid))
			{
				if (forValidator)
				{
					if (rettypeid == ANYARRAYOID)
						rettypeid = INT4ARRAYOID;
					else
						rettypeid = INT4OID;
					/* XXX what could we use for ANYENUM? */
				}
				else
				{
					rettypeid = get_fn_expr_rettype(fcinfo->flinfo);
					if (!OidIsValid(rettypeid))
						ereport(ERROR,
								(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
							 errmsg("could not determine actual return type "
									"for polymorphic function \"%s\"",
									plpgpsm_error_funcname)));
				}
			}

			/*
			 * Normal function has a defined returntype
			 */
			function->fn_rettype = rettypeid;
			function->fn_retset = procStruct->proretset;

			/*
			 * Lookup the function's return type
			 */
			typeTup = SearchSysCache(TYPEOID,
									 ObjectIdGetDatum(rettypeid),
									 0, 0, 0);
			if (!HeapTupleIsValid(typeTup))
				elog(ERROR, "cache lookup failed for type %u", rettypeid);
			typeStruct = (Form_pg_type) GETSTRUCT(typeTup);

			/* Disallow pseudotype result, except VOID or RECORD */
			/* (note we already replaced polymorphic types) */
			if (typeStruct->typtype == TYPTYPE_PSEUDO)
			{
				if (rettypeid == VOIDOID ||
					rettypeid == RECORDOID)
					 /* okay */ ;
				else if (rettypeid == TRIGGEROID)
					ereport(ERROR,
							(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
							 errmsg("trigger functions may only be called as triggers")));
				else
					ereport(ERROR,
							(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
							 errmsg("plpgpsm functions cannot return type %s",
									format_type_be(rettypeid))));
			}

			if (typeStruct->typrelid != InvalidOid ||
				rettypeid == RECORDOID)
				function->fn_retistuple = true;
			else
			{
				function->fn_retbyval = typeStruct->typbyval;
				function->fn_rettyplen = typeStruct->typlen;
				function->fn_rettypioparam = getTypeIOParam(typeTup);
				fmgr_info(typeStruct->typinput, &(function->fn_retinput));

				/*
				 * install $0 reference, but only for polymorphic return
				 * types, and not when the return is specified through an
				 * output parameter.
				 */
				if (IsPolymorphicType(procStruct->prorettype) &&
					num_out_args == 0)
				{
					(void) plpgpsm_build_variable("$0", 0,
												  build_datatype(typeTup, -1),
												  true, NULL);
				}
			}
			ReleaseSysCache(typeTup);
			break;

		case T_TRIGGER:
			/* Trigger procedure's return type is unknown yet */
			function->fn_rettype = InvalidOid;
			function->fn_retbyval = false;
			function->fn_retistuple = true;
			function->fn_retset = false;

			/* shouldn't be any declared arguments */
			if (procStruct->pronargs != 0)
				ereport(ERROR,
						(errcode(ERRCODE_INVALID_FUNCTION_DEFINITION),
				  errmsg("trigger functions cannot have declared arguments"),
						 errhint("You probably want to use TG_NARGS and TG_ARGV instead.")));

			/* Add the record for referencing NEW */
			rec = palloc0(sizeof(PLpgPSM_rec));
			rec->dtype = PLPGPSM_DTYPE_REC;
			rec->refname = pstrdup("new");
			rec->tup = NULL;
			rec->tupdesc = NULL;
			rec->freetup = false;
			plpgpsm_adddatum((PLpgPSM_datum *) rec);
			plpgpsm_ns_additem(PLPGPSM_NSTYPE_REC, rec->dno, rec->refname);
			function->new_varno = rec->dno;

			/* Add the record for referencing OLD */
			rec = palloc0(sizeof(PLpgPSM_rec));
			rec->dtype = PLPGPSM_DTYPE_REC;
			rec->refname = pstrdup("old");
			rec->tup = NULL;
			rec->tupdesc = NULL;
			rec->freetup = false;
			plpgpsm_adddatum((PLpgPSM_datum *) rec);
			plpgpsm_ns_additem(PLPGPSM_NSTYPE_REC, rec->dno, rec->refname);
			function->old_varno = rec->dno;

			/* Add the variable tg_name */
			var = plpgpsm_build_variable("tg_name", 0,
										 plpgpsm_build_datatype(NAMEOID, -1),
										 true, NULL);
			function->tg_name_varno = var->dno;

			/* Add the variable tg_when */
			var = plpgpsm_build_variable("tg_when", 0,
										 plpgpsm_build_datatype(TEXTOID, -1),
										 true, NULL);
			function->tg_when_varno = var->dno;

			/* Add the variable tg_level */
			var = plpgpsm_build_variable("tg_level", 0,
										 plpgpsm_build_datatype(TEXTOID, -1),
										 true, NULL);
			function->tg_level_varno = var->dno;

			/* Add the variable tg_op */
			var = plpgpsm_build_variable("tg_op", 0,
										 plpgpsm_build_datatype(TEXTOID, -1),
										 true, NULL);
			function->tg_op_varno = var->dno;

			/* Add the variable tg_relid */
			var = plpgpsm_build_variable("tg_relid", 0,
										 plpgpsm_build_datatype(OIDOID, -1),
										 true, NULL);
			function->tg_relid_varno = var->dno;

			/* Add the variable tg_relname */
			var = plpgpsm_build_variable("tg_relname", 0,
										 plpgpsm_build_datatype(NAMEOID, -1),
										 true, NULL);
			function->tg_relname_varno = var->dno;

			/* tg_table_name is now preferred to tg_relname */
			var = plpgpsm_build_variable("tg_table_name", 0,
										 plpgpsm_build_datatype(NAMEOID, -1),
										 true, NULL);
			function->tg_table_name_varno = var->dno;


			/* add variable tg_table_schema */
			var = plpgpsm_build_variable("tg_table_schema", 0,
										 plpgpsm_build_datatype(NAMEOID, -1),
										 true, NULL);
			function->tg_table_schema_varno = var->dno;


			/* Add the variable tg_nargs */
			var = plpgpsm_build_variable("tg_nargs", 0,
										 plpgpsm_build_datatype(INT4OID, -1),
										 true, NULL);
			function->tg_nargs_varno = var->dno;

			break;

		default:
			elog(ERROR, "unrecognized function typecode: %u", functype);
			break;
	}

	/* Remember if function is STABLE/IMMUTABLE */
	function->fn_readonly = (procStruct->provolatile != PROVOLATILE_VOLATILE);

	/*
	 * Create the magic FOUND variable.
	 */
	var = plpgpsm_build_variable("found", 0,
								 plpgpsm_build_datatype(BOOLOID, -1),
								 true, NULL);
	function->found_varno = var->dno;

	function->sqlstate_varno = PLPGPSM_INVALID_VARNO;

	/*
	 * Forget about the above created variables
	 */
	plpgpsm_add_initdatums(NULL);

	/*
	 * Now parse the function's text
	 */
	parse_rc = plpgpsm_yyparse();
	
	
	//elog(ERROR, "zatim konec");
	if (parse_rc != 0)
		elog(ERROR, "plpgpsm parser returned %d", parse_rc);
	
	function->action = plpgpsm_yylval.program->body;
	function->sqlstate_varno = plpgpsm_yylval.program->sqlstate_varno;;

	plpgpsm_scanner_finish();
	pfree(proc_source);

	/*
	 * If it has OUT parameters or returns VOID or returns a set, we allow
	 * control to fall off the end without an explicit RETURN statement. 
	 * Implement new outer compound_statement with statement and RETURN statement.
	 */
	if (num_out_args > 0 || function->fn_rettype == VOIDOID ||
		function->fn_retset)
	{
		PLpgPSM_stmt_block *block;
		PLpgPSM_stmt_return *ret;
		
		block = palloc0(sizeof(PLpgPSM_stmt_block));
		block->cmd_type = PLPGPSM_STMT_BLOCK;
		/* implicit compund statement */
		block->generated = true;
		block->body = list_make1(function->action);
		
		block->sqlstate_varno = plpgpsm_yylval.program->sqlstate_varno;
		
		function->action = (PLpgPSM_stmt*) block;
		
		ret = palloc0(sizeof(PLpgPSM_stmt_return));
		ret->cmd_type = PLPGPSM_STMT_RETURN;
		ret->expr = NULL;
		ret->retvarno = function->out_param_varno;
		
		block->body = lappend(block->body, ret);
	}
	
	/*
	 * Complete the function's info
	 */
	function->fn_nargs = procStruct->pronargs;
	for (i = 0; i < function->fn_nargs; i++)
		function->fn_argvarnos[i] = in_arg_varnos[i];
	function->ndatums = plpgpsm_nDatums;
	function->datums = palloc(sizeof(PLpgPSM_datum *) * plpgpsm_nDatums);
	for (i = 0; i < plpgpsm_nDatums; i++)
		function->datums[i] = plpgpsm_Datums[i];

	/* Debug dump for completed functions */
	if (plpgpsm_DumpExecTree)
		plpgpsm_dumptree(function);

	/*
	 * add it to the hash table
	 */
	plpgpsm_HashTableInsert(function, hashkey);

	/*
	 * Pop the error context stack
	 */
	error_context_stack = plerrcontext.previous;
	plpgpsm_error_funcname = NULL;
	plpgpsm_error_lineno = 0;

	plpgpsm_check_syntax = false;

	MemoryContextSwitchTo(compile_tmp_cxt);
	compile_tmp_cxt = NULL;
	
	return function;
}


/*
 * error context callback to let us supply a call-stack traceback. If
 * we are validating, the function source is passed as an
 * argument. This function is public only for the sake of an assertion
 * in gram.y
 */
void
plpgpsm_compile_error_callback(void *arg)
{
	if (arg)
	{
		/*
		 * Try to convert syntax error position to reference text of original
		 * CREATE FUNCTION command.
		 */
		if (function_parse_error_transpose((const char *) arg))
			return;

		/*
		 * Done if a syntax error position was reported; otherwise we have to
		 * fall back to a "near line N" report.
		 */
	}

	if (plpgpsm_error_funcname)
		errcontext("compile of PL/pgSQL function \"%s\" near line %d",
				   plpgpsm_error_funcname, plpgpsm_error_lineno);
}


/* ----------
 * plpgpsm_parse_word		The scanner calls this to postparse
 *				any single word not found by a
 *				keyword rule.
 * ----------
 */
int
plpgpsm_parse_word(const char *word)
{
	PLpgPSM_nsitem *nse;
	char	   *cp[1];

	/* Do case conversion and word separation */
	plpgpsm_convert_ident(word, cp, 1);

	/*
	 * Recognize tg_argv when compiling triggers
	 */
	if (plpgpsm_curr_compile->fn_functype == T_TRIGGER)
	{
		if (strcmp(cp[0], "tg_argv") == 0)
		{
			bool		save_spacescanned = plpgpsm_SpaceScanned;
			PLpgPSM_trigarg *trigarg;

			trigarg = palloc0(sizeof(PLpgPSM_trigarg));
			trigarg->dtype = PLPGPSM_DTYPE_TRIGARG;

			if (plpgpsm_yylex() != '[')
				plpgpsm_yyerror("expected \"[\"");

			trigarg->argnum = plpgpsm_read_expression(']', "]");

			plpgpsm_adddatum((PLpgPSM_datum *) trigarg);
			plpgpsm_yylval.scalar = (PLpgPSM_datum *) trigarg;

			plpgpsm_SpaceScanned = save_spacescanned;
			pfree(cp[0]);
			return T_SCALAR;
		}
	}

	/*
	 * Do a lookup on the compiler's namestack
	 */
	nse = plpgpsm_ns_lookup(cp[0], NULL, NULL, NULL);
	pfree(cp[0]);
	if (nse != NULL)
	{
		switch (nse->itemtype)
		{
			case PLPGPSM_NSTYPE_VAR:
				plpgpsm_yylval.scalar = plpgpsm_Datums[nse->itemno];
				return T_SCALAR;

			case PLPGPSM_NSTYPE_REC:
				plpgpsm_yylval.rec = (PLpgPSM_rec *) (plpgpsm_Datums[nse->itemno]);
				return T_RECORD;

			case PLPGPSM_NSTYPE_ROW:
				plpgpsm_yylval.row = (PLpgPSM_row *) (plpgpsm_Datums[nse->itemno]);
				return T_ROW;
				
			case PLPGPSM_NSTYPE_CONDITION:
				plpgpsm_yylval.condvar = (PLpgPSM_cond_var *) (plpgpsm_Datums[nse->itemno]);
				return T_CONDITION;

			default:
				return T_ERROR;
		}
	}

	/*
	 * Nothing found - up to now it's a word without any special meaning for
	 * us.
	 */
	return T_WORD;
}


/* ----------
 * plpgpsm_parse_dblword		Same lookup for two words
 *					separated by a dot.
 * ----------
 */
int
plpgpsm_parse_dblword(const char *word)
{
	PLpgPSM_nsitem *ns;
	char	   *cp[2];
	int			nnames;

	/* Do case conversion and word separation */
	plpgpsm_convert_ident(word, cp, 2);

	/*
	 * Lookup the first word
	 */
	ns = plpgpsm_ns_lookup(cp[0], cp[1], NULL, &nnames);
	if (ns == NULL)
	{
		pfree(cp[0]);
		pfree(cp[1]);
		return T_ERROR;
	}

	switch (ns->itemtype)
	{
		case PLPGPSM_NSTYPE_VAR:
			/* Block-qualified reference to scalar variable. */
			plpgpsm_yylval.scalar = plpgpsm_Datums[ns->itemno];
			pfree(cp[0]);
			pfree(cp[1]);
			return T_SCALAR;

		case PLPGPSM_NSTYPE_REC:
			if (nnames == 1)
			{
				/*
				 * First word is a record name, so second word must be a field
				 * in this record.
				 */
				PLpgPSM_recfield *new;

				new = palloc(sizeof(PLpgPSM_recfield));
				new->dtype = PLPGPSM_DTYPE_RECFIELD;
				new->fieldname = pstrdup(cp[1]);
				new->recparentno = ns->itemno;

				plpgpsm_adddatum((PLpgPSM_datum *) new);

				plpgpsm_yylval.scalar = (PLpgPSM_datum *) new;

				pfree(cp[0]);
				pfree(cp[1]);
				return T_SCALAR;
			}
			else
			{
				/* Block-qualified reference to recod variable. */
				plpgpsm_yylval.rec = (PLpgPSM_rec *) plpgpsm_Datums[ns->itemno];
				pfree(cp[0]);
				pfree(cp[1]);
				return T_RECORD;    
			}

		case PLPGPSM_NSTYPE_ROW:
			if (nnames == 1)
			{
				/*
				 * First word is a row name, so second word must be a field in
				 * this row.
				 */
				PLpgPSM_row *row;
				int			i;

				row = (PLpgPSM_row *) (plpgpsm_Datums[ns->itemno]);
				for (i = 0; i < row->nfields; i++)
				{
					if (row->fieldnames[i] &&
						strcmp(row->fieldnames[i], cp[1]) == 0)
					{
						plpgpsm_yylval.scalar = plpgpsm_Datums[row->varnos[i]];
						pfree(cp[0]);
						pfree(cp[1]);
						return T_SCALAR;
					}
				}
				ereport(ERROR,
						(errcode(ERRCODE_UNDEFINED_COLUMN),
						 errmsg("row \"%s\" has no field \"%s\"",
								cp[0], cp[1])));
			}
			else
			{
				/* Block-qualified reference to row variable. */
				plpgpsm_yylval.row = (PLpgPSM_row *) (plpgpsm_Datums[ns->itemno]);
				pfree(cp[0]);
				pfree(cp[1]);	
				return T_ROW;
			}

		default:
			break;
	}

	pfree(cp[0]);
	pfree(cp[1]);
	return T_ERROR;
}


/* ----------
 * plpgpsm_parse_tripword		Same lookup for three words
 *					separated by dots.
 * ----------
 */
int
plpgpsm_parse_tripword(const char *word)
{
	PLpgPSM_nsitem *ns;
	char	   *cp[3];
	int			nnames;

	/* Do case conversion and word separation */
	plpgpsm_convert_ident(word, cp, 3);

	/*
	 * Do a lookup on the compiler's namestack.
	 * Must find a qualified reference.
	 */
	ns = plpgpsm_ns_lookup(cp[0], cp[1], cp[2], &nnames);
	if (ns == NULL || nnames != 2)
	{
		pfree(cp[0]);
		pfree(cp[1]);
		pfree(cp[2]);
		return T_ERROR;
	}

	switch (ns->itemtype)
	{
		case PLPGPSM_NSTYPE_REC:
			{
				/*
				 * words 1/2 are a record name, so third word must be a field
				 * in this record.
				 */
				PLpgPSM_recfield *new;

				new = palloc(sizeof(PLpgPSM_recfield));
				new->dtype = PLPGPSM_DTYPE_RECFIELD;
				new->fieldname = pstrdup(cp[2]);
				new->recparentno = ns->itemno;

				plpgpsm_adddatum((PLpgPSM_datum *) new);

				plpgpsm_yylval.scalar = (PLpgPSM_datum *) new;

				pfree(cp[0]);
				pfree(cp[1]);
				pfree(cp[2]);

				return T_SCALAR;
			}

		case PLPGPSM_NSTYPE_ROW:
			{
				/*
				 * words 1/2 are a row name, so third word must be a field in
				 * this row.
				 */
				PLpgPSM_row *row;
				int			i;

				row = (PLpgPSM_row *) (plpgpsm_Datums[ns->itemno]);
				for (i = 0; i < row->nfields; i++)
				{
					if (row->fieldnames[i] &&
						strcmp(row->fieldnames[i], cp[2]) == 0)
					{
						plpgpsm_yylval.scalar = plpgpsm_Datums[row->varnos[i]];

						pfree(cp[0]);
						pfree(cp[1]);
						pfree(cp[2]);

						return T_SCALAR;
					}
				}
				ereport(ERROR,
						(errcode(ERRCODE_UNDEFINED_COLUMN),
						 errmsg("row \"%s.%s\" has no field \"%s\"",
								cp[0], cp[1], cp[2])));
			}

		default:
			break;
	}

	pfree(cp[0]);
	pfree(cp[1]);
	pfree(cp[2]);
	return T_ERROR;
}

/* ----------
 * plpgpsm_parse_wordtype	The scanner found word%TYPE. word can be
 *				a variable name or a basetype.
 * ----------
 */
int
plpgpsm_parse_wordtype(char *word)
{
	PLpgPSM_nsitem *nse;
	bool		old_nsstate;
	HeapTuple	typeTup;
	char	   *cp[2];
	int			i;

	/* Do case conversion and word separation */
	/* We convert %type to .type momentarily to keep converter happy */
	i = strlen(word) - 5;
	Assert(word[i] == '%');
	word[i] = '.';
	plpgpsm_convert_ident(word, cp, 2);
	word[i] = '%';
	pfree(cp[1]);

	/*
	 * Do a lookup on the compiler's namestack.  Ensure we scan all levels.
	 */
	old_nsstate = plpgpsm_ns_setlocal(false);
	nse = plpgpsm_ns_lookup(cp[0], NULL, NULL, NULL);
	plpgpsm_ns_setlocal(old_nsstate);

	if (nse != NULL)
	{
		pfree(cp[0]);
		switch (nse->itemtype)
		{
			case PLPGPSM_NSTYPE_VAR:
				plpgpsm_yylval.dtype = ((PLpgPSM_var *) (plpgpsm_Datums[nse->itemno]))->datatype;
				return T_DTYPE;

				/* XXX perhaps allow REC here? */

			default:
				return T_ERROR;
		}
	}

	/*
	 * Word wasn't found on the namestack. Try to find a data type with that
	 * name, but ignore shell types and complex types.
	 */
	typeTup = LookupTypeName(NULL, makeTypeName(cp[0]), NULL);
	if (typeTup)
	{
		Form_pg_type typeStruct = (Form_pg_type) GETSTRUCT(typeTup);

		if (!typeStruct->typisdefined ||
			typeStruct->typrelid != InvalidOid)
		{
			ReleaseSysCache(typeTup);
			pfree(cp[0]);
			return T_ERROR;
		}

		plpgpsm_yylval.dtype = build_datatype(typeTup, -1);

		ReleaseSysCache(typeTup);
		pfree(cp[0]);
		return T_DTYPE;
	}

	/*
	 * Nothing found - up to now it's a word without any special meaning for
	 * us.
	 */
	pfree(cp[0]);
	return T_ERROR;
}


/* ----------
 * plpgpsm_parse_dblwordtype		Same lookup for word.word%TYPE
 * ----------
 */
int
plpgpsm_parse_dblwordtype(char *word)
{
	PLpgPSM_nsitem *nse;
	bool		old_nsstate;
	Oid			classOid;
	HeapTuple	classtup = NULL;
	HeapTuple	attrtup = NULL;
	HeapTuple	typetup = NULL;
	Form_pg_class classStruct;
	Form_pg_attribute attrStruct;
	char	   *cp[3];
	int			i;
	MemoryContext oldCxt;
	int			result = T_ERROR;

	/* Avoid memory leaks in the long-term function context */
	oldCxt = MemoryContextSwitchTo(compile_tmp_cxt);

	/* Do case conversion and word separation */
	/* We convert %type to .type momentarily to keep converter happy */
	i = strlen(word) - 5;
	Assert(word[i] == '%');
	word[i] = '.';
	plpgpsm_convert_ident(word, cp, 3);
	word[i] = '%';
	pfree(cp[2]);

	/*
	 * Do a lookup on the compiler's namestack.  Ensure we scan all levels.
	 * We don't need to check number of names matched, because we will only
	 * consider scalar variables.
	 */
	old_nsstate = plpgpsm_ns_setlocal(false);
	nse = plpgpsm_ns_lookup(cp[0], cp[1], NULL, NULL);
	plpgpsm_ns_setlocal(old_nsstate);

	if (nse != NULL && nse->itemtype == PLPGPSM_NSTYPE_VAR)
	{
		plpgpsm_yylval.dtype = ((PLpgPSM_var *) (plpgpsm_Datums[nse->itemno]))->datatype;
		result = T_DTYPE;
		goto done;
	}

	/*
	 * First word could also be a table name
	 */
	classOid = RelnameGetRelid(cp[0]);
	if (!OidIsValid(classOid))
		goto done;

	classtup = SearchSysCache(RELOID,
							  ObjectIdGetDatum(classOid),
							  0, 0, 0);
	if (!HeapTupleIsValid(classtup))
		goto done;
	classStruct = (Form_pg_class) GETSTRUCT(classtup);

	/*
	 * It must be a relation, sequence, view, or type
	 */
	if (classStruct->relkind != RELKIND_RELATION &&
		classStruct->relkind != RELKIND_SEQUENCE &&
		classStruct->relkind != RELKIND_VIEW &&
		classStruct->relkind != RELKIND_COMPOSITE_TYPE)
		goto done;

	/*
	 * Fetch the named table field and its type
	 */
	attrtup = SearchSysCacheAttName(classOid, cp[1]);
	if (!HeapTupleIsValid(attrtup))
		goto done;
	attrStruct = (Form_pg_attribute) GETSTRUCT(attrtup);

	typetup = SearchSysCache(TYPEOID,
							 ObjectIdGetDatum(attrStruct->atttypid),
							 0, 0, 0);
	if (!HeapTupleIsValid(typetup))
		elog(ERROR, "cache lookup failed for type %u", attrStruct->atttypid);

	/*
	 * Found that - build a compiler type struct in the caller's cxt and
	 * return it
	 */
	MemoryContextSwitchTo(oldCxt);
	plpgpsm_yylval.dtype = build_datatype(typetup, attrStruct->atttypmod);
	MemoryContextSwitchTo(compile_tmp_cxt);
	result = T_DTYPE;

done:
	if (HeapTupleIsValid(classtup))
		ReleaseSysCache(classtup);
	if (HeapTupleIsValid(attrtup))
		ReleaseSysCache(attrtup);
	if (HeapTupleIsValid(typetup))
		ReleaseSysCache(typetup);

	MemoryContextSwitchTo(oldCxt);
	return result;
}

/* ----------
 * plpgpsm_parse_tripwordtype		Same lookup for word.word.word%TYPE
 * ----------
 */
int
plpgpsm_parse_tripwordtype(char *word)
{
	Oid			classOid;
	HeapTuple	classtup = NULL;
	HeapTuple	attrtup = NULL;
	HeapTuple	typetup = NULL;
	Form_pg_class classStruct;
	Form_pg_attribute attrStruct;
	char	   *cp[4];
	int			i;
	RangeVar   *relvar;
	MemoryContext oldCxt;
	int			result = T_ERROR;

	/* Avoid memory leaks in the long-term function context */
	oldCxt = MemoryContextSwitchTo(compile_tmp_cxt);

	/* Do case conversion and word separation */
	/* We convert %type to .type momentarily to keep converter happy */
	i = strlen(word) - 5;
	Assert(word[i] == '%');
	word[i] = '.';
	plpgpsm_convert_ident(word, cp, 4);
	word[i] = '%';
	pfree(cp[3]);

	relvar = makeRangeVar(cp[0], cp[1], -1);
	classOid = RangeVarGetRelid(relvar, true);
	if (!OidIsValid(classOid))
		goto done;

	classtup = SearchSysCache(RELOID,
							  ObjectIdGetDatum(classOid),
							  0, 0, 0);
	if (!HeapTupleIsValid(classtup))
		goto done;
	classStruct = (Form_pg_class) GETSTRUCT(classtup);

	/*
	 * It must be a relation, sequence, view, or type
	 */
	if (classStruct->relkind != RELKIND_RELATION &&
		classStruct->relkind != RELKIND_SEQUENCE &&
		classStruct->relkind != RELKIND_VIEW &&
		classStruct->relkind != RELKIND_COMPOSITE_TYPE)
		goto done;

	/*
	 * Fetch the named table field and its type
	 */
	attrtup = SearchSysCacheAttName(classOid, cp[2]);
	if (!HeapTupleIsValid(attrtup))
		goto done;
	attrStruct = (Form_pg_attribute) GETSTRUCT(attrtup);

	typetup = SearchSysCache(TYPEOID,
							 ObjectIdGetDatum(attrStruct->atttypid),
							 0, 0, 0);
	if (!HeapTupleIsValid(typetup))
		elog(ERROR, "cache lookup failed for type %u", attrStruct->atttypid);

	/*
	 * Found that - build a compiler type struct in the caller's cxt and
	 * return it
	 */
	MemoryContextSwitchTo(oldCxt);
	plpgpsm_yylval.dtype = build_datatype(typetup, attrStruct->atttypmod);
	MemoryContextSwitchTo(compile_tmp_cxt);
	result = T_DTYPE;

done:
	if (HeapTupleIsValid(classtup))
		ReleaseSysCache(classtup);
	if (HeapTupleIsValid(attrtup))
		ReleaseSysCache(attrtup);
	if (HeapTupleIsValid(typetup))
		ReleaseSysCache(typetup);

	MemoryContextSwitchTo(oldCxt);
	return result;
}

/* ----------
 * plpgpsm_parse_wordrowtype		Scanner found word%ROWTYPE.
 *					So word must be a table name.
 * ----------
 */
int
plpgpsm_parse_wordrowtype(char *word)
{
	Oid			classOid;
	char	   *cp[2];
	int			i;

	/* Do case conversion and word separation */
	/* We convert %rowtype to .rowtype momentarily to keep converter happy */
	i = strlen(word) - 8;
	Assert(word[i] == '%');
	word[i] = '.';
	plpgpsm_convert_ident(word, cp, 2);
	word[i] = '%';

	/* Lookup the relation */
	classOid = RelnameGetRelid(cp[0]);
	if (!OidIsValid(classOid))
		ereport(ERROR,
				(errcode(ERRCODE_UNDEFINED_TABLE),
				 errmsg("relation \"%s\" does not exist", cp[0])));

	/*
	 * Build and return the row type struct
	 */
	plpgpsm_yylval.dtype = plpgpsm_build_datatype(get_rel_type_id(classOid),
												  -1);

	pfree(cp[0]);
	pfree(cp[1]);

	return T_DTYPE;
}

/* ----------
 * plpgpsm_parse_dblwordrowtype		Scanner found word.word%ROWTYPE.
 *			So word must be a namespace qualified table name.
 * ----------
 */
int
plpgpsm_parse_dblwordrowtype(char *word)
{
	Oid			classOid;
	char	   *cp[3];
	int			i;
	RangeVar   *relvar;
	MemoryContext oldCxt;

	/* Avoid memory leaks in long-term function context */
	oldCxt = MemoryContextSwitchTo(compile_tmp_cxt);

	/* Do case conversion and word separation */
	/* We convert %rowtype to .rowtype momentarily to keep converter happy */
	i = strlen(word) - 8;
	Assert(word[i] == '%');
	word[i] = '.';
	plpgpsm_convert_ident(word, cp, 3);
	word[i] = '%';

	/* Lookup the relation */
	relvar = makeRangeVar(cp[0], cp[1], -1);
	classOid = RangeVarGetRelid(relvar, true);
	if (!OidIsValid(classOid))
		ereport(ERROR,
				(errcode(ERRCODE_UNDEFINED_TABLE),
				 errmsg("relation \"%s.%s\" does not exist", cp[0], cp[1])));

	/* Build and return the row type struct */
	plpgpsm_yylval.dtype = plpgpsm_build_datatype(get_rel_type_id(classOid),
												  -1);

	MemoryContextSwitchTo(oldCxt);
	return T_DTYPE;
}


/*
 * plpgpsm_build_variable - build a datum-array entry of a given
 * datatype
 *
 * The returned struct may be a PLpgPSM_var, PLpgPSM_row, or
 * PLpgPSM_rec depending on the given datatype, and is allocated via
 * palloc.	The struct is automatically added to the current datum
 * array, and optionally to the current namespace.
 */
PLpgPSM_variable *
plpgpsm_build_variable(const char *refname, int lineno, PLpgPSM_type *dtype,
					   bool add2namespace, List ** initvars)
{
	PLpgPSM_variable *result;

	switch (dtype->ttype)
	{
		case PLPGPSM_TTYPE_SCALAR:
			{
				/* Ordinary scalar datatype */
				PLpgPSM_var *var;

				var = palloc0(sizeof(PLpgPSM_var));
				var->dtype = PLPGPSM_DTYPE_VAR;
				var->refname = pstrdup(refname);
				var->lineno = lineno;
				var->datatype = dtype;
				/* other fields might be filled by caller */

				/* preset to NULL */
				var->value = 0;
				var->isnull = true;
				var->freeval = false;

				plpgpsm_adddatum((PLpgPSM_datum *) var);
				if (add2namespace)
					plpgpsm_ns_additem(PLPGPSM_NSTYPE_VAR,
									   var->dno,
									   refname);
				result = (PLpgPSM_variable *) var;
				if (initvars)
					*initvars = lappend(*initvars, (void *) var->dno);

				break;
			}
		case PLPGPSM_TTYPE_ROW:
			{
				/* Composite type -- build a row variable */
				PLpgPSM_row *row;

				row = build_row_from_class(dtype->typrelid, initvars);

				row->dtype = PLPGPSM_DTYPE_ROW;
				row->refname = pstrdup(refname);
				row->lineno = lineno;

				plpgpsm_adddatum((PLpgPSM_datum *) row);
				if (add2namespace)
					plpgpsm_ns_additem(PLPGPSM_NSTYPE_ROW,
									   row->dno,
									   refname);
				result = (PLpgPSM_variable *) row;
				break;
			}
		case PLPGPSM_TTYPE_REC:
			{
				/*
				 * "record" type -- build a variable-contents record variable
				 */
				PLpgPSM_rec *rec;

				rec = palloc0(sizeof(PLpgPSM_rec));
				rec->dtype = PLPGPSM_DTYPE_REC;
				rec->refname = pstrdup(refname);
				rec->lineno = lineno;

				plpgpsm_adddatum((PLpgPSM_datum *) rec);
				if (add2namespace)
					plpgpsm_ns_additem(PLPGPSM_NSTYPE_REC,
									   rec->dno,
									   refname);
				result = (PLpgPSM_variable *) rec;
				break;
			}
		case PLPGPSM_TTYPE_CONDITION:
			{
				PLpgPSM_cond_var  *var;
				
				var = palloc0(sizeof(PLpgPSM_cond_var));
				var->dtype = PLPGPSM_DTYPE_CONDITION;
				var->lineno = lineno;
				
				plpgpsm_adddatum((PLpgPSM_datum *) var);
				if (add2namespace)
					plpgpsm_ns_additem(PLPGPSM_NSTYPE_CONDITION, 
										    var->cno, refname);
				result = (PLpgPSM_variable *)var;
				break;
			}
		case PLPGPSM_TTYPE_PSEUDO:
			ereport(ERROR,
					(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
					 errmsg("variable \"%s\" has pseudo-type %s",
							refname, format_type_be(dtype->typoid))));
			result = NULL;		/* keep compiler quiet */
			break;
		default:
			elog(ERROR, "unrecognized ttype: %d", dtype->ttype);
			result = NULL;		/* keep compiler quiet */
			break;
	}

	return result;
}

/*
 * Build a row-variable data structure given the pg_class OID.
 */
static PLpgPSM_row *
build_row_from_class(Oid classOid, List **initvars)
{
	PLpgPSM_row *row;
	Relation	rel;
	Form_pg_class classStruct;
	const char *relname;
	int			i;

	/*
	 * Open the relation to get info.
	 */
	rel = relation_open(classOid, AccessShareLock);
	classStruct = RelationGetForm(rel);
	relname = RelationGetRelationName(rel);

	/* accept relation, sequence, view, or composite type entries */
	if (classStruct->relkind != RELKIND_RELATION &&
		classStruct->relkind != RELKIND_SEQUENCE &&
		classStruct->relkind != RELKIND_VIEW &&
		classStruct->relkind != RELKIND_COMPOSITE_TYPE)
		ereport(ERROR,
				(errcode(ERRCODE_WRONG_OBJECT_TYPE),
				 errmsg("relation \"%s\" is not a table", relname)));

	/*
	 * Create a row datum entry and all the required variables that it will
	 * point to.
	 */
	row = palloc0(sizeof(PLpgPSM_row));
	row->dtype = PLPGPSM_DTYPE_ROW;
	row->rowtupdesc = CreateTupleDescCopy(RelationGetDescr(rel));
	row->nfields = classStruct->relnatts;
	row->fieldnames = palloc(sizeof(char *) * row->nfields);
	row->varnos = palloc(sizeof(int) * row->nfields);
	row->isnull = true;

	for (i = 0; i < row->nfields; i++)
	{
		Form_pg_attribute attrStruct;

		/*
		 * Get the attribute and check for dropped column
		 */
		attrStruct = row->rowtupdesc->attrs[i];

		if (!attrStruct->attisdropped)
		{
			char	   *attname;
			char		refname[(NAMEDATALEN * 2) + 100];
			PLpgPSM_variable *var;

			attname = NameStr(attrStruct->attname);
			snprintf(refname, sizeof(refname), "%s.%s", relname, attname);

			/*
			 * Create the internal variable for the field
			 *
			 * We know if the table definitions contain a default value or if
			 * the field is declared in the table as NOT NULL. But it's
			 * possible to create a table field as NOT NULL without a default
			 * value and that would lead to problems later when initializing
			 * the variables due to entering a block at execution time. Thus
			 * we ignore this information for now.
			 */
			var = plpgpsm_build_variable(refname, 0,
								 plpgpsm_build_datatype(attrStruct->atttypid,
													  attrStruct->atttypmod),
										 false, NULL);

			if (var->dtype == PLPGPSM_DTYPE_VAR)
				((PLpgPSM_var *) var)->parent = row;

			/* Add the variable to the row */
			row->fieldnames[i] = attname;
			row->varnos[i] = var->dno;
			if (initvars)
				*initvars = lappend(*initvars, (void *) var->dno);
		}
		else
		{
			/* Leave a hole in the row structure for the dropped col */
			row->fieldnames[i] = NULL;
			row->varnos[i] = -1;
		}
	}

	relation_close(rel, AccessShareLock);

	return row;
}

/*
 * Build a row-variable data structure given the component variables.
 */
static PLpgPSM_row *
build_row_from_vars(PLpgPSM_variable **vars, int numvars)
{
	PLpgPSM_row *row;
	int			i;

	row = palloc0(sizeof(PLpgPSM_row));
	row->dtype = PLPGPSM_DTYPE_ROW;
	row->rowtupdesc = CreateTemplateTupleDesc(numvars, false);
	row->nfields = numvars;
	row->fieldnames = palloc(numvars * sizeof(char *));
	row->varnos = palloc(numvars * sizeof(int));

	for (i = 0; i < numvars; i++)
	{
		PLpgPSM_variable *var = vars[i];
		Oid			typoid = RECORDOID;
		int32		typmod = -1;

		switch (var->dtype)
		{
			case PLPGPSM_DTYPE_VAR:
				typoid = ((PLpgPSM_var *) var)->datatype->typoid;
				typmod = ((PLpgPSM_var *) var)->datatype->atttypmod;
				break;

			case PLPGPSM_DTYPE_REC:
				break;

			case PLPGPSM_DTYPE_ROW:
				if (((PLpgPSM_row *) var)->rowtupdesc)
				{
					typoid = ((PLpgPSM_row *) var)->rowtupdesc->tdtypeid;
					typmod = ((PLpgPSM_row *) var)->rowtupdesc->tdtypmod;
				}
				break;

			default:
				elog(ERROR, "unrecognized dtype: %d", var->dtype);
		}

		row->fieldnames[i] = var->refname;
		row->varnos[i] = var->dno;

		TupleDescInitEntry(row->rowtupdesc, i + 1,
						   var->refname,
						   typoid, typmod,
						   0);
	}

	return row;
}


/* ----------
 * plpgpsm_parse_datatype			Scanner found something that should
 *					be a datatype name.
 * ----------
 */
PLpgPSM_type *
plpgpsm_parse_datatype(const char *string)
{
	Oid			type_id;
	int32		typmod;

	/* Let the main parser try to parse it under standard SQL rules */
	parseTypeString(string, &type_id, &typmod);

	/* Okay, build a PLpgPSM_type data structure for it */
	return plpgpsm_build_datatype(type_id, typmod);
}

/*
 * plpgpsm_build_datatype
 *		Build PLpgPSM_type struct given type OID and typmod.
 */
PLpgPSM_type *
plpgpsm_build_datatype(Oid typeOid, int32 typmod)
{
	HeapTuple	typeTup;
	PLpgPSM_type *typ;

	typeTup = SearchSysCache(TYPEOID,
							 ObjectIdGetDatum(typeOid),
							 0, 0, 0);
	if (!HeapTupleIsValid(typeTup))
		elog(ERROR, "cache lookup failed for type %u", typeOid);

	typ = build_datatype(typeTup, typmod);

	ReleaseSysCache(typeTup);

	return typ;
}

/*
 * Utility subroutine to make a PLpgPSM_type struct given a pg_type entry
 */
static PLpgPSM_type *
build_datatype(HeapTuple typeTup, int32 typmod)
{
	Form_pg_type typeStruct = (Form_pg_type) GETSTRUCT(typeTup);
	PLpgPSM_type *typ;

	if (!typeStruct->typisdefined)
		ereport(ERROR,
				(errcode(ERRCODE_UNDEFINED_OBJECT),
				 errmsg("type \"%s\" is only a shell",
						NameStr(typeStruct->typname))));

	typ = (PLpgPSM_type *) palloc(sizeof(PLpgPSM_type));

	typ->typname = pstrdup(NameStr(typeStruct->typname));
	typ->typoid = HeapTupleGetOid(typeTup);
	switch (typeStruct->typtype)
	{
		case TYPTYPE_BASE:			/* base type */
		case TYPTYPE_DOMAIN:			/* domain */
		case TYPTYPE_ENUM:			/* enum */
			typ->ttype = PLPGPSM_TTYPE_SCALAR;
			break;
		case TYPTYPE_COMPOSITE:			/* composite, ie, rowtype */
			Assert(OidIsValid(typeStruct->typrelid));
			typ->ttype = PLPGPSM_TTYPE_ROW;
			break;
		case TYPTYPE_PSEUDO:			/* pseudo */
			if (typ->typoid == RECORDOID)
				typ->ttype = PLPGPSM_TTYPE_REC;
			else
				typ->ttype = PLPGPSM_TTYPE_PSEUDO;
			break;
		default:
			elog(ERROR, "unrecognized typtype: %d",
				 (int) typeStruct->typtype);
			break;
	}
	typ->typlen = typeStruct->typlen;
	typ->typbyval = typeStruct->typbyval;
	typ->typrelid = typeStruct->typrelid;
	typ->typioparam = getTypeIOParam(typeTup);
	fmgr_info(typeStruct->typinput, &(typ->typinput));
	typ->atttypmod = typmod;

	return typ;
}

/*
 * plpgpsm_parse_err_condition
 *		Generate PLpgPSM_condition entry(s) for an exception condition name
 *
 * This has to be able to return a list because there are some duplicate
 * names in the table of error code names.
 */
PLpgPSM_condition *
plpgpsm_parse_err_condition(char *condname)
{
	int			i;
	PLpgPSM_condition *new;
	PLpgPSM_condition *prev;
//	PLpgPSM_nsitem	*nse;
//	char *cp[1];
	

	/*
	 * XXX Eventually we will want to look for user-defined exception names
	 * here.
	 */

	/*
	 * OTHERS is represented as code 0 (which would map to '00000', but we
	 * have no need to represent that as an exception condition).
	 */
	if (strcmp(condname, "others") == 0)
	{
		new = palloc(sizeof(PLpgPSM_condition));
		new->sqlerrstate = 0;
		new->condname = condname;
		new->next = NULL;
		return new;
	}

	prev = NULL;
	for (i = 0; exception_label_map[i].label != NULL; i++)
	{
		if (strcmp(condname, exception_label_map[i].label) == 0)
		{
			new = palloc(sizeof(PLpgPSM_condition));
			new->sqlerrstate = exception_label_map[i].sqlerrstate;
			new->condname = condname;
			new->next = prev;
			prev = new;
		}
	}

	if (!prev)
		ereport(ERROR,
				(errcode(ERRCODE_UNDEFINED_OBJECT),
				 errmsg("unrecognized exception condition \"%s\"",
						condname)));

	return prev;
}

/* ----------
 * plpgpsm_adddatum			Add a variable, record or row
 *					to the compiler's datum list.
 * ----------
 */
void
plpgpsm_adddatum(PLpgPSM_datum *new)
{
	if (plpgpsm_nDatums == datums_alloc)
	{
		datums_alloc *= 2;
		plpgpsm_Datums = repalloc(plpgpsm_Datums, sizeof(PLpgPSM_datum *) * datums_alloc);
	}

	new->dno = plpgpsm_nDatums;
	plpgpsm_Datums[plpgpsm_nDatums++] = new;
}


/* ----------
 * plpgpsm_add_initdatums		Put all datum entries created
 *					since the last call into the
 *					finishing code block so the
 *					block knows which variables to
 *					reinitialize when entered.
 * ----------
 */
int
plpgpsm_add_initdatums(int **varnos)
{
	int			i;
	int			n = 0;
	
	for (i = datums_last; i < plpgpsm_nDatums; i++)
	{
		switch (plpgpsm_Datums[i]->dtype)
		{
			case PLPGPSM_DTYPE_VAR:
				n++;
				break;

			default:
				break;
		}
	}

	if (varnos != NULL)
	{
		if (n > 0)
		{
			*varnos = (int *) palloc(sizeof(int) * n);

			n = 0;
			for (i = datums_last; i < plpgpsm_nDatums; i++)
			{
				switch (plpgpsm_Datums[i]->dtype)
				{
					case PLPGPSM_DTYPE_VAR:
						(*varnos)[n++] = plpgpsm_Datums[i]->dno;

					default:
						break;
				}
			}
		}
		else
			*varnos = NULL;
	}

	datums_last = plpgpsm_nDatums;
	return n;
}


/*
 * Compute the hashkey for a given function invocation
 *
 * The hashkey is returned into the caller-provided storage at *hashkey.
 */
static void
compute_function_hashkey(FunctionCallInfo fcinfo,
						 Form_pg_proc procStruct,
						 PLpgPSM_func_hashkey *hashkey,
						 bool forValidator)
{
	/* Make sure any unused bytes of the struct are zero */
	MemSet(hashkey, 0, sizeof(PLpgPSM_func_hashkey));

	/* get function OID */
	hashkey->funcOid = fcinfo->flinfo->fn_oid;

	/*
	 * if trigger, get relation OID.  In validation mode we do not know what
	 * relation is intended to be used, so we leave trigrelOid zero; the hash
	 * entry built in this case will never really be used.
	 */
	if (CALLED_AS_TRIGGER(fcinfo) && !forValidator)
	{
		TriggerData *trigdata = (TriggerData *) fcinfo->context;

		hashkey->trigrelOid = RelationGetRelid(trigdata->tg_relation);
	}

	if (procStruct->pronargs > 0)
	{
		/* get the argument types */
		memcpy(hashkey->argtypes, procStruct->proargtypes.values,
			   procStruct->pronargs * sizeof(Oid));

		/* resolve any polymorphic argument types */
		plpgpsm_resolve_polymorphic_argtypes(procStruct->pronargs,
											 hashkey->argtypes,
											 NULL,
											 fcinfo->flinfo->fn_expr,
											 forValidator,
											 NameStr(procStruct->proname));
	}
}

/*
 * This is the same as the standard resolve_polymorphic_argtypes() function,
 * but with a special case for validation: assume that polymorphic arguments
 * are integer or integer-array.  Also, we go ahead and report the error
 * if we can't resolve the types.
 */
static void
plpgpsm_resolve_polymorphic_argtypes(int numargs,
									 Oid *argtypes, char *argmodes,
									 Node *call_expr, bool forValidator,
									 const char *proname)
{
	int			i;

	if (!forValidator)
	{
		/* normal case, pass to standard routine */
		if (!resolve_polymorphic_argtypes(numargs, argtypes, argmodes,
										  call_expr))
			ereport(ERROR,
					(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
					 errmsg("could not determine actual argument "
							"type for polymorphic function \"%s\"",
							proname)));
	}
	else
	{
		/* special validation case */
		for (i = 0; i < numargs; i++)
		{
			switch (argtypes[i])
			{
				case ANYELEMENTOID:
				case ANYENUMOID:		 /* XXX dubious */
					argtypes[i] = INT4OID;
					break;
				case ANYARRAYOID:
					argtypes[i] = INT4ARRAYOID;
					break;
				default:
					break;
			}
		}
	}
}

/*                                                                                                   
 * delete_function - clean up as much as possible of a stale function cache                          
 *                                                                                                   
 * We can't release the PLpgPSM_function struct itself, because of the                               
 * possibility that there are fn_extra pointers to it.  We can release                               
 * the subsidiary storage, but only if there are no active evaluations                               
 * in progress.  Otherwise we'll just leak that storage.  Since the                                  
 * case would only occur if a pg_proc update is detected during a nested                             
 * recursive call on the function, a leak seems acceptable.                                          
 *                                                                                                   
 * Note that this can be called more than once if there are multiple fn_extra                        
 * pointers to the same function cache.  Hence be careful not to do things                           
 * twice.                                                                                            
 */                                                                                                  
static void
delete_function(PLpgPSM_function *func)
{
        /* remove function from hash table (might be done already) */
        plpgpsm_HashTableDelete(func);

        /* release the function's storage if safe and not done already */
        if (func->use_count == 0 && func->fn_cxt)
        {
                MemoryContextDelete(func->fn_cxt);
                func->fn_cxt = NULL;
        }
}

/* exported so we can call it from plpgpsm_init() */
void
plpgpsm_HashTableInit(void)
{
	HASHCTL		ctl;

	/* don't allow double-initialization */
	Assert(plpgpsm_HashTable == NULL);

	memset(&ctl, 0, sizeof(ctl));
	ctl.keysize = sizeof(PLpgPSM_func_hashkey);
	ctl.entrysize = sizeof(plpgpsm_HashEnt);
	ctl.hash = tag_hash;
	plpgpsm_HashTable = hash_create("PLpgPSM function cache",
									FUNCS_PER_USER,
									&ctl,
									HASH_ELEM | HASH_FUNCTION);
}

static PLpgPSM_function *
plpgpsm_HashTableLookup(PLpgPSM_func_hashkey *func_key)
{
	plpgpsm_HashEnt *hentry;

	hentry = (plpgpsm_HashEnt *) hash_search(plpgpsm_HashTable,
											 (void *) func_key,
											 HASH_FIND,
											 NULL);
	if (hentry)
		return hentry->function;
	else
		return NULL;
}

static void
plpgpsm_HashTableInsert(PLpgPSM_function *function,
						PLpgPSM_func_hashkey *func_key)
{
	plpgpsm_HashEnt *hentry;
	bool		found;

	hentry = (plpgpsm_HashEnt *) hash_search(plpgpsm_HashTable,
											 (void *) func_key,
											 HASH_ENTER,
											 &found);
	if (found)
		elog(WARNING, "trying to insert a function that already exists");

	hentry->function = function;
	/* prepare back link from function to hashtable key */
	function->fn_hashkey = &hentry->key;
}

static void
plpgpsm_HashTableDelete(PLpgPSM_function *function)
{
	plpgpsm_HashEnt *hentry;

	/* do nothing if not in table */
	if (function->fn_hashkey == NULL)
		return;

	hentry = (plpgpsm_HashEnt *) hash_search(plpgpsm_HashTable,
											 (void *) function->fn_hashkey,
											 HASH_REMOVE,
											 NULL);
	if (hentry == NULL)
		elog(WARNING, "trying to delete function that does not exist");

	/* remove back link, which no longer points to allocated storage */
	function->fn_hashkey = NULL;
}
