/*
* MyGoGrinder - a program to practice Go problems
* Copyright (c) 2004-2006 Tim Kington
*   timkington@users.sourceforge.net
* Portions Copyright (C) Ruediger Klehn (2014)
*   RuediRf@users.sourceforge.net
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
*
*/

package GoGrinder;

import java.io.*;
import java.util.*;
import javax.swing.JOptionPane;
import javax.swing.tree.*;

import GoGrinder.ui.*;

/**
 *
 * @author  tkington
 * @author  Ruediger Klehn
 */
public class ProbCollection extends DefaultMutableTreeNode {
//  public ProbCollection() {/*EMPTY*/}
//  public ProbCollection(ProbCollection parent, File dir, StatusListener status) {
//  public void getSelectionStats(SelectionStats s, ArrayList tags, int matchType) {
//  public void getProblems(ArrayList problems, ArrayList tags, int matchType) {
//  public void probDone(boolean right, long time) {
//  public void startNewSet() { // handle highscores
//  public ProbCollection findCollectionByDir(File d) {
//  public ProbData findProblem(ProbData p) {
//  private String getDatName() { ### gives path of collection.dat with path
//  private void loadStats() {
//  private void saveStats() {
//  public void deleteStats() {
//  public boolean equals(Object o) {
//  private void removeStats(int tr, int ri, long time) {
//  private void resetStats(HashSet colls, ProgressDialog d) {
//  public void resetStats(ProgressDialog d) {
//  public void addTag(String t, ProgressDialog d) {
//  public void removeTags(ArrayList tags, ProgressDialog d) {

    private static final long serialVersionUID = -4391979107075368395L;
    private static final int NUM_HIGH_SCORES = 10;
    
    private int size;
    private ArrayList probs;
    private File dir;
    private ProbCollection parent;
    
    private int numTried;
    private int numRight;
    private long totalTime;
    
    private int cursetNumTried;
    private int cursetNumRight;
    private long cursetTotalTime;
    private TreeSet highScores;
    
    /** Creates a new instance of ProbCollection */
    public ProbCollection() {
        //  This is just here for Serialization
    }
    
    public ProbCollection(ProbCollection parent, File dir, StatusListener status) {
        super(dir.getName()); // getName: last part of the path
        this.dir = dir;
        this.parent = parent;
        highScores = new TreeSet();
        
        String dirString = dir.getPath();
        int dirStrLength = dirString.length();
        String displayDirString = "";
        if (dirStrLength > 55){ // ? change here to My problems/... like in status bar of ProbFrame?
          displayDirString = dirString.substring(0, 7) + " ... " + dirString.substring(dirStrLength - 40, dirStrLength);
        } // so this becomes something like "c:/User ... rinderSettings/problems/beginners"
        else displayDirString = dirString;
        status.setStatus(Messages.getString("loading") + " " + displayDirString); //$NON-NLS-1$ //$NON-NLS-2$

        loadStats();
        
        probs = new ArrayList();
        
        File [] files = dir.listFiles();
        if (GS.getSelectionOrderedDirs())
          Arrays.sort(files); // file lists were not sorted, when got from some file system (FAT32, ext4 on a Linux system):
          // "There is no guarantee that the name strings in the resulting array will appear in any
          //   specific order; they are not, in particular, guaranteed to appear in alphabetical order. "
        for(int i = 0; i < files.length; i++) {
            if(files[i].isDirectory()) {
                ProbCollection child = new ProbCollection(this, files[i], status);
                add(child);
                size += child.getSize();
            }
            else if(files[i].getName().toLowerCase().endsWith(".sgf")) { //$NON-NLS-1$
                probs.add(new ProbData(this, files[i]));
                size++;
            }
        }
        
        Collections.sort(probs);
        for(int i = 0; i < probs.size(); i++) {
            ProbData p = (ProbData)probs.get(i);
            p.setNum(i);
        }
    }
    
    public void getSelectionStats(SelectionStats s, ArrayList tags, int matchType) {
        for(int i = 0; i < probs.size(); i++) {
            ProbData p = (ProbData)probs.get(i);
            if(p.matchesTags(tags, matchType))
                p.getSelectionStats(s);
        }
        
        for(int i = 0; i < getChildCount(); i++) {
            ProbCollection c = (ProbCollection)getChildAt(i);
            c.getSelectionStats(s, tags, matchType);
        }
    }
    
    public void getProblems(ArrayList problems, ArrayList tags, int matchType) { // here with request "isExport" or so we could change this to relative path
// /*DEBUG*/    JOptionPane.showMessageDialog(null, problems.size() + " <- from there - should be also here 0 - to there -> " + probs.size());
        for(int i = 0; i < probs.size(); i++) {
            ProbData p = (ProbData)probs.get(i);
            if(p.matchesTags(tags, matchType))
                problems.add(p);
// /*DEBUG*/    JOptionPane.showMessageDialog(null, problems.get(i) + " <- from there - should be also here 0 - to there ->" + probs.get(i));
        }
        
        for(int i = 0; i < getChildCount(); i++) {
            ProbCollection c = (ProbCollection)getChildAt(i);
            c.getProblems(problems, tags, matchType);
        }
    }
    
    public void probDone(boolean right, long time) {
        numTried++;
        if(right) {
            numRight++;
            totalTime += time;
        }
        
        cursetNumTried++;
        if(right) {
            cursetNumRight++;
            cursetTotalTime += time;
        }
        
        saveStats();
        if(parent != null)
            parent.probDone(right, time);
    }
    
    public void startNewSet() {
//    	 Main.logSilent("Set=\"" + this + "\", tried=" + cursetNumTried  //$NON-NLS-1$ //$NON-NLS-2$
//    			+ ", size=" + size); //$NON-NLS-1$                // just some logging for debugging? (could be switched on for LogAll)
        if(cursetNumTried == size && cursetNumTried != 0) {
            HighScore h = new HighScore(cursetNumRight, cursetNumTried, cursetTotalTime);
            HighScore worst = null;
            if(!highScores.isEmpty())
                worst = (HighScore)highScores.last();
            if(highScores.size() < NUM_HIGH_SCORES || h.compareTo(worst) < 0) {
                highScores.add(h);
                if(highScores.size() > NUM_HIGH_SCORES)
                    highScores.remove(worst);
                new HighScoreDialog(ProbFrame.inst, (String)getUserObject(), highScores, h);
            }
        }
        
        cursetNumTried = 0;
        cursetNumRight = 0;
        cursetTotalTime = 0;
        
        saveStats();
        
        for(int i = 0; i < getChildCount(); i++) {
            ProbCollection ch = (ProbCollection)getChildAt(i);
            ch.startNewSet();
        }
    }
    
    public ProbCollection findCollectionByDir(File d) {
        if(dir.equals(d))
            return this;
        
        for(int i = 0; i < getChildCount(); i++) {
            ProbCollection ch = (ProbCollection)getChildAt(i);
            ProbCollection ret = ch.findCollectionByDir(d);
            if(ret != null)
                return ret;
        }
        
        return null;
    }
    
    public ProbData findProblem(ProbData p) {
        File parentDir = p.getFile().getParentFile();
        ProbCollection c = findCollectionByDir(parentDir);
        
        if(c == null)
            return null;
        
        try {
            ProbData prob = c.getAt(p.getNum());
            if(prob.getFile().equals(p.getFile()))
                return prob;
            return null;
        }
        catch(IndexOutOfBoundsException e) {
            return null;
        }
    }
    
    private String getDatName() {
      String path = "";
      try
      {
        path = dir.getPath() + File.separator + "collection.dat"; //$NON-NLS-1$
        int index = path.indexOf(Main.pathToProblems);
        path = path.substring(0, index) + Main.pathToStats + path.substring(index + Main.pathToProblems.length());
      }
      catch (Exception localException){
        System.out.println("Wham!! (read log file)");
        ExceptionHandler.logCommonProblem(localException, "Wham! (in ProbCollection.getDatName() )");
      }
 //d.b.g(path);
        return path;
    }

    
    private static final int REVISION = 2;
    
    private void loadStats() {
        try {
            File f = new File(getDatName());
            
            File parentFile = f.getParentFile();
            if(!parentFile.exists()) {
                if(!parentFile.mkdirs()) {
                    JOptionPane.showMessageDialog(null, Messages.getString("couldnt_create_dir") //$NON-NLS-1$
                                                        + " " + parent.getPath()); //$NON-NLS-1$
                    throw new Exception(Messages.getString("couldnt_create_dir")  //$NON-NLS-1$
                                                    + " " + parent.getPath()); //$NON-NLS-1$
                }
            }
            
            ObjectInputStream in = new ObjectInputStream(new FileInputStream(f));
            
            int rev = in.readInt();
            if(rev > REVISION) {
                String msg = Messages.getString("err_data_file_newer"); //$NON-NLS-1$
                JOptionPane.showMessageDialog(null, msg);
                Main.logSilent(new Exception(msg));
                System.exit(-1);
            }
            
            numTried = in.readInt();
            numRight = in.readInt();
            totalTime = in.readLong();
            
            if(rev < 2) {
                cursetNumTried = in.readInt();
                cursetNumRight = in.readInt();
                cursetTotalTime = in.readLong();
            }
            
            highScores = (TreeSet)in.readObject();
            
            in.close();
        }
        catch(FileNotFoundException e) {
            //  Do nothing
        }
        catch(Exception e) {
            JOptionPane.showMessageDialog(null, Messages.getString("err_reading") + " " + getDatName()); //$NON-NLS-1$ //$NON-NLS-2$
            Main.logSilent(e);
        }
    }
    
    private void saveStats() {
        String filename = getDatName();
        try {
            ObjectOutputStream out = new ObjectOutputStream(new FileOutputStream(filename));
            out.writeInt(REVISION);
            out.writeInt(numTried);
            out.writeInt(numRight);
            out.writeLong(totalTime);
            out.writeObject(highScores);
            out.close();
            
            if(parent != null)
                parent.saveStats();
        }
        catch(IOException e) {
            JOptionPane.showMessageDialog(null, Messages.getString("err_writing") + " " + filename); //$NON-NLS-1$ //$NON-NLS-2$
            Main.logSilent(e);
        }
    }
    
    public void deleteStats() {
        File f = new File(getDatName());
        if(f.exists()) {
            if(!f.delete())
                JOptionPane.showMessageDialog(null, Messages.getString("err_deleting") + " " + f.getPath()); //$NON-NLS-1$ //$NON-NLS-2$
        }
    }
    
    public boolean equals(Object o) {
        ProbCollection c = (ProbCollection)o;
        return dir.equals(c.dir) && size == c.size;
    }
    
    private void removeStats(int tr, int ri, long time) {
        numTried -= tr;
        numRight -= ri;
        totalTime -= time;
        
        if(parent != null)
            parent.removeStats(tr, ri, time);
    }
    
    private void resetStats(HashSet colls, ProgressDialog d) {
        colls.add(this);
        if(getChildCount() == 0)
            removeStats(numTried, numRight, totalTime);
        else {
            for(int i = 0; i < getChildCount(); i++) {
                ProbCollection c = (ProbCollection)getChildAt(i);
                c.resetStats(colls, d);
            }
        }
        
        for(int i = 0; i < probs.size(); i++) {
            ProbData p = (ProbData)probs.get(i);
            p.resetStats();
            d.bump();
        }
    }
    
    public void resetStats(ProgressDialog d) {
        HashSet colls = new HashSet();
        resetStats(colls, d);
        
        Iterator it = colls.iterator();
        while(it.hasNext()) {
            ProbCollection c = (ProbCollection)it.next();
            c.deleteStats();
        }
    }
    
    public void addTag(String t, ProgressDialog d) {
        for(int i = 0; i < getChildCount(); i++) {
            ProbCollection c = (ProbCollection)getChildAt(i);
            c.addTag(t, d);
        }
        
        for(int i = 0; i < probs.size(); i++) {
            ProbData p = (ProbData)probs.get(i);
            p.addTag(t);
            d.bump();
        }
    }
    
    public void removeTags(ArrayList tags, ProgressDialog d) {
        for(int i = 0; i < getChildCount(); i++) { // dive into the subfolders
            ProbCollection c = (ProbCollection)getChildAt(i); // how does it know now, if child is file or folder?
            c.removeTags(tags, d);                            //   --> look at the top, the 2nd method!
        }
        
        for(int i = 0; i < probs.size(); i++) {
            ProbData p = (ProbData)probs.get(i);
            for(int j = 0; j < tags.size(); j++)
                p.removeTag((String)tags.get(j));
            d.bump();
        }
    }
    
    public int getSize() { return size; }
    public ProbData getAt(int i) { return (ProbData)probs.get(i); }
    public File getDir() { return dir; }  // #######
    public int getCursetNumTried() { return cursetNumTried; }
    public void setCursetNumTried(int n) { cursetNumTried = n; }
    public int getCursetNumRight() { return cursetNumRight; }
    public void setCursetNumRight(int n) { cursetNumRight = n; }
    public long getCursetTotalTime() { return cursetTotalTime; }
    public void setCursetTotalTime(long t) { cursetTotalTime = t; }
    public TreeSet getHighScores() { return highScores; }
    public int hashCode() { return dir.hashCode(); }
}
