# frozen_string_literal: true

module Faraday
  # Adds the ability for other modules to manage autoloadable
  # constants.
  #
  # @api private
  module AutoloadHelper
    # Registers the constants to be auto loaded.
    #
    # @param prefix [String] The require prefix. If the path is inside Faraday,
    #           then it will be prefixed with the root path of this loaded
    #           Faraday version.
    # @param options [{ Symbol => String }] library names.
    #
    # @example
    #
    #   Faraday.autoload_all 'faraday/foo',
    #     Bar: 'bar'
    #
    #   # requires faraday/foo/bar to load Faraday::Bar.
    #   Faraday::Bar
    #
    # @return [void]
    def autoload_all(prefix, options)
      if prefix.match? %r{^faraday(/|$)}i
        prefix = File.join(Faraday.root_path, prefix)
      end

      options.each do |const_name, path|
        autoload const_name, File.join(prefix, path)
      end
    end

    # Loads each autoloaded constant.  If thread safety is a concern,
    # wrap this in a Mutex.
    #
    # @return [void]
    def load_autoloaded_constants
      constants.each do |const|
        const_get(const) if autoload?(const)
      end
    end

    # Filters the module's contents with those that have been already
    # autoloaded.
    #
    # @return [Array<Class, Module>]
    def all_loaded_constants
      constants
        .map { |c| const_get(c) }
        .select { |a| a.respond_to?(:loaded?) && a.loaded? }
    end
  end

  # Adapter is the base class for all Faraday adapters.
  # @see lib/faraday/adapter.rb Original class location
  class Adapter
    extend AutoloadHelper
    autoload_all 'faraday/adapter',
                 Typhoeus: 'typhoeus',
                 Test: 'test'
  end

  # Request represents a single HTTP request for a Faraday adapter to make.
  # @see lib/faraday/request.rb Original class location
  class Request
    extend AutoloadHelper
    autoload_all 'faraday/request',
                 Authorization: 'authorization',
                 BasicAuthentication: 'basic_authentication',
                 Instrumentation: 'instrumentation',
                 Json: 'json',
                 Multipart: 'multipart',
                 Retry: 'retry',
                 TokenAuthentication: 'token_authentication',
                 UrlEncoded: 'url_encoded'
  end

  # Response represents the returned value of a sent Faraday request.
  # @see lib/faraday/response.rb Original class location
  class Response
    extend AutoloadHelper
    autoload_all 'faraday/response',
                 Json: 'json',
                 Logger: 'logger',
                 RaiseError: 'raise_error'
  end
end
