//
// $Id: LayeredOutStream.h,v 1.14 2007/03/06 20:42:19 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#if !defined(LAYEREDOUTSTREAM_OL_GUARD)
#define LAYEREDOUTSTREAM_OL_GUARD

#include <ObjectiveLib/OutStream.h>

/**
 * @class OLLayeredOutStream LayeredOutStream.h ObjectiveLib/LayeredOutStream.h
 *
 * An out stream that is layered over another out stream. Layered streams can
 * filter data that are ultimately written out to connected streams. The methods
 * in OLLayeredOutStream merely forward messages to the underlying out stream.
*
 * @sa Streams
 *
 * @ingroup Streams
 */
@interface OLLayeredOutStream : OLOutStream
{
@protected
    /**
     * The stream on top of which this stream is layered
     */
    OLOutStream* stream;
}

/**
 * @name Initializers and Deallocators
 */
/* @{ */
/**
 * Initialize the stream.
 *
 * @param underStream the stream that functions as the layer under this one
 * @return a reference to this stream
 */
- (id) initWithOutStream: (OLOutStream*)underStream;

#if !defined(OL_NO_OPENSTEP)
/**
 * Finalize the stream and deallocate any allocated memory.
 */
- (void) dealloc;
#endif
/* @} */

- (void) close;

/**
 * Return the OLConnectedOutStream at the bottom of the stream stack. All layered streams
 * in this stack of layered streams will return the same value of the connected stream.
 *
 * @note The returned value should not be released, as that will destroy the integrity
 * of the stream stack. The returned object will be a type of OLConnectedOutStream, but
 * it is returned as an id to enable more freedom in calling its methods.
 *
 * @return the connected stream
 */
- (id) connectedOutStream;
- (void) flush;
#if defined(OL_NO_OPENSTEP)
- (void) freeStreamResources;
#endif

/**
 * Write bytes to the stream. This implementation merely forwards the message
 * to the next lower stream in the stream stack.
 *
 * @exception OLInputOutputException if there is an error writing to the stream
 * @param bytes the bytes to write
 * @param count the number of bytes to write
 * @return the number of bytes written
 */
- (unsigned) writeBytes: (const uint8_t*)bytes count: (unsigned)count;

@end

#endif
