/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   sv.c								*
*									*
*	Sync vector routines.  A sync vector keeps track of successes	*
*	and failures across multiple replies from a single request	*
*	(from multiple servers), and maintains an "all or nothing"	*
*	success value for the whole transaction.			*
*									*
************************************************************************/
#include <sys/time.h>
#include <sys/types.h>
#ifdef _AIX
#include <sys/select.h>
#endif
#include "xmx.h"
#include "sv.h"
#include "incl/sv.pvt.h"

static sv_t *svfree;		/* free list */

/************************************************************************
*									*
*  sv_new								*
*  sv_free								*
*									*
*	Allocate/free a sync vector.					*
*									*
************************************************************************/
sv_t *
sv_new
   VOID
{
   register sv_t *svp;

   if (svfree) {
      svp = svfree;
      svfree = svfree->next;
   }
   else if (MALLOC(svp, sv_t *, sizeof(sv_t)))
      return 0;

   sv_clearall(svp);

   return svp;
}

void
sv_free
   AL((svp))
   DB sv_t *svp
   DE
{
   svp->next = svfree;
   svfree = svp;
}

/************************************************************************
*									*
*   sv_free_freelist							*
*									*
************************************************************************/
void
sv_free_freelist
   VOID
{
   register sv_t *svp, *last;

   for (svp=svfree; last=svp;) {
      svp = svp->next;
      free(last);
   }
   svfree = 0;
}

/************************************************************************
*									*
*   sv_set								*
*									*
*	Set a bit.							*
*									*
************************************************************************/
void
sv_set
   AL((svp, fd))
   DB sv_t *svp
   DD int fd
   DE
{
   svp->nset++;
   FD_SET(fd, &svp->fds);
}

/************************************************************************
*									*
*   sv_isset								*
*									*
*	Is bit set?							*
*									*
************************************************************************/
int
sv_isset
   AL((svp, fd))
   DB sv_t *svp
   DD int fd
   DE
{
   return FD_ISSET(fd, &svp->fds);
}

/************************************************************************
*									*
*   sv_clear								*
*									*
*	Clear a bit.							*
*									*
************************************************************************/
void
sv_clear
   AL((svp, fd))
   DB sv_t *svp
   DD int fd
   DE
{
   svp->nset--;
   FD_CLR(fd, &svp->fds);
}

/************************************************************************
*									*
*   sv_fail								*
*									*
*	Record failure (non-specific).					*
*									*
************************************************************************/
void
sv_fail
   AL((svp))
   DB sv_t *svp
   DE
{
   svp->fail++;
}

/************************************************************************
*									*
*   sv_failed								*
*									*
*	Report if the current transaction has failed.			*
*									*
************************************************************************/
int
sv_failed
   AL((svp))
   DB sv_t *svp
   DE
{
   return svp->fail;
}

/************************************************************************
*									*
*   sv_clearall								*
*									*
*	Clear a sync vector.						*
*									*
************************************************************************/
void
sv_clearall
   AL((svp))
   DB sv_t *svp
   DE
{
   svp->nset = 0;
   svp->fail = 0;
   FD_ZERO(&svp->fds);
}
