/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   root.c								*
*									*
*	Routines for generating X protocol requests to set up root	*
*	windows and other resources.					*
*									*
************************************************************************/
#include <sys/time.h>
#include <sys/types.h>
#ifdef _AIX
#include <sys/select.h>
#endif
#define NEED_REPLIES
#define NEED_EVENTS
#include <X11/X.h>
			/* Xutil.h needs Xlib.h */
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xproto.h>
#include <X11/Xatom.h>

#include "xmx.h"
#include "df.h"
#include "cx.h"
#include "rx.h"
#include "zb.h"
#include "incl/root.pvt.h"
#include "fd.h"
#include "tphot.h"

		/* bitmap definitions should have unsigned data */
#define NBITMAPS	3
#define char u8_t
#include "xmx.bm"
#include "tpsrc.bm"
#include "tpmsk.bm"
#undef char

/************************************************************************
*									*
*   root_owncmap							*
*									*
************************************************************************/
chunk_t *
root_owncmap
   VOID
{
   register int i, dynamic, alloc;
   register chunk_t *chp;
   register list_t *lstp;

   switch (vscreen.wp->vp->class) {
      case GrayScale:
      case PseudoColor:
      case DirectColor:
         dynamic = 1;
         break;
      case StaticGray:
      case StaticColor:
      case TrueColor:
         dynamic = 0;
         break;
   }
   alloc = opt.owncmap ? dynamic : 0;

   proto_CreateColormap(vscreen.rootcmap,
			vscreen.wp->vp->cid,
			vscreen.vservroot,
			alloc);
   if (alloc) {
      lstp = list_new();
      cmap_all_pixels(vscreen.wp->mp->cmap, lstp);
      proto_QueryColors(vscreen.vservcmap, lstp);
      list_free(lstp);
      rx_begin_repeat(zrxqp);
      rx_queue(zrxqp, X_QueryColors, 0, zb_seqno(), query_colors_reply);
      rx_end_repeat(zrxqp);
   }
}

static void
query_colors_reply
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp
   DD client_t *cp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DD chunk_t *chp
   DE
{
   register int i, j, n;
   register u16_t *valp;
   xColorItem cbuf[256];
   xQueryColorsReply *rp = (xQueryColorsReply *)buf_data(chp);

   if (rp->type == X_Error) {
      warn("root.c/query_colors_reply: request failed\n");
      dprx_error((xError *)rp);
      return;
   }
   valp = (u16_t *)(rp+1);

   zb_dest(ZD_SERV, sp);
   j = 0;
   while (n = cmap_get_cells(vscreen.wp->mp->cmap, CS_ALL, 256, cbuf)){
      for (i=0; i<n; i++) {
         cbuf[i].flags = DoRed | DoGreen | DoBlue;
         cbuf[i].red = valp[j++];
         cbuf[i].green = valp[j++];
         cbuf[i].blue = valp[j++];
         j++;			/* unused field */
      }
      proto_StoreColors(vscreen.rootcmap, n, cbuf, vscreen.wp->mp->mappixels);
   }
}

/************************************************************************
*									*
*   root_free_colormaps							*
*									*
************************************************************************/
chunk_t *
root_free_colormaps
   VOID
{
   proto_FreeColormap(vscreen.rootcmap);
}

/************************************************************************
*									*
*   root_own_shell							*
*									*
*	Generate X protocol requests to create the root shell window.	*
*	This routine is called in cases where the window referenced	*
*	by shellroot does not already exist.				*
*									*
************************************************************************/
void
root_own_shell
   AL((sp))
   DB server_t *sp
   DE
{
   register window_t *wp;
   register geom_t *geomp;
   s16_t x, y;
   u32_t values[2];

   wp = vscreen.wp;

   geomp = sp->geomp ? sp->geomp : opt.geomp ? opt.geomp : 0;
   if (geomp && geomp->what & G_SETPOS)
      util_adjloc(      geomp,
			sp->smap.root.pixWidth,
			sp->smap.root.pixHeight,
			sp->smap.root.width,
			sp->smap.root.height,
			&x, &y);
   else
      x = y = 0;

   /*
   **  need to set BorderPixel in case parent is of a different depth
   */
   values[0] = vscreen.blackPixel;
   values[1] = vscreen.rootcmap;

   proto_CreateWindow(	wp->dwb.depth,
			vscreen.shellroot,
			vscreen.vservroot,
			x, y,
			sp->smap.root.width,
			sp->smap.root.height,
			wp->borderwidth,
			wp->class,
			wp->vp->cid,
			CWBorderPixel | CWColormap,
			values,
			wp->mp->mappixels);
}

/************************************************************************
*									*
*   root_resources							*
*									*
*	Generate X protocol requests to create sundry root resources.	*
*	The shell root window must exist at this point.			*
*									*
************************************************************************/
int
root_resources
   VOID
{
   int n = 0;
   register int j;
   register buffer_t *bp;	/* for bitmaps only */
   register chunk_t *chp;
   rid_t wid, gid, pid, bpid, cmid, gc1bit;
   window_t *wp;
   mask_t fbmask = GCForeground | GCBackground;
   pixel_t white[2], black[2];
   xColorItem cvec[2];
   u32_t values[5];
   mask_t valuemask;
   static u16_t inverse_rootweave[] = {	/* x,y pairs for polypoint */
	0, 3,  1, 1,  2, 2,  3, 0
   };
   static u16_t rootweave[] = {
	0, 0,  0, 1,  0, 2,
	1, 0,         1, 2,  1, 3,
	2, 0,  2, 1,         2, 3,
	       3, 1,  3, 2,  3, 3
   };
   XSizeHints normhints;
   XWMHints wmhints;

   bp = buf_new(B_FREEONWRITE);

   white[0] = vscreen.whitePixel;
   white[1] = vscreen.blackPixel;
   black[0] = vscreen.blackPixel;
   black[1] = vscreen.whitePixel;

   wid = vscreen.vservroot;
   gid = vscreen.dp->gc;
   pid = vscreen.rootpixmap;
   bpid = vscreen.borderpixmap;
   cmid = vscreen.rootcmap;

   wp = vscreen.wp;

   values[0] = black[0];	/* foreground pixel */
   values[1] = black[1];	/* background pixel */
   values[2] = False;	/* graphics exposures */

   proto_CreatePixmap(pid, wid, 4, 4, wp->dwb.depth);
   proto_CreatePixmap(bpid, wid, 4, 4, wp->dwb.depth);
   for (j=0; j<vscreen.ndepths; j++) {
      proto_CreatePixmap(vscreen.dv[j].pixmap, wid, 1, 1,
					vscreen.dv[j].depth);
      proto_CreateGC(	vscreen.dv[j].gc,
				vscreen.dv[j].pixmap,
				fbmask | GCGraphicsExposures,
				values,
				wp->mp->mappixels);
      if (vscreen.dv[j].depth == 1)
         gc1bit = vscreen.dv[j].gc;
   }
   /*
   **	draw rootweave pattern on root background pixmap
   */
   proto_PolyFillRectangle(pid, gid, 0, 0, 4, 4);
   proto_ChangeGC(gid, fbmask, white, wp->mp->mappixels);
   proto_PolyPoint(CoordModeOrigin, pid, gid, 4, inverse_rootweave);
   proto_ChangeGC(gid, fbmask, black, wp->mp->mappixels);
   /*
   **	draw black rectangle on border pixmap
   */
   proto_PolyFillRectangle(bpid, gid, 0, 0, 4, 4);
   /*
   **	icon image contained in xmx.bm
   */
   proto_CreatePixmap(vscreen.iconpixmap, wid, xmx_width, xmx_height, 1);
   proto_PutImage(	XYBitmap,
			vscreen.iconpixmap,
			gc1bit,
			0, 0,
			xmx_width, xmx_height,
			0, 1);
   zb_queue(0);
   n = ((int)(xmx_width + 7)/8) * xmx_height;
   buf_put(bp, (char *)xmx_bits, n);
   chp = image_source(bmiord, xmx_width, 1, bp, n);
   zb_queue_chunk(chp);
   buf_clear(chp);
   /*
   **	Customize the shell window
   */
   values[0] = black[0];	/* background pixel */
   values[1] = black[0];	/* border pixel */
   values[2] = NotUseful;	/* backing store (always obscured) */
   values[3] = IsShellMask;	/* event mask (request_scan fills it in) */
   values[4] = cmid;		/* colormap */

   valuemask =	CWBackPixel |
			CWBorderPixel |
			CWBackingStore |
			CWEventMask |
			CWColormap;

   proto_ChangeWindowAttributes(vscreen.shellroot,
				valuemask,
				values,
				wp->mp->mappixels);
   proto_ChangeProperty(	vscreen.shellroot,
				XA_WM_NAME,
				XA_STRING,
				PropModeReplace,
				8, 3,
				"xmx");
   proto_ChangeProperty(	vscreen.shellroot,
				XA_WM_ICON_NAME,
				XA_STRING,
				PropModeReplace,
				8, 3,
				"xmx");
   normhints.flags = USSize;
   normhints.width = wp->dwb.width;		/* for old window managers */
   normhints.height = wp->dwb.height;	/* for old window managers */
   if (!opt.resize) {
      normhints.flags |= PMinSize | PMaxSize;
      normhints.min_width = wp->dwb.width;
      normhints.min_height = wp->dwb.height;
      normhints.max_width = wp->dwb.width;
      normhints.max_height = wp->dwb.height;
   }
   if (opt.geomp && opt.geomp->what & G_SETPOS) {
      normhints.flags |= USPosition;
      normhints.x = wp->x;			/* for old window managers */
      normhints.y = wp->y;			/* for old window managers */
   }
   proto_ChangeProperty(	vscreen.shellroot,
				XA_WM_NORMAL_HINTS,
				XA_WM_SIZE_HINTS,
				PropModeReplace,
				32, sizeof(XSizeHints)/4,
				(char *)&normhints);
   if (vscreen.iconpixmap) {
      wmhints.flags = IconPixmapHint;
      wmhints.icon_pixmap = vscreen.iconpixmap;
      proto_ChangeProperty(	vscreen.shellroot,
				XA_WM_HINTS,
				XA_WM_HINTS,
				PropModeReplace,
				32, sizeof(XWMHints)/4,
				(char *)&wmhints);
   }
}

/************************************************************************
*									*
*   root_free_shell							*
*									*
*	Generate X protocol requests to free the root shell window	*
*	(and sundry associated resources).				*
*									*
*	This routine undoes root_shell.					*
*									*
************************************************************************/
void
root_free_shell
   VOID
{
   register int i;

   proto_FreePixmap(vscreen.rootpixmap);
   proto_FreePixmap(vscreen.borderpixmap);
   for (i=0; i<vscreen.ndepths; i++) {
      proto_FreePixmap(vscreen.dv[i].pixmap);
      proto_FreeGC(vscreen.dv[i].gc);
   }
   proto_FreePixmap(vscreen.iconpixmap);
   proto_DestroyWindow(vscreen.shellroot);
}

/************************************************************************
*									*
*   root_tptr_dimen							*
*									*
*	Just returns the dimensions of the default root telepointer	*
*	image.								*
*									*
************************************************************************/
void
root_tptr_dimen
   AL((hotxp, hotyp, widthp, heightp))
   DB s16_t *hotxp
   DD s16_t *hotyp
   DD u16_t *widthp
   DD u16_t *heightp
   DE
{
   if (tpsrc_width != tpmsk_width || tpsrc_height != tpmsk_height)
      warn("root_tptr: source and mask bitmaps are unequal sizes\n");

   *hotxp = tp_hotx;
   *hotyp = tp_hoty;
   *widthp = tpsrc_width;
   *heightp = tpsrc_height;
}

/************************************************************************
*									*
*   root_tptr_image							*
*									*
*	Draw root telepointer image.					*
*									*
************************************************************************/
void
root_tptr_image
   VOID
{
   register int n;
   register buffer_t *bp;	/* for bitmaps only */
   register chunk_t *chp;
   register tpim_t *tpimp;
   pixel_t vals[2];

   bp = buf_new(B_FREEONWRITE);

   tpimp = tptr_default_image();

   /*
   **  create and draw the bitmaps first - this is really only needed
   **  for creating the cursor in root_cursor, below
   */
   proto_CreatePixmap(tpimp->sbmid, vscreen.shellroot,
					tpsrc_width, tpsrc_height, 1);
   if (vscreen.blackPixel == 0) {
      vals[0] = 1;	/* foreground */
      vals[1] = 0;	/* background */
      proto_ChangeGC(	vscreen.dv[0].gc,
			GCForeground | GCBackground,
			vals, 0);
   }
   proto_PutImage(XYBitmap, tpimp->sbmid, vscreen.dv[0].gc,
					0, 0, tpsrc_width, tpsrc_height, 0, 1);
   zb_queue(0);
   n = ((int)(tpsrc_width + 7)/8) * tpsrc_height;
   buf_put(bp, (char *)tpsrc_bits, n);
   chp = image_source(bmiord, tpsrc_width, 1, bp, n);
   zb_queue_chunk(chp);
   buf_clear(chp);

   proto_CreatePixmap(tpimp->mbmid, vscreen.shellroot,
					tpmsk_width, tpmsk_height, 1);
   proto_PutImage(XYBitmap, tpimp->mbmid, vscreen.dv[0].gc,
					0, 0, tpmsk_width, tpmsk_height, 0, 1);
   zb_queue(0);

   n = ((int)(tpmsk_width + 7)/8) * tpmsk_height;
   buf_put(bp, (char *)tpmsk_bits, n);
   chp = image_source(bmiord, tpmsk_width, 1, bp, n);
   zb_queue_chunk(chp);
   buf_clear(chp);
   /*
   **	default telepointer image contained in tpsrc.bm
   */
   proto_CreatePixmap(tpimp->spmid, vscreen.shellroot,
			tpsrc_width, tpsrc_height, vscreen.wp->dwb.depth);
   proto_PutImage(XYBitmap, tpimp->spmid, vscreen.dp->gc,
					0, 0, tpsrc_width, tpsrc_height, 0, 1);
   zb_queue(0);

   n = ((int)(tpsrc_width + 7)/8) * tpsrc_height;
   buf_put(bp, (char *)tpsrc_bits, n);
   chp = image_source(bmiord, tpsrc_width, 1, bp, n);
   zb_queue_chunk(chp);
   buf_clear(chp);
   /*
   **	default telepointer image mask contained in tpmsk.bm
   */
   proto_CreatePixmap(tpimp->mpmid, vscreen.shellroot,
			tpmsk_width, tpmsk_height, vscreen.wp->dwb.depth);
   proto_PutImage(XYBitmap, tpimp->mpmid, vscreen.dp->gc,
					0, 0, tpmsk_width, tpmsk_height, 0, 1);
   zb_queue(0);

   n = ((int)(tpmsk_width + 7)/8) * tpmsk_height;
   buf_put(bp, (char *)tpmsk_bits, n);
   chp = image_source(bmiord, tpmsk_width, 1, bp, n);
   zb_queue_chunk(chp);
   buf_clear(chp);
}

void
root_cursor
   VOID
{
   register tpim_t *tpimp = tptr_default_image();

   proto_CreateCursor(	vscreen.cursor,
			tpimp->sbmid,
			tpimp->mbmid,
			0, 0, 0,
			MAXCOLVAL, MAXCOLVAL, MAXCOLVAL,
			tpimp->hotx, tpimp->hoty);
}
