/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   rgb.c								*
*									*
*	Originally written to extract colors from the ascii version	*
*	of the rgb database.  Now uses (n)dbm, but can be compiled	*
*	to use the old code if RGBTXT is defined.			*
*									*
************************************************************************/
#include <sys/param.h>
#include <stdio.h>
#include "xmx.h"

#ifdef NDBM
#include <ndbm.h>
#else
#ifndef RGBTXT
#ifdef SVR4
#include <rpcsvc/dbm.h>
#else
#include <dbm.h>
#endif
#endif
#endif

typedef struct {
   char *	name;
   u16_t	namelen;
   u16_t	red;
   u16_t	green;
   u16_t	blue;
}rgb_t;

#include "incl/rgb.pvt.h"

static int usedbm;

static union {
   rgb_t *txt;
#ifdef NDBM
   DBM *ndbm;
#endif
}rgb;

static int sz = 0;

/************************************************************************
*									*
*   rgb_init								*
*									*
************************************************************************/
void
rgb_init
   VOID
{
   register int len;
   char dbname[MAXPATHLEN];

   len = strlen(opt.co);

   if (len > MAXPATHLEN - 5) {
      warn("%s: rgb database path too long\n", opt.co);
      return;
   }
   (void) strcpy(dbname, opt.co);

#ifndef RGBTXT
   /*
   **	if the color database path is a .txt file, trim that suffix
   **	before handing it to (n)dbm.
   */
   if (len > 4 && strcmp(&dbname[len-4], ".txt") == 0)
      dbname[len-4] = '\0';

#ifdef NDBM
   if (rgb.ndbm = dbm_open(dbname, 0, 0))
      usedbm = 1;
#else
   if (dbminit(dbname) == 0)
      usedbm = 1;
#endif
#endif

   if (usedbm == 0)
      if (txt_open(dbname, len))
         warn("can't open rgb database '%s'\n", dbname);
}

static int
txt_open
   AL((dbname, len))
   DB char *dbname
   DD int len
   DE
{
   register int i;
   register char *cp, *ep;
   register FILE *fp;
   char buf[1024];

   if (len < 5 || strcmp(&dbname[len-4], ".txt"))	/* be tolerant */
      (void) strcat(dbname, ".txt");

   if ((fp = fopen(dbname, "r")) == 0)
      return -1;
   while (fgets(buf, 1024, fp)) {
      for (cp=buf; *cp == ' ' || *cp == '\t'; cp++);
      if (*cp == '#')
         continue;
      sz++;
   }
   if (CALLOC(rgb.txt, rgb_t *, sz, sizeof(rgb_t))) {
      sz = 0;
      return -1;
   }
   rewind(fp);
   for (i=0; fgets(buf, 1024, fp); i++) {
      for (cp=buf; *cp == ' ' || *cp == '\t'; cp++);
      if (*cp == '#')
         continue;
      if (isdigit(*cp)) {
         for (ep=cp; isdigit(*ep); ep++);
         *ep = '\0';
         if (ep != cp)
            rgb.txt[i].red = atoi(cp);

         for (cp=ep+1; *cp == ' ' || *cp == '\t'; cp++);
         for (ep=cp; isdigit(*ep); ep++);
         *ep = '\0';
         if (ep != cp)
            rgb.txt[i].green = atoi(cp);

         for (cp=ep+1; *cp == ' ' || *cp == '\t'; cp++);
         for (ep=cp; isdigit(*ep); ep++);
         *ep = '\0';
         if (ep != cp)
            rgb.txt[i].blue = atoi(cp);

         for (cp=ep+1; *cp == ' ' || *cp == '\t'; cp++);
         rgb.txt[i].namelen = dropwhite(cp, strlen(cp));
         if (rgb.txt[i].namelen) {
            if (MALLOC(rgb.txt[i].name, char *, rgb.txt[i].namelen+1) == 0)
               bcopy(cp, rgb.txt[i].name, rgb.txt[i].namelen);
            rgb.txt[i].name[rgb.txt[i].namelen] = '\0';
         }
      }
   }
   return 0;
}

/************************************************************************
*									*
*   rgb_find								*
*									*
************************************************************************/
int
rgb_find
   AL((name, namelen, red, green, blue))
   DB char *name
   DD u16_t namelen
   DD u16_t *red
   DD u16_t *green
   DD u16_t *blue
   DE
{
   register int i;
   char buf[256];
#ifndef RGBTXT
   datum item;
   u16_t val[3];
#endif

   i = MIN(namelen, 256);
   bcopy(name, buf, i);
   namelen = dropwhite(buf, i);

   if (usedbm) {
#ifndef RGBTXT
      item.dptr = buf;
      item.dsize = namelen;
#ifdef NDBM
      item = dbm_fetch(rgb.ndbm, item);
#else
      item = fetch(item);
#endif
      if (item.dptr) {
         bcopy(item.dptr, (char *)val, sizeof(val));
         *red = val[0];
         *green = val[1];
         *blue = val[2];
         return 0;
      }
#endif
   }
   else {
      for (i=0; i<sz; i++)
         if (namelen == rgb.txt[i].namelen)
            if (strncmp(buf, rgb.txt[i].name, namelen) == 0) {
               *red = (rgb.txt[i].red + 1) * 256 - 1;
               *green = (rgb.txt[i].green + 1) * 256 - 1;
               *blue = (rgb.txt[i].blue + 1) * 256 - 1;
               return 0;
            }
   }
   return -1;
}

/*
**   drop whitespace from a string.  converts uppercase letters to
**   lowercase.  returns new length
*/
static int
dropwhite
   AL((cp, n))
   DB char *cp
   DD int n
   DE
{
   register int i, j;

   for (i=j=0; j<n; j++)
      if (cp[j] != ' ' && cp[j] != '\t')
      if (!isspace(cp[j]))
         if (isupper(cp[j]))
            cp[i++] = tolower(cp[j]);
         else
            cp[i++] = cp[j];

   return i;
}
