/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   font.c								*
*									*
************************************************************************/
#define NEED_REPLIES
#include <X11/Xproto.h>
#include <X11/X.h>
#include "xmx.h"
#include "df.h"
#include "zb.h"
#include "res.h"
#include "incl/font.pvt.h"

static font_t *fonts;

static char **path;	/* font path */
static int pathsz;
static int pathlen;

/************************************************************************
*									*
*   font_open								*
*									*
************************************************************************/
int
font_open
   AL((cp, p))
   DB client_t *cp
   DD xOpenFontReq *p
   DE
{
   register font_t *fp;

   if (MALLOC(fp, font_t *, sizeof(font_t))) {
      proto_Error(cp, BadAlloc, 0, 0, X_OpenFont);
      return -1;
   }
   if (MALLOC(fp->name, char *, p->nbytes + 1)) {
      proto_Error(cp, BadAlloc, 0, 0, X_OpenFont);
      free(fp);
      return -1;
   }
   if ((fp->vid = hash_add_client_id(p->fid, (char *)fp)) == 0) {
      free(fp->name);
      free(fp);
      proto_Error(cp, BadIDChoice, p->fid, 0, X_OpenFont);
      return -1;
   }
   fp->res.client = cp;
   fp->cid = p->fid;
   fp->refs = 1;
   fp->namelen = p->nbytes;

   bcopy((char *)(p+1), fp->name, p->nbytes);
   fp->name[p->nbytes] = '\0';

   fp->last = 0;
   fp->next = fonts;
   if (fp->next)
      fp->next->last = fp;
   fonts = fp;

   cp->refs++;
   return 0;
}

/************************************************************************
*									*
*   font_close								*
*									*
************************************************************************/
int
font_close
   AL((cp, p))
   DB client_t *cp
   DD xResourceReq *p
   DE
{
   font_t *fp;

   if ((fp = (font_t *)hash_data(vmap, p->id)) == 0) {
      proto_Error(cp, BadPixmap, p->id, 0, X_CloseFont);
      return -1;
   }
   return free_font(fp);
}

/************************************************************************
*									*
*   font_set_path							*
*   font_get_path_reply							*
*									*
*	Store the font path, whether it is being set or retrieved.	*
*	This is extremely passive, and relies on the relative		*
*	homogeneity of the participating systems.			*
*									*
*	The saved font path is used to set the path when adding a	*
*	display.  See the ketchup routine below.			*
*									*
************************************************************************/
void
font_set_path
   AL((cp, p))
   DB client_t *cp
   DD xSetFontPathReq *p
   DE
{
   (void) font_store_path((char *)(p + 1), p->nFonts);
}

void
font_get_path_reply
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp
   DD client_t *cp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DD chunk_t *chp
   DE
{
   register xGetFontPathReply *rp = (xGetFontPathReply *)buf_data(chp);

#ifdef NOT_NECESSARY_I_THINK
   if (rp->type == X_Reply)	/* in case it's an error */
      font_store_path((char *)(rp + 1), rp->nPaths);
#endif

   queue_add(cp->qp, chp);
}

int
font_store_path
   AL((strs, nstrs))
   DB char *strs	/* LISTofSTRING8 */
   DD int nstrs		/* number of strings */
   DE
{
   register int i, len;
   register char *cp;

   if (path) {
      /*
      **  see if it has changed
      */
      if (nstrs == pathlen) {
         cp = strs;
         for (i=0; i<nstrs; i++) {
            len = (int)*cp++;
            if (strncmp(cp, path[i], len))
               break;
            cp += len;
         }
         if (i == nstrs)
            return 0;	/* no change, bye bye */
      }
      /*
      **  free path elements
      */
      for (i=0; i<pathlen; i++)
         if (path[i])
            free(path[i]);
   }
   if (nstrs > pathsz) {
      /*
      **  adjust size of path
      */
      if (path)
         free(path);

      len = RUP(nstrs, 32);
      if (MALLOC(path, char **, len * sizeof(char *))) {
         pathsz = 0;
         pathlen = 0;
         return -1;
      }
      pathsz = len;
   }
   /*
   **  store path
   */
   cp = strs;
   for (i=0; i<nstrs; i++) {
      len = (int)*cp;
      if (MALLOC(path[i], char *, len + 2)) {
         pathlen = i;
         return -1;
      }
      bcopy(cp, path[i], len + 1);	/* copy leading length byte, too */
      path[i][len + 1] = '\0';
      cp += len + 1;
   }
   pathlen = nstrs;

   return 0;
}

/************************************************************************
*									*
*   font_client_death							*
*									*
************************************************************************/
void
font_client_death
   AL((client))
   DB client_t *client
   DE
{
   register rid_t cid;
   register font_t *fp, *nfp;

   for (nfp=fonts; fp=nfp;) {
      nfp = nfp->next;
      if (client_cmp(client, fp->res.client)) {
         cid = fp->cid;
         if (free_font(fp) == 0)
            proto_CloseFont(cid);
         else {
            client->refs--;
            fp->res.client = 0;
         }
      }
   }
}

/************************************************************************
*									*
*   font_reset								*
*									*
************************************************************************/
void
font_reset
   VOID
{
   register rid_t cid;
   register font_t *fp, *nfp;

   for (nfp=fonts; fp=nfp;) {
      nfp = nfp->next;
      cid = fp->cid;
      if (free_font(fp) == 0)
         proto_CloseFont(cid);
   }
}

/************************************************************************
*									*
*   font_ketchup							*
*									*
************************************************************************/
void
font_ketchup
   VOID
{
   register font_t *fp;

   if (pathlen)
      proto_SetFontPath(pathlen, path);

   for (fp=fonts; fp; fp=fp->next)
      proto_OpenFont(	fp->cid,
			fp->name,
			fp->namelen);
}

/************************************************************************
*									*
*   font_ref								*
*   font_deref								*
*   font_assign								*
*									*
************************************************************************/
void
font_ref
   AL((fp))
   DB font_t *fp
   DE
{
   fp->refs++;
}
 
void
font_deref
   AL((fp))
   DB font_t *fp
   DE
{
   register chunk_t *chp;

   if (fp->refs == 1) {
      zb_dest(ZD_ALLSERV, 0);
      proto_CloseFont(fp->cid);
   }
   free_font(fp);
}
 
void
font_assign
   AL((fpp, fp))
   DB font_t **fpp
   DD font_t *fp
   DE
{
   if (*fpp != fp) {  
      if (*fpp)
         font_deref(*fpp);
      if (fp)
         font_ref(fp);
      *fpp = fp;
   }
}


static int
free_font
   AL((fp))
   DB font_t *fp
   DE
{
   if (--fp->refs == 0) {
      if (fp->last)
         fp->last->next = fp->next;
      if (fonts == fp)
         fonts = fp->next;
      if (fp->next)
         fp->next->last = fp->last;
      hash_mark(vmap, fp->cid);
      fp->res.client->refs--;
      free(fp);
      return 0;
   }
   return -1;
}
