/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   df.c								*
*									*
*	Data frames.  Allocate "frames" of data values.  These are	*
*	allocated from a series of blocks.  Only the current frame	*
*	(the last one new'd) can grow.  Frames may be free'd in any	*
*	order.  These are used to save arbitrary data during		*
*	request/reply dialogs between xmx and individual X servers.	*
*	It just makes it easier to write those dialogs.			*
*									*
************************************************************************/
#include "xmx.h"
#include "df.h"
#include "incl/df.pvt.h"

df_t *curdfp;		/* currently "open" frame */
static df_block_t *blkhead;	/* start of active block list */
static df_block_t *blktail;	/* last in active block list */

static df_t *freedfp;		/* data frame free list */
static df_block_t *freeblkp;	/* block free list */

/************************************************************************
*									*
*   df_current								*
*									*
************************************************************************/
df_t *
df_current
   VOID
{
   register df_t *dfp;

   dfp = curdfp;
   curdfp = 0;

   return dfp;
}

/************************************************************************
*									*
*   df_free								*
*									*
*	Free a data frame.						*
*									*
************************************************************************/
void
df_free
   AL((dfp))
   DB df_t *dfp
   DE
{
   int lastdex;
   df_block_t *blkp, *lastp;

   lastdex = dfp->index + dfp->count - 1;

   for (blkp=dfp->blkp; lastp=blkp;) {
      blkp = blkp->next;
      free_block(lastp);

      if (lastdex < DF_NVALS)
         break;
      else
         lastdex -= DF_NVALS;
   }
   dfp->next = freedfp;
   freedfp = dfp;
}

/************************************************************************
*									*
*   df_put_i								*
*   df_put_p								*
*									*
*	Add a value to the current data frame.				*
*									*
************************************************************************/
void
df_put_i
   AL((i))
   DB int i
   DE
{
   if (curdfp == 0)
      curdfp = new_df();

   if (curdfp->curblkp->index >= DF_NVALS) {
      new_block();
      blktail->refs++;
      curdfp->curblkp = blktail;
   }
   curdfp->curblkp->val[curdfp->curblkp->index++].i = i;
   curdfp->count++;
}

void
df_put_p
   AL((p))
   DB void *p
   DE
{
   if (curdfp == 0)
      curdfp = new_df();

   if (curdfp->curblkp->index >= DF_NVALS) {
      new_block();
      blktail->refs++;
      curdfp->curblkp = blktail;
   }
   curdfp->curblkp->val[curdfp->curblkp->index++].p = p;
   curdfp->count++;
}

/************************************************************************
*									*
*   df_reset								*
*									*
*	Must always be done before doing any get's.			*
*									*
************************************************************************/
void
df_reset
   AL((dfp))
   DB df_t *dfp
   DE
{
   dfp->curblkp = dfp->blkp;
   dfp->curdex = dfp->index;
}

/************************************************************************
*									*
*   df_get_i								*
*   df_get_p								*
*									*
*	Return the next value in the given data frame.			*
*									*
*	There is no checking that what was put in (integer or pointer)	*
*	is the same type as that returned here.	  You're in big		*
*	trouble if you pull them out in the wrong order anyway.		*
*	This is probably a good source of bugs.				*
*									*
************************************************************************/
int
df_get_i
   AL((dfp))
   DB df_t *dfp
   DE
{
   int i;

#ifdef DEBUG
   if (dfp->curblkp == dfp->blkp) {
      if (dfp->count <= dfp->curdex - dfp->index)
         warn("df_get_i: getting a value beyond frame!\n");
   }
   else
      if (dfp->count <= dfp->curdex + DF_NVALS - dfp->index)
         warn("df_get_i: getting a value beyond frame! (across blocks)\n");
#endif

   i = dfp->curblkp->val[dfp->curdex++].i;

   if (dfp->curdex >= DF_NVALS) {
      dfp->curblkp = dfp->curblkp->next;
      dfp->curdex = 0;
   }
   return i;
}

void *
df_get_p
   AL((dfp))
   DB df_t *dfp
   DE
{
   void *p;

#ifdef DEBUG
   if (dfp->curblkp == dfp->blkp) {
      if (dfp->count <= dfp->curdex - dfp->index)
         warn("df_get_p: getting a value beyond frame!\n");
   }
   else
      if (dfp->count <= dfp->curdex + DF_NVALS - dfp->index)
         warn("df_get_p: getting a value beyond frame! (across blocks)\n");
#endif

   p = dfp->curblkp->val[dfp->curdex++].p;

   if (dfp->curdex >= DF_NVALS) {
      dfp->curblkp = dfp->curblkp->next;
      dfp->curdex = 0;
   }
   return p;
}

/*
**   new_df
**
**	Allocate a new data frame.
*/
static df_t *
new_df
   VOID
{
   register df_t *dfp;

   if (freedfp) {
      dfp = freedfp;
      freedfp = freedfp->next;
   }
   else if (MALLOC(dfp, df_t *, sizeof(df_t)))
      return 0;

   if (blktail == 0 || blktail->index >= DF_NVALS)
      new_block();

   blktail->refs++;
   dfp->blkp = dfp->curblkp = blktail;
   dfp->index = dfp->curdex = blktail->index;
   dfp->count = 0;

   return dfp;
}

/*
**  new_block
**
**	allocate and link a new block - always onto the tail
*/
static void
new_block
   VOID
{
   df_block_t *blkp;

   if (freeblkp) {
      blkp = freeblkp;
      freeblkp = freeblkp->next;
   }
   else if (MALLOC(blkp, df_block_t *, sizeof(df_block_t)))
      quit(-1, "df.c/new_block: out of memory, can't recover\n");

   blkp->next = 0;
   if (blktail)
      blktail->next = blkp;
   else
      blkhead = blkp;
   blktail = blkp;

   blkp->refs = 0;
   blkp->index = 0;
}

/*
**  free_block
**
**	unlink block and put it on the free list
*/
static void
free_block
   AL((blkp))
   DB df_block_t *blkp
   DE
{
   register df_block_t *tblkp;

   if (--blkp->refs == 0) {
      /*
      **  unlink it
      */
      if (blkhead == blkp) {	/* most common case */
         if ((blkhead = blkp->next) == 0)
            blktail = 0;
      }
      else {
         for (tblkp=blkhead; tblkp && tblkp->next != blkp; tblkp=tblkp->next);
         if (tblkp) {
            if ((tblkp->next = blkp->next) == 0)
               blktail = tblkp;
         }
         else {
            warn("df.c/free_block: block not in active list, aborting...\n");
            abort();
         }
      }
      /*
      **  free it
      */
      blkp->next = freeblkp;
      freeblkp = blkp;
   }
}
