/*
 * Copyright 1991-1998, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*   cliz.c								*
*									*
*	Client Zero routines.  Client zero is the X client activity	*
*	of xmx itself.							*
*									*
************************************************************************/
#include <sys/time.h>
#include <sys/types.h>
#ifdef _AIX
#include <sys/select.h>
#endif
#include <netinet/in.h>		/* ip.h needs this */
#include <netinet/in_systm.h>	/* ...and this */
#include <netinet/ip.h>		/* for IP_MAXPACKET */
#include <stdlib.h>
#include <X11/X.h>
#include <X11/Xatom.h>
#define NEED_REPLIES
#define NEED_EVENTS
#include <X11/Xproto.h>

#include <xmc.h>
#include <xmcp.h>

#include "xmx.h"
#include "am.h"
#include "res.h"
#include "df.h"
#include "cx.h"
#include "rx.h"
#include "zb.h"
#include "ptc.h"
#include "kpm.h"
#include "incl/cliz.pvt.h"
#include "fd.h"

#ifndef IP_MAXPACKET
#define IP_MAXPACKET	8192
#endif

/************************************************************************
*									*
*   cliz_init								*
*									*
************************************************************************/
void
cliz_init
   VOID
{
   zrxqp = rx_new();	/* client zero reply context queue */
}

/************************************************************************
*									*
*   Adding servers							*
*   ------ -------							*
*									*
*	Adding one or more servers requires a dialog with each		*
*	server, and, at certain points, all servers being added must	*
*	be in sync.  One or more calls to cliz_add_server causes the	*
*	following series of routines to be called, in this order:	*
*									*
*		foreach server:						*
*			cliz_add_server					*
*			list_extensions_reply				*
*			intern_atom_reply				*
*		once, synced:						*
*			cliz_vconfig_cont				*
*			ketchup						*
*		once, synced:						*
*			setup_roots_cont				*
*		once, synced:						*
*			final_ketchup_cont				*
*		foreach server:						*
*			finish_server					*
*		once, synced:						*
*			done_cont					*
*									*
************************************************************************/

/************************************************************************
*									*
*   cliz_add_server							*
*									*
*	Begin new server setup.  Query server for config info.		*
*									*
*	This routine creates a window in time during which server	*
*	adds become part of the "current join."  This window extends	*
*	from the first add to the "last" response (before the		*
*	ketchup phase).  Collecting multiple adds in this way is	*
*	a big win, as adds are expensive overall, but each server	*
*	added requires only a small additional effort.			*
*									*
*	The window is defined using the "pending" mechanism of the	*
*	zb module, whereby the "current" forward buffer block is	*
*	saved between calls to zb_queue, until the destination is	*
*	changed (which does not happen here until all replies are	*
*	received).							*
*									*
*	TODO: need to prevent a new join from starting before a		*
*	previous one completes.						*
*									*
************************************************************************/

static int in_join;	/* lock */

int
cliz_add_server
   AL((sp))
   DB server_t *sp
   DE
{
   int rv;

   DEBUG2(D_RESET, "cliz_add_server: adding [%d/%s]\n", sp->fd, sp->tag);
   if ((rv = connect_as_client(sp)) < 0)
      return -1;

   if (fd_inreset() && ! in_join) {
      /*
      **  Attempt to add during an add (but after join window has closed).
      **  Block it until current add is done.
      */
      DEBUG2(D_RESET, "...blocking [%d/%s]\n", sp->fd, sp->tag);
      server_state(sp, S_BLOCKED);
      return 0;
   }
   /*
   **  all systems are go
   */
   poke(sp);

   return 0;
}

static void
poke
   AL((sp))
   DB server_t *sp
   DE
{
   DEBUG2(D_RESET, "poke: queuing client block [%d/%s]\n",sp->fd,sp->tag);

   time_start(sp->fd, opt.sto);	/* begin timer on server */

   fd_reset(sp->fd);

   zb_dest(ZD_SERV_ASYNC, sp);

   if (! in_join) {
      in_join = 1;

      rx_begin_repeat(zrxqp);
      rx_queue(zrxqp, 0, 0, 0, poke_reply);
      rx_end_repeat(zrxqp);

      cx_push(zrxqp, done_cont);	/* final step */
      cx_push(zrxqp, poke_cont);	/* next step */
   }
   connect_client_out(sp);
   zb_queue(0);

   server_state(sp, S_POKED);
}

void
poke_reply
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp
   DD client_t *cp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DD chunk_t *chp
   DE
{
   time_end(sp->fd);	/* made it! */

   DEBUG2(D_RESET, "poke_reply: [%d/%s]\n", sp->fd, sp->tag);

   /*
   **  deal with connection block
   */
   if (connect_server_in(sp, chp)) {
      server_state(sp, S_ZOMBIE);	/* got reply, not S_POKE anymore */
      xmc_drop(sp);
      return;	/* connection failed */
   }
   /*
   **  choose query server, if need be
   */
   if (qservp == 0)
      qservp = sp;
   else if (sp->mode > qservp->mode)
      qservp = sp;

   server_state(sp, S_KETCHUP);
}

void
poke_cont
   AL((dfp))
   DB df_t *dfp
   DE
{
   register atom_t i;

   in_join = 0;
   fd_reset(-1);

   DEBUG0(D_RESET, "poke_cont: called\n");
   if (server_count(S_KETCHUP)) {
      zb_dest(ZD_KETCHUP, 0);

      rx_begin_repeat(zrxqp);

      proto_ListExtensions();
      rx_queue(zrxqp, X_ListExtensions, 0, zb_seqno(), list_extensions_reply);

      for (i=0; am_atom_names[i]; i++) {
         proto_InternAtom(0, am_atom_names[i], strlen(am_atom_names[i]));
         df_put_i((int)i);
         rx_queue(zrxqp, X_InternAtom, 0, zb_seqno(), intern_atom_reply);
      }
      rx_end_repeat(zrxqp);

      zb_queue(0);

      cx_push(zrxqp, cliz_vconfig_cont);	/* next step */
   }
}

static void
list_extensions_reply
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp
   DD client_t *cp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DD chunk_t *chp
   DE
{
   register int i, sz;
   register char *strp;
   xListExtensionsReply *rp = (xListExtensionsReply *)buf_data(chp);

   DEBUG3(D_RESET, "list_extensions_reply: [%d/%s] %s\n", sp->fd, sp->tag,
						dprx_event_str(rp->type));
   if (rp->type == X_Error) {
      warn("cliz.c/list_extensions_reply: requst failed\n");
      dprx_error((xError *)rp);
      xmc_drop(sp);
      return;
   }
   if (rp->nExtensions) {
      if (MALLOC(sp->exts, ext_t *, sizeof(ext_t) * rp->nExtensions)) {
         xmc_drop(sp);
         return;        /* handle this error better... */
      }
   }
   else
      sp->exts = 0;
 
   strp = (char *)(rp + 1);
   for (i=0; i<(int)rp->nExtensions; i++) {
      sz = (int)*strp++;
      if (MALLOC(sp->exts[i].name, char *, sz + 1)) {
         sp->exts[i].name = 0;
         xmc_drop(sp);
         return;       /* bad, bad, TODO */
      }
      bcopy(strp, sp->exts[i].name, sz);
      sp->exts[i].name[sz] = '\0';
 
      strp += sz;
   }
   sp->nexts = rp->nExtensions;
}

static void
intern_atom_reply
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp
   DD client_t *cp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DD chunk_t *chp
   DE
{
   xInternAtomReply *rp = (xInternAtomReply *)buf_data(chp);
   atom_t atom = (atom_t)df_get_i(dfp);

   DEBUG3(D_RESET, "intern_atom_reply: [%d/%s] %s\n", sp->fd, sp->tag,
						dprx_event_str(rp->type));
   if (rp->type == X_Error) {
      warn("cliz.c/intern_atom_reply: InternAtom failed\n");
      dprx_error((xError *)rp);
      xmc_drop(sp);
      return;
   }
   am_store(sp->smap.amp, atom, rp->atom);
}

/************************************************************************
*									*
*   cliz_vconfig_cont							*
*									*
*	Establish the virtual configuration, map all servers that are	*
*	ready.								*
*									*
*	This can be called as part of a server add, or a reset.		*
*									*
************************************************************************/
void
cliz_vconfig_cont
   AL((dfp))
   DB df_t *dfp
   DE
{
   register int i;

   DEBUG0(D_RESET, "cliz_vconfig_cont: called\n");
   /*
   **  establish virtual config, if not already
   */
   if (vcstate != VC_FIXED)
      if (vconf_fix())
         return;
   /*
   **  map all new servers
   */
   for (i=0; i<num_serv; i++)
      if (servers[i]->state == S_KETCHUP)
         if (vconf_map(servers[i])) {
            warn("can't map server %s:%d.%d to virtual config\n",
		servers[i]->tag, servers[i]->display, servers[i]->screen);
            xmc_drop(servers[i--]);	/* slides servers array left */
         }

   if (i)
      ketchup();
}

/************************************************************************
*									*
*   ketchup								*
*									*
*	Generate all current resources on all servers waiting to	*
*	join the session.						*
*									*
************************************************************************/
static void
ketchup
   VOID
{
   register int i;
   register chunk_t *chp;

   DEBUG0(D_RESET, "ketchup: called\n");

   if (server_count(S_KETCHUP)) {
      zb_dest(ZD_KETCHUP, 0);

      ext_ketchup();

      /*
      **  retrieve key and pointer mappings
      */
      for (i=0; i<num_serv; i++)
         if (servers[i]->state == S_KETCHUP) {
            zb_dest(ZD_SERV, servers[i]);
            kpm_update_keys(servers[i]->smap.minkey, servers[i]->smap.keycnt);
         }
      zb_dest(ZD_KETCHUP, 0);
      rx_begin_repeat(zrxqp);
      kpm_update_mods();
      kpm_update_buttons();
      rx_end_repeat(zrxqp);
      zb_queue(0);
      /*
      **  create own cmap if unshared default root cmap
      */
      for (i=0; i<num_serv; i++)
         if (	servers[i]->state == S_KETCHUP &&
		servers[i]->smap.root.flags & RF_OWNCMAP) {
            zb_dest(ZD_SERV, servers[i]);
            root_owncmap();
         }
      zb_dest(ZD_KETCHUP, 0);
      color_ketchup();	/* this does a sync */

      cx_push(zrxqp, setup_roots_cont);
   }
}

/*
**	All servers now have valid root colormaps.
*/
static void
setup_roots_cont
   AL((dfp))
   DB df_t *dfp
   DE
{
   register int i;

   DEBUG0(D_RESET, "setup_roots_cont: called\n");

   if (server_count(S_KETCHUP)) {
      /*
      **  create own shell root window if needed
      */
      for (i=0; i<num_serv; i++)
         if (	servers[i]->state == S_KETCHUP &&
		servers[i]->smap.root.flags & RF_OWNROOT) {
            zb_dest(ZD_SERV, servers[i]);
            root_own_shell(servers[i]);
         }
      zb_dest(ZD_KETCHUP, 0);
      /*
      **  create screen resources
      */
      root_resources();
      root_tptr_image();
      root_cursor();
      /*
      **  create all pixmaps
      */
      pixmap_ketchup();
      tptr_pixmap_ketchup();
      cursor_pixmap_ketchup();

      /*
      **  [ do pixmap caching thing here ]
      */

      zb_dest(ZD_QSERV, 0);
      /*
      **  retrieve pixmap contents from query server
      */
      pixmap_get_images(ZD_KETCHUP);
      tptr_get_images(ZD_KETCHUP);
      cursor_get_images(ZD_KETCHUP);
      /*
      **  sync (in case there are no pixmaps)
      */
      proto_GetInputFocus();
      rx_queue(zrxqp, X_GetInputFocus, 0, zb_seqno(), 0);
      /*
      **  push next operation
      */
      cx_push(zrxqp, final_ketchup_cont);
   }
}

/*
**  final_ketchup_cont
**
**	All servers have colormaps, shell windows (unmapped), and all
**	pixmaps have been created and filled in.
*/
static void
final_ketchup_cont
   AL((dfp))
   DB df_t *dfp
   DE
{
   register int i;

   DEBUG0(D_RESET, "final_ketchup_cont: called\n");

   if (server_count(S_KETCHUP)) {
      zb_dest(ZD_KETCHUP, 0);
   
      font_ketchup();
      cursor_ketchup();
      gc_ketchup();
      window_ketchup();
      color_update_winvec();	/* creates WM_COLORMAP_WINDOWS property */
      tptr_ketchup();
      /*
      **  make everything appear and generate exposures
      */
      proto_MapWindow(vscreen.shellroot);
      /*
      **  final sync
      */
      zb_dest(ZD_KETCHUP, 0);
      proto_GetInputFocus();
      rx_begin_repeat(zrxqp);
      rx_queue(zrxqp, X_GetInputFocus, 0, zb_seqno(), finish_server);
      rx_end_repeat(zrxqp);
   }
}

/*
**  finish_server
**
**	Called once for each server at the point where it has all
**	resources set up.
*/
static void
finish_server
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp
   DD client_t *cp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DD chunk_t *chp
   DE
{
   DEBUG2(D_RESET, "finish_server: [%d/%s]\n", sp->fd, sp->tag);

   server_state(sp, S_READY);
   xmc_display_event(DisplayInEvent, sp->id);
}

/*
**  done_cont
**
**	Called once after all servers have been set up.
*/
static void
done_cont
   AL((dfp))
   DB df_t *dfp
   DE
{
   register int i;
   register tp_t *tpp;
   static int once;

   DEBUG0(D_RESET, "done_cont: called\n");

   if (! opt.notp) {
      tpp = tptr_default();

      for (i=0; i<num_serv; i++)
         if (servers[i]->tpp == 0)
            tptr_set(servers[i], tpp);
   }
   if (once == 0 && server_count(S_READY)) {
      util_ready();	/* this can be annoying */
      once = 1;
   }
   fd_unreset();
   /*
   **  If any adds were attempted while the one we just finished was
   **  going on, they are queued up and ready to go now.  Begin again.
   */
   if (server_count(S_BLOCKED)) {
      DEBUG0(D_RESET, "done_cont: processing blocked adds...\n");
      for (i=0; i<num_serv; i++)
         if (servers[i]->state == S_BLOCKED)
            poke(servers[i]);
   }
}

/************************************************************************
*									*
*   cliz_sync								*
*									*
*	Send a request to all X servers (in current destination) and	*
*	wait for a reply from each, then invoke the given procedure.	*
*									*
************************************************************************/
void
cliz_sync
   AL((proc))
   DB cont_proc_t proc
   DE
{
   proto_GetInputFocus();
   rx_begin_repeat(zrxqp);
   rx_queue(zrxqp, X_GetInputFocus, 0, zb_seqno(), 0);
   rx_end_repeat(zrxqp);

   cx_push(zrxqp, proc);
   cx_push(zrxqp, sync_cont);

   fd_reset(-1);
}

static void
sync_cont
   AL((dfp))
   DB df_t *dfp
   DE
{
   fd_unreset();
}

/************************************************************************
*									*
*   cliz_root_expose							*
*									*
*	Paint the virtual root window in response to an expose		*
*	event.								*
*									*
************************************************************************/
void
cliz_root_expose
   AL((sp, chp, n))
   DB server_t *sp
   DD chunk_t *chp
   DD int n		/* number of events in chunk */
   DE
{
   register int i;
   register xEvent *ev;

   ev = (xEvent *)buf_data(chp);

   zb_dest(ZD_SERV, sp);

   for (i=0; i<n; ev++, i++)
      proto_ClearArea(	ev->u.expose.window,
			ev->u.expose.x,
			ev->u.expose.y,
			ev->u.expose.width,
			ev->u.expose.height,
			xFalse);
}

/************************************************************************
*									*
*   cliz_grab_keyboard							*
*   cliz_ungrab_keyboard						*
*									*
************************************************************************/
void
cliz_grab_keyboard
   AL((xp, sp))
   DB xmc_t *xp
   DD server_t *sp
   DE
{
   register cx_t *cxp;

   zb_dest(ZD_SERV, sp);

   proto_GrabKeyboard(	vscreen.shellroot,
			xTrue,
			GrabModeAsync,
			GrabModeAsync,
			CurrentTime);
   df_put_p((void *)xp);
   rx_queue(zrxqp, X_GrabKeyboard, 0, zb_seqno(), grab_keyboard_reply);

   fd_pending(sp->fd);
}

static void
grab_keyboard_reply
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp
   DD client_t *cp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DD chunk_t *chp
   DE
{
   register xmc_t *xp = (xmc_t *)df_get_p(dfp);
   register xGrabKeyboardReply *rp = (xGrabKeyboardReply *)buf_data(chp);

   if (rp->type == X_Error) {
      warn("cliz.c/grab_keyboard_reply: request failed\n");
      dprx_error((xError *)rp);
      return;
   }
#ifdef GARBAGE_TODO
   if (rp->status == Success && cxp->u.grab.ok)
      xmc_grab_succeeded(cxp->u.grab.xp, sp);
   else {
      if (cxp->u.grab.ok) {
         /* release pointer TODO */
      }
      else if (rp->status == Success) {
         /* release keyboard TODO */
      }
      xmc_grab_failed(cxp->u.grab.xp, sp);
   }
#endif
   fd_unpending();	/* release block on server input */
}

void
cliz_ungrab_keyboard
   AL((sp))
   DB server_t *sp
   DE
{
   register chunk_t *chp;

   zb_dest(ZD_SERV, sp);

   proto_UngrabKeyboard(CurrentTime);
}

/************************************************************************
*									*
*   cliz_grab_pointer							*
*   cliz_ungrab_pointer							*
*									*
************************************************************************/
void
cliz_grab_pointer
   AL((xp, sp))
   DB xmc_t *xp
   DD server_t *sp
   DE
{
   register cx_t *cxp;

   zb_dest(ZD_SERV, sp);

   proto_GrabPointer(	vscreen.shellroot,
			xTrue,
			0,
			GrabModeAsync,
			GrabModeAsync,
			None,
			None,
			CurrentTime);
   df_put_p((void *)xp);
   rx_queue(zrxqp, X_GrabPointer, 0, zb_seqno(), grab_pointer_reply);

   fd_pending(sp->fd);
}

static void
grab_pointer_reply
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp
   DD client_t *cp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DD chunk_t *chp
   DE
{
   register xmc_t *xp = (xmc_t *)df_get_p(dfp);
   register xGrabPointerReply *rp = (xGrabPointerReply *)buf_data(chp);

   if (rp->type == X_Error) {
      warn("cliz.c/grab_pointer_reply: request failed\n");
      dprx_error((xError *)rp);
      return;
   }
#ifdef GARBAGE_TODO
   if (rp->status == Success && cxp->u.grab.ok)
      xmc_grab_succeeded(cxp->u.grab.xp, sp);
   else {
      if (cxp->u.grab.ok) {
         /* release pointer TODO */
      }
      else if (rp->status == Success) {
         /* release keyboard TODO */
      }
      xmc_grab_failed(cxp->u.grab.xp, sp);
   }
#endif
   fd_unpending();	/* release block on server input */
}

void
cliz_ungrab_pointer
   AL((sp))
   DB server_t *sp
   DE
{
   register chunk_t *chp;

   zb_dest(ZD_SERV, sp);

   proto_UngrabPointer(CurrentTime);
}

/************************************************************************
*									*
*   cliz_kill_client							*
*									*
************************************************************************/
void
cliz_kill_client
   AL((cp))
   DB client_t *cp
   DE
{
   zb_dest(ZD_ALLSERV, 0);

   zb_client_death(cp);

   if (cp) {
      sel_client_death(cp);
      inp_client_death(cp);	/* clears input mask */
   }
   if (cp == 0 || cp->state == K_DEAD || cp->closedownmode == DestroyAll) {
      window_client_death(cp);
      gc_client_death(cp);
      cursor_client_death(cp);
      pixmap_client_death(cp);
      font_client_death(cp);
      color_client_death(cp);
   }
   if (cp) {
      /*
      **  if it's dead, the connection is already closed
      */
      if (cp->state != K_DEAD)
         main_close_conn(cp->fd); /* calls fd_clear which drops server grabs */
      client_free(cp);
   }
}

/************************************************************************
*									*
*   cliz_scan_fontprops							*
*									*
*	This is a hack.							*
*									*
*	This routine intercepts atoms allocated by X servers, and not	*
*	known by xmx, which occur as fontprop names or values in	*
*	QueryFont and ListFontsWithInfo requests.			*
*									*
*	If a reply to one of those requests contains unknown atoms,	*
*	a GetAtomName request is generated for each one and sent back	*
*	to the server - before the reply is returned to the client.	*
*	All activity stops until all GetAtomName requests are answered.	*
*	As the replies come in, the string names are intern'd by xmx	*
*	and the resulting atoms replace the old values in the font	*
*	reply.  When all atoms are accounted for, the reply is finally	*
*	forwarded to the client.					*
*									*
************************************************************************/
/*
**	boolean array - if atom is fontprop, is it of type Atom?
*/
static atom_t typeatom[] = {
/*  0			*/	0,
/*  1 PRIMARY		*/	0,
/*  2 SECONDARY		*/	0,
/*  3 ARC		*/	0,
/*  4 ATOM		*/	0,
/*  5 BITMAP		*/	0,
/*  6 CARDINAL		*/	0,
/*  7 COLORMAP		*/	0,
/*  8 CURSOR		*/	0,
/*  9 CUT_BUFFER0	*/	0,
/* 10 CUT_BUFFER1	*/	0,
/* 11 CUT_BUFFER2	*/	0,
/* 12 CUT_BUFFER3	*/	0,
/* 13 CUT_BUFFER4	*/	0,
/* 14 CUT_BUFFER5	*/	0,
/* 15 CUT_BUFFER6	*/	0,
/* 16 CUT_BUFFER7	*/	0,
/* 17 DRAWABLE		*/	0,
/* 18 FONT		*/	1,
/* 19 INTEGER		*/	0,
/* 20 PIXMAP		*/	0,
/* 21 POINT		*/	0,
/* 22 RECTANGLE		*/	0,
/* 23 RESOURCE_MANAGER	*/	0,
/* 24 RGB_COLOR_MAP	*/	0,
/* 25 RGB_BEST_MAP	*/	0,
/* 26 RGB_BLUE_MAP	*/	0,
/* 27 RGB_DEFAULT_MAP	*/	0,
/* 28 RGB_GRAY_MAP	*/	0,
/* 29 RGB_GREEN_MAP	*/	0,
/* 30 RGB_RED_MAP	*/	0,
/* 31 STRING		*/	0,
/* 32 VISUALID		*/	0,
/* 33 WINDOW		*/	0,
/* 34 WM_COMMAND	*/	0,
/* 35 WM_HINTS		*/	0,
/* 36 WM_CLIENT_MACHINE	*/	0,
/* 37 WM_ICON_NAME	*/	0,
/* 38 WM_ICON_SIZE	*/	0,
/* 39 WM_NAME		*/	0,
/* 40 WM_NORMAL_HINTS	*/	0,
/* 41 WM_SIZE_HINTS	*/	0,
/* 42 WM_ZOOM_HINTS	*/	0,
/* 43 MIN_SPACE		*/	0,
/* 44 NORM_SPACE	*/	0,
/* 45 MAX_SPACE		*/	0,
/* 46 END_SPACE		*/	0,
/* 47 SUPERSCRIPT_X	*/	0,
/* 48 SUPERSCRIPT_Y	*/	0,
/* 49 SUBSCRIPT_X	*/	0,
/* 50 SUBSCRIPT_Y	*/	0,
/* 51 UNDERLINE_POSITION*/	0,
/* 52 UNDERLINE_THICKNESS*/	0,
/* 53 STRIKEOUT_ASCENT	*/	0,
/* 54 STRIKEOUT_DESCENT	*/	0,
/* 55 ITALIC_ANGLE	*/	0,
/* 56 X_HEIGHT		*/	0,
/* 57 QUAD_WIDTH	*/	0,
/* 58 WEIGHT		*/	0,
/* 59 POINT_SIZE	*/	0,
/* 60 RESOLUTION	*/	0,
/* 61 COPYRIGHT		*/	1,
/* 62 NOTICE		*/	1,
/* 63 FONT_NAME		*/	1,
/* 64 FAMILY_NAME	*/	1,
/* 65 FULL_NAME		*/	1,
/* 66 CAP_HEIGHT	*/	0,
/* 67 WM_CLASS		*/	0,
/* 68 WM_TRANSIENT_FOR	*/	0,
	/* XLFD property atoms */
/* 69 FOUNDRY		*/	1,
/* 70 FAMILY_NAME	*/	1,
/* 71 WEIGHT_NAME	*/	1,
/* 72 SLANT		*/	1,
/* 73 SETWIDTH_NAME	*/	1,
/* 74 ADD_STYLE_NAME	*/	1,
/* 75 PIXEL_SIZE	*/	0,
/* 76 POINT_SIZE	*/	0,
/* 77 RESOLUTION_X	*/	0,
/* 78 RESOLUTION_Y	*/	0,
/* 79 SPACING		*/	1,
/* 80 AVERAGE_WIDTH	*/	0,
/* 81 CHARSET_REGISTRY	*/	1,
/* 82 CHARSET_ENCODING	*/	1,
/* 83 MIN_SPACE		*/	0,
/* 84 NORM_SPACE	*/	0,
/* 85 MAX_SPACE		*/	0,
/* 86 END_SPACE		*/	0,
/* 87 AVG_CAPITAL_WIDTH	*/	0,
/* 88 AVG_LOWERCASE_WIDTH*/	0,
/* 89 QUAD_WIDTH	*/	0,
/* 90 FIGURE_WIDTH	*/	0,
/* 91 SUPERSCRIPT_X	*/	0,
/* 92 SUPERSCRIPT_Y	*/	0,
/* 93 SUBSCRIPT_X	*/	0,
/* 94 SUBSCRIPT_Y	*/	0,
/* 95 SUPERSCRIPT_SIZE	*/	0,
/* 96 SUBSCRIPT_SIZE	*/	0,
/* 97 SMALL_CAP_SIZE	*/	0,
/* 98 UNDERLINE_POSITION*/	0,
/* 99 UNDERLINE_THICKNESS*/	0,
/*100 STRIKEOUT_ASCENT	*/	0,
/*101 STRIKEOUT_DESCENT	*/	0,
/*102 ITALIC_ANGLE	*/	0,
/*103 CAP_HEIGHT	*/	0,
/*104 X_HEIGHT		*/	0,
/*105 RELATIVE_SETWIDTH	*/	0,
/*106 RELATIVE_WEIGHT	*/	0,
/*107 WEIGHT		*/	0,
/*108 RESOLUTION	*/	0,
/*109 FACE_NAME		*/	1,
/*110 COPYRIGHT		*/	1,
/*111 NOTICE		*/	1,
/*112 DESTINATION	*/	0,
};

/*
**  cliz_scan_fontprops
**
**	Scan a font query reply for alien atoms.  If none, send it on.
**	Otherwise, put it on hold and find out what those atoms are
**	first.
*/
void
cliz_scan_fontprops
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp
   DD client_t *cp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DD chunk_t *chp
   DE
{
   register int i, off, nreq;
   atom_t atom;
   xQueryFontReply *rp; /* this is the same for xListFontsWithInfoReply */
   xFontProp *fp;

   rp = (xQueryFontReply *)buf_data(chp);
   if (rp->type == X_Reply && (
		major == X_QueryFont ||
		((xListFontsWithInfoReply *)buf_data(chp))->nameLength)) {
      nreq = 0;
      fp = (xFontProp *)(rp + 1);
      off = sz_xQueryFontReply;
      for (i=0; i<(int)rp->nFontProps; i++) {
         /*
         **  Is fontprop name an atom not yet seen?
         **
         **  (need to record atoms we've seen and check for them here)
         */
         if (fp[i].name > MAX_PREDEF_ATOM) {
            zb_dest(ZD_SERV, sp);
            proto_GetAtomName(fp[i].name);
            df_put_p((void *)chp);
            df_put_i(off);
            df_put_i(1);		/* boolean: is name? */
            rx_queue(zrxqp, X_GetAtomName, 0, zb_seqno(), fontprop_reply);
            nreq++;
         }
         /*
         **  Is fontprop value an atom not yet seen?
         */
	 else if (typeatom[fp[i].name] && fp[i].value > MAX_PREDEF_ATOM) {
            zb_dest(ZD_SERV, sp);
            proto_GetAtomName(fp[i].value);
            df_put_p((void *)chp);
            df_put_i(off);
            df_put_i(0);		/* boolean: is name? */
            rx_queue(zrxqp, X_GetAtomName, 0, zb_seqno(), fontprop_reply);
            nreq++;
         }
         off += sz_xFontProp;
      }
      if (nreq) {
         queue_hold(cp->qp);	/* keep client queue from flushing */
         fd_pending(sp->fd);	/* block on it */
         /*
         **  do this when done
         */
         df_put_p((void *)cp->qp);
         cx_push(zrxqp, fontprop_done_cont);
      }
   }
   queue_add(cp->qp, chp);
}

/*
**	fontprop_reply
**
**	Handle a GetAtomName reply issued to retrieve an unknown atom
**	from a fontprop buried in a font query reply.
*/
static void
fontprop_reply
   AL((dfp, sp, cp, major, minor, seqno, chp))
   DB df_t *dfp
   DD server_t *sp
   DD client_t *cp
   DD u8_t major
   DD u16_t minor
   DD u16_t seqno
   DD chunk_t *chp
   DE
{
   atom_t atom;
   xFontProp *fp;
   xGetAtomNameReply *rp = (xGetAtomNameReply *)buf_data(chp);
   chunk_t *ochp = (chunk_t *)df_get_p(dfp);
   int off = df_get_i(dfp);
   int isname = df_get_i(dfp);

   if (rp->type == X_Error) {
      warn("fontprop_reply: GetAtomName failed for server [%d] [%s]\n",
							sp->fd, sp->tag);
      dprx_error((xError *)rp);
      return;
   }
   atom = atom_store((char *)(rp+1), rp->nameLength, 0);
   fp = (xFontProp *)(buf_data(ochp) + off);
   if (isname) {
      fp->name = atom;
      /*
      **  is the type of this fontprop an atom, and if so, is the
      **  value an unknown one?
      */
      if (	fp->name <= MAX_KNOWN_ATOM &&
		typeatom[fp->name] &&
		fp->value > MAX_PREDEF_ATOM) {

         zb_dest(ZD_SERV, sp);
         proto_GetAtomName(fp->value);
         df_put_p((void *)ochp);
         df_put_i(off);
         df_put_i(0);		/* boolean: is name? */
         rx_queue(zrxqp, X_GetAtomName, 0, zb_seqno(), fontprop_reply);
      }
   }
   else
      fp->value = atom;
}

/*
**	fontprop_done_cont
**
**	Release reply and unblock main loop.
*/
static void
fontprop_done_cont
   AL((dfp))
   DB df_t *dfp
   DE
{
   register queue_t *qp = (queue_t *)df_get_p(dfp);

   fd_unpending();
   queue_release(qp);
}
