/*
 * Copyright 1991-1997, Brown University, Providence, RI.
 * 
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose other than its incorporation into a
 * commercial product is hereby granted without fee, provided that the
 * above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Brown University not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 * 
 * BROWN UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR ANY
 * PARTICULAR PURPOSE.  IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
/************************************************************************
*									*
*	pathfind.c							*
*									*
*	Find a file in a path.						*
*									*
************************************************************************/
#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <pwd.h>
#include <grp.h>
#include <string.h>
#include <unistd.h>

char *
pathfind
#ifdef NEED_PROTO
   (char *path, char *name, char* mode)
#else
   (path, name, mode)
   char *path, *name, *mode;
#endif
{
   register int i, j, okay, chkgrp;
   register char *cp, *mp;
   register uid_t uid;
   register gid_t gid;
   static char pbuf[MAXPATHLEN];
   DIR *dirp;
   struct dirent *dep;
   struct stat st;
   struct passwd *pwd;
   struct group *grp;

   uid = getuid();
   gid = getgid();

   cp = path;

   while (*cp) {
      for (i=0; *cp && *cp != ':'; i++)
         pbuf[i] = *cp++;
      pbuf[i] = '\0';

      if (*cp)
         cp++;

      if (i) {
         if ((dirp = opendir(pbuf)) == 0)
            continue;
         while (dep = readdir(dirp)) {
            if (strcmp(name, dep->d_name) == 0) {
               pbuf[i] = '\0';		/* reset path */
               (void)strcat(pbuf, "/");
               (void)strcat(pbuf, name);

               if (stat(pbuf, &st))
                  continue;

               okay = 1;
               chkgrp = 0;
               for (mp=mode; *mp && okay; mp++) {
                  switch (*mp) {
                     case 'r':
                        if (	st.st_mode & S_IROTH == 0 &&
				(st.st_mode & S_IRUSR == 0 || uid != st.st_uid))
                           if (st.st_mode & S_IRGRP == 0 || gid != st.st_gid)
                              okay = 0;
                           else
                              chkgrp = 1;
                        break;
                     case 'w':
                        if (	st.st_mode & S_IWOTH == 0 &&
				(st.st_mode & S_IWUSR == 0 || uid != st.st_uid))
                           if (st.st_mode & S_IWGRP == 0 || gid != st.st_gid)
                              okay = 0;
                           else
                              chkgrp = 1;
                        break;
                     case 'x':
                        if (	st.st_mode & S_IXOTH == 0 &&
				(st.st_mode & S_IXUSR == 0 || uid != st.st_uid))
                           if (st.st_mode & S_IXGRP == 0 || gid != st.st_gid)
                              okay = 0;
                           else
                              chkgrp = 1;
                        break;
                     case 'f':
                        okay = S_ISREG(st.st_mode);
                        break;
                     case 'b':
                        okay = S_ISBLK(st.st_mode);
                        break;
                     case 'c':
                        okay = S_ISCHR(st.st_mode);
                        break;
                     case 'd':
                        okay = S_ISDIR(st.st_mode);
                        break;
                     case 'p':
                        okay = S_ISFIFO(st.st_mode);
                        break;
                     case 'u':
                        okay = (st.st_mode & S_ISUID != 0);
                        break;
                     case 'g':
                        okay = (st.st_mode & S_ISGID != 0);
                        break;
                     case 'k':
                        okay = (st.st_mode & S_ISVTX != 0);
                        break;
                     case 's':
                        okay = (st.st_size != 0);
                        break;
                  }
               }
               if (okay)
                  if (chkgrp) {
                     grp = getgrgid(st.st_gid);
                     pwd = getpwuid(uid);
                     for (j=0; grp->gr_mem[j]; j++)
                        if (strcmp(grp->gr_mem[j], pwd->pw_name) == 0)
                           return pbuf;
                  }
                  else
                     return pbuf;
            }
         }
         (void)closedir(dirp);
      }
   }
   return 0;
}
