/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.streaming.connectors.kinesis.internals;

import java.io.IOException;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import org.apache.flink.annotation.Internal;
import org.apache.flink.kinesis.shaded.com.amazonaws.services.kinesis.clientlibrary.types.UserRecord;
import org.apache.flink.kinesis.shaded.com.amazonaws.services.kinesis.model.ExpiredIteratorException;
import org.apache.flink.kinesis.shaded.com.amazonaws.services.kinesis.model.GetRecordsResult;
import org.apache.flink.kinesis.shaded.com.amazonaws.services.kinesis.model.Record;
import org.apache.flink.kinesis.shaded.com.amazonaws.services.kinesis.model.ShardIteratorType;
import org.apache.flink.streaming.connectors.kinesis.internals.KinesisDataFetcher;
import org.apache.flink.streaming.connectors.kinesis.metrics.ShardMetricsReporter;
import org.apache.flink.streaming.connectors.kinesis.model.SentinelSequenceNumber;
import org.apache.flink.streaming.connectors.kinesis.model.SequenceNumber;
import org.apache.flink.streaming.connectors.kinesis.model.StreamShardHandle;
import org.apache.flink.streaming.connectors.kinesis.proxy.KinesisProxyInterface;
import org.apache.flink.streaming.connectors.kinesis.serialization.KinesisDeserializationSchema;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Internal
public class ShardConsumer<T>
implements Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(ShardConsumer.class);
    private static final long KINESIS_SHARD_BYTES_PER_SECOND_LIMIT = 0x200000L;
    private final KinesisDeserializationSchema<T> deserializer;
    private final KinesisProxyInterface kinesis;
    private final int subscribedShardStateIndex;
    private final KinesisDataFetcher<T> fetcherRef;
    private final StreamShardHandle subscribedShard;
    private int maxNumberOfRecordsPerFetch;
    private final long fetchIntervalMillis;
    private final boolean useAdaptiveReads;
    private final ShardMetricsReporter shardMetricsReporter;
    private SequenceNumber lastSequenceNum;
    private Date initTimestamp;

    public ShardConsumer(KinesisDataFetcher<T> fetcherRef, Integer subscribedShardStateIndex, StreamShardHandle subscribedShard, SequenceNumber lastSequenceNum, KinesisProxyInterface kinesis, ShardMetricsReporter shardMetricsReporter) {
        this.fetcherRef = (KinesisDataFetcher)Preconditions.checkNotNull(fetcherRef);
        this.subscribedShardStateIndex = (Integer)Preconditions.checkNotNull((Object)subscribedShardStateIndex);
        this.subscribedShard = (StreamShardHandle)Preconditions.checkNotNull((Object)subscribedShard);
        this.lastSequenceNum = (SequenceNumber)Preconditions.checkNotNull((Object)lastSequenceNum);
        this.shardMetricsReporter = (ShardMetricsReporter)Preconditions.checkNotNull((Object)shardMetricsReporter);
        Preconditions.checkArgument((!lastSequenceNum.equals(SentinelSequenceNumber.SENTINEL_SHARD_ENDING_SEQUENCE_NUM.get()) ? 1 : 0) != 0, (Object)"Should not start a ShardConsumer if the shard has already been completely read.");
        this.deserializer = fetcherRef.getClonedDeserializationSchema();
        Properties consumerConfig = fetcherRef.getConsumerConfiguration();
        this.kinesis = kinesis;
        this.maxNumberOfRecordsPerFetch = Integer.valueOf(consumerConfig.getProperty("flink.shard.getrecords.maxrecordcount", Integer.toString(10000)));
        this.fetchIntervalMillis = Long.valueOf(consumerConfig.getProperty("flink.shard.getrecords.intervalmillis", Long.toString(200L)));
        this.useAdaptiveReads = Boolean.valueOf(consumerConfig.getProperty("flink.shard.adaptivereads", Boolean.toString(false)));
        if (lastSequenceNum.equals(SentinelSequenceNumber.SENTINEL_AT_TIMESTAMP_SEQUENCE_NUM.get())) {
            String timestamp = consumerConfig.getProperty("flink.stream.initpos.timestamp");
            try {
                String format = consumerConfig.getProperty("flink.stream.initpos.timestamp.format", "yyyy-MM-dd'T'HH:mm:ss.SSSXXX");
                SimpleDateFormat customDateFormat = new SimpleDateFormat(format);
                this.initTimestamp = customDateFormat.parse(timestamp);
            }
            catch (IllegalArgumentException | NullPointerException exception) {
                throw new IllegalArgumentException(exception);
            }
            catch (ParseException exception) {
                this.initTimestamp = new Date((long)(Double.parseDouble(timestamp) * 1000.0));
            }
        } else {
            this.initTimestamp = null;
        }
    }

    protected String getShardIterator(SequenceNumber sequenceNumber) throws Exception {
        if (SentinelSequenceNumber.isSentinelSequenceNumber(sequenceNumber)) {
            return this.getShardIteratorForSentinel(sequenceNumber);
        }
        return this.getShardIteratorForRealSequenceNumber(sequenceNumber);
    }

    protected String getShardIteratorForSentinel(SequenceNumber sentinelSequenceNumber) throws InterruptedException {
        String nextShardItr;
        if (sentinelSequenceNumber.equals(SentinelSequenceNumber.SENTINEL_LATEST_SEQUENCE_NUM.get())) {
            nextShardItr = this.subscribedShard.isClosed() ? null : this.kinesis.getShardIterator(this.subscribedShard, ShardIteratorType.LATEST.toString(), null);
        } else if (sentinelSequenceNumber.equals(SentinelSequenceNumber.SENTINEL_EARLIEST_SEQUENCE_NUM.get())) {
            nextShardItr = this.kinesis.getShardIterator(this.subscribedShard, ShardIteratorType.TRIM_HORIZON.toString(), null);
        } else if (sentinelSequenceNumber.equals(SentinelSequenceNumber.SENTINEL_SHARD_ENDING_SEQUENCE_NUM.get())) {
            nextShardItr = null;
        } else if (sentinelSequenceNumber.equals(SentinelSequenceNumber.SENTINEL_AT_TIMESTAMP_SEQUENCE_NUM.get())) {
            nextShardItr = this.kinesis.getShardIterator(this.subscribedShard, ShardIteratorType.AT_TIMESTAMP.toString(), this.initTimestamp);
        } else {
            throw new RuntimeException("Unknown sentinel type: " + sentinelSequenceNumber);
        }
        return nextShardItr;
    }

    protected String getShardIteratorForRealSequenceNumber(SequenceNumber sequenceNumber) throws Exception {
        if (sequenceNumber.isAggregated()) {
            return this.getShardIteratorForAggregatedSequenceNumber(sequenceNumber);
        }
        return this.kinesis.getShardIterator(this.subscribedShard, ShardIteratorType.AFTER_SEQUENCE_NUMBER.toString(), sequenceNumber.getSequenceNumber());
    }

    protected String getShardIteratorForAggregatedSequenceNumber(SequenceNumber sequenceNumber) throws Exception {
        String itrForLastAggregatedRecord = this.kinesis.getShardIterator(this.subscribedShard, ShardIteratorType.AT_SEQUENCE_NUMBER.toString(), sequenceNumber.getSequenceNumber());
        GetRecordsResult getRecordsResult = this.getRecords(itrForLastAggregatedRecord, 1);
        List<UserRecord> fetchedRecords = ShardConsumer.deaggregateRecords(getRecordsResult.getRecords(), this.subscribedShard.getShard().getHashKeyRange().getStartingHashKey(), this.subscribedShard.getShard().getHashKeyRange().getEndingHashKey());
        long lastSubSequenceNum = sequenceNumber.getSubSequenceNumber();
        for (UserRecord record : fetchedRecords) {
            if (record.getSubSequenceNumber() <= lastSubSequenceNum) continue;
            this.deserializeRecordForCollectionAndUpdateState(record);
        }
        return getRecordsResult.getNextShardIterator();
    }

    @Override
    public void run() {
        try {
            String nextShardItr = this.getShardIterator(this.lastSequenceNum);
            long processingStartTimeNanos = System.nanoTime();
            while (this.isRunning()) {
                if (nextShardItr == null) {
                    this.fetcherRef.updateState(this.subscribedShardStateIndex, SentinelSequenceNumber.SENTINEL_SHARD_ENDING_SEQUENCE_NUM.get());
                    break;
                }
                this.shardMetricsReporter.setMaxNumberOfRecordsPerFetch(this.maxNumberOfRecordsPerFetch);
                GetRecordsResult getRecordsResult = this.getRecords(nextShardItr, this.maxNumberOfRecordsPerFetch);
                List<Record> aggregatedRecords = getRecordsResult.getRecords();
                int numberOfAggregatedRecords = aggregatedRecords.size();
                this.shardMetricsReporter.setNumberOfAggregatedRecords(numberOfAggregatedRecords);
                List<UserRecord> fetchedRecords = ShardConsumer.deaggregateRecords(aggregatedRecords, this.subscribedShard.getShard().getHashKeyRange().getStartingHashKey(), this.subscribedShard.getShard().getHashKeyRange().getEndingHashKey());
                long recordBatchSizeBytes = 0L;
                for (UserRecord record : fetchedRecords) {
                    recordBatchSizeBytes += (long)record.getData().remaining();
                    this.deserializeRecordForCollectionAndUpdateState(record);
                }
                int numberOfDeaggregatedRecords = fetchedRecords.size();
                this.shardMetricsReporter.setNumberOfDeaggregatedRecords(numberOfDeaggregatedRecords);
                nextShardItr = getRecordsResult.getNextShardIterator();
                long adjustmentEndTimeNanos = this.adjustRunLoopFrequency(processingStartTimeNanos, System.nanoTime());
                long runLoopTimeNanos = adjustmentEndTimeNanos - processingStartTimeNanos;
                this.maxNumberOfRecordsPerFetch = this.adaptRecordsToRead(runLoopTimeNanos, fetchedRecords.size(), recordBatchSizeBytes, this.maxNumberOfRecordsPerFetch);
                this.shardMetricsReporter.setRunLoopTimeNanos(runLoopTimeNanos);
                processingStartTimeNanos = adjustmentEndTimeNanos;
            }
        }
        catch (Throwable t) {
            this.fetcherRef.stopWithError(t);
        }
    }

    protected long adjustRunLoopFrequency(long processingStartTimeNanos, long processingEndTimeNanos) throws InterruptedException {
        long processingTimeNanos;
        long sleepTimeMillis;
        long endTimeNanos = processingEndTimeNanos;
        if (this.fetchIntervalMillis != 0L && (sleepTimeMillis = this.fetchIntervalMillis - (processingTimeNanos = processingEndTimeNanos - processingStartTimeNanos) / 1000000L) > 0L) {
            Thread.sleep(sleepTimeMillis);
            endTimeNanos = System.nanoTime();
            this.shardMetricsReporter.setSleepTimeMillis(sleepTimeMillis);
        }
        return endTimeNanos;
    }

    private int adaptRecordsToRead(long runLoopTimeNanos, int numRecords, long recordBatchSizeBytes, int maxNumberOfRecordsPerFetch) {
        if (this.useAdaptiveReads && numRecords != 0 && runLoopTimeNanos != 0L) {
            long averageRecordSizeBytes = recordBatchSizeBytes / (long)numRecords;
            double loopFrequencyHz = 1.0E9 / (double)runLoopTimeNanos;
            double bytesPerRead = 2097152.0 / loopFrequencyHz;
            maxNumberOfRecordsPerFetch = (int)(bytesPerRead / (double)averageRecordSizeBytes);
            maxNumberOfRecordsPerFetch = Math.max(1, Math.min(maxNumberOfRecordsPerFetch, 10000));
            this.shardMetricsReporter.setAverageRecordSizeBytes(averageRecordSizeBytes);
            this.shardMetricsReporter.setLoopFrequencyHz(loopFrequencyHz);
            this.shardMetricsReporter.setBytesPerRead(bytesPerRead);
        }
        return maxNumberOfRecordsPerFetch;
    }

    private boolean isRunning() {
        return !Thread.interrupted();
    }

    private void deserializeRecordForCollectionAndUpdateState(UserRecord record) throws IOException {
        ByteBuffer recordData = record.getData();
        byte[] dataBytes = new byte[recordData.remaining()];
        recordData.get(dataBytes);
        long approxArrivalTimestamp = record.getApproximateArrivalTimestamp().getTime();
        T value = this.deserializer.deserialize(dataBytes, record.getPartitionKey(), record.getSequenceNumber(), approxArrivalTimestamp, this.subscribedShard.getStreamName(), this.subscribedShard.getShard().getShardId());
        SequenceNumber collectedSequenceNumber = record.isAggregated() ? new SequenceNumber(record.getSequenceNumber(), record.getSubSequenceNumber()) : new SequenceNumber(record.getSequenceNumber());
        this.fetcherRef.emitRecordAndUpdateState(value, approxArrivalTimestamp, this.subscribedShardStateIndex, collectedSequenceNumber);
        this.lastSequenceNum = collectedSequenceNumber;
    }

    private GetRecordsResult getRecords(String shardItr, int maxNumberOfRecords) throws Exception {
        GetRecordsResult getRecordsResult = null;
        while (getRecordsResult == null) {
            try {
                getRecordsResult = this.kinesis.getRecords(shardItr, maxNumberOfRecords);
                Long millisBehindLatest = getRecordsResult.getMillisBehindLatest();
                if (millisBehindLatest == null) continue;
                this.shardMetricsReporter.setMillisBehindLatest(millisBehindLatest);
            }
            catch (ExpiredIteratorException eiEx) {
                LOG.warn("Encountered an unexpected expired iterator {} for shard {}; refreshing the iterator ...", (Object)shardItr, (Object)this.subscribedShard);
                shardItr = this.getShardIterator(this.lastSequenceNum);
                if (this.fetchIntervalMillis == 0L) continue;
                Thread.sleep(this.fetchIntervalMillis);
            }
        }
        return getRecordsResult;
    }

    protected static List<UserRecord> deaggregateRecords(List<Record> records, String startingHashKey, String endingHashKey) {
        return UserRecord.deaggregate(records, new BigInteger(startingHashKey), new BigInteger(endingHashKey));
    }
}

