/*
 * Decompiled with CFR 0.152.
 */
package com.amazonaws.services.kinesisanalytics.flink.connectors.producer.impl;

import com.amazonaws.services.kinesisanalytics.flink.connectors.config.ProducerConfigConstants;
import com.amazonaws.services.kinesisanalytics.flink.connectors.exception.FlinkKinesisFirehoseException;
import com.amazonaws.services.kinesisanalytics.flink.connectors.exception.RecordCouldNotBeSentException;
import com.amazonaws.services.kinesisanalytics.flink.connectors.exception.TimeoutExpiredException;
import com.amazonaws.services.kinesisanalytics.flink.connectors.producer.IProducer;
import com.amazonaws.services.kinesisfirehose.AmazonKinesisFirehose;
import com.amazonaws.services.kinesisfirehose.model.AmazonKinesisFirehoseException;
import com.amazonaws.services.kinesisfirehose.model.PutRecordBatchRequest;
import com.amazonaws.services.kinesisfirehose.model.PutRecordBatchResponseEntry;
import com.amazonaws.services.kinesisfirehose.model.PutRecordBatchResult;
import com.amazonaws.services.kinesisfirehose.model.Record;
import com.amazonaws.services.kinesisfirehose.model.ServiceUnavailableException;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.util.ArrayDeque;
import java.util.Properties;
import java.util.Queue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;
import org.apache.commons.lang3.RandomUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public class FirehoseProducer<O extends UserRecordResult, R extends Record>
implements IProducer<O, R> {
    private static final Logger LOGGER = LoggerFactory.getLogger(FirehoseProducer.class);
    private final int maxBufferSize;
    private final long bufferTimeoutInMillis;
    private final long bufferFullWaitTimeoutInMillis;
    private final long bufferTimeoutBetweenFlushes;
    private final int numberOfRetries;
    private final long maxBackOffInMillis;
    private final long baseBackOffInMillis;
    private final long maxOperationTimeoutInMillis;
    private final AmazonKinesisFirehose firehoseClient;
    private final String deliveryStream;
    private final ExecutorService flusher;
    @GuardedBy(value="this")
    private final Object producerBufferLock = new Object();
    private volatile Queue<Record> producerBuffer;
    private volatile Queue<Record> flusherBuffer;
    private volatile long lastSucceededFlushTimestamp;
    private volatile boolean isDestroyed;
    private volatile boolean syncFlush;
    private volatile boolean isFlusherFailed;

    public FirehoseProducer(String deliveryStream, AmazonKinesisFirehose firehoseClient, Properties configProps) {
        this.firehoseClient = (AmazonKinesisFirehose)Validate.notNull((Object)firehoseClient, (String)"Kinesis Firehose client cannot be null", (Object[])new Object[0]);
        this.deliveryStream = (String)Validate.notBlank((CharSequence)deliveryStream, (String)"Kinesis Firehose delivery stream cannot be null or empty.", (Object[])new Object[0]);
        Validate.notNull((Object)configProps, (String)"Firehose producer configuration properties cannot be null", (Object[])new Object[0]);
        this.maxBufferSize = Integer.valueOf(configProps.getProperty("firehose.producer.batch.size", String.valueOf(500)));
        Validate.isTrue((this.maxBufferSize <= 0 || this.maxBufferSize <= 500 ? 1 : 0) != 0, (String)String.format("Buffer size cannot be <= 0 or > %s", 500), (Object[])new Object[0]);
        this.bufferTimeoutInMillis = Long.valueOf(configProps.getProperty("firehose.producer.buffer.timeout", String.valueOf(ProducerConfigConstants.DEFAULT_MAX_BUFFER_TIMEOUT)));
        Validate.isTrue((this.bufferTimeoutInMillis >= 0L ? 1 : 0) != 0, (String)"Flush timeout should be > 0.", (Object[])new Object[0]);
        this.numberOfRetries = Integer.valueOf(configProps.getProperty("firehose.producer.buffer.flush.retries", String.valueOf(10)));
        Validate.isTrue((this.numberOfRetries >= 0 ? 1 : 0) != 0, (String)"Number of retries cannot be negative.", (Object[])new Object[0]);
        this.bufferFullWaitTimeoutInMillis = Long.valueOf(configProps.getProperty("firehose.producer.buffer.full.wait.timeout", String.valueOf(100L)));
        Validate.isTrue((this.bufferFullWaitTimeoutInMillis >= 0L ? 1 : 0) != 0, (String)"Buffer full waiting timeout should be > 0.", (Object[])new Object[0]);
        this.bufferTimeoutBetweenFlushes = Long.valueOf(configProps.getProperty("firehose.producer.buffer.flush.timeout", String.valueOf(50L)));
        Validate.isTrue((this.bufferTimeoutBetweenFlushes >= 0L ? 1 : 0) != 0, (String)"Interval between flushes cannot be negative.", (Object[])new Object[0]);
        this.maxBackOffInMillis = Long.valueOf(configProps.getProperty("firehose.producer.buffer.max.backoff", String.valueOf(100L)));
        Validate.isTrue((this.maxBackOffInMillis >= 0L ? 1 : 0) != 0, (String)"Max backoff timeout should be > 0.", (Object[])new Object[0]);
        this.baseBackOffInMillis = Long.valueOf(configProps.getProperty("firehose.producer.buffer.base.backoff", String.valueOf(10L)));
        Validate.isTrue((this.baseBackOffInMillis >= 0L ? 1 : 0) != 0, (String)"Base backoff timeout should be > 0.", (Object[])new Object[0]);
        this.maxOperationTimeoutInMillis = Long.valueOf(configProps.getProperty("firehose.producer.operation.timeout", String.valueOf(ProducerConfigConstants.DEFAULT_MAX_OPERATION_TIMEOUT)));
        Validate.isTrue((this.maxOperationTimeoutInMillis >= 0L ? 1 : 0) != 0, (String)"Max operation timeout should be > 0.", (Object[])new Object[0]);
        this.producerBuffer = new ArrayDeque<Record>(this.maxBufferSize);
        this.flusherBuffer = new ArrayDeque<Record>(this.maxBufferSize);
        this.flusher = Executors.newSingleThreadExecutor(new ThreadFactoryBuilder().setDaemon(false).setNameFormat("kda-writer-thread-%d").build());
        this.flusher.submit(() -> this.flushBuffer());
    }

    @Override
    public ListenableFuture<O> addUserRecord(R record) throws Exception {
        return this.addUserRecord(record, this.maxOperationTimeoutInMillis);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public ListenableFuture<O> addUserRecord(R record, long timeoutInMillis) throws TimeoutExpiredException, InterruptedException {
        Validate.notNull(record, (String)"Record cannot be null.", (Object[])new Object[0]);
        Validate.isTrue((timeoutInMillis > 0L ? 1 : 0) != 0, (String)"Operation timeout should be > 0.", (Object[])new Object[0]);
        long operationTimeoutInNanos = TimeUnit.MILLISECONDS.toNanos(timeoutInMillis);
        Object object = this.producerBufferLock;
        synchronized (object) {
            long lastTimestamp = System.nanoTime();
            while (this.producerBuffer.size() >= this.maxBufferSize) {
                if (System.nanoTime() - lastTimestamp >= operationTimeoutInNanos) {
                    throw new TimeoutExpiredException("Timeout has expired for the given operation");
                }
                if (this.flusherBuffer.isEmpty()) {
                    this.producerBufferLock.notify();
                }
                this.producerBufferLock.wait(this.bufferFullWaitTimeoutInMillis);
            }
            this.producerBuffer.offer((Record)record);
            if (this.producerBuffer.size() >= this.maxBufferSize && this.flusherBuffer.isEmpty()) {
                this.producerBufferLock.notify();
            }
        }
        UserRecordResult recordResult = new UserRecordResult().setSuccessful(true);
        SettableFuture<UserRecordResult> futureResult = SettableFuture.create();
        futureResult.set(recordResult);
        return futureResult;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void flushBuffer() {
        this.lastSucceededFlushTimestamp = System.nanoTime();
        long bufferTimeoutInNanos = TimeUnit.MILLISECONDS.toNanos(this.bufferTimeoutInMillis);
        while (true) {
            boolean timeoutFlush = System.nanoTime() - this.lastSucceededFlushTimestamp >= bufferTimeoutInNanos;
            Object object = this.producerBufferLock;
            synchronized (object) {
                Validate.validState((boolean)this.flusherBuffer.isEmpty());
                if (this.isDestroyed) {
                    return;
                }
                if (!(this.syncFlush || this.producerBuffer.size() >= this.maxBufferSize || timeoutFlush && this.producerBuffer.size() > 0)) {
                    try {
                        this.producerBufferLock.wait(this.bufferTimeoutBetweenFlushes);
                    }
                    catch (InterruptedException e) {
                        LOGGER.info("An interrupted exception has been thrown, while trying to sleep and release the lock during a flush.", (Throwable)e);
                    }
                    continue;
                }
                Queue<Record> tmpBuffer = this.flusherBuffer;
                this.flusherBuffer = this.producerBuffer;
                this.producerBuffer = tmpBuffer;
                this.producerBufferLock.notify();
            }
            try {
                this.submitBatchWithRetry(this.flusherBuffer);
                ArrayDeque<Record> emptyFlushBuffer = new ArrayDeque<Record>(this.maxBufferSize);
                Object e = this.producerBufferLock;
                synchronized (e) {
                    Validate.validState((!this.flusherBuffer.isEmpty() ? 1 : 0) != 0);
                    this.flusherBuffer = emptyFlushBuffer;
                    if (this.syncFlush) {
                        this.syncFlush = false;
                        this.producerBufferLock.notify();
                    }
                }
            }
            catch (Exception ex) {
                String errorMsg = "An error has occurred while trying to send data to Kinesis Firehose.";
                if (ex instanceof AmazonKinesisFirehoseException && ((AmazonKinesisFirehoseException)ex).getStatusCode() == 413) {
                    LOGGER.error(String.valueOf(errorMsg) + "Batch of records too large. Please try to reduce your batch size by passing " + "FIREHOSE_PRODUCER_BUFFER_MAX_SIZE into your configuration.", (Throwable)ex);
                } else {
                    LOGGER.error(errorMsg, (Throwable)ex);
                }
                Object object2 = this.producerBufferLock;
                synchronized (object2) {
                    this.isFlusherFailed = true;
                }
                throw ex;
            }
        }
    }

    private void submitBatchWithRetry(Queue<Record> records) throws AmazonKinesisFirehoseException, RecordCouldNotBeSentException {
        String warnMessage = null;
        int attempts = 0;
        while (attempts < this.numberOfRetries) {
            try {
                LOGGER.debug("Trying to flush Buffer of size: {} on attempt: {}", (Object)records.size(), (Object)attempts);
                PutRecordBatchResult lastResult = this.submitBatch(records);
                if (lastResult.getFailedPutCount() == null || lastResult.getFailedPutCount() == 0) {
                    this.lastSucceededFlushTimestamp = System.nanoTime();
                    LOGGER.debug("Firehose Buffer has been flushed with size: {} on attempt: {}", (Object)records.size(), (Object)attempts);
                    return;
                }
                PutRecordBatchResponseEntry failedRecord = lastResult.getRequestResponses().stream().filter(r -> r.getRecordId() == null).findFirst().orElse(null);
                warnMessage = String.format("Number of failed records: %s.", lastResult.getFailedPutCount());
                if (failedRecord != null) {
                    warnMessage = String.format("Last Kinesis Firehose putRecordBath encountered an error and failed trying to put: %s records with error: %s - %s.", lastResult.getFailedPutCount(), failedRecord.getErrorCode(), failedRecord.getErrorMessage());
                }
                LOGGER.warn(warnMessage);
                long timeToSleep = RandomUtils.nextLong((long)0L, (long)Math.min(this.maxBackOffInMillis, this.baseBackOffInMillis * 2L * (long)attempts));
                LOGGER.info("Sleeping for: {}ms on attempt: {}", (Object)timeToSleep, (Object)attempts);
                Thread.sleep(timeToSleep);
            }
            catch (ServiceUnavailableException ex) {
                LOGGER.info("Kinesis Firehose has thrown a recoverable exception.", (Throwable)ex);
            }
            catch (InterruptedException e) {
                LOGGER.info("An interrupted exception has been thrown between retry attempts.", (Throwable)e);
            }
            catch (AmazonKinesisFirehoseException ex) {
                throw ex;
            }
            ++attempts;
        }
        throw new RecordCouldNotBeSentException("Exceeded number of attempts! " + warnMessage);
    }

    private PutRecordBatchResult submitBatch(Queue<Record> records) throws AmazonKinesisFirehoseException {
        LOGGER.debug("Sending {} records to Kinesis Firehose on stream: {}", (Object)records.size(), (Object)this.deliveryStream);
        PutRecordBatchResult result = this.firehoseClient.putRecordBatch(new PutRecordBatchRequest().withDeliveryStreamName(this.deliveryStream).withRecords(records));
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void destroy() throws Exception {
        Object object = this.producerBufferLock;
        synchronized (object) {
            this.isDestroyed = true;
            this.producerBuffer = null;
            this.producerBufferLock.notify();
        }
        if (!this.flusher.isShutdown() && !this.flusher.isTerminated()) {
            LOGGER.info("Shutting down scheduled service.");
            this.flusher.shutdown();
            try {
                LOGGER.info("Awaiting executor service termination...");
                this.flusher.awaitTermination(1L, TimeUnit.MINUTES);
            }
            catch (InterruptedException e) {
                LOGGER.error("Error waiting executor writer termination.", (Throwable)e);
                throw new FlinkKinesisFirehoseException("Error waiting executor writer termination.", e);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public boolean isDestroyed() {
        Object object = this.producerBufferLock;
        synchronized (object) {
            return this.isDestroyed;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public int getOutstandingRecordsCount() {
        Object object = this.producerBufferLock;
        synchronized (object) {
            return this.producerBuffer.size() + this.flusherBuffer.size();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public boolean isFlushFailed() {
        Object object = this.producerBufferLock;
        synchronized (object) {
            return this.isFlusherFailed;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void flush() {
        Object object = this.producerBufferLock;
        synchronized (object) {
            this.syncFlush = true;
            this.producerBufferLock.notify();
        }
    }

    @Override
    public void flushSync() {
        while (this.getOutstandingRecordsCount() > 0 && !this.isFlushFailed()) {
            this.flush();
            try {
                Thread.sleep(500L);
            }
            catch (InterruptedException interruptedException) {
                LOGGER.warn("An interruption has happened while trying to flush the buffer synchronously.");
                Thread.currentThread().interrupt();
            }
        }
        if (this.isFlushFailed()) {
            LOGGER.warn("The flusher thread has failed trying to synchronously flush the buffer.");
        }
    }

    public static class UserRecordResult {
        private Throwable exception;
        private boolean successful;

        public Throwable getException() {
            return this.exception;
        }

        public UserRecordResult setException(Throwable exception) {
            this.exception = exception;
            return this;
        }

        public boolean isSuccessful() {
            return this.successful;
        }

        public UserRecordResult setSuccessful(boolean successful) {
            this.successful = successful;
            return this;
        }
    }
}

