/* riemann/event.c -- Riemann C client library
 * SPDX-FileCopyrightText: 2013-2024 Gergely Nagy
 * SPDX-FileContributor: Gergely Nagy
 *
 * SPDX-License-Identifier: EUPL-1.2
 */

#include <riemann/event.h>
#include "riemann/_private.h"

#include <errno.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>

riemann_event_t *
riemann_event_new (void)
{
  riemann_event_t *event;

  event = (riemann_event_t *)malloc (sizeof (riemann_event_t));
  event__init (event);
  return event;
}

void
riemann_event_free (riemann_event_t *event)
{
  if (!event)
    {
      errno = EINVAL;
      return;
    }

  event__free_unpacked (event, NULL);
}

int
riemann_event_set_va (riemann_event_t *event,
                      riemann_event_field_t first_field, va_list aq)
{
  va_list ap;
  riemann_event_field_t field;

  if (!event)
    return -EINVAL;

  va_copy (ap, aq);
  field = first_field;
  do
    {
      switch (field)
        {
        case RIEMANN_EVENT_FIELD_NONE:
          break;

        case RIEMANN_EVENT_FIELD_TIME:
          event->time = (int64_t) va_arg (ap, int64_t);
          event->has_time = 1;
          break;

        case RIEMANN_EVENT_FIELD_TIME_MICROS:
          event->time_micros = (int64_t) va_arg (ap, int64_t);
          event->has_time_micros = 1;
          break;

        case RIEMANN_EVENT_FIELD_STATE:
          _riemann_set_string (&event->state, va_arg (ap, char *));
          break;

        case RIEMANN_EVENT_FIELD_SERVICE:
          _riemann_set_string (&event->service, va_arg (ap, char *));
          break;

        case RIEMANN_EVENT_FIELD_HOST:
          _riemann_set_string (&event->host, va_arg (ap, char *));
          break;

        case RIEMANN_EVENT_FIELD_DESCRIPTION:
          _riemann_set_string (&event->description, va_arg (ap, char *));
          break;

        case RIEMANN_EVENT_FIELD_TAGS:
          {
            char *tag;
            size_t n;

            for (n = 0; n < event->n_tags; n++)
              free (event->tags[n]);
            if (event->tags)
              free (event->tags);
            event->tags = NULL;
            event->n_tags = 0;

            while ((tag = va_arg (ap, char *)) != NULL)
              {
                event->tags = (char **)
                  realloc (event->tags, sizeof (char *) * (event->n_tags + 1));
                event->tags[event->n_tags] = strdup (tag);
                event->n_tags++;
              }

            break;
          }

        case RIEMANN_EVENT_FIELD_TTL:
          event->ttl = (float) va_arg (ap, double);
          event->has_ttl = 1;
          break;

        case RIEMANN_EVENT_FIELD_ATTRIBUTES:
          {
            riemann_attribute_t *attrib;
            size_t n;

            for (n = 0; n < event->n_attributes; n++)
              riemann_attribute_free (event->attributes[n]);
            if (event->attributes)
              free (event->attributes);
            event->attributes = NULL;
            event->n_attributes = 0;

            while ((attrib = va_arg (ap, riemann_attribute_t *)) != NULL)
              {
                event->attributes = (riemann_attribute_t **)
                  realloc (event->attributes,
                           sizeof (riemann_attribute_t *) * (event->n_attributes + 1));
                event->attributes[event->n_attributes] = attrib;
                event->n_attributes++;
              }

            break;
          }

        case RIEMANN_EVENT_FIELD_STRING_ATTRIBUTES:
          {
            const char *key, *value;
            size_t n;

            for (n = 0; n < event->n_attributes; n++)
              riemann_attribute_free (event->attributes[n]);
            if (event->attributes)
              free (event->attributes);
            event->attributes = NULL;
            event->n_attributes = 0;

            while ((key = va_arg (ap, const char *)) != NULL)
              {
                value = va_arg (ap, const char *);

                event->attributes = (riemann_attribute_t **)
                  realloc (event->attributes,
                           sizeof (riemann_attribute_t *) * (event->n_attributes + 1));
                event->attributes[event->n_attributes] =
                  riemann_attribute_create (key, value);
                event->n_attributes++;
              }

            break;
          }

        case RIEMANN_EVENT_FIELD_METRIC_S64:
          event->metric_sint64 = va_arg (ap, int64_t);
          event->has_metric_sint64 = 1;
          break;

        case RIEMANN_EVENT_FIELD_METRIC_D:
          event->metric_d = va_arg (ap, double);
          event->has_metric_d = 1;
          break;

        case RIEMANN_EVENT_FIELD_METRIC_F:
          event->metric_f = (float) va_arg (ap, double);
          event->has_metric_f = 1;
          break;

        default:
          va_end (ap);
          return -EPROTO;
        }

      if (field != RIEMANN_EVENT_FIELD_NONE)
        field = (riemann_event_field_t) va_arg (ap, int);
    }
  while (field != RIEMANN_EVENT_FIELD_NONE);
  va_end (ap);

  return 0;
}

int
riemann_event_set (riemann_event_t *event, ...)
{
  va_list ap;
  int r;
  riemann_event_field_t first_field;

  va_start (ap, event);
  first_field = (riemann_event_field_t) va_arg (ap, int);
  r = riemann_event_set_va (event, first_field, ap);
  va_end (ap);
  return r;
}

int
riemann_event_tag_add (riemann_event_t *event, const char *tag)
{
  if (!event || !tag)
    return -EINVAL;

  event->tags = (char **)
    realloc (event->tags, sizeof (char *) * (event->n_tags + 1));
  event->tags[event->n_tags] = strdup (tag);
  event->n_tags++;

  return 0;
}

int
riemann_event_attribute_add (riemann_event_t *event,
                             riemann_attribute_t *attrib)
{
  if (!event || !attrib)
    return -EINVAL;

  event->attributes = (riemann_attribute_t **)
    realloc (event->attributes,
             sizeof (riemann_attribute_t *) * (event->n_attributes + 1));
  event->attributes[event->n_attributes] = attrib;
  event->n_attributes++;

  return 0;
}

int
riemann_event_string_attribute_add (riemann_event_t *event,
                                    const char *key,
                                    const char *value)
{
  if (!key || !value)
    return -EINVAL;

  return riemann_event_attribute_add (event,
                                      riemann_attribute_create (key, value));
}

riemann_event_t *
riemann_event_create_va (riemann_event_field_t field, va_list aq)
{
  va_list ap;
  riemann_event_t *event;
  int e;

  event = riemann_event_new ();

  va_copy (ap, aq);
  if ((e = riemann_event_set_va (event, field, ap)) != 0)
    {
      va_end (ap);
      riemann_event_free (event);
      errno = -e;
      return NULL;
    }
  va_end (ap);

  return event;
}

riemann_event_t *
riemann_event_create (riemann_event_field_t field, ...)
{
  riemann_event_t *event;
  va_list ap;

  va_start (ap, field);
  event = riemann_event_create_va (field, ap);
  va_end (ap);

  return event;
}

riemann_event_t *
riemann_event_clone (const riemann_event_t *event)
{
  riemann_event_t *clone;
  size_t n;

  if (!event)
    {
      errno = EINVAL;
      return NULL;
    }

  clone = riemann_event_new ();

  /* Copy non-pointer properties */
  clone->time = event->time;
  clone->time_micros = event->time_micros;
  clone->ttl = event->ttl;
  clone->metric_sint64 = event->metric_sint64;
  clone->metric_d = event->metric_d;
  clone->metric_f = event->metric_f;

  /* Copy strings */
  if (event->state)
    clone->state = strdup (event->state);
  if (event->host)
    clone->host = strdup (event->host);
  if (event->service)
    clone->service = strdup (event->service);
  if (event->description)
    clone->description = strdup (event->description);

  /* Copy deeper structures */
  clone->n_tags = event->n_tags;
  clone->tags = (char **) calloc (clone->n_tags, sizeof (char *));
  for (n = 0; n < clone->n_tags; n++)
    clone->tags[n] = strdup (event->tags[n]);

  clone->n_attributes = event->n_attributes;
  clone->attributes = (riemann_attribute_t **)
    calloc (clone->n_attributes, sizeof (riemann_attribute_t *));
  for (n = 0; n < clone->n_attributes; n++)
    clone->attributes[n] = riemann_attribute_clone (event->attributes[n]);

  return clone;
}

int
riemann_event_set_tags (riemann_event_t *event, ...)
{
  va_list ap;
  int r;

  va_start (ap, event);
  r = riemann_event_set_tags_va (event, ap);
  va_end (ap);

  return r;
}

int
riemann_event_set_tags_va (riemann_event_t *event, va_list aq)
{
  va_list ap;
  const char *tag;

  if (!event)
    return -EINVAL;

  for (size_t n = 0; n < event->n_tags; n++)
    free (event->tags[n]);
  if (event->tags)
    free (event->tags);

  event->tags = NULL;
  event->n_tags = 0;

  va_copy (ap, aq);
  while ((tag = va_arg (ap, char *)) != NULL)
    {
      event->tags = (char **)
        realloc (event->tags, sizeof (char *) * (event->n_tags + 1));
      event->tags[event->n_tags] = strdup (tag);
      event->n_tags++;
    }
  va_end (ap);

  return 0;
}

int
riemann_event_set_tags_n (riemann_event_t *event, size_t n_tags, const char **tags)
{
  if (!event)
    return -EINVAL;

  if (n_tags == 0 && tags)
    return -ERANGE;

  if (n_tags > 0 && !tags)
    return -EINVAL;

  for (size_t n = 0; n < event->n_tags; n++)
    free (event->tags[n]);
  if (event->tags)
    free (event->tags);

  event->n_tags = n_tags;
  event->tags = (char **) calloc (n_tags, sizeof (char *));
  for (size_t n = 0; n < n_tags; n++)
    event->tags[n] = strdup (tags[n]);

  return 0;
}

int
riemann_event_set_attributes (riemann_event_t *event, ...)
{
  va_list ap;
  int r;

  va_start (ap, event);
  r = riemann_event_set_attributes_va (event, ap);
  va_end (ap);

  return r;
}

int
riemann_event_set_attributes_va (riemann_event_t *event, va_list aq)
{
  va_list ap;
  riemann_attribute_t *attrib;

  if (!event)
    return -EINVAL;

  for (size_t n = 0; n < event->n_attributes; n++)
    riemann_attribute_free (event->attributes[n]);
  if (event->attributes)
    free (event->attributes);
  event->attributes = NULL;
  event->n_attributes = 0;

  va_copy (ap, aq);
  while ((attrib = va_arg (ap, riemann_attribute_t *)) != NULL)
    {
      event->attributes = (riemann_attribute_t **)
        realloc (event->attributes,
                 sizeof (riemann_attribute_t *) * (event->n_attributes + 1));
      event->attributes[event->n_attributes] = riemann_attribute_clone (attrib);
      event->n_attributes++;
    }
  va_end (ap);

  return 0;
}

int
riemann_event_set_attributes_n (riemann_event_t *event, size_t n_attributes,
                                riemann_attribute_t **attributes)
{
  if (!event)
    return -EINVAL;

  if (n_attributes == 0 && attributes)
    return -ERANGE;

  if (n_attributes > 0 && !attributes)
    return -EINVAL;

  for (size_t n = 0; n < event->n_attributes; n++)
    riemann_attribute_free (event->attributes[n]);
  if (event->attributes)
    free (event->attributes);

  event->n_attributes = n_attributes;
  event->attributes = (riemann_attribute_t **)
    calloc (n_attributes, sizeof (riemann_attribute_t *));
  for (size_t n = 0; n < n_attributes; n++)
    event->attributes[n] = riemann_attribute_clone (attributes[n]);

  return 0;
}

int
riemann_event_set_string_attributes_n (riemann_event_t *event,
                                       size_t n_attributes,
                                       const char **keys,
                                       const char **values)
{
  if (!event)
    return -EINVAL;

  if (n_attributes == 0 && keys && values)
    return -ERANGE;

  if (n_attributes > 0 && (!keys || !values))
    return -EINVAL;

  for (size_t n = 0; n < event->n_attributes; n++)
    riemann_attribute_free (event->attributes[n]);
  if (event->attributes)
    free (event->attributes);

  event->n_attributes = n_attributes;
  event->attributes = (riemann_attribute_t **)
    calloc (n_attributes, sizeof (riemann_attribute_t *));
  for (size_t n = 0; n < n_attributes; n++) {
    event->attributes[n] = riemann_attribute_create(keys[n], values[n]);
  }

  return 0;
}

riemann_event_t *
riemann_event_create_full (int has_time, int64_t time,
                           char *state,
                           char *service,
                           char *host,
                           char *desc,
                           size_t n_tags, char **tags,
                           int has_ttl, float ttl,
                           size_t n_attributes, riemann_attribute_t **attributes,
                           int has_time_micros, int64_t time_micros,
                           int has_metric_s64, int64_t metric_s64,
                           int has_metric_d, double metric_d,
                           int has_metric_f, float metric_f)
{
  riemann_event_t *event = riemann_event_new ();

  event->has_time = has_time;
  event->time = time;

  if (state)
    event->state = strdup (state);
  if (service)
    event->service = strdup (service);
  if (host)
    event->host = strdup (host);
  if (desc)
    event->description = strdup (desc);

  event->n_tags = n_tags;
  event->tags = (char **) calloc (n_tags, sizeof (char *));
  for (size_t n = 0; n < n_tags; n++)
    event->tags[n] = strdup (tags[n]);

  event->has_ttl = has_ttl;
  event->ttl = ttl;

  event->n_attributes = n_attributes;
  event->attributes = (riemann_attribute_t **)
    calloc (n_attributes, sizeof (riemann_attribute_t *));
  for (size_t n = 0; n < n_attributes; n++)
    event->attributes[n] = riemann_attribute_clone (attributes[n]);

  event->has_time_micros = has_time_micros;
  event->time_micros = time_micros;

  event->has_metric_sint64 = has_metric_s64;
  event->metric_sint64 = metric_s64;

  event->has_metric_d = has_metric_d;
  event->metric_d = metric_d;

  event->has_metric_f = has_metric_f;
  event->metric_f = metric_f;

  return event;
}
