/* riemann/client/tls-openssl.c -- Riemann C client library
 * SPDX-FileCopyrightText: 2021-2024 Gergely Nagy
 * SPDX-FileContributor: Gergely Nagy
 *
 * SPDX-License-Identifier: EUPL-1.2
 */

#define _GNU_SOURCE
#include <stdio.h>

#include "riemann/_private.h"
#include "riemann/platform.h"

#include "riemann/client/tls.h"

#include <string.h>

#include <openssl/err.h>
#include <openssl/ssl.h>

typedef struct {
  struct
  {
    SSL_CTX *ctx;
    BIO *bio;

    riemann_client_tls_options_t options;
  } tls;
} riemann_client_connection_openssl_t;

int
_riemann_client_disconnect_tls (riemann_client_t *client)
{
  riemann_client_connection_openssl_t *conn =
    (riemann_client_connection_openssl_t *) client->connection;

  if (!conn || client->type != RIEMANN_CLIENT_TLS)
    return -ENOTCONN;

  BIO_free_all (conn->tls.bio);
  SSL_CTX_free (conn->tls.ctx);

  _riemann_client_tls_options_free (&conn->tls.options);
  free (client->connection);
  client->connection = NULL;

  return 0;
}

static int
_riemann_client_get_fd_openssl (riemann_client_t *client)
{
  riemann_client_connection_openssl_t *conn =
    (riemann_client_connection_openssl_t *) client->connection;

  return BIO_get_fd (conn->tls.bio, NULL);
}

static int
_riemann_client_set_timeout_openssl (riemann_client_t *client,
                                     struct timeval *timeout)
{
  riemann_client_connection_openssl_t *conn =
    (riemann_client_connection_openssl_t *) client->connection;

  SSL_CTX_set_timeout(conn->tls.ctx, timeout->tv_sec);

  return 0;
}

int
_riemann_client_connect_setup_tls (riemann_client_t *client,
                                   va_list aq)
{
  va_list ap;
  riemann_client_option_t option;
  riemann_client_connection_openssl_t *conn;

  conn = (riemann_client_connection_openssl_t *)
    calloc (1, sizeof (riemann_client_connection_openssl_t));

  client->type = RIEMANN_CLIENT_TLS;
  client->send = _riemann_client_send_message_tls;
  client->recv = _riemann_client_recv_message_tls;
  client->connect = _riemann_client_connect_tls;
  client->disconnect = _riemann_client_disconnect_tls;
  client->get_fd = _riemann_client_get_fd_openssl;
  client->set_timeout = _riemann_client_set_timeout_openssl;
  client->connection = (void *) conn;

  va_copy (ap, aq);

  option = (riemann_client_option_t) va_arg (ap, int);
  do
    {
      switch (option)
        {
        case RIEMANN_CLIENT_OPTION_NONE:
          break;

        case RIEMANN_CLIENT_OPTION_TLS_CA_FILE:
          _riemann_set_string (&(conn->tls.options.cafn), va_arg (ap, char *));
          break;

        case RIEMANN_CLIENT_OPTION_TLS_CERT_FILE:
          _riemann_set_string (&(conn->tls.options.certfn), va_arg (ap, char *));
          break;

        case RIEMANN_CLIENT_OPTION_TLS_KEY_FILE:
          _riemann_set_string (&(conn->tls.options.keyfn), va_arg (ap, char *));
          break;

        case RIEMANN_CLIENT_OPTION_TLS_HANDSHAKE_TIMEOUT:
          conn->tls.options.handshake_timeout = va_arg (ap, unsigned int);
          break;

        case RIEMANN_CLIENT_OPTION_TLS_PRIORITIES:
          _riemann_set_string (&(conn->tls.options.priorities), va_arg (ap, char *));
          break;

        default:
          va_end (ap);
          goto err;
        }

      if (option != RIEMANN_CLIENT_OPTION_NONE)
        option = (riemann_client_option_t) va_arg (ap, int);
    }
  while (option != RIEMANN_CLIENT_OPTION_NONE);
  va_end (ap);

  if (!conn->tls.options.cafn || !conn->tls.options.certfn || !conn->tls.options.keyfn)
    {
      goto err;
    }

  return 0;

 err:
  _riemann_client_tls_options_free (&conn->tls.options);
  free (conn);
  client->connection = NULL;
  return -EINVAL;
}

int
_riemann_client_connect_tls (riemann_client_t *client,
                             const char *hostname,
                             int port)
{
  riemann_client_connection_openssl_t *conn =
    (riemann_client_connection_openssl_t *) client->connection;
  char port_s[32];

  if (conn->tls.options.handshake_timeout != 0)
    {
      client->disconnect (client);
      return -ENOTSUP;
    }

  conn->tls.ctx = SSL_CTX_new (TLS_client_method());

  if (conn->tls.options.certfn)
    {
      int r = SSL_CTX_use_certificate_file (conn->tls.ctx,
                                            conn->tls.options.certfn,
                                            SSL_FILETYPE_PEM);
      if (!r)
        {
          client->disconnect (client);
          return -EPROTO;
        }
    }
  if (conn->tls.options.keyfn)
    {
      int r = SSL_CTX_use_PrivateKey_file (conn->tls.ctx,
                                           conn->tls.options.keyfn,
                                           SSL_FILETYPE_PEM);
      if (!r)
        {
          client->disconnect (client);
          return -EPROTO;
        }
    }
  if (conn->tls.options.cafn)
    {
      int r = SSL_CTX_load_verify_locations (conn->tls.ctx,
                                             conn->tls.options.cafn,
                                             NULL);
      if (!r)
        {
          client->disconnect (client);
          return -EPROTO;
        }
    }
  if (conn->tls.options.priorities)
    {
      int r = SSL_CTX_set_cipher_list (conn->tls.ctx,
                                       conn->tls.options.priorities);
      r |= SSL_CTX_set_ciphersuites (conn->tls.ctx,
                                     conn->tls.options.priorities);
      if (!r) {
        client->disconnect (client);
        return -EPROTO;
      }
    }

  conn->tls.bio = BIO_new_ssl_connect (conn->tls.ctx);

  memset (port_s, 0, sizeof (port_s));
  snprintf (port_s, sizeof (port_s) - 1, "%d", port);
  BIO_set_conn_hostname (conn->tls.bio, hostname);
  BIO_set_conn_port (conn->tls.bio, port_s);

  if (BIO_do_connect (conn->tls.bio) != 1)
    {
      client->disconnect (client);
      return -ERR_GET_REASON (ERR_get_error ());
    }

  return 0;
}

int
_riemann_client_send_message_tls (riemann_client_t *client,
                                  riemann_message_t *message)
{
  uint8_t *buffer;
  size_t len;
  ssize_t sent;
  riemann_client_connection_openssl_t *conn =
    (riemann_client_connection_openssl_t *) client->connection;

  buffer = riemann_message_to_buffer (message, &len);
  if (!buffer)
    return -errno;

  sent = BIO_write (conn->tls.bio, buffer, len);
  if (sent < 0 || (size_t)sent != len)
    {
      free (buffer);
      return -EPROTO;
    }
  free (buffer);
  return 0;
}

riemann_message_t *
_riemann_client_recv_message_tls (riemann_client_t *client)
{
  uint32_t header, len;
  uint8_t *buffer;
  ssize_t received;
  riemann_message_t *message;
  riemann_client_connection_openssl_t *conn =
    (riemann_client_connection_openssl_t *) client->connection;

  received = BIO_read (conn->tls.bio, &header, sizeof (header));
  if (received != sizeof (header))
    {
      errno = EPROTO;
      return NULL;
    }
  len = ntohl (header);

  buffer = (uint8_t *) calloc (len, sizeof (uint8_t));
  received = BIO_read (conn->tls.bio, buffer, len);
  if (received != len)
    {
      free (buffer);
      errno = EPROTO;
      return NULL;
    }

  message = riemann_message_from_buffer (buffer, len);
  if (message == NULL)
    {
      int e = errno;

      free (buffer);
      errno = e;
      return NULL;
    }
  free (buffer);

  return message;
}

riemann_client_tls_library_t
riemann_client_get_tls_library (void)
{
  return RIEMANN_CLIENT_TLS_LIBRARY_OPENSSL;
}
