/* riemann/client/tcp.c -- Riemann C client library
 * SPDX-FileCopyrightText: 2013-2024 Gergely Nagy
 * SPDX-FileContributor: Gergely Nagy
 *
 * SPDX-License-Identifier: EUPL-1.2
 */

#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>

#include "riemann/client/tcp.h"
#include "riemann/client/socket.h"
#include "riemann/_private.h"

void
_riemann_client_connect_setup_tcp (riemann_client_t *client)
{
  _riemann_client_connect_setup_socket (client);

  client->type = RIEMANN_CLIENT_TCP;
  client->send = _riemann_client_send_message_tcp;
  client->recv = _riemann_client_recv_message_tcp;
}

int
_riemann_client_send_message_tcp (riemann_client_t *client,
                                  riemann_message_t *message)
{
  riemann_client_connection_socket_t *conn =
    (riemann_client_connection_socket_t *) client->connection;

  uint8_t *buffer;
  size_t len;
  ssize_t sent;

  buffer = riemann_message_to_buffer (message, &len);
  if (!buffer)
    return -errno;

  sent = send (conn->sock, buffer, len, 0);
  if (sent == -1 || (size_t)sent != len)
    {
      int e = errno;

      free (buffer);
      return -e;
    }
  free (buffer);
  return 0;
}

riemann_message_t *
_riemann_client_recv_message_tcp (riemann_client_t *client)
{
  uint32_t header, len;
  uint8_t *buffer;
  ssize_t received;
  riemann_message_t *message;
  riemann_client_connection_socket_t *conn =
    (riemann_client_connection_socket_t *) client->connection;

  received = recv (conn->sock, &header, sizeof (header), MSG_WAITALL);
  if (received != sizeof (header))
    return NULL;
  len = ntohl (header);

  buffer = (uint8_t *) calloc (len, sizeof (uint8_t));

  received = recv (conn->sock, buffer, len, MSG_WAITALL);
  if (received != len)
    {
      int e = errno;

      free (buffer);
      errno = e;
      return NULL;
    }

  message = riemann_message_from_buffer (buffer, len);
  if (message == NULL)
    {
      int e = errno;

      free (buffer);
      errno = e;
      return NULL;
    }
  free (buffer);

  return message;
}
