/* riemann/client.c -- Riemann C client library
 * SPDX-FileCopyrightText: 2013-2024 Gergely Nagy
 * SPDX-FileContributor: Gergely Nagy
 *
 * SPDX-License-Identifier: EUPL-1.2
 */

#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include "riemann/riemann-client.h"

#include "riemann/_private.h"
#include "riemann/platform.h"

#include "riemann/client/tcp.h"
#include "riemann/client/tls.h"
#include "riemann/client/udp.h"

const char *
riemann_client_version (void)
{
  return PACKAGE_VERSION;
}

const char *
riemann_client_version_string (void)
{
  return PACKAGE_STRING;
}

int
riemann_client_check_version (int major, int minor, int patch)
{
  return _riemann_client_compare_versions (RCC_MAJOR_VERSION,
                                           RCC_MINOR_VERSION,
                                           RCC_PATCH_VERSION,
                                           major, minor, patch);
}

riemann_client_t *
riemann_client_new (void)
{
  riemann_client_t *client;

  client = (riemann_client_t *) calloc (1, sizeof (riemann_client_t));

  return client;
}

int
riemann_client_disconnect (riemann_client_t *client)
{
  if (!client || !client->disconnect)
    return -ENOTCONN;

  return client->disconnect (client);
}

void
riemann_client_free (riemann_client_t *client)
{
  if (!client)
    {
      errno = EINVAL;
      return;
    }

  errno = -riemann_client_disconnect (client);

  free (client);
}

int
riemann_client_get_fd (riemann_client_t *client)
{
  if (!client || !client->get_fd)
    return -EINVAL;

  return client->get_fd (client);
}

int
riemann_client_set_timeout (riemann_client_t *client, struct timeval *timeout)
{
  if (!client)
    return -EINVAL;

  if (!client->connection || !timeout)
    return -EINVAL;

  if (!client->set_timeout)
    return -EINVAL;

  return client->set_timeout (client, timeout);
}

static int
riemann_client_connect_va (riemann_client_t *client,
                           riemann_client_type_t type,
                           const char *hostname, int port,
                           va_list aq)
{
  if (!client || !hostname)
    return -EINVAL;
  if (port <= 0)
    return -ERANGE;

  switch (type)
    {
    case RIEMANN_CLIENT_TCP:
      _riemann_client_connect_setup_tcp (client);
      break;
    case RIEMANN_CLIENT_UDP:
      _riemann_client_connect_setup_udp (client);
      break;
    case RIEMANN_CLIENT_TLS:
      {
        va_list ap;
        int e;

        va_copy (ap, aq);
        e = _riemann_client_connect_setup_tls (client, ap);
        va_end (ap);

        if (e != 0)
          {
            return e;
          }

        break;
      }
    default:
      return -EINVAL;
    }

  return client->connect (client, hostname, port);
}

int
riemann_client_connect (riemann_client_t *client,
                        riemann_client_type_t type,
                        const char *hostname, int port, ...)
{
  va_list ap;
  int r;

  va_start (ap, port);
  r = riemann_client_connect_va (client, type, hostname, port, ap);
  va_end (ap);
  return r;
}

riemann_client_t *
riemann_client_create (riemann_client_type_t type,
                       const char *hostname, int port, ...)
{
  riemann_client_t *client;
  int e;
  va_list ap;

  client = riemann_client_new ();

  va_start (ap, port);
  e = riemann_client_connect_va (client, type, hostname, port, ap);
  if (e != 0)
    {
      riemann_client_free (client);
      va_end (ap);
      errno = -e;
      return NULL;
    }
  va_end (ap);

  return client;
}

int
riemann_client_send_message (riemann_client_t *client,
                             riemann_message_t *message)
{
  if (!client)
    return -ENOTCONN;
  if (!message)
    return -EINVAL;

  if (!client->send)
    return -ENOTCONN;

  return client->send (client, message);
}

int
riemann_client_send_message_oneshot (riemann_client_t *client,
                                     riemann_message_t *message)
{
  int ret = 0;

  ret = riemann_client_send_message (client, message);
  riemann_message_free (message);

  return ret;
}

riemann_message_t *
riemann_client_recv_message (riemann_client_t *client)
{
  if (!client || !client->recv)
    {
      errno = ENOTCONN;
      return NULL;
    }

  return client->recv (client);
}
