/*  QuadruplebackEngine.h - Game engine.

    quadrupleback - A video game where intruders must be circled.
    Copyright (C) 2012-2024 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
    02110-1301, USA.
*/

#ifndef _H_QuadruplebackEngine
#define _H_QuadruplebackEngine

#include "CircularPointBuffer.h"

class Enemy;

#include <flatzebra/GameEngine.h>
#include <flatzebra/Joystick.h>
#include <flatzebra/KeyState.h>
#include <flatzebra/SoundMixer.h>

#include <string>
#include <vector>

using namespace std;


class QuadruplebackEngine : public flatzebra::GameEngine
{
public:

    /*  See base class.
        'windowManagerCaption' must contain the title to give to the window.
        Throws an error message string or an integer error code upon failure.
    */
    QuadruplebackEngine(const string &windowManagerCaption,
                        bool _useSound,
                        bool _useAcceleratedRendering);

    virtual ~QuadruplebackEngine();

    virtual void processKey(SDL_Keycode keysym, bool pressed) override;

    virtual bool tick() override;

    virtual void processActivation(bool appActive) override;

private:

    enum
    {
        FPS = 20,  // frames (ticks) per second
        CHAR_WIDTH  = 9,  // in pixels
        CHAR_HEIGHT = 18,
        MARGIN = 8,
    };

    class Snake
    {
    public:
        double dirInRadians;
        CircularPointBuffer points;  // in pixels; points are relative to "game area", not to window
        double maxSpeedInPixels;
        flatzebra::RCouple lastJoystickPos;

        Snake()
        :   dirInRadians(0),
            points(1),
            maxSpeedInPixels(0),
            lastJoystickPos(0, 0)
        {
        }

        void reset(double _snakeDirInRadians,
                   const flatzebra::RCouple &_firstPoint,
                   size_t _maxSnakeLenInPoints,
                   double _maxSnakeSpeedInPixels,
                   const flatzebra::RCouple &_initJoystickPos)
        {
            dirInRadians = _snakeDirInRadians;
            points.reset(_maxSnakeLenInPoints);
            points.addPoint(_firstPoint);
            maxSpeedInPixels = _maxSnakeSpeedInPixels;
            lastJoystickPos = _initJoystickPos;
        }

        flatzebra::RCouple getHeadPos() const
        {
            assert(points.size() > 0);
            return points.back();
        }
    };

    enum GamePhase
    {
        WAITING_FOR_PLAYER,
        DELAY_BEFORE_LIFE_START,
        PLAYING,
        PLAYER_DYING
    };

    class Area
    {
    public:
        Area(double x, double y,
             double width, double height,
             SDL_Color _fgColor,
             SDL_Color _bgColor)
        :   topLeft(x, y),
            size(width, height),
            fgColor(_fgColor),
            bgColor(_bgColor)
        {
        }

        flatzebra::RCouple topLeft;
        flatzebra::RCouple size;
        SDL_Color fgColor;
        SDL_Color bgColor;
    };

    class FloatingNumber  // used to show scored points in game area
    {
    public:
        FloatingNumber(long _number,
                       const flatzebra::RCouple &_pos,
                       double _speed,
                       size_t _numTicksToLive)
        :   number(_number),
            pos(_pos),
            speed(_speed),
            numTicksLeft(_numTicksToLive)
        {
        }

        long number;
        flatzebra::RCouple pos;
        double speed;  // positive number of pixels per tick
        size_t numTicksLeft;

    private:
        // Forbidden operations:
        FloatingNumber(const FloatingNumber &);
        FloatingNumber &operator = (const FloatingNumber &);
    };

private:

    void loadPixmap(const char *filePath, flatzebra::PixmapArray &pa, size_t index);
    void resetGame();
    void initNewLife();
    void moveSnakeRandomly();
    void play();
    bool isPositionNearSnakeHead(const flatzebra::RCouple &pos, double maxDistance) const;
    void drawFilledRect(const flatzebra::RCouple &topLeft,
                        const flatzebra::RCouple &size,
                        SDL_Color color);
    void drawAreaBG(const Area &area);
    void drawScene();
    void writeCenteredString(Sint16 y, const char *s, SDL_Color color);
    flatzebra::RCouple addPointToSnakeAndFindIntersections(double directionInRadians,
                                                           double speed);
    void addPointToSnakeAndFindIntersections(const flatzebra::RCouple &newSegEnd);
    size_t killCircledEnemies(const vector<flatzebra::RCouple> &loop);
    void drawLoop(const vector<flatzebra::RCouple> &loop);
    bool isPointOnEnemy(const flatzebra::RCouple &p) const;
    size_t getNumSkulls() const;
    void playSoundEffect(flatzebra::SoundMixer::Chunk &wb);

    // Forbidden operations:
    QuadruplebackEngine(const QuadruplebackEngine &);
    QuadruplebackEngine &operator = (const QuadruplebackEngine &);

private:

    bool paused;
    size_t tickCount;
    string errorMessage;

    flatzebra::KeyState quitKS;
    flatzebra::KeyState startKS;
    flatzebra::KeyState fullScreenKS;
    flatzebra::KeyState pauseResumeKS;
    flatzebra::KeyState upKS;
    flatzebra::KeyState downKS;
    flatzebra::KeyState leftKS;
    flatzebra::KeyState rightKS;
    flatzebra::KeyState throttleKS;

    Area livesArea;
    Area scoreArea;
    Area gameArea;

    SDL_Color presentationColor;
    SDL_Color errorMessageColor;
    SDL_Color pauseBGColor;
    SDL_Color snakeColor;
    SDL_Color loopColor;
    SDL_Color intersectionColor;
    SDL_Color currentLifeColor;
    SDL_Color otherLivesColor;
    SDL_Color floatingNumberColor;
    SDL_Color yoyoWireColor;
    SDL_Color deathColor;

    flatzebra::Joystick joystick;

    Snake snake;

    vector<Enemy *> enemies;  // elements must come from 'new'
    size_t maxNumEnemies;
    size_t minTimeBeforeNextEnemy;  // in ticks
    size_t maxTimeBeforeNextEnemy;
    size_t nextEnemyTime;
    size_t numEnemiesKilled;

    GamePhase gamePhase;
    size_t preLifeEndTime;  // tick at which pre-life delay ends
    size_t deathEndTime;  // tick at which death sequence ends
    size_t numLives;
    long score;
    vector<FloatingNumber *> floatingNumbers;  // elements must come from 'new'

    flatzebra::PixmapArray cherryPA;
    flatzebra::PixmapArray applePA;
    flatzebra::PixmapArray pearPA;
    flatzebra::PixmapArray horseshoePA;
    flatzebra::PixmapArray skullPA;
    flatzebra::PixmapArray skatePA;
    flatzebra::PixmapArray spiderPA;
    flatzebra::PixmapArray yoyoPA;

    std::string fontDefBuffer;

    bool useSound;
    flatzebra::SoundMixer *theSoundMixer;  // see method playSoundEffect()
    flatzebra::SoundMixer::Chunk enemyAppearsSound;
    flatzebra::SoundMixer::Chunk enemyCapturedSound;
    flatzebra::SoundMixer::Chunk enemyTouchedSound;

};


#endif  /* _H_QuadruplebackEngine */
