/*  $Id: Enemy.cpp,v 1.1 2012/07/08 09:35:52 sarrazip Exp $
    Enemy.cpp - Object to be circled.

    quadrupleback - A video game where intruders must be circled.
    Copyright (C) 2012 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
    02110-1301, USA.
*/

#include "Enemy.h"


using namespace flatzebra;
using namespace std;


Enemy::Enemy(Type _type,
             flatzebra::RSprite *_sprite,
             const char **_xpm,
             double _speed,
             size_t _tickOfDeath)
:   type(_type),
    sprite(_sprite),
    xpmPixels(_xpm + 1),  // starting pointing to 2nd line of XPM definition
    speed(_speed),
    tickOfDeath(_tickOfDeath),
    targetPos(0, 0),
    targetChangeTime(0),  // zero requests a change at 1st call to isTimeToChangeTarget()
    yoyoDescending(true),
    yoyoLength(0),
    yoyoMaxLength(300)
{
    assert(strcmp(*xpmPixels, " \tc None") == 0);  // first color def must be space->None

    do
    {
        ++xpmPixels;
    } while (strlen(*xpmPixels) == 11 && (*xpmPixels)[2] == 'c');  // color def lines assumed to be 11 chars long
}


Enemy::~Enemy()
{
    delete sprite;
}


Enemy::Type
Enemy::getType() const
{
    return type;
}


bool
Enemy::touchesPosition(const RCouple &p) const
{
    RCouple offset = p - sprite->getPos();  // in sprite image
    const RCouple &size = sprite->getSize();
    if (offset.x < 0 || offset.x >= size.x
            || offset.y < 0 || offset.y >= size.y)
        return false;

    assert(xpmPixels[int(offset.y)] != NULL);
    return xpmPixels[int(offset.y)][int(offset.x)] != ' ';
}


bool
Enemy::isTimeToChangeTarget(size_t currentTick) const
{
    if (currentTick >= targetChangeTime)  // if current target has expired
        return true;

    // If target near enough (this avoids dithering around the target).
    return (targetPos - sprite->getCenterPos()).squaredLength() <= 50 * 50;
}


void
Enemy::setTargetPos(const RCouple &p, size_t timeOfNextChange)
{
    targetPos = p;
    targetChangeTime = timeOfNextChange;
}


RCouple
Enemy::getTargetPos() const
{
    return targetPos;
}


bool
Enemy::movesHorizOnly() const
{
    return type == SKATE;
}


RCouple
Enemy::getCapturePos() const
{
    return sprite->getCenterPos();
}


bool
Enemy::isCapturedByLoop(const vector<RCouple> &loop) const
{
    if (type == SKULL)
        return false;

    if (!isPointInPolygon(getCapturePos(), loop))
        return false;

    if (type != YOYO)
        return true;

    // In the case of a yoyo, the wire must also be inside the loop.
    // We test several points along the wire's rectangle.
    //
    RCouple wireTopLeft, wireSize;
    getWireRect(wireTopLeft, wireSize);

    enum { STEPS = 4 };
    for (size_t j = 0; j <= STEPS; ++j)
    {
        RCouple p = wireTopLeft + double(j) / STEPS * wireSize;
        if (!isPointInPolygon(p, loop))
            return false;
    }

    return true;
}


double 
Enemy::getSpeed() const
{
    return speed;
}


long
Enemy::getPointValue() const
{
    switch (type)
    {
    case CHERRY:    return 100;
    case APPLE:     return  70;
    case PEAR:      return 300;
    case HORSESHOE: return 150;
    case SKULL:     return 600;  // only for prob. distribution purposes
    case SKATE:     return 200;
    case SPIDER:    return 500;
    case YOYO:      return 250;
    default: assert(false); return 0;
    }
}


const unsigned short Enemy::typeDistribution0[NUM_ENEMY_TYPES] =
{
    100, 142, 33, 66, 16, 50, 20, 40
};


//static
Enemy::Type
Enemy::chooseRandomly()
{
    unsigned short sum = 0;
    for (int i = 0; i < NUM_ENEMY_TYPES; ++i)
        sum += typeDistribution0[i];

    unsigned short r = rand() % sum;
    for (int i = 0; i < NUM_ENEMY_TYPES; ++i)
    {
        if (r < typeDistribution0[i])
            return Type(i);
        r -= typeDistribution0[i];
    }

    assert(false);
    return APPLE;  // fallback
}


size_t
Enemy::getTickOfDeath() const
{
    return tickOfDeath;
}


size_t
Enemy::getAnimationFreq() const
{
    switch (type)
    {
    case SKATE:     return 2;
    case SPIDER:    return 2;
    default:        return 1;
    }
}


void
Enemy::updatePixmapIndex()
{
    size_t numPixmaps = sprite->getPixmapArray()->getNumImages();
    size_t add = (yoyoDescending ? 1 : numPixmaps - 1);
    sprite->currentPixmapIndex = (sprite->currentPixmapIndex + add) % numPixmaps;
}


void
Enemy::setYoyoMaxLength(double maxLengthInPixels)
{
    assert(maxLengthInPixels > 1 && maxLengthInPixels < 10000);
    yoyoMaxLength = maxLengthInPixels;
}


double
Enemy::getYoyoLength() const
{
    return yoyoLength;
}


void
Enemy::animateYoyo()
{
    if (yoyoDescending)
    {
        if (yoyoLength >= yoyoMaxLength)
            yoyoDescending = false;
        else
        {
            yoyoLength += speed;
            sprite->setPos(sprite->getPos() + RCouple(0, speed));
        }
    }
    else
    {
        if (yoyoLength <= 0)
            yoyoDescending = true;
        else
        {
            yoyoLength -= speed;
            sprite->setPos(sprite->getPos() - RCouple(0, speed));
        }
    }
}


void
Enemy::getWireRect(RCouple &wireTopLeft, RCouple &wireSize) const
{
    wireSize.x = 3;
    wireSize.y = yoyoLength;
    wireTopLeft.x = sprite->getLowerRightPos().x - wireSize.x;
    wireTopLeft.y = sprite->getCenterPos().y - wireSize.y;
}
