"""
SocketWrapper method tests for pygnssutils

Created on 20 Aug 2024

*** NB: must be saved in UTF-8 format ***

@author: semuadmin
"""

# pylint: disable=line-too-long

import os
import sys
import unittest
from io import StringIO

from pyubx2 import ERR_LOG, UBXReader
from pynmeagps import SocketWrapper
from pygnssutils import (
    DEFAULT_BUFSIZE,
    ENCODE_CHUNKED,
    ENCODE_GZIP,
    ENCODE_NONE,
)
from tests.dummysocket import DummySocket

DIRNAME = os.path.dirname(__file__)


class StreamTest(unittest.TestCase):
    def setUp(self):
        self.maxDiff = None

    def tearDown(self):
        pass

    def catchio(self):
        """
        Capture stdout as string.
        """

        self._saved_stdout = sys.stdout
        self._strout = StringIO()
        sys.stdout = self._strout

    def restoreio(self) -> str:
        """
        Return captured output and restore stdout.
        """

        sys.stdout = self._saved_stdout
        return self._strout.getvalue().strip()

    def testnoencode(self):  # test socket read with no encoding
        EXPECTED_RESULT = "<RTCM(1127, DF002=1127, DF003=25, DF427=76457000, DF393=1, DF409=0, DF001_7=0, DF411=1, DF412=0, DF417=0, DF418=0, DF394=585527394058633216, NSat=10, DF395=1090781569, NSig=6, DF396=732080671570230775, NCell=44, PRN_01=005, PRN_02=011, PRN_03=019, PRN_04=020, PRN_05=022, PRN_06=023, PRN_07=028, PRN_08=037, PRN_09=040, PRN_10=043, DF397_01=137, DF397_02=84, DF397_03=78, DF397_04=80, DF397_05=85, DF397_06=85, DF397_07=83, DF397_08=73, DF397_09=135, DF397_10=82, ExtSatInfo_01=0, ExtSatInfo_02=0, ExtSatInfo_03=0, ExtSatInfo_04=0, ExtSatInfo_05=0, ExtSatInfo_06=0, ExtSatInfo_07=0, ExtSatInfo_08=0, ExtSatInfo_09=0, ExtSatInfo_10=0, DF398_01=0.1943359375, DF398_02=0.7177734375, DF398_03=0.5888671875, DF398_04=0.96875, DF398_05=0.75, DF398_06=0.4169921875, DF398_07=0.2880859375, DF398_08=0.798828125, DF398_09=0.720703125, DF398_10=0.7587890625, DF399_01=0, DF399_02=237, DF399_03=49, DF399_04=551, DF399_05=-482, DF399_06=560, DF399_07=-519, DF399_08=202, DF399_09=250, DF399_10=64, CELLPRN_01=005, CELLSIG_01=2I, CELLPRN_02=005, CELLSIG_02=7I, CELLPRN_03=011, CELLSIG_03=2I, CELLPRN_04=011, CELLSIG_04=7I, CELLPRN_05=019, CELLSIG_05=2I, CELLPRN_06=019, CELLSIG_06=6I, CELLPRN_07=019, CELLSIG_07=5X, CELLPRN_08=019, CELLSIG_08=7D, CELLPRN_09=019, CELLSIG_09=1X, CELLPRN_10=020, CELLSIG_10=2I, CELLPRN_11=020, CELLSIG_11=6I, CELLPRN_12=020, CELLSIG_12=5X, CELLPRN_13=020, CELLSIG_13=7D, CELLPRN_14=020, CELLSIG_14=1X, CELLPRN_15=022, CELLSIG_15=2I, CELLPRN_16=022, CELLSIG_16=6I, CELLPRN_17=022, CELLSIG_17=5X, CELLPRN_18=022, CELLSIG_18=7D, CELLPRN_19=022, CELLSIG_19=1X, CELLPRN_20=023, CELLSIG_20=2I, CELLPRN_21=023, CELLSIG_21=6I, CELLPRN_22=023, CELLSIG_22=5X, CELLPRN_23=023, CELLSIG_23=7D, CELLPRN_24=023, CELLSIG_24=1X, CELLPRN_25=028, CELLSIG_25=2I, CELLPRN_26=028, CELLSIG_26=6I, CELLPRN_27=028, CELLSIG_27=5X, CELLPRN_28=028, CELLSIG_28=7D, CELLPRN_29=028, CELLSIG_29=1X, CELLPRN_30=037, CELLSIG_30=2I, CELLPRN_31=037, CELLSIG_31=6I, CELLPRN_32=037, CELLSIG_32=5X, CELLPRN_33=037, CELLSIG_33=7D, CELLPRN_34=037, CELLSIG_34=1X, CELLPRN_35=040, CELLSIG_35=2I, CELLPRN_36=040, CELLSIG_36=6I, CELLPRN_37=040, CELLSIG_37=5X, CELLPRN_38=040, CELLSIG_38=7D, CELLPRN_39=040, CELLSIG_39=1X, CELLPRN_40=043, CELLSIG_40=2I, CELLPRN_41=043, CELLSIG_41=6I, CELLPRN_42=043, CELLSIG_42=5X, CELLPRN_43=043, CELLSIG_43=7D, CELLPRN_44=043, CELLSIG_44=1X, DF405_01=-0.00027126073837280273, DF405_02=-0.000278569757938385, DF405_03=-0.0004104096442461014, DF405_04=-0.0004260316491127014, DF405_05=0.000143418088555336, DF405_06=0.00010621920228004456, DF405_07=0.00013359449803829193, DF405_08=0.00013173185288906097, DF405_09=0.00013707391917705536, DF405_10=-1.6732141375541687e-05, DF405_11=-5.449168384075165e-05, DF405_12=-2.5577843189239502e-05, DF405_13=-2.8640031814575195e-05, DF405_14=-2.1709129214286804e-05, DF405_15=0.00021881423890590668, DF405_16=0.00017960742115974426, DF405_17=0.00020647607743740082, DF405_18=0.00020122528076171875, DF405_19=0.0002170950174331665, DF405_20=0.00044489651918411255, DF405_21=0.0003968421369791031, DF405_22=0.00042641907930374146, DF405_23=0.00042051635682582855, DF405_24=0.00043869391083717346, DF405_25=0.00037656165659427643, DF405_26=0.0003569386899471283, DF405_27=0.0003777872771024704, DF405_28=0.00037570297718048096, DF405_29=0.00037261471152305603, DF405_30=0.0004130285233259201, DF405_31=0.0003992561250925064, DF405_32=0.00041648000478744507, DF405_33=0.0004133004695177078, DF405_34=0.00041049905121326447, DF405_35=-0.00020756013691425323, DF405_36=-0.0002356264740228653, DF405_37=-0.00021158717572689056, DF405_38=-0.0002105310559272766, DF405_39=-0.0002106744796037674, DF405_40=0.0003317147493362427, DF405_41=0.00030823424458503723, DF405_42=0.0003336295485496521, DF405_43=0.000326789915561676, DF405_44=0.00032790936529636383, DF406_01=-0.00013775844126939774, DF406_02=-6.562424823641777e-05, DF406_03=-0.00044764718040823936, DF406_04=-0.0004188176244497299, DF406_05=0.00017263321205973625, DF406_06=0.00017758505418896675, DF406_07=0.0001785140484571457, DF406_08=0.00018015969544649124, DF406_09=0.00015710527077317238, DF406_10=-1.124013215303421e-05, DF406_11=-2.8881710022687912e-05, DF406_12=1.2540258467197418e-06, DF406_13=2.5243498384952545e-06, DF406_14=2.2971536964178085e-05, DF406_15=0.00023131771013140678, DF406_16=0.00018697651103138924, DF406_17=0.0002232315018773079, DF406_18=0.00020561553537845612, DF406_19=0.00022930558770895004, DF406_20=0.0005551809445023537, DF406_21=0.0006061764433979988, DF406_22=0.0005877683870494366, DF406_23=0.0005839508958160877, DF406_24=0.0005522309802472591, DF406_25=0.00033460604026913643, DF406_26=0.0002748500555753708, DF406_27=0.0003119930624961853, DF406_28=0.00030980492010712624, DF406_29=0.0003337021917104721, DF406_30=0.000517421867698431, DF406_31=0.0004334920085966587, DF406_32=0.000607010442763567, DF406_33=0.00044425809755921364, DF406_34=0.0005145617760717869, DF406_35=-0.00018185051158070564, DF406_36=-0.00021443702280521393, DF406_37=-0.0002012345939874649, DF406_38=-0.0001888033002614975, DF406_39=-0.00014277733862400055, DF406_40=0.0003635748289525509, DF406_41=0.0003521735779941082, DF406_42=0.0003685951232910156, DF406_43=0.0003663073293864727, DF406_44=0.0003600991331040859, DF407_01=492, DF407_02=396, DF407_03=588, DF407_04=588, DF407_05=610, DF407_06=610, DF407_07=610, DF407_08=610, DF407_09=610, DF407_10=637, DF407_11=637, DF407_12=637, DF407_13=637, DF407_14=637, DF407_15=530, DF407_16=530, DF407_17=530, DF407_18=530, DF407_19=530, DF407_20=648, DF407_21=648, DF407_22=648, DF407_23=648, DF407_24=648, DF407_25=552, DF407_26=552, DF407_27=552, DF407_28=552, DF407_29=552, DF407_30=633, DF407_31=616, DF407_32=633, DF407_33=616, DF407_34=633, DF407_35=633, DF407_36=633, DF407_37=633, DF407_38=633, DF407_39=633, DF407_40=595, DF407_41=595, DF407_42=595, DF407_43=595, DF407_44=595, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF420_13=0, DF420_14=0, DF420_15=0, DF420_16=0, DF420_17=0, DF420_18=0, DF420_19=0, DF420_20=0, DF420_21=0, DF420_22=0, DF420_23=0, DF420_24=0, DF420_25=0, DF420_26=0, DF420_27=0, DF420_28=0, DF420_29=0, DF420_30=0, DF420_31=0, DF420_32=0, DF420_33=0, DF420_34=0, DF420_35=0, DF420_36=0, DF420_37=0, DF420_38=0, DF420_39=0, DF420_40=0, DF420_41=0, DF420_42=0, DF420_43=0, DF420_44=0, DF408_01=34.3125, DF408_02=39.375, DF408_03=41.6875, DF408_04=43.8125, DF408_05=47.8125, DF408_06=49.375, DF408_07=51.125, DF408_08=45.8125, DF408_09=47.5, DF408_10=47.125, DF408_11=47.0, DF408_12=48.875, DF408_13=44.875, DF408_14=45.8125, DF408_15=41.8125, DF408_16=45.625, DF408_17=45.1875, DF408_18=40.375, DF408_19=42.875, DF408_20=41.5, DF408_21=45.6875, DF408_22=47.5, DF408_23=42.875, DF408_24=42.1875, DF408_25=44.3125, DF408_26=45.125, DF408_27=45.625, DF408_28=41.875, DF408_29=43.6875, DF408_30=51.1875, DF408_31=51.6875, DF408_32=54.625, DF408_33=48.875, DF408_34=51.875, DF408_35=37.375, DF408_36=40.375, DF408_37=42.5, DF408_38=39.1875, DF408_39=37.3125, DF408_40=44.6875, DF408_41=43.6875, DF408_42=46.6875, DF408_43=40.125, DF408_44=44.875, DF404_01=0.33680000000000004, DF404_02=0.3366, DF404_03=-0.0038, DF404_04=-0.0039000000000000003, DF404_05=0.3136, DF404_06=0.31320000000000003, DF404_07=0.31320000000000003, DF404_08=0.3129, DF404_09=0.3136, DF404_10=0.058100000000000006, DF404_11=0.058100000000000006, DF404_12=0.0587, DF404_13=0.058100000000000006, DF404_14=0.0585, DF404_15=0.1827, DF404_16=0.1827, DF404_17=0.1829, DF404_18=0.18280000000000002, DF404_19=0.1824, DF404_20=0.1394, DF404_21=0.1393, DF404_22=0.1399, DF404_23=0.1394, DF404_24=0.1398, DF404_25=0.46340000000000003, DF404_26=0.4635, DF404_27=0.4637, DF404_28=0.463, DF404_29=0.4631, DF404_30=0.2033, DF404_31=0.20370000000000002, DF404_32=0.2031, DF404_33=0.203, DF404_34=0.2034, DF404_35=0.2882, DF404_36=0.2881, DF404_37=0.2879, DF404_38=0.2886, DF404_39=0.2882, DF404_40=0.1449, DF404_41=0.1449, DF404_42=0.14500000000000002, DF404_43=0.14500000000000002, DF404_44=0.1446)>"

        filename = os.path.join(DIRNAME, "ntrip_encode_none.bin")
        sock = SocketWrapper(
            DummySocket(filename, DEFAULT_BUFSIZE * 3), encoding=ENCODE_NONE
        )
        ubr = UBXReader(sock, quitonerror=ERR_LOG)
        count = 0
        for raw, parsed in ubr:
            if parsed is not None:
                count += 1
        # print(parsed)
        # print(f"{count=}")
        self.assertEqual(str(parsed), EXPECTED_RESULT)
        self.assertEqual(count, 53)

    def testchunked(self):  # test socket read with chunking
        EXPECTED_RESULT = "<RTCM(1097, DF002=1097, DF003=25, DF248=76471000, DF393=1, DF409=0, DF001_7=0, DF411=1, DF412=0, DF417=0, DF418=0, DF394=7566049025397358592, NSat=7, DF395=134287616, NSig=4, DF396=268435455, NCell=28, PRN_01=002, PRN_02=003, PRN_03=005, PRN_04=008, PRN_05=024, PRN_06=025, PRN_07=033, DF397_01=84, DF397_02=79, DF397_03=85, DF397_04=82, DF397_05=86, DF397_06=77, DF397_07=92, ExtSatInfo_01=0, ExtSatInfo_02=0, ExtSatInfo_03=0, ExtSatInfo_04=0, ExtSatInfo_05=0, ExtSatInfo_06=0, ExtSatInfo_07=0, DF398_01=0.1611328125, DF398_02=0.806640625, DF398_03=0.3740234375, DF398_04=0.572265625, DF398_05=0.9501953125, DF398_06=0.7265625, DF398_07=0.884765625, DF399_01=-457, DF399_02=267, DF399_03=597, DF399_04=-283, DF399_05=463, DF399_06=8, DF399_07=7, CELLPRN_01=002, CELLSIG_01=1X, CELLPRN_02=002, CELLSIG_02=7X, CELLPRN_03=002, CELLSIG_03=8X, CELLPRN_04=002, CELLSIG_04=5X, CELLPRN_05=003, CELLSIG_05=1X, CELLPRN_06=003, CELLSIG_06=7X, CELLPRN_07=003, CELLSIG_07=8X, CELLPRN_08=003, CELLSIG_08=5X, CELLPRN_09=005, CELLSIG_09=1X, CELLPRN_10=005, CELLSIG_10=7X, CELLPRN_11=005, CELLSIG_11=8X, CELLPRN_12=005, CELLSIG_12=5X, CELLPRN_13=008, CELLSIG_13=1X, CELLPRN_14=008, CELLSIG_14=7X, CELLPRN_15=008, CELLSIG_15=8X, CELLPRN_16=008, CELLSIG_16=5X, CELLPRN_17=024, CELLSIG_17=1X, CELLPRN_18=024, CELLSIG_18=7X, CELLPRN_19=024, CELLSIG_19=8X, CELLPRN_20=024, CELLSIG_20=5X, CELLPRN_21=025, CELLSIG_21=1X, CELLPRN_22=025, CELLSIG_22=7X, CELLPRN_23=025, CELLSIG_23=8X, CELLPRN_24=025, CELLSIG_24=5X, CELLPRN_25=033, CELLSIG_25=1X, CELLPRN_26=033, CELLSIG_26=7X, CELLPRN_27=033, CELLSIG_27=8X, CELLPRN_28=033, CELLSIG_28=5X, DF405_01=-7.630698382854462e-05, DF405_02=-7.011927664279938e-05, DF405_03=-6.508827209472656e-05, DF405_04=-6.460584700107574e-05, DF405_05=-0.00010639987885951996, DF405_06=-9.954534471035004e-05, DF405_07=-9.430758655071259e-05, DF405_08=-9.482912719249725e-05, DF405_09=0.0002455785870552063, DF405_10=0.00026315636932849884, DF405_11=0.0002693980932235718, DF405_12=0.00026905909180641174, DF405_13=-7.494911551475525e-05, DF405_14=-7.195211946964264e-05, DF405_15=-6.722286343574524e-05, DF405_16=-6.61015510559082e-05, DF405_17=-0.00023757852613925934, DF405_18=-0.00023729167878627777, DF405_19=-0.00023064576089382172, DF405_20=-0.0002303197979927063, DF405_21=0.0003365352749824524, DF405_22=0.00034263357520103455, DF405_23=0.00034783221781253815, DF405_24=0.0003476366400718689, DF405_25=0.0004724990576505661, DF405_26=0.0004754960536956787, DF405_27=0.0004833526909351349, DF405_28=0.00048507191240787506, DF406_01=1.4388933777809143e-07, DF406_02=6.136717274785042e-05, DF406_03=6.769131869077682e-05, DF406_04=7.537147030234337e-05, DF406_05=-6.63958489894867e-05, DF406_06=-3.1387899070978165e-05, DF406_07=-2.8406735509634018e-05, DF406_08=-2.7540605515241623e-05, DF406_09=0.0002525872550904751, DF406_10=0.00027188751846551895, DF406_11=0.000331144779920578, DF406_12=0.000273305457085371, DF406_13=-3.942009061574936e-05, DF406_14=-1.1799391359090805e-05, DF406_15=-7.2303228080272675e-06, DF406_16=-6.454996764659882e-06, DF406_17=-0.00019253278151154518, DF406_18=-0.00014961371198296547, DF406_19=-0.0001438758336007595, DF406_20=-0.00014359736815094948, DF406_21=0.0004492630250751972, DF406_22=0.0004980978555977345, DF406_23=0.00047367624938488007, DF406_24=0.0005161161534488201, DF406_25=0.00048280367627739906, DF406_26=0.0004954561591148376, DF406_27=0.0005019824020564556, DF406_28=0.0005047735758125782, DF407_01=586, DF407_02=586, DF407_03=586, DF407_04=586, DF407_05=640, DF407_06=640, DF407_07=640, DF407_08=640, DF407_09=659, DF407_10=659, DF407_11=659, DF407_12=659, DF407_13=604, DF407_14=604, DF407_15=604, DF407_16=604, DF407_17=643, DF407_18=643, DF407_19=643, DF407_20=643, DF407_21=624, DF407_22=624, DF407_23=624, DF407_24=624, DF407_25=564, DF407_26=564, DF407_27=564, DF407_28=564, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF420_13=0, DF420_14=0, DF420_15=0, DF420_16=0, DF420_17=0, DF420_18=0, DF420_19=0, DF420_20=0, DF420_21=0, DF420_22=0, DF420_23=0, DF420_24=0, DF420_25=0, DF420_26=0, DF420_27=0, DF420_28=0, DF408_01=48.0, DF408_02=48.6875, DF408_03=52.625, DF408_04=49.1875, DF408_05=50.875, DF408_06=53.5, DF408_07=57.1875, DF408_08=52.8125, DF408_09=44.1875, DF408_10=45.875, DF408_11=49.625, DF408_12=46.125, DF408_13=50.375, DF408_14=50.625, DF408_15=54.375, DF408_16=50.625, DF408_17=47.5, DF408_18=48.6875, DF408_19=52.5, DF408_20=48.875, DF408_21=51.3125, DF408_22=53.625, DF408_23=56.6875, DF408_24=53.0, DF408_25=38.6875, DF408_26=40.1875, DF408_27=46.0, DF408_28=44.125, DF404_01=-0.3032, DF404_02=-0.3034, DF404_03=-0.30360000000000004, DF404_04=-0.3029, DF404_05=0.0118, DF404_06=0.0115, DF404_07=0.0112, DF404_08=0.012, DF404_09=-0.442, DF404_10=-0.4425, DF404_11=-0.4425, DF404_12=-0.44260000000000005, DF404_13=0.3909, DF404_14=0.391, DF404_15=0.3909, DF404_16=0.3907, DF404_17=-0.3426, DF404_18=-0.3426, DF404_19=-0.3422, DF404_20=-0.3428, DF404_21=-0.2938, DF404_22=-0.2944, DF404_23=-0.2944, DF404_24=-0.2944, DF404_25=0.3427, DF404_26=0.3428, DF404_27=0.343, DF404_28=0.3422)>"

        filename = os.path.join(DIRNAME, "ntrip_encode_chunked.bin")
        sock = SocketWrapper(
            DummySocket(filename, DEFAULT_BUFSIZE * 3), encoding=ENCODE_CHUNKED
        )
        ubr = UBXReader(sock, quitonerror=ERR_LOG)
        count = 0
        parsed = None
        for raw, parsed in ubr:
            if parsed is not None:
                count += 1
        # print(parsed)
        # print(f"{count=}")
        self.assertEqual(str(parsed), EXPECTED_RESULT)
        self.assertEqual(count, 54)

    def testchunkedgzip(self):  # test socket read with chunking & gzip
        EXPECTED_RESULT = "<RTCM(1019, DF002=1019, DF009=17, DF076=280, DF077=0, DF078=0, DF079=-1.8769696907838807e-10, DF071=60, DF081=79200, DF082=0.0, DF083=-9.777068044058979e-12, DF084=0.0006512394174933434, DF085=60, DF086=28.78125, DF087=1.2496457202360034e-09, DF088=-0.9548183786682785, DF089=1.5757977962493896e-06, DF090=0.01348085340578109, DF091=1.1881813406944275e-05, DF092=5153.782318115234, DF093=79200, DF094=1.4901161193847656e-07, DF095=-0.14867734163999557, DF096=4.470348358154297e-08, DF097=0.3080817819572985, DF098=154.6875, DF099=-0.41714857798069715, DF100=-2.3395614334731363e-09, DF101=-1.1175870895385742e-08, DF102=0, DF103=0, DF137=1)>"

        filename = os.path.join(DIRNAME, "ntrip_encode_chunkedgzip.bin")
        sock = SocketWrapper(
            DummySocket(filename, DEFAULT_BUFSIZE * 3),
            encoding=ENCODE_CHUNKED | ENCODE_GZIP,
        )
        ubr = UBXReader(sock, quitonerror=ERR_LOG)
        count = 0
        parsed = None
        for raw, parsed in ubr:
            if parsed is not None:
                count += 1
        # print(parsed)
        # print(f"{count=}")
        self.assertEqual(str(parsed), EXPECTED_RESULT)
        self.assertEqual(count, 49)

    def testreadline(self):  # test socket readline with nmea data
        EXPECTED_RESULT = (
            "<NMEA(GNZDA, time=10:36:07, day=6, month=3, year=2021, ltzh=00, ltzn=00)>"
        )

        filename = os.path.join(DIRNAME, "pygpsdata-nmea.log")
        sock = SocketWrapper(
            DummySocket(filename, DEFAULT_BUFSIZE * 3), encoding=ENCODE_NONE
        )
        ubr = UBXReader(sock, quitonerror=ERR_LOG)
        count = 0
        parsed = None
        for raw, parsed in ubr:
            if parsed is not None:
                count += 1
        print(parsed)
        # print(f"{count=}")
        self.assertEqual(str(parsed), EXPECTED_RESULT)
        self.assertEqual(count, 26)

    def testsend(self):  # test socket send
        filename = os.path.join(DIRNAME, "ntrip_chunked.bin")
        self.catchio()
        sock = SocketWrapper(
            DummySocket(filename, DEFAULT_BUFSIZE * 3), encoding=ENCODE_CHUNKED
        )
        data = b"this is a test data sequence"
        n = sock.write(data)
        res = self.restoreio()
        self.assertEqual(n, len(data))
        self.assertEqual(res, "data sent: b'this is a test data sequence'")

    def testtimeout(self):  # test simulated socket timeout
        filename = os.path.join(DIRNAME, "ntrip_chunked.bin")
        sock = SocketWrapper(
            DummySocket(filename, DEFAULT_BUFSIZE, False), encoding=ENCODE_CHUNKED
        )
        b = sock.read(16)
        print(b)
        self.assertEqual(b, b"=\x86C\xc0\xe1\x92v63c>F@3~@")
        sock = SocketWrapper(
            DummySocket(filename, DEFAULT_BUFSIZE, True), encoding=ENCODE_CHUNKED
        )
        b = sock.read(16)
        self.assertEqual(b, b"")
