//
// $Id: TextTest.m,v 1.9 2007/04/15 23:33:01 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */


#import "TextTest.h"
#import "Number.h"
#import "Random.h"
#import "ThreadServices.h"
#import <ObjectiveLib/Text.h>
#import <ObjectiveLib/Character.h>
#import <ObjectiveLib/DataOutStream.h>
#import <ObjectiveLib/ObjectOutStream.h>
#import <ObjectiveLib/DataInStream.h>
#import <ObjectiveLib/ObjectInStream.h>
#import <ObjectiveLib/FileInStream.h>
#if defined(OL_NO_OPENSTEP)
#import <ObjectiveLib/Exception.h>
#import <ObjectiveLib/Reaper.h>
#else
#import <Foundation/NSException.h>
#import <Foundation/NSString.h>
#import <Foundation/NSArchiver.h>
#import <Foundation/NSData.h>
#if defined(HAVE_KEYED_ARCHIVES)
#import <Foundation/NSKeyedArchiver.h>
#endif
#endif
#import <limits.h>
#import <fcntl.h>
#import <string.h>
#import <errno.h>
#import <unistd.h>
#import <sys/types.h>
#import <sys/stat.h>
#include <stdlib.h>
#import <time.h>
#if defined(__NEXT_RUNTIME__)
#import <objc/objc-class.h>
#endif
#if defined(OL_HAVE_WIN32_SLEEP)
#include <windows.h>
#endif

struct ThreadInfo
{
    UnitTest*   test;
    OLText*     text;
};

#define INNER_LOOP_SIZE 200
#define OUTER_LOOP_SIZE 1000

static void* threadMain(void* arg)
{
    struct ThreadInfo* info = arg;
    OLText** locals = malloc(INNER_LOOP_SIZE * sizeof(OLText*));
#if defined(OL_HAVE_NANOSLEEP)
    struct timespec speck;
#endif
    unsigned i;
    unsigned j;

#if defined(OL_HAVE_NANOSLEEP)
    speck.tv_sec = OLRandom() % 5;
    speck.tv_nsec = OLRandom() % 1000000000;
    nanosleep(&speck, NULL);
#elif defined(OL_HAVE_WIN32_SLEEP)
    Sleep(((OLRandom() % 5) * 1000) + (OLRandom() % 1000));
#endif
    [info->test logMessage: "Thread %p: Starting", OLCurrentThread()];
    for (i = 0; i < OUTER_LOOP_SIZE; i++)
    {
        for (j = 0; j < INNER_LOOP_SIZE; j++)
            locals[j] = [info->text copy];
        for (j = 0; j < INNER_LOOP_SIZE; j++)
            [locals[j] RELEASE];
    }
    free(locals);
    [info->test logMessage: "Thread %p: Done", OLCurrentThread()];
    return NULL;
}

@implementation TextTest

#if !defined(OL_NO_OPENSTEP)
- (void) testCoding
{
    NSArchiver* archiver;
    NSMutableData* data;
    NSData* archData;
    OLText* text;
    OLText* readText;

    data = [[NSMutableData alloc] initWithCapacity: 50];
    archiver = [[NSArchiver alloc] initForWritingWithMutableData: data];
    text = [[OLText alloc] initWithCString: "My dog has fleas"];
    [archiver encodeRootObject: text];
    [archiver RELEASE];
    readText = [NSUnarchiver unarchiveObjectWithData: data];
    [data RELEASE];
    if (![readText IS_MEMBER_OF: [OLText class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected OLText class, but got %s", ((Class)[readText class])->name];
    }
    if (![readText isEqual: text])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"%s\", but got \"%s\"",
            [text cString], [readText cString]];
    }
#if defined(HAVE_KEYED_ARCHIVES)
    [text RELEASE];
    text = [[OLText alloc] initWithCString: "My cat has fleas"];
    archData = [NSKeyedArchiver archivedDataWithRootObject: text];
    readText = [NSKeyedUnarchiver unarchiveObjectWithData: archData];
    if (![readText IS_MEMBER_OF: [OLText class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected OLText class, but got %s", ((Class)[readText class])->name];
    }
    if (![readText isEqual: text])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"%s\", but got \"%s\"",
            [text cString], [readText cString]];
    }
#endif
    [text RELEASE];
}
#endif

- (void) testCompare
{
    OLText* t1;
    OLText* t2;
    olchar chars[128];
    unsigned i;
    int rc;

    for (i = 0; i < 128; i++)
        chars[i] = 0x0500 + i;
    t1 = [[OLText alloc] initWithChars: chars count: 128];
    t2 = [[OLText alloc] initWithChars: chars count: 128];
    rc = [t1 compare: t2];
    if (rc != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %i", rc];
    }
    if (![t1 isEqual: t2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "\"%s\" should be equal to \"%s\"",
            [t1 cString], [t2 cString]];
    }
    [t2 RELEASE];
    chars[100] = chars[100] - 1;
    t2 = [[OLText alloc] initWithChars: chars count: 128];
    rc = [t1 compare: t2];
    if (rc <= 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected > 0, but got %i", rc];
    }
    if ([t1 isEqual: t2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "\"%s\" should not be equal to \"%s\"",
            [t1 cString], [t2 cString]];
    }
    [t2 RELEASE];
    chars[100] = chars[100] + 2;
    t2 = [[OLText alloc] initWithChars: chars count: 128];
    rc = [t1 compare: t2];
    if (rc >= 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected < 0, but got %i", rc];
    }
    if ([t1 isEqual: t2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "\"%s\" should not be equal to \"%s\"",
            [t1 cString], [t2 cString]];
    }
    [t2 RELEASE];
    chars[100] = chars[100] - 1;
    t2 = [[OLText alloc] initWithChars: chars count: 127];
    rc = [t1 compare: t2];
    if (rc <= 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected > 0, but got %i", rc];
    }
    if ([t1 isEqual: t2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "\"%s\" should not be equal to \"%s\"",
            [t1 cString], [t2 cString]];
    }
    rc = [t2 compare: t1];
    if (rc >= 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected < 0, but got %i", rc];
    }
    if ([t1 isEqual: t2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "\"%s\" should not be equal to \"%s\"",
            [t1 cString], [t2 cString]];
    }
    [t2 RELEASE];
    [t1 RELEASE];
    t1 = [[OLText alloc] initWithCString: "doggyboy"];
#if defined(OL_NO_OPENSTEP)
    rc = [@"doggyboy" compare: t1];
    if (rc != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %i", rc];
    }
    if (![@"doggyboy" isEqual: t1])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "\"doggyboy\" should be equal to \"%s\"",
            [t1 cString]];
    }
#endif
    rc = [t1 compare: @"doggyboy"];
    if (rc != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %i", rc];
    }
    if (![t1 isEqual: @"doggyboy"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "\"doggyboy\" should be equal to \"%s\"",
            [t1 cString]];
    }
#if defined(OL_NO_OPENSTEP)
    rc = [@"doggycoy" compare: t1];
    if (rc <= 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected > 0, but got %i", rc];
    }
    if ([@"doggycoy" isEqual: t1])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "\"doggycoy\" should not be equal to \"%s\"",
            [t1 cString]];
    }
#endif
    rc = [t1 compare: @"doggycoy"];
    if (rc >= 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected < 0, but got %i", rc];
    }
    if ([t1 isEqual: @"doggycoy"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "\"doggycoy\" should not be equal to \"%s\"",
            [t1 cString]];
    }
#if defined(OL_NO_OPENSTEP)
    rc = [@"doggxcoy" compare: t1];
    if (rc >= 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected < 0, but got %i", rc];
    }
    if ([@"doggxcoy" isEqual: t1])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "\"doggxcoy\" should not be equal to \"%s\"",
            [t1 cString]];
    }
#endif
    rc = [t1 compare: @"doggxcoy"];
    if (rc <= 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected > 0, but got %i", rc];
    }
    if ([t1 isEqual: @"doggxcoy"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "\"doggxcoy\" should not be equal to \"%s\"",
            [t1 cString]];
    }
    [t1 RELEASE];
}

- (void) testConvenienceAllocators
{
    OLText* text;
    OLText* text2;
    uint8_t vasyaBytes[] = { 0xb2, 0xd0, 0xe1, 0xef };
    olchar vasyaChars[] = { 0x0412, 0x0430, 0x0441, 0x044f };
    olchar doggyChars[] =
    {
        OL_SMALL_D_CHAR,
        OL_SMALL_O_CHAR,
        OL_SMALL_G_CHAR,
        OL_SMALL_G_CHAR,
        OL_SMALL_Y_CHAR
    };
    unsigned i;

    TRY

        text = REAP([OLText textWithBytes: vasyaBytes count: sizeof(vasyaBytes)
            encoding: "ISO8859-5"]);
        if ([text size] != 4)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected 4, but got %u", [text size]];
        }
        for (i = 0; i < [text size]; i++)
        {
            if ([text at: i] != vasyaChars[i])
            {
                [self errInFile: __FILE__ line: __LINE__
                    format: "Expected %0.4x, but got %0.4x", vasyaChars[i], [text at: i]];
            }
        }

    CATCH

        [self errInFile: __FILE__ line: __LINE__
            format: "%s - %s", EXCEPTION_NAME, EXCEPTION_MESSAGE];

    END_CATCH

    text = REAP([OLText textWithCString: "doggy"]);
    if ([text size] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5, but got %u", [text size]];
    }
    for (i = 0; i < [text size]; i++)
    {
        if ([text at: i] != doggyChars[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %0.4x, but got %0.4x", doggyChars[i], [text at: i]];
        }
    }
    text2 = REAP([OLText textWithText: text]);
    if ([text2 size] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5, but got %u", [text2 size]];
    }
    for (i = 0; i < [text2 size]; i++)
    {
        if ([text2 at: i] != doggyChars[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %0.4x, but got %0.4x", doggyChars[i], [text2 at: i]];
        }
    }
    if (![text isEqual: text2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The objects should be equal"];
    }
}

- (void) testCopy
{
    struct ThreadInfo info;
    info.text = [[OLText alloc] init];
    info.test = self;
    OLThread threads[10];
    unsigned i;

    [self logMessage: "Begin scrambling eggs..."];
    for (i = 0; i < 10; i++)
        OLCreateThread(&threads[i], threadMain, &info);
    for (i = 0; i < 10; i++)
        OLWaitForThread(threads[i]);
    [info.text RELEASE];
}

- (void) testExtraction
{
    OLText* t1;
    OLText* t2;
    olchar chars[128];
    olchar chars2[128];
    unsigned i;

    for (i = 0; i < 128; i++)
        chars[i] = 0x0500 + i;
    t1 = [[OLText alloc] initWithChars: chars count: 128];
    for (i = 0; i < 128; i++)
    {
        if ([t1 at: i] != chars[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                chars[i], [t1 at: i]];
            break;
        }
    }
    [t1 getCharacters: chars2 fromOffset: 64 count: 50];
    for (i = 0; i < 50; i++)
    {
        if (chars[i + 64] != chars2[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                chars[i + 64], chars2[i]];
            break;
        }
    }
    t2 = REAP([t1 substrFromOffset: 100 count: 28]);
    for (i = 0; i < 28; i++)
    {
        if ([t1 at: i + 100] != [t2 at: i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \\u%0.4X, but got \\u%0.4X",
                [t1 at: i + 100], [t2 at: i]];
            break;
        }
    }
    [t1 RELEASE];
}

- (void) testFindChar
{
    OLText* t1;
    olchar chars[128];
    unsigned i;
    unsigned rc;

    for (i = 0; i < 128; i++)
        chars[i] = 0x0500 + i;
    t1 = [[OLText alloc] initWithChars: chars count: 128];
    rc = [t1 findChar: 0x0500 + 100 fromOffset: 0];
    if (rc != 100)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 100, but got %u", rc];
    }
    rc = [t1 findChar: 77 fromOffset: 0];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    rc = [t1 findChar: 0x0500 + 110 fromOffset: 100];
    if (rc != 110)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 110, but got %u", rc];
    }
    rc = [t1 findChar: 0x0510 fromOffset: 100];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    [t1 RELEASE];
}

- (void) testFindFirstLastNotOf
{
    OLText* t1;
    OLText* t2;
    OLTextBuffer* buf;
    unsigned rc;
    int i;

    buf = [[OLTextBuffer alloc] init];
    for (i = 0; i < 10; i++)
        [buf appendChar: 0x0500 + i];
    t2 = REAP([buf text]);
    [buf clear];
    for (i = 0; i < 5000; i++)
    {
        if (i == 2499)
            [buf appendChar: 0x050A];
        else if (i == 2500)
            [buf appendChar: 0x050B];
        else
            [buf appendChar: 0x0500 + (OLRandom() % 10)];
    }
    t1 = REAP([buf text]);
    rc = [t1 findFirstNotOf: t2 fromOffset: 0];
    if (rc != 2499)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2499, but got %u", rc];
    }
    rc = [t1 findFirstNotOf: t2 fromOffset: 2499];
    if (rc != 2499)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2499, but got %u", rc];
    }
    rc = [t1 findFirstNotOf: t2 fromOffset: 2501];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    rc = [t1 findLastNotOf: t2 fromOffset: UINT_MAX];
    if (rc != 2500)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2500, but got %u", rc];
    }
    rc = [t1 findLastNotOf: t2 fromOffset: 2499];
    if (rc != 2499)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2499, but got %u", rc];
    }
    rc = [t1 findLastNotOf: t2 fromOffset: 2498];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    [buf RELEASE];
}

- (void) testFindFirstLastOf
{
    OLText* t1;
    OLText* t2;
    OLTextBuffer* buf;
    olchar seq[] = { 0x0010, 0x0020, 0x0550, 0x0551, 0x9000, 0xA000 };
    unsigned rc;
    int i;

    buf = [[OLTextBuffer alloc] init];
    for (i = 0; i < 128; i++)
        [buf appendChar: 0x0500 + i];
    t1 = REAP([buf text]);
    [buf clear];
    [buf appendChars: seq fromOffset: 0 count: sizeof(seq) / sizeof(olchar)];
    t2 = REAP([buf text]);
    rc = [t1 findFirstOf: t2 fromOffset: 0];
    if (rc != 0x50)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", 0x50, rc];
    }
    rc = [t1 findFirstOf: t2 fromOffset: 0x50];
    if (rc != 0x50)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", 0x50, rc];
    }
    rc = [t1 findFirstOf: t2 fromOffset: 0x52];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    rc = [t1 findLastOf: t2 fromOffset: UINT_MAX];
    if (rc != 0x51)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", 0x51, rc];
    }
    rc = [t1 findLastOf: t2 fromOffset: 0x51];
    if (rc != 0x51)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", 0x51, rc];
    }
    rc = [t1 findLastOf: t2 fromOffset: 0x49];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    [buf assignAt: 2 character: 0x8000];
    [buf assignAt: 3 character: 0x8001];
    t2 = REAP([buf text]);
    rc = [t1 findFirstOf: t2 fromOffset: 0];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    rc = [t1 findFirstOf: t2 fromOffset: 0x50];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    rc = [t1 findFirstOf: t2 fromOffset: 0x51];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    rc = [t1 findLastOf: t2 fromOffset: UINT_MAX];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    rc = [t1 findLastOf: t2 fromOffset: 0x51];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    rc = [t1 findLastOf: t2 fromOffset: 0x49];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    [buf RELEASE];
}

- (void) testFindText
{
    uint8_t* mapped;
    OLText* text;
    OLText* target;
    struct stat statBuf;
    OLFileInStream* finStream =
        REAP([OLFileInStream streamWithPath: "test/HamletComplete.utf16-le.txt"]);
    unsigned pos;
    olchar charsNotThere[] = { 0x000D, 0x000A, 0xFFFF, 0xFFFE, 0xFFFD, 0xFFFC, 0xFFFB, 0xFFFA };
    unsigned amountRead;

    if (stat("test/HamletComplete.utf16-le.txt", &statBuf) == -1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Count not stat test/HamletComplete.utf16-le.txt"];
        return;
    }
    mapped = (uint8_t*)malloc(statBuf.st_size);
    amountRead = [finStream readBytes: mapped count: statBuf.st_size];
    if (amountRead < statBuf.st_size)
    {
        free(mapped);
        [self errInFile: __FILE__ line: __LINE__
            format: "Couldn't read the entire file. Read %u or %u bytes.",
            amountRead, statBuf.st_size];
        return;
    }
    text = [[OLText alloc] initWithBytes: mapped + 2 count: statBuf.st_size - 2
        encoding: "UTF-16LE"];
    free(mapped);
    target = REAP([text substrFromOffset: 0 count: 20]);
    pos = [text findText: target fromOffset: 0];
    if (pos != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %u", pos];
    }
    target = REAP([text substrFromOffset: [text size] - 100 count: 100]);
    pos = [text findText: target fromOffset: [text size] / 2];
    if (pos != ([text size] - 100))
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", [text size] - 100, pos];
    }
    target = REAP([text substrFromOffset: 0x25674 / 2 count: 50]);
    pos = [text findText: target fromOffset: 0];
    if (pos != (0x25674 / 2))
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", 0x25674 / 2, pos];
    }
    target = [[OLText alloc] initWithChars: charsNotThere count: 8];
    pos = [text findText: target fromOffset: 0];
    if (pos != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, pos];
    }
    [target RELEASE];
    [text RELEASE];
}

- (void) testInitializers
{
    OLText* text;
    OLText* text2;
    uint8_t vasyaBytes[] = { 0xb2, 0xd0, 0xe1, 0xef };
    olchar vasyaChars[] = { 0x0412, 0x0430, 0x0441, 0x044f };
    olchar fleasChars[] =
    {
        OL_SMALL_F_CHAR,
        OL_SMALL_L_CHAR,
        OL_SMALL_E_CHAR,
        OL_SMALL_A_CHAR,
        OL_SMALL_S_CHAR
    };
    int i;

    text = [[OLText alloc] init];
    if ([text size] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %u", [text size]];
    }
    [text RELEASE];

    TRY

        text = [[OLText alloc] initWithBytes: vasyaBytes count: sizeof(vasyaBytes)
            encoding: "ISO8859-5"];
        if ([text size] != 4)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected 4, but got %u", [text size]];
        }
        [self logMessage: "Vasya data is:"];
        for (i = 0; i < [text size]; i++)
        {
            if ([text at: i] != vasyaChars[i])
            {
                [self errInFile: __FILE__ line: __LINE__
                    format: "Expected %0.4x, but got %0.4x", vasyaChars[i], [text at: i]];
            }
            [self logMessage: "  \\u%0.4X", [text at: i]];
        }

    CATCH

        [self errInFile: __FILE__ line: __LINE__
            format: "%s - %s", EXCEPTION_NAME, EXCEPTION_MESSAGE];

    END_CATCH

    [text RELEASE];

    TRY

        text = [[OLText alloc] initWithBytes: vasyaBytes count: sizeof(vasyaBytes)
            encoding: "MY-DOG-IS-COOLER-THAN-YOUR-DOG"];
        [self errInFile: __FILE__ line: __LINE__
            format: "The encoding named MY-DOG-IS-COOLER-THAN-YOUR-DOG should have generated an exception"];
        [text RELEASE];

    CATCH

        [self logMessage: "Got expected exception:"];
        [self logMessage: "  %s: %s", EXCEPTION_NAME, EXCEPTION_MESSAGE];

    END_CATCH

    text = [[OLText alloc] initWithChars: vasyaChars count: 4];
    if ([text size] != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 4, but got %u", [text size]];
    }
    for (i = 0; i < [text size]; i++)
    {
        if ([text at: i] != vasyaChars[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %0.4x, but got %0.4x", vasyaChars[i], [text at: i]];
        }
    }
    [text RELEASE];

    text = [[OLText alloc] initWithCString: "fleas"];
    if ([text size] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5, but got %u", [text size]];
    }
    for (i = 0; i < [text size]; i++)
    {
        if ([text at: i] != fleasChars[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %0.4x, but got %0.4x", fleasChars[i], [text at: i]];
        }
    }
    text2 = [[OLText alloc] initWithText: text];
    if ([text2 size] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5, but got %u", [text2 size]];
    }
    for (i = 0; i < [text2 size]; i++)
    {
        if ([text2 at: i] != fleasChars[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %0.4x, but got %0.4x", fleasChars[i], [text2 at: i]];
        }
    }
    if (![text isEqual: text2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The objects should be equal"];
    }
    [text2 RELEASE];

    text2 = [[OLText alloc] initWithText: text offset: 2 count: 2];
    if ([text2 size] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %u", [text2 size]];
    }
    for (i = 0; i < [text2 size]; i++)
    {
        if ([text2 at: i] != fleasChars[i + 2])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %0.4x, but got %0.4x", fleasChars[i + 2], [text2 at: i]];
        }
    }
    if ([text isEqual: text2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The objects should not be equal"];
    }
    [text RELEASE];
    [text2 RELEASE];
}

- (void) testProperties
{
    OLText* t1;
    OLTextBuffer* buf = REAP([OLTextBuffer textBuffer]);
    OLNumber* hash;
    OLSet* set = REAP([OLSet set]);
    int i;
    int j;
    unsigned collisionCount = 0;

    t1 = [[OLText alloc] init];
    if (![t1 empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The text should be empty"];
    }
    [t1 RELEASE];
    for (i = 0; i < 5000; i++)
        [buf appendChar: OLRandom() & 0xFFFF];
    t1 = REAP([buf text]);
    if ([t1 length] != 5000)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5000, but got %u", [t1 length]];
    }
    if ([t1 size] != 5000)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5000, but got %u", [t1 size]];
    }
    [self logMessage: "The max size of OLText is %u", [t1 maxSize]];
    [self logMessage: "Checking 10,000 hashes"];
    for (i = 0; i < 10000; i++)
    {
        [buf clear];
        for (j = 0; j < 5000; j++)
            [buf appendChar: OLRandom() & 0xFFFF];
        hash = [[OLNumber alloc] initWithUnsignedInt: [REAP([buf text]) hash]];
        if (![[REAP([set insert: hash]) second] boolValue])
            collisionCount++;
        [hash RELEASE];
        if (!(i % 500))
        {
            fprintf([self outStream], ".");
            fflush([self outStream]);
        }
    }
    fprintf([self outStream], "\n");
    [self logMessage: "10,000 random texts of 5,000 characters each produced %u hash collisions",
        collisionCount];
}

- (void) testRFindChar
{
    OLTextBuffer* buf;
    OLText* text;
    int i;
    unsigned rc;

    buf = [[OLTextBuffer alloc] init];
    for (i = 0; i < 5000; i++)
    {
        if (i == 2501)
            [buf appendChar: 0x0500 + i - 1];
        else
            [buf appendChar: 0x0500 + i];
    }
    text = REAP([buf text]);
    rc = [text rfindChar: 0x0500 + 2500 fromOffset: UINT_MAX];
    if (rc != 2501)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2501, but got %u", rc];
    }
    rc = [text rfindChar: 0x0500 + 2500 fromOffset: 2500];
    if (rc != 2500)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2500, but got %u", rc];
    }
    rc = [text rfindChar: 0x0500 + 2500 fromOffset: 2499];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    rc = [text rfindChar: 0x000A fromOffset: UINT_MAX];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    rc = [text rfindChar: 0x0500 + 2500 fromOffset: 0];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    rc = [text rfindChar: 0x000A fromOffset: 0];
    if (rc != UINT_MAX)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected %u, but got %u", UINT_MAX, rc];
    }
    [buf RELEASE];
}

- (void) testStreaming
{
    OLDataOutStream* dout;
    OLObjectOutStream* oout;
    OLObjectInStream* oin;
    OLText* text;
    OLText* readText;

    dout = REAP([OLDataOutStream stream]);
    oout = REAP([OLObjectOutStream streamWithOutStream: dout]);
    text = REAP([OLText textWithCString: "My dog has fleas"]);
    [oout writeObject: text];
    oin = REAP([OLObjectInStream streamWithInStream:
        REAP([OLDataInStream streamWithBytes: [dout bytes] count: [dout count]])]);
    readText = REAP([oin readObject]);
    if (![readText IS_MEMBER_OF: [OLText class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected OLText class, but got %s", ((Class)[readText class])->name];
    }
    if (![readText isEqual: text])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"%s\", but got \"%s\"",
            [text cString], [readText cString]];
    }
}

@end
