//
// $Id: ListTest.m,v 1.25 2007/03/28 03:16:52 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "ListTest.h"
#import "Random.h"
#import <ObjectiveLib/List.h>
#import <ObjectiveLib/Vector.h>
#import <ObjectiveLib/DataOutStream.h>
#import <ObjectiveLib/DataInStream.h>
#import <ObjectiveLib/ObjectOutStream.h>
#import <ObjectiveLib/ObjectInStream.h>
#import "Number.h"
#if defined(OL_NO_OPENSTEP)
#import <ObjectiveLib/Reaper.h>
#import <ObjectiveLib/Text.h>
#else
#import <Foundation/NSString.h>
#import <Foundation/NSData.h>
#import <Foundation/NSArchiver.h>
#if defined(HAVE_KEYED_ARCHIVES)
#import <Foundation/NSKeyedArchiver.h>
#endif
#endif
#include <stdlib.h>
#if defined(__NEXT_RUNTIME__)
#import <objc/objc-class.h>
#endif

@interface FirstLetterSame :
#if defined(OL_NO_OPENSTEP)
    Object
#else
    NSObject
#endif
    <OLBoolBinaryFunction>
{
}

- (BOOL) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2;

@end

@implementation ListTest

- (void) testAccessors
{
    OLList* l;
    CONSTSTR* strs[] = { @"one", @"two", @"three", @"four", @"five" };
    int i;

    l = [[OLList alloc] init];
    for (i = 0; i < 5; i++)
        [l pushBack: strs[i]];
    if (![[l back] isEqual: @"five"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"five\", but got \"%s\"", [[l back] cString]];
    }
    if (![[l front] isEqual: @"one"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"one\", but got \"%s\"", [[l front] cString]];
    }
    [l RELEASE];
}

- (void) testAssign
{
    OLList* l;
    OLList* l2;
    OLListIterator* itor;
    OLListIterator* end;

    l = [[OLList alloc] init];
    [l pushBack: @"one"];
    [l pushBack: @"two"];
    [l pushBack: @"three"];
    l2 = [l copy];

    [l assign: 2 filledWith: @"four"];
    if ([l size] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %i", [l size]];
    }
    end = REAP([l end]);
    for (itor = REAP([l begin]); ![itor isEqual: end]; [itor advance])
    {
        if (![[itor dereference] isEqual: @"four"])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"four\", but got \"%s\"", [[itor dereference] cString]];
        }
    }
    [l assignFrom: REAP([l2 begin]) to: REAP([l2 end])];
    [l2 RELEASE];
    if ([l size] != 3)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 3, but got %i", [l size]];
    }
    itor = REAP([l begin]);
    if (![[itor dereference] isEqual: @"one"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"one\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"two"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"two\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"three"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"three\", but got \"%s\"", [[itor dereference] cString]];
    }
    [l RELEASE];
}

#if !defined(OL_NO_OPENSTEP)
- (void) testCoding
{
    NSMutableData* data;
    NSArchiver* archiver;
    NSData* archData;
    OLList* l1;
    OLList* l2;
    OLNumber* num;
    int i;


    l1 = [[OLList alloc] init];
    for (i = 0; i< 5000; i++)
    {
        num = [[OLNumber alloc] initWithInt: OLRandom() % 10000];
        [l1 pushBack: num];
        [num RELEASE];
    }
    data = [[NSMutableData alloc] initWithCapacity: 5000];
    archiver = [[NSArchiver alloc] initForWritingWithMutableData: data];
    [archiver encodeRootObject: l1];
    [archiver RELEASE];
    l2 = [NSUnarchiver unarchiveObjectWithData: data];
    [data RELEASE];
    if (![l1 isEqual: l2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The lists should be equal"];
    }
#if defined(HAVE_KEYED_ARCHIVES)
    archData = [NSKeyedArchiver archivedDataWithRootObject: l1];
    l2 = [NSKeyedUnarchiver unarchiveObjectWithData: archData];
    if (![l1 isEqual: l2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The lists should be equal"];
    }
#endif
    [l1 RELEASE];
}
#endif

- (void) testConvenienceAllocators
{
    OLList* l;
    OLList* l2;
    OLNumber* num;

    l = REAP([OLList list]);
    if (![l IS_MEMBER_OF: [OLList class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"%s\", but got \"%s\"",
            ((Class)[OLList class])->name, ((Class)[l class])->name];
    }
    if (![l empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The list should be empty"];
    }
    num = [[OLNumber alloc] initWithInt: 5];
    [l pushBack: num];
    [num RELEASE];
    num = [[OLNumber alloc] initWithInt: 6];
    [l pushBack: num];
    [num RELEASE];
    num = [[OLNumber alloc] initWithInt: 7];
    [l pushBack: num];
    [num RELEASE];
    l2 = REAP([OLList listFrom: REAP([l begin]) to: REAP([l end])]);
    if (![l isEqual: l2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The lists should be equal"];
    }
    num = [[OLNumber alloc] initWithInt: 8];
    [l pushBack: num];
    [num RELEASE];
    num = [[OLNumber alloc] initWithInt: 9];
    [l pushBack: num];
    [num RELEASE];
    l2 = REAP([OLList listWithList: l]);
    if (![l isEqual: l2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The lists should be equal"];
    }
    num = [[OLNumber alloc] initWithInt: 10];
    l = REAP([OLList listWithSize: 666 filledWith: num]);
    [num RELEASE];
    if (![l IS_MEMBER_OF: [OLList class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"%s\", but got \"%s\"",
            ((Class)[OLList class])->name, ((Class)[l class])->name];
    }
    if ([l size] != 666)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 666, but got %u", [l size]];
    }
}

- (void) testErase
{
    CONSTSTR* strs[] = { @"one", @"two", @"three" };
    OLList* l;
    int i;
    OLListIterator* itor;
    OLGreater* greater;
    OLBoolBinder2nd* binder;

    l = [[OLList alloc] init];
    for (i = 0; i < 3; i++)
        [l pushBack: strs[i]];
    itor = REAP([l begin]);
    [itor advance];
    itor = REAP([l erase: itor]);
    if ([l size] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %i", [l size]];
    }
    if (![[itor dereference] isEqual: strs[2]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"three\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor reverse];
    if (![[itor dereference] isEqual: strs[0]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"one\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    itor = REAP([l eraseFrom: itor to: REAP([l end])]);
    if ([l size] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %i", [l size]];
    }
    if (![itor isEqual: REAP([l end])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The iterator should be at the end"];
    }
    [itor reverse];
    if (![[itor dereference] isEqual: strs[0]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"one\", but got \"%s\"", [[itor dereference] cString]];
    }
    [l clear];
    if ([l size] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %i", [l size]];
    }
    for (i = 0; i < 3; i++)
        [l pushBack: strs[i]];
    [l popFront];
    if ([l size] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %i", [l size]];
    }
    itor = REAP([l begin]);
    if (![[itor dereference] isEqual: strs[1]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"two\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: strs[2]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"three\", but got \"%s\"", [[itor dereference] cString]];
    }
    [l popBack];
    if ([l size] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %i", [l size]];
    }
    if (![[l front] isEqual: strs[1]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"two\", but got \"%s\"", [[l front] cString]];
    }
    [l clear];
    for (i = 0; i < 3; i++)
        [l pushBack: strs[i]];
    [l remove: @"two"];
    if ([l size] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %i", [l size]];
    }
    itor = REAP([l begin]);
    if (![[itor dereference] isEqual: strs[0]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"one\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: strs[2]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"three\", but got \"%s\"", [[itor dereference] cString]];
    }
    [l clear];
    for (i = 0; i < 3; i++)
        [l pushBack: strs[i]];
    greater = [[OLGreater alloc] init];
    binder = [[OLBoolBinder2nd alloc] initWithBoolFunction: greater andRightArg: @"socks"];
    [greater RELEASE];
    [l removeIf: binder];
    [binder RELEASE];
    if ([l size] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %i", [l size]];
    }
    if (![[l front] isEqual: strs[0]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"one\", but got \"%s\"", [[l front] cString]];
    }
    [l RELEASE];
}

- (void) testInitializers
{
    OLList* l;
    OLVector* v;
    CONSTSTR* strs[] = { @"one", @"two", @"three", @"four", @"five" };
    int i;
    OLList* l2;

    l = [[OLList alloc] init];
    if ([l size] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %i", [l size]];
    }
    if (![l empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The list must be empty"];
    }
    [l RELEASE];

    l = [[OLList alloc] initWithSize: 5 filledWith: @"one"];
    if ([l size] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5, but got %i", [l size]];
    }
    for (i = 0; i < 5; i++)
    {
        if (![[l front] isEqual: @"one"])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"one\", but got \"%s\"", [[l front] cString]];
        }
        [l popFront];
    }
    if (![l empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The list should be empty"];
    }
    [l RELEASE];


    v = [[OLVector alloc] init];
    for (i = 0; i < 5; i++)
        [v pushBack: strs[i]];
    l = [[OLList alloc] initFrom: REAP([v begin]) to: REAP([v end])];
    if ([l size] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5, but got %i", [l size]];
    }
    for (i = 0; i < 5; i ++)
    {
        if (![[l front] isEqual: strs[i]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Item at index %i is wrong", i];
        }
        [l popFront];
    }
    if (![l empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The list should be empty"];
    }
    [l RELEASE];

    l = [[OLList alloc] initFrom: REAP([v begin]) to: REAP([v end])];
    [v RELEASE];
    l2 = [[OLList alloc] initWithList: l];
    if ([l2 size] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5, but got %i", [l2 size]];
    }
    if (![l isEqual: l2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The lists should be equal"];
    }
    for (i = 0; i < 5; i++)
    {
        if (![[l front] isEqual: [l2 front]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "The elements at index %i should be equal", i];
        }
        [l popFront];
        [l2 popFront];
    }
    [l RELEASE];
    [l2 RELEASE];
}

- (void) testInsert
{
    OLList* l;
    CONSTSTR* strs[] = { @"one", @"two", @"three", @"four", @"five" };
    int i;
    OLListIterator* itor;
    OLListIterator* where;
    OLListIterator* end;
    OLVector* v;

    l = [[OLList alloc] init];
    REAP([l insertAt: REAP([l begin]) value: strs[0]]);
    if ([l size] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %i", [l size]];
    }
    if (![[l front] isEqual: strs[0]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"one\", but got %s", [[l front] cString]];
    }
    itor = REAP([l insertAt: REAP([l end]) value: strs[2]]);
    if ([l size] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %i", [l size]];
    }
    if (![[l front] isEqual: strs[0]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"one\", but got %i", [[l front] cString]];
    }
    if (![[l back] isEqual: strs[2]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"three\", but got %s", [[l back] cString]];
    }
    where = REAP([l insertAt: itor value: strs[1]]);
    if ([l size] != 3)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 3, but got %i", [l size]];
    }
    end = REAP([l end]);
    for (itor = REAP([l begin]), i = 0; ![itor isEqual: end]; [itor advance], i++)
    {
        if (![[itor dereference] isEqual: strs[i]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"%s\", but got \"%s\"",
                [strs[i] cString], [[itor dereference] cString]];
        }
    }
    v = [[OLVector alloc] init];
    [v pushBack: REAP([OLNumber numberWithInt: 1])];
    [v pushBack: REAP([OLNumber numberWithInt: 2])];
    [v pushBack: REAP([OLNumber numberWithInt: 3])];
    [l insertAt: where from: REAP([v begin]) to: REAP([v end])];
    if ([l size] != 6)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 6, but got %i", [l size]];
    }
    itor = REAP([l begin]);
    if (![[itor dereference] isEqual: strs[0]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"%s\", but got \"%s\"",
            [strs[0] cString], [[itor dereference] cString]];
    }
    for (i = 0; i < 3; i++)
    {
        if (![[[itor advance] dereference] isEqual: [v at: i]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"%s\", but got \"%s\"",
                [[v at: i] cString], [[itor dereference] cString]];
        }
    }
    [v RELEASE];
    for (i = 1; i < 3; i++)
    {
        if (![[[itor advance] dereference] isEqual: strs[i]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"%s\", but got \"%s\"",
                [strs[i] cString], [[itor dereference] cString]];
        }
    }
    [l insertAt: REAP([l begin]) count: 3 filledWith: @"doggy"];
    itor = REAP([l begin]);
    for (i = 0; i < 3; i++)
    {
        if (![[itor dereference] isEqual: @"doggy"])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"doggy\", but got \"%s\"", [[itor dereference] cString]];
        }
        [itor advance];
    }
    [l RELEASE];
}

- (void) testIterators
{
    OLList* l;
    CONSTSTR* strs[] = { @"one", @"two", @"three", @"four", @"five" };
    int i;
    OLListIterator* itor;
    OLReverseBidiIterator* ritor;
    OLReverseBidiIterator* rend;

    l = [[OLList alloc] init];
    for (i = 0; i < 5; i++)
        [l pushBack: strs[i]];
    itor = REAP([l begin]);
    for (i = 0; i < 5; i++)
    {
        if (![[itor dereference] isEqual: strs[i]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"%s\", but got \"%s\"",
                [strs[i] cString], [[itor dereference] cString]];
        }
        [itor advance];
    }
    if (![itor isEqual: REAP([l end])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The iterator should be at the end"];
    }
    for (i = 4; i >= 0; i--)
    {
        [itor reverse];
        if (![[itor dereference] isEqual: strs[i]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"%s\", but got \"%s\"",
                [strs[i] cString], [[itor dereference] cString]];
        }
    }
    rend = REAP([l rend]);
    for (ritor = REAP([l rbegin]), i = 4; ![ritor isEqual: rend]; [ritor advance], i--)
    {
        if (![[ritor dereference] isEqual: strs[i]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"%s\", but got \"%s\"",
                [strs[i] cString], [[itor dereference] cString]];
        }
    }
    ritor = [ritor copy];
    if (![ritor isEqual: REAP([l rend])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The reverse iterator should be at the reverse end"];
    }
    [ritor RELEASE];
    itor = [itor copy];
    if (![itor isEqual: REAP([l begin])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The iterator should be at the beginning"];
    }
    [itor RELEASE];
    [l RELEASE];
}

- (void) testProperties
{
    OLList* l;
    OLList* l2;
    CONSTSTR* strs[] = { @"one", @"two", @"three", @"four", @"five" };
    int i;

    l = [[OLList alloc] init];
    if (![l empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The list must report that it is empty"];
    }
    l2 = [[OLList alloc] init];
    for (i = 0; i < 5; i++)
    {
        [l pushBack: strs[i]];
        [l2 pushBack: strs[i]];
    }
    if (![l isEqual: l2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The lists must be equal"];
    }
    if ([l size] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5, but got %i", [l size]];
    }
    if ([l empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The list cannot report that it is empty"];
    }
    [l clear];
    if ([l isEqual: l2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The lists cannot be equal"];
    }
    if ([l maxSize] != [l2 maxSize])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "All lists must have the same max size"];
    }
    [self logMessage: "The max size of a list is %u", [l maxSize]];
    [l RELEASE];
    [l2 RELEASE];
}

- (void) testMerge
{
    CONSTSTR* strs1[] = { @"Daniela", @"Helena" };
    CONSTSTR* strs2[] = { @"Clara", @"Yehudi" };
    int i;
    OLList* l;
    OLList* l2;
    OLListIterator* itor;
    OLGreater* greater;

    l = [[OLList alloc] init];
    for (i = 0; i < 2; i++)
        [l pushBack: strs1[i]];
    l2 = [[OLList alloc] init];
    for (i = 0; i < 2; i++)
        [l2 pushBack: strs2[i]];
    [l merge: l2];
    if ([l size] != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 4, but got %i", [l size]];
    }
    itor = REAP([l begin]);
    if (![[itor dereference] isEqual: @"Clara"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"Clara\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"Daniela"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"Daniela\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"Helena"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"Helena\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"Yehudi"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"Yehudi\", but got \"%s\"", [[itor dereference] cString]];
    }
    if (![l2 empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The list must be empty"];
    }
    [l clear];
    for (i = 1; i >= 0; i--)
        [l pushBack: strs1[i]];
    for (i = 1; i >= 0; i--)
        [l2 pushBack: strs2[i]];
    greater = [[OLGreater alloc] init];
    [l merge: l2 withOrder: greater];
    [greater RELEASE];
    if ([l size] != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 4, but got %i", [l size]];
    }
    itor = REAP([l begin]);
    if (![[itor dereference] isEqual: @"Yehudi"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"Yehudi\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"Helena"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"Helena\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"Daniela"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"Daniela\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"Clara"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"Clara\", but got \"%s\"", [[itor dereference] cString]];
    }
    if (![l2 empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The list must be empty"];
    }
    [l RELEASE];
    [l2 RELEASE];
}

- (void) testResize
{
    CONSTSTR* strs[] = { @"one", @"two", @"three" };
    OLList* l;
    OLListIterator* itor;
    int i;

    l = [[OLList alloc] init];
    for (i = 0; i < 3; i++)
        [l pushBack: strs[i]];
    [l resize: 2 filledWith: @"doggy"];
    if ([l size] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %i", [l size]];
    }
    if (![[l front] isEqual: @"one"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"one\", but got \"%s\"", [[l front] cString]];
    }
    if (![[l back] isEqual: @"two"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"two\", but got \"%s\"", [[l back] cString]];
    }
    [l resize: 4 filledWith: @"doggy"];
    if ([l size] != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 4, but got %i", [l size]];
    }
    itor = REAP([l begin]);
    if (![[itor dereference] isEqual: @"one"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"one\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"two"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"two\", but got \"%s\"", [[itor dereference] cString]];
    }
    for (i = 0; i < 2; i++)
    {
        if (![[[itor advance] dereference] isEqual: @"doggy"])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"doggy\", but got \"%s\"", [[itor dereference] cString]];
        }
    }
    [l RELEASE];
}

- (void) testReverse
{
    CONSTSTR* strs[] = { @"one", @"two", @"three", @"four", @"five" };
    OLList* l;
    int i;
    OLListIterator* itor;
    OLListIterator* end;

    l = [[OLList alloc] init];
    for (i = 0; i < 5; i++)
        [l pushBack: strs[i]];
    [l reverse];
    end = REAP([l end]);
    for (itor = REAP([l begin]), i = 4; ![itor isEqual: end]; [itor advance], i--)
    {
        if (![[itor dereference] isEqual: strs[i]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"%s\", but got \"%s\"",
                [strs[i] cString], [[itor dereference] cString]];
        }
    }
    [l clear];
    [l reverse];
    if ([l size] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The list must be empty"];
    }
    [l RELEASE];
}

- (void) testSort
{
    CONSTSTR* strs[] = { @"Yehudi", @"Helena", @"Paula", @"Clara", @"Daniela" };
    OLList* l;
    OLListIterator* itor;
    int i;

    l = [[OLList alloc] init];
    for (i = 0; i < 5; i++)
        [l pushBack: strs[i]];
    [l sort];
    itor = REAP([l begin]);
    if (![[itor dereference] isEqual: @"Clara"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"Clara\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"Daniela"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"Daniela\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"Helena"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"Helena\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"Paula"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"Paula\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"Yehudi"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"Yehudi\", but got \"%s\"", [[itor dereference] cString]];
    }
    [l RELEASE];
}

- (void) testSplice
{
    OLList* l;
    OLList* l2;
    CONSTSTR* strs1[] = { @"one", @"five" };
    CONSTSTR* strs2[] = { @"two", @"three", @"four" };
    OLListIterator* itor;
    int i;

    l = [[OLList alloc] init];
    for (i = 0; i < 2; i++)
        [l pushBack: strs1[i]];
    l2 = [[OLList alloc] init];
    for (i = 0; i < 3; i++)
        [l2 pushBack: strs2[i]];
    [l spliceAt: [REAP([l begin]) advance] list: l2
        from: [REAP([l2 begin]) advance] to: [[REAP([l2 begin]) advance] advance]];
    if ([l size] != 3)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 3, but got %i", [l size]];
    }
    itor = REAP([l begin]);
    if (![[itor dereference] isEqual: @"one"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"one\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"three"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"three\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"five"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"five\", but got \"%s\"", [[itor dereference] cString]];
    }
    if ([l2 size] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %i", [l2 size]];
    }
    if (![[l2 front] isEqual: @"two"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"two\", but got \"%s\"", [[itor dereference] cString]];
    }
    if (![[l2 back] isEqual: @"four"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"four\", but got \"%s\"", [[itor dereference] cString]];
    }
    [l clear];
    for (i = 0; i < 2; i++)
        [l pushBack: strs1[i]];
    [l2 clear];
    for (i = 0; i < 3; i++)
        [l2 pushBack: strs2[i]];
    [l spliceAt: [REAP([l begin]) advance] list: l2 from: [REAP([l2 begin]) advance]];
    if ([l size] != 3)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 3, but got %i", [l size]];
    }
    itor = REAP([l begin]);
    if (![[itor dereference] isEqual: @"one"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"one\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"three"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"three\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"five"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"five\", but got \"%s\"", [[itor dereference] cString]];
    }
    if ([l2 size] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %i", [l2 size]];
    }
    if (![[l2 front] isEqual: @"two"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"two\", but got \"%s\"", [[l2 front] cString]];
    }
    if (![[l2 back] isEqual: @"four"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"four\", but got \"%s\"", [[l2 back] cString]];
    }
    [l clear];
    for (i = 0; i < 2; i++)
        [l pushBack: strs1[i]];
    [l2 clear];
    for (i = 0; i < 3; i++)
        [l2 pushBack: strs2[i]];
    [l spliceAt: [REAP([l begin]) advance] list: l2];
    if ([l size] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5, but got %i", [l size]];
    }
    itor = REAP([l begin]);
    if (![[itor dereference] isEqual: @"one"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"one\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"two"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"two\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"three"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"three\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"four"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"four\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"five"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"five\", but got \"%s\"", [[itor dereference] cString]];
    }
    if (![l2 empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The list must be empty"];
    }
    [l RELEASE];
    [l2 RELEASE];
}

- (void) testStreaming
{
    OLDataOutStream* dout;
    OLObjectOutStream* oout;
    OLObjectInStream* oin;
    OLList* list;
    OLList* readList;
    OLNumber* num;
    int i;

    dout = REAP([OLDataOutStream stream]);
    oout = REAP([OLObjectOutStream streamWithOutStream: dout]);
    list = [[OLList alloc] init];
    for (i = 0; i< 10000; i++)
    {
        num = [[OLNumber alloc] initWithInt: OLRandom() % 10000];
        [list pushBack: num];
        [num RELEASE];
    }
    [oout writeObject: list];
    oin = REAP([OLObjectInStream streamWithInStream:
        REAP([OLDataInStream streamWithBytes: [dout bytes] count: [dout count]])]);
    readList = REAP([oin readObject]);
    if (![readList IS_MEMBER_OF: [OLList class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected OLList class, but got %s", ((Class)[readList class])->name];
    }
    if (![readList isEqual: list])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The lists should be equal"];
    }
    [list RELEASE];
}

- (void) testSwap
{
    OLList* l;
    OLList* l2;
    CONSTSTR* strs1[] = { @"one", @"two", @"three" };
    CONSTSTR* strs2[] = { @"four", @"five" };
    int i;
    OLListIterator* itor;
    OLListIterator* end;

    l = [[OLList alloc] init];
    for (i = 0; i < 3; i++)
        [l pushBack: strs1[i]];
    l2 = [[OLList alloc] init];
    for (i = 0; i < 2; i++)
        [l2 pushBack: strs2[i]];
    [l swap: l2];
    if ([l size] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %i", [l size]];
    }
    if ([l2 size] != 3)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 3, but got %i", [l2 size]];
    }
    end = REAP([l end]);
    for (itor = REAP([l begin]), i = 0; ![itor isEqual: end]; [itor advance], i++)
    {
        if (![[itor dereference] isEqual: strs2[i]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"%s\", but got \"%s\"",
                [strs2[i] cString], [[itor dereference] cString]];
        }
    }
    [l RELEASE];
    end = REAP([l2 end]);
    for (itor = REAP([l2 begin]), i = 0; ![itor isEqual: end]; [itor advance], i++)
    {
        if (![[itor dereference] isEqual: strs1[i]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"%s\", but got \"%s\"",
                [strs1[i] cString], [[itor dereference] cString]];
        }
    }
    [l2 RELEASE];
}

- (void) testUnique
{
    CONSTSTR* strs[] = { @"Clara", @"Clara", @"Daniela", @"Danny", @"Helena", @"Helena", @"Helena", @"Yehudi" };
    CONSTSTR* strs2[] = { @"Clara", @"Daniela", @"Danny", @"Helena", @"Yehudi" };
    OLList* l;
    OLListIterator* itor;
    OLListIterator* end;
    int i;
    FirstLetterSame* pred;

    l = [[OLList alloc] init];
    for (i = 0; i < 8; i++)
        [l pushBack: strs[i]];
    [l unique];
    if ([l size] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5, but got %i", [l size]];
    }
    end = REAP([l end]);
    for (itor = REAP([l begin]), i = 0; ![itor isEqual: end]; [itor advance], i++)
    {
        if (![[itor dereference] isEqual: strs2[i]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"%s\", but got \"%s\"",
                [strs[i] cString], [[itor dereference] cString]];
        }
    }
    [l clear];
    for (i = 0; i < 8; i++)
        [l pushBack: strs[i]];
    pred = [[FirstLetterSame alloc] init];
    [l uniqueWith: pred];
    [pred RELEASE];
    if ([l size] != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 4, but got %i", [l size]];
    }
    itor = REAP([l begin]);
    if (![[itor dereference] isEqual: @"Clara"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"Clara\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"Daniela"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"Daniela\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"Helena"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"Helena\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![[itor dereference] isEqual: @"Yehudi"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"Yehudi\", but got \"%s\"", [[itor dereference] cString]];
    }
    [l RELEASE];
}

@end

@implementation FirstLetterSame

- (BOOL) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2
{
#if defined(OL_NO_OPENSTEP)
    return [((OLConstantString*)arg1) cString][0] ==
           [((OLConstantString*)arg2) cString][0] ? YES : NO;
#else
    return [arg1 characterAtIndex: 0] == [arg2 characterAtIndex: 0] ? YES : NO;
#endif
}

@end


