//
// $Id: ZlibOutStream.h,v 1.13 2007/03/06 20:42:20 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#if !defined(ZLIBOUTSTREAM_OL_GUARD)
#define ZLIBOUTSTREAM_OL_GUARD

#include <ObjectiveLib/LayeredOutStream.h>

struct z_stream_s;

/**
 * @class OLZlibOutStream ZlibOutStream.h ObjectiveLib/ZlibOutStream.h
 *
 * A stream for zlib compression. This is a concrete layered stream that compresses all
 * data written using the library zlib.
 *
 * @sa OLZlibInStream
 *
 * @ingroup Streams
 */
@interface OLZlibOutStream : OLLayeredOutStream
{
@protected
    /**
     * The opaque data type used by zlib to represent the stream
     */
    struct z_stream_s*  zstream;

    /**
     * The buffer required by zlib
     */
    uint8_t*            buffer;

    /**
     * The size of the zlib buffer
     */
    unsigned            bufferSize;

    /**
     * Indicator of whether the compression has finished yet
     */
    BOOL                isFinished;
}

/**
 * Create and return a new compression stream. The stream uses
 * the default compression level and a buffer size of 1,024. The given out stream
 * @a underStream is the next lower stream in the stream stack.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @exception OLInputOutputException if the stream cannot be created for deflation
 * @param underStream the next lower stream in the stack
 * @return a reference to the newly created stream
 */
+ (id) streamWithOutStream: (OLOutStream*)underStream;

/**
 * Create and return a new compression stream. The stream uses
 * the given compression level and a buffer size of 1,024. The given out stream
 * @a underStream is the next lower stream in the stream stack.
 *
 * @pre The indicated compression level must be in the range [0,9].
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @if OL_NO_OPENSTEP
 * @exception OLInvalidArgumentException if the compression level is not in the range [0,9]
 * @else
 * @exception NSInvalidArgumentException if the compression level is not in the range [0,9]
 * @endif
 * @exception OLInputOutputException if the stream cannot be created for deflation
 * @param underStream the next lower stream in the stack
 * @param level the compression level which must be in the range [0,9]
 * @return a reference to the newly created stream
 */
+ (id) streamWithOutStream: (OLOutStream*)underStream compressionLevel: (int)level;

/**
 * @name Initializers and Deallocators
 */
/* @{ */
/**
 * Initialize the compression stream. The stream uses
 * the default compression level and a buffer size of 1,024. The given out stream
 * @a underStream is the next lower stream in the stream stack.
 *
 * @exception OLInputOutputException if the stream cannot be created for deflation
 * @param underStream the next lower stream in the stack
 * @return a reference to this stream
 */
- (id) initWithOutStream: (OLOutStream*)underStream;

/**
 * Initialize the compression stream. The stream uses
 * the given compression level and a buffer size of 1,024. The given out stream
 * @a underStream is the next lower stream in the stream stack.
 *
 * @pre The indicated compression level must be in the range [0,9].
 *
 * @if OL_NO_OPENSTEP
 * @exception OLInvalidArgumentException if the compression level is not in the range [0,9]
 * @else
 * @exception NSInvalidArgumentException if the compression level is not in the range [0,9]
 * @endif
 * @exception OLInputOutputException if the stream cannot be created for deflation
 * @param underStream the next lower stream in the stack
 * @param level the compression level which must be in the range [0,9]
 * @return a reference to this stream
 */
- (id) initWithOutStream: (OLOutStream*)underStream compressionLevel: (int)level;

/**
 * Initialize the compression stream. The stream uses
 * the given compression level and buffer size. The given out stream
 * @a underStream is the next lower stream in the stream stack.
 *
 * @pre The indicated compression level must be in the range [0,9].
 *
 * @if OL_NO_OPENSTEP
 * @exception OLInvalidArgumentException if the compression level is not in the range [0,9]
 * @else
 * @exception NSInvalidArgumentException if the compression level is not in the range [0,9]
 * @endif
 * @exception OLInputOutputException if the stream cannot be created for deflation
 * @param underStream the next lower stream in the stack
 * @param level the compression level which must be in the range [0,9]
 * @param size the size of the internal buffer
 * @return a reference to this stream
 */
- (id) initWithOutStream: (OLOutStream*)underStream compressionLevel: (int)level bufferSize: (unsigned)size;

/**
 * Initialize the compression stream. The stream uses
 * the given compression level and buffer size. The given out stream
 * @a underStream is the next lower stream in the stream stack. If @a zlibHeader is
 * YES, then the zlib header will be written to the stream, otherwise it will not be
 * written.
 *
 * @attention You probably do not want to use this initializer. This initializer exists so that
 * subclasses will have maximum control over how the zstream is created. Do not use this
 * initializer unless you know what you are doing.
 *
 * @pre The indicated compression level must be in the range [0,9].
 *
 * @exception NSInvalidArgumentException if the compression level is not in the range [0,9]
 * @exception OLInputOutputException if the stream cannot be created for deflation
 * @param underStream the next lower stream in the stack
 * @param level the compression level which must be in the range [0,9]
 * @param size the size of the internal buffer
 * @param zlibHeader should be YES if the zlib header should be written to the stream,
 * otherwise NO
 * @return a reference to this stream
 */
- (id) initWithOutStream: (OLOutStream*)underStream compressionLevel: (int)level bufferSize: (unsigned)size writeZlibHeader: (BOOL)zlibHeader;
#if !defined(OL_NO_OPENSTEP)
- (void) dealloc;
#endif
/* @} */

- (void) close;

#if defined(OL_NO_OPENSTEP)
- (void) freeStreamResources;
#endif
/**
 * Write a sequence of bytes to the stream. Compress up to @a num bytes and return the
 * number actually written.
 *
 * @exception OLInputOutputException if there is an error writing to the stream
 * @param bytes the bytes to write
 * @param num the number of bytes to write
 * @return the number of bytes written
 */
- (unsigned) writeBytes: (const uint8_t*)bytes count: (unsigned)num;

@end

#endif
