//
// $Id: HashMap.m,v 1.17 2007/03/06 20:42:19 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "HashMap.h"
#import "HashTable.h"
#import "Macros.h"
#import "RunTime.h"
#import "ObjectInStream.h"
#import "ObjectOutStream.h"
#import "Utility.h"
#if !defined(OL_NO_OPENSTEP)
#import <Foundation/NSCoder.h>
#import <Foundation/NSAutoreleasePool.h>
#import <Foundation/NSString.h>
#endif
#import <stdlib.h>

#if !defined(OL_NO_OPENSTEP)
NSString* const KEYEQUAL_KEY = @"OL_KEYEQUAL_KEY";
NSString* const TABLE_SIZE_KEY = @"OL_TABLE_SIZE_KEY";
#endif

@implementation OLHashMap

+ (id) hashMap
{
    OL_BEGIN_AUTO_CTOR(OLHashMap)
        init
    OL_END_AUTO_CTOR;
}

+ (id) hashMapFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    OL_BEGIN_AUTO_CTOR(OLHashMap)
        initFrom: first to: last
    OL_END_AUTO_CTOR;
}

+ (id) hashMapWithHashMap: (OLHashMap*)right
{
    OL_BEGIN_AUTO_CTOR(OLHashMap)
        initWithHashMap: right
    OL_END_AUTO_CTOR;
}

- (id) init
{
    return [self initWithTableSize: 100];
}

- (id) initFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    [self init];
    [self insertFrom: first to: last];
    return self;
}

- (id) initFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last tableSize: (unsigned)size
{
    [self initWithTableSize: size];
    [self insertFrom: first to: last];
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) initFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last tableSize: (unsigned)size keyEqual: (Object<OLBoolBinaryFunction>*)eq
#else
- (id) initFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last tableSize: (unsigned)size keyEqual: (NSObject<OLBoolBinaryFunction>*)eq
#endif
{
    [self initWithTableSize: size keyEqual: eq];
    [self insertFrom: first to: last];
    return self;
}

#if !defined(OL_NO_OPENSTEP)
- (id) initWithCoder: (NSCoder*)decoder
{
    NSObject<OLBoolBinaryFunction>* eq;
    unsigned tableSize;

    [super init];
    if ([decoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [decoder allowsKeyedCoding])
    {
        eq = [decoder decodeObjectForKey: KEYEQUAL_KEY];
        tableSize = [decoder decodeIntForKey: TABLE_SIZE_KEY];
    }
    else
    {
        eq = [decoder decodeObject];
        [decoder decodeValueOfObjCType: @encode(unsigned) at: &tableSize];
    }
    table = [[OLHashTableMap alloc] initWithSize: tableSize equalFunc: eq];
    readContainerWithInsertRange(self, decoder, @selector(decodeObject));
    return self;
}
#endif

- (id) initWithHashMap: (OLHashMap*)right
{
    [super init];
    table = [[OLHashTableMap alloc] initWithHashTable: right->table];
    return self;
}

- (id) initWithTableSize: (unsigned)size
{
    OLEqualTo* eq = [[OLEqualTo alloc] init];

    [self initWithTableSize: size keyEqual: eq];
    OBJ_RELEASE(eq);
    return self;
}

- (id) initWithObjectInStream: (OLObjectInStream*)stream
{
#if defined(OL_NO_OPENSTEP)
    Object<OLBoolBinaryFunction>* eq = [stream readObject];
#else
    NSObject<OLBoolBinaryFunction>* eq = [stream readObject];
#endif
    unsigned tableSize = [stream readInt];

    [super init];
    table = [[OLHashTableMap alloc] initWithSize: tableSize equalFunc: eq];
#if defined(OL_NO_OPENSTEP)
    [eq free];
#endif
    readContainerWithInsertRange(self, stream, @selector(readObject));
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) initWithTableSize: (unsigned)size keyEqual: (Object<OLBoolBinaryFunction>*)eq
#else
- (id) initWithTableSize: (unsigned)size keyEqual: (NSObject<OLBoolBinaryFunction>*)eq
#endif
{
    [super init];
    table = [[OLHashTableMap alloc] initWithSize: size equalFunc: eq];
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    OBJ_RELEASE(table);
    SUPER_FREE;
}

- (void) assignKey: (id)key value: (id)value
{
    [table assignUniqueKey: key value: value];
}

- (OLHashIterator*) begin
{
    return [table begin];
}

- (void) clear
{
    [table clear];
}

- (int) compare: (id)other
{
    if (IS_KIND_OF(other, OLHashMap))
        return [table compare: ((OLHashMap*)other)->table];
    return -1;
}

#if defined(OL_NO_OPENSTEP)

- (id) copy
{
    return [[OLHashMap alloc] initWithHashMap: self];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [[OLHashMap allocWithZone: zone] initWithHashMap: self];
}

#endif

- (unsigned) count: (id)key
{
    return [table count: key];
}

- (BOOL) empty
{
    return [table empty];
}

#if !defined(OL_NO_OPENSTEP)
- (void) encodeWithCoder: (NSCoder*)encoder
{
    unsigned tableSize = [table tableSize];

    if ([encoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [encoder allowsKeyedCoding])
    {
        [encoder encodeObject: [table keyEqual] forKey: KEYEQUAL_KEY];
        [encoder encodeInt: tableSize forKey: TABLE_SIZE_KEY];
    }
    else
    {
        [encoder encodeObject: [table keyEqual]];
        [encoder encodeValueOfObjCType: @encode(unsigned) at: &tableSize];
    }
    writeContainer(table, @selector(beginImpl), @selector(endImpl),
        encoder, @selector(encodeObject:));
}
#endif

- (OLHashIterator*) end
{
    return [table end];
}

- (OLPair*) equalRange: (id)key
{
    return [table equalRange: key];
}

- (void) erase: (OLHashIterator*)where
{
    [table erase: where];
}

- (void) eraseFrom: (OLHashIterator*)first to: (OLHashIterator*)last
{
    [table eraseFrom: first to: last];
}

- (unsigned) eraseKey: (id)key
{
    return [table eraseKey: key];
}

- (OLHashIterator*) find: (id)key
{
    return [table find: key];
}

- (id) insert: (OLPair*)keyValue
{
    return [table insertUnique: keyValue];
}

- (void) insertFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    [table insertUniqueFrom: first to: last];
}

- (id) insertKey: (id)key value: (id)val
{
    OLPair* p = [[OLPair alloc] initWithFirst: key second: val];
    id r = [self insert: p];

    OBJ_RELEASE(p);
    return r;
}

- (BOOL) isEqual: (id)object
{
    return IS_KIND_OF(object, OLHashMap) &&
           [((OLHashMap*)object)->table isEqualUnique: table];
}

#if defined(OL_NO_OPENSTEP)
- (Object<OLBoolBinaryFunction>*) keyEqual
#else
- (NSObject<OLBoolBinaryFunction>*) keyEqual
#endif
{
    return [table keyEqual];
}

- (unsigned) maxSize
{
    return [table maxSize];
}

- (unsigned) size
{
    return [table size];
}

- (void) swap: (OLHashMap*)right
{
    [table swap: right->table];
}

- (id) valueForKey: (id)key
{
    return [table findValue: key];
}

- (void) writeSelfToStream: (OLObjectOutStream*)stream
{
    [stream writeObject: [table keyEqual]];
    [stream writeInt: [table tableSize]];
    writeContainer(table, @selector(beginImpl), @selector(endImpl),
        stream, @selector(writeObject:));
}

@end

@implementation OLHashMultiMap

+ (id) hashMultiMap
{
    OL_BEGIN_AUTO_CTOR(OLHashMultiMap)
        init
    OL_END_AUTO_CTOR;
}

+ (id) hashMultiMapFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    OL_BEGIN_AUTO_CTOR(OLHashMultiMap)
        initFrom: first to: last
    OL_END_AUTO_CTOR;
}

+ (id) hashMultiMapWithHashMap: (OLHashMap*)right
{
    OL_BEGIN_AUTO_CTOR(OLHashMultiMap)
        initWithHashMap: right
    OL_END_AUTO_CTOR;
}

- (void) assignKey: (id)key value: (id)value
{
    [table assignEqualKey: key value: value];
}

- (id) insert: (OLPair*)keyValue
{
    return [table insertEqual: keyValue];
}

- (void) insertFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    [table insertEqualFrom: first to: last];
}

- (BOOL) isEqual: (id)object
{
    return IS_KIND_OF(object, OLHashMultiMap) &&
           [((OLHashMultiMap*)object)->table isEqualNonUnique: table];
}

- (id) valueForKey: (id)key
{
    return nil;
}

@end
