//
// $Id: Functional.m,v 1.29 2007/03/25 18:12:15 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "Functional.h"
#import "Arithmetic.h"
#import "Macros.h"
#import "RunTime.h"
#import "ObjectInStream.h"
#import "ObjectOutStream.h"
#if !defined(OL_NO_OPENSTEP)
#import <Foundation/NSCoder.h>
#import <Foundation/NSString.h>
#endif
#import <string.h>

#if !defined(OL_NO_OPENSTEP)
NSString* const FUNCTION_KEY = @"OL_FUNCTION_KEY";
NSString* const ARGUMENT_KEY = @"OL_ARGUMENT_KEY";
#endif

@protocol OLFunctionalUnknownMethods

- (BOOL) boolValue;

@end

@implementation OLFunctor

+ (id) functorOfType: (OLFunctorType)type
{
    static BOOL initialized = NO;
    static Class typeClasses[OLFunctorType_Plus + 1];

    if (!initialized)
    {
        typeClasses[OLFunctorType_Divides] = [OLDivides class];
        typeClasses[OLFunctorType_EqualTo] = [OLEqualTo class];
        typeClasses[OLFunctorType_Greater] = [OLGreater class];
        typeClasses[OLFunctorType_GreaterEqual] = [OLGreaterEqual class];
        typeClasses[OLFunctorType_Less] = [OLLess class];
        typeClasses[OLFunctorType_LessEqual] = [OLLessEqual class];
        typeClasses[OLFunctorType_LogicalAnd] = [OLLogicalAnd class];
        typeClasses[OLFunctorType_LogicalNot] = [OLLogicalNot class];
        typeClasses[OLFunctorType_LogicalOr] = [OLLogicalOr class];
        typeClasses[OLFunctorType_Minus] = [OLMinus class];
        typeClasses[OLFunctorType_Modulus] = [OLModulus class];
        typeClasses[OLFunctorType_Multiplies] = [OLMultiplies class];
        typeClasses[OLFunctorType_Negate] = [OLNegate class];
        typeClasses[OLFunctorType_NotEqualTo] = [OLNotEqualTo class];
        typeClasses[OLFunctorType_Plus] = [OLPlus class];
        initialized = YES;
    }
    return OBJ_AUTORELEASE([[typeClasses[type] alloc] init]);
}

@end

@implementation OLStreamableFunctor

#if !defined(OL_NO_OPENSTEP)
- (id) initWithCoder: (NSCoder*)decoder
{
    return [super init];
}
#endif

- (id) initWithObjectInStream: (OLObjectInStream*)stream
{
    return [self init];
}

#if !defined(OL_NO_OPENSTEP)
- (void) encodeWithCoder: (NSCoder*)encoder
{
}
#endif

- (void) writeSelfToStream: (OLObjectOutStream*)stream
{
}

@end

@implementation OLBinaryNegate

+ (id) functorWithBinaryFunction: (OLStreamableFunctor<OLBoolBinaryFunction>*)function
{
    OL_BEGIN_AUTO_CTOR(OLBinaryNegate)
        initWithBinaryFunction: function
    OL_END_AUTO_CTOR;
}

#if !defined(OL_NO_OPENSTEP)
- (id) initWithCoder: (NSCoder*)decoder
{
    [super initWithCoder: decoder];
    if ([decoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [decoder allowsKeyedCoding])
    {
        fn = OBJ_RETAIN_AUTO([decoder decodeObjectForKey: FUNCTION_KEY]);
    }
    else
    {
        fn = OBJ_RETAIN_AUTO([decoder decodeObject]);
    }
    return self;
}
#endif

- (id) initWithBinaryFunction: (OLStreamableFunctor<OLBoolBinaryFunction>*)function
{
    [super init];
    fn = OBJ_RETAIN(function);
    return self;
}

- (id) initWithObjectInStream: (OLObjectInStream*)stream;
{
    [super initWithObjectInStream: stream];
    fn = OBJ_RETAIN_AUTO([stream readObject]);
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    OBJ_RELEASE(fn);
    SUPER_FREE;
}

#if !defined(OL_NO_OPENSTEP)
- (void) encodeWithCoder: (NSCoder*)encoder
{
    [super encodeWithCoder: encoder];
    if ([encoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [encoder allowsKeyedCoding])
    {
        [encoder encodeObject: fn forKey: FUNCTION_KEY];
    }
    else
    {
        [encoder encodeObject: fn];
    }
}
#endif

- (BOOL) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2
{
    return [fn performBinaryFunctionWithArg: arg1 andArg: arg2] ? NO : YES;
}

- (void) writeSelfToStream: (OLObjectOutStream*)stream
{
    [super writeSelfToStream: stream];
    [stream writeObject: fn];
}

@end

@implementation OLBinder1st

+ (id) functorWithFunction: (OLStreamableFunctor<OLBinaryFunction>*)function andLeftArg: (id)left
{
    OL_BEGIN_AUTO_CTOR(OLBinder1st)
        initWithFunction: function andLeftArg: left
    OL_END_AUTO_CTOR;
}

#if !defined(OL_NO_OPENSTEP)
- (id) initWithCoder: (NSCoder*)decoder
{
    [super initWithCoder: decoder];
    if ([decoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [decoder allowsKeyedCoding])
    {
        fn = OBJ_RETAIN_AUTO([decoder decodeObjectForKey: FUNCTION_KEY]);
        lft = OBJ_RETAIN_AUTO([decoder decodeObjectForKey: ARGUMENT_KEY]);
    }
    else
    {
        fn = OBJ_RETAIN_AUTO([decoder decodeObject]);
        lft = OBJ_RETAIN_AUTO([decoder decodeObject]);
    }
    return self;
}
#endif

- (id) initWithFunction: (OLStreamableFunctor<OLBinaryFunction>*)function andLeftArg: (id)left
{
    [super init];
    fn = OBJ_RETAIN(function);
    lft = OBJ_RETAIN(left);
    return self;
}

- (id) initWithObjectInStream: (OLObjectInStream*)stream
{
    [super initWithObjectInStream: stream];
    fn = OBJ_RETAIN_AUTO([stream readObject]);
    lft = OBJ_RETAIN_AUTO([stream readObject]);
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    OBJ_RELEASE(fn);
    OBJ_RELEASE(lft);
    SUPER_FREE;
}

#if !defined(OL_NO_OPENSTEP)
- (void) encodeWithCoder: (NSCoder*)encoder
{
    [super encodeWithCoder: encoder];
    if ([encoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [encoder allowsKeyedCoding])
    {
        [encoder encodeObject: fn forKey: FUNCTION_KEY];
        [encoder encodeObject: lft forKey: ARGUMENT_KEY];
    }
    else
    {
        [encoder encodeObject: fn];
        [encoder encodeObject: lft];
    }
}
#endif

- (id) performUnaryFunctionWithArg: (id)arg
{
    return [fn performBinaryFunctionWithArg: lft andArg: arg];
}

- (void) writeSelfToStream: (OLObjectOutStream*)stream
{
    [super writeSelfToStream: stream];
    [stream writeObject: fn];
    [stream writeObject: lft];
}

@end

@implementation OLBinder2nd

+ (id) functorWithFunction: (OLStreamableFunctor<OLBinaryFunction>*)function andRightArg: (id)right
{
    OL_BEGIN_AUTO_CTOR(OLBinder2nd)
        initWithFunction: function andRightArg: right
    OL_END_AUTO_CTOR;
}

#if !defined(OL_NO_OPENSTEP)
- (id) initWithCoder: (NSCoder*)decoder
{
    [super initWithCoder: decoder];
    if ([decoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [decoder allowsKeyedCoding])
    {
        fn = OBJ_RETAIN_AUTO([decoder decodeObjectForKey: FUNCTION_KEY]);
        rght = OBJ_RETAIN_AUTO([decoder decodeObjectForKey: ARGUMENT_KEY]);
    }
    else
    {
        fn = OBJ_RETAIN_AUTO([decoder decodeObject]);
        rght = OBJ_RETAIN_AUTO([decoder decodeObject]);
    }
    return self;
}
#endif

- (id) initWithFunction: (OLStreamableFunctor<OLBinaryFunction>*)function andRightArg: (id)right
{
    [super init];
    fn = OBJ_RETAIN(function);
    rght = OBJ_RETAIN(right);
    return self;
}

- (id) initWithObjectInStream: (OLObjectInStream*)stream
{
    [super initWithObjectInStream: stream];
    fn = OBJ_RETAIN_AUTO([stream readObject]);
    rght = OBJ_RETAIN_AUTO([stream readObject]);
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    OBJ_RELEASE(fn);
    OBJ_RELEASE(rght);
    SUPER_FREE;
}

#if !defined(OL_NO_OPENSTEP)
- (void) encodeWithCoder: (NSCoder*)encoder
{
    [super encodeWithCoder: encoder];
    if ([encoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [encoder allowsKeyedCoding])
    {
        [encoder encodeObject: fn forKey: FUNCTION_KEY];
        [encoder encodeObject: rght forKey: ARGUMENT_KEY];
    }
    else
    {
        [encoder encodeObject: fn];
        [encoder encodeObject: rght];
    }
}
#endif

- (id) performUnaryFunctionWithArg: (id)arg
{
    return [fn performBinaryFunctionWithArg: arg andArg: rght];
}

- (void) writeSelfToStream: (OLObjectOutStream*)stream
{
    [super writeSelfToStream: stream];
    [stream writeObject: fn];
    [stream writeObject: rght];
}

@end

@implementation OLBoolBinder1st

+ (id) functorWithBoolFunction: (OLStreamableFunctor<OLBoolBinaryFunction>*)function andLeftArg: (id)left
{
    OL_BEGIN_AUTO_CTOR(OLBoolBinder1st)
        initWithBoolFunction: function andLeftArg: left
    OL_END_AUTO_CTOR;
}

- (id) initWithBoolFunction: (OLStreamableFunctor<OLBoolBinaryFunction>*)function andLeftArg: (id)left
{
    [super init];
    fn = OBJ_RETAIN(function);
    lft = OBJ_RETAIN(left);
    return self;
}

#if !defined(OL_NO_OPENSTEP)
- (id) initWithCoder: (NSCoder*)decoder
{
    [super initWithCoder: decoder];
    if ([decoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [decoder allowsKeyedCoding])
    {
        fn = OBJ_RETAIN_AUTO([decoder decodeObjectForKey: FUNCTION_KEY]);
        lft = OBJ_RETAIN_AUTO([decoder decodeObjectForKey: ARGUMENT_KEY]);
    }
    else
    {
        fn = OBJ_RETAIN_AUTO([decoder decodeObject]);
        lft = OBJ_RETAIN_AUTO([decoder decodeObject]);
    }
    return self;
}
#endif

- (id) initWithObjectInStream: (OLObjectInStream*)stream
{
    [super initWithObjectInStream: stream];
    fn = OBJ_RETAIN_AUTO([stream readObject]);
    lft = OBJ_RETAIN_AUTO([stream readObject]);
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    OBJ_RELEASE(fn);
    OBJ_RELEASE(lft);
    SUPER_FREE;
}

#if !defined(OL_NO_OPENSTEP)
- (void) encodeWithCoder: (NSCoder*)encoder
{
    [super encodeWithCoder: encoder];
    if ([encoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [encoder allowsKeyedCoding])
    {
        [encoder encodeObject: fn forKey: FUNCTION_KEY];
        [encoder encodeObject: lft forKey: ARGUMENT_KEY];
    }
    else
    {
        [encoder encodeObject: fn];
        [encoder encodeObject: lft];
    }
}
#endif

- (BOOL) performUnaryFunctionWithArg: (id)arg
{
    return [fn performBinaryFunctionWithArg: lft andArg: arg];
}

- (void) writeSelfToStream: (OLObjectOutStream*)stream
{
    [super writeSelfToStream: stream];
    [stream writeObject: fn];
    [stream writeObject: lft];
}

@end

@implementation OLBoolBinder2nd

+ (id) functorWithBoolFunction: (OLStreamableFunctor<OLBoolBinaryFunction>*)function andRightArg: (id)right
{
    OL_BEGIN_AUTO_CTOR(OLBoolBinder2nd)
        initWithBoolFunction: function andRightArg: right
    OL_END_AUTO_CTOR;
}

- (id) initWithBoolFunction: (OLStreamableFunctor<OLBoolBinaryFunction>*)function andRightArg: (id)right
{
    [super init];
    fn = OBJ_RETAIN(function);
    rght = OBJ_RETAIN(right);
    return self;
}

#if !defined(OL_NO_OPENSTEP)
- (id) initWithCoder: (NSCoder*)decoder
{
    [super initWithCoder: decoder];
    if ([decoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [decoder allowsKeyedCoding])
    {
        fn = OBJ_RETAIN_AUTO([decoder decodeObjectForKey: FUNCTION_KEY]);
        rght = OBJ_RETAIN_AUTO([decoder decodeObjectForKey: ARGUMENT_KEY]);
    }
    else
    {
        fn = OBJ_RETAIN_AUTO([decoder decodeObject]);
        rght = OBJ_RETAIN_AUTO([decoder decodeObject]);
    }
    return self;
}
#endif

- (id) initWithObjectInStream: (OLObjectInStream*)stream
{
    [super initWithObjectInStream: stream];
    fn = OBJ_RETAIN_AUTO([stream readObject]);
    rght = OBJ_RETAIN_AUTO([stream readObject]);
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    OBJ_RELEASE(fn);
    OBJ_RELEASE(rght);
    SUPER_FREE;
}

#if !defined(OL_NO_OPENSTEP)
- (void) encodeWithCoder: (NSCoder*)encoder
{
    [super encodeWithCoder: encoder];
    if ([encoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [encoder allowsKeyedCoding])
    {
        [encoder encodeObject: fn forKey: FUNCTION_KEY];
        [encoder encodeObject: rght forKey: ARGUMENT_KEY];
    }
    else
    {
        [encoder encodeObject: fn];
        [encoder encodeObject: rght];
    }
}
#endif

- (BOOL) performUnaryFunctionWithArg: (id)arg
{
    return [fn performBinaryFunctionWithArg: arg andArg: rght];
}

- (void) writeSelfToStream: (OLObjectOutStream*)stream
{
    [super writeSelfToStream: stream];
    [stream writeObject: fn];
    [stream writeObject: rght];
}

@end

@implementation OLDivides

- (id) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2
{
    return [arg1 arithmeticDivideBy: arg2];
}

@end

@implementation OLEqualTo

- (BOOL) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2
{
    return [arg1 isEqual: arg2];
}

@end

@implementation OLGreater

- (BOOL) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2
{
    return [arg1 compare: arg2] > 0 ? YES : NO;
}

@end

@implementation OLGreaterEqual

- (BOOL) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2
{
    return [arg1 compare: arg2] >= 0 ? YES : NO;
}

@end

@implementation OLLess

- (BOOL) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2
{
    return [arg1 compare: arg2] < 0 ? YES : NO;
}

@end

@implementation OLLessEqual

- (BOOL) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2
{
    return [arg1 compare: arg2] <= 0 ? YES : NO;
}

@end

@implementation OLLogicalAnd

- (BOOL) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2
{
    return [arg1 boolValue] && [arg2 boolValue] ? YES : NO;
}

@end

@implementation OLLogicalNot

- (BOOL) performUnaryFunctionWithArg: (id)arg
{
    return ![arg boolValue] ? YES : NO;
}

@end

@implementation OLLogicalOr

- (BOOL) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2
{
    return [arg1 boolValue] || [arg2 boolValue] ? YES : NO;
}

@end

@implementation OLMemFun

+ (id) functorWithSelector: (SEL)selector
{
    OL_BEGIN_AUTO_CTOR(OLMemFun)
        initWithSelector: selector
    OL_END_AUTO_CTOR;
}

#if !defined(OL_NO_OPENSTEP)
- (id) initWithCoder: (NSCoder*)decoder
{
    [super initWithCoder: decoder];
    [decoder decodeValueOfObjCType: @encode(SEL) at: &sel];
    return self;
}
#endif

- (id) initWithObjectInStream: (OLObjectInStream*)stream
{
    [super initWithObjectInStream: stream];
    sel = [stream readSelector];
    return self;
}

- (id) initWithSelector: (SEL)selector
{
    [super init];
    sel = selector;
    return self;
}

#if !defined(OL_NO_OPENSTEP)
- (void) encodeWithCoder: (NSCoder*)encoder
{
    [super encodeWithCoder: encoder];
    [encoder encodeValueOfObjCType: @encode(SEL) at: &sel];
}
#endif

- (id) performUnaryFunctionWithArg: (id)arg
{
#if defined(OL_NO_OPENSTEP)
    return [arg perform: sel];
#else
    return [arg performSelector: sel];
#endif
}

- (void) writeSelfToStream: (OLObjectOutStream*)stream
{
    [super writeSelfToStream: stream];
    [stream writeSelector: sel];
}

@end

@implementation OLMemFun1

+ (id) functorWithSelector: (SEL)selector
{
    OL_BEGIN_AUTO_CTOR(OLMemFun1)
        initWithSelector: selector
    OL_END_AUTO_CTOR;
}

#if !defined(OL_NO_OPENSTEP)
- (id) initWithCoder: (NSCoder*)decoder
{
    [super initWithCoder: decoder];
    [decoder decodeValueOfObjCType: @encode(SEL) at: &sel];
    return self;
}
#endif

- (id) initWithObjectInStream: (OLObjectInStream*)stream
{
    [super initWithObjectInStream: stream];
    sel = [stream readSelector];
    return self;
}

- (id) initWithSelector: (SEL)selector
{
    [super init];
    sel = selector;
    return self;
}

#if !defined(OL_NO_OPENSTEP)
- (void) encodeWithCoder: (NSCoder*)encoder
{
    [super encodeWithCoder: encoder];
    [encoder encodeValueOfObjCType: @encode(SEL) at: &sel];
}
#endif

- (id) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2
{
#if defined(OL_NO_OPENSTEP)
    return [arg1 perform: sel with: arg2];
#else
    return [arg1 performSelector: sel withObject: arg2];
#endif
}

- (void) writeSelfToStream: (OLObjectOutStream*)stream
{
    [super writeSelfToStream: stream];
    [stream writeSelector: sel];
}

@end

@implementation OLMinus

- (id) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2
{
    return [arg1 arithmeticSubtract: arg2];
}

@end

@implementation OLModulus

- (id) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2
{
    return [arg1 arithmeticModulus: arg2];
}

@end

@implementation OLMultiplies

- (id) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2
{
    return [arg1 arithmeticMultiply: arg2];
}

@end

@implementation OLNegate

- (id) performUnaryFunctionWithArg: (id)arg
{
    return [arg arithmeticNegate];
}

@end

@implementation OLNotEqualTo

- (BOOL) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2
{
    return ![arg1 isEqual: arg2];
}

@end

@implementation OLPlus

- (id) performBinaryFunctionWithArg: (id)arg1 andArg: (id)arg2
{
    return [arg1 arithmeticAdd: arg2];
}

@end

@implementation OLUnaryNegate

+ (id) functorWithUnaryFunction: (OLStreamableFunctor<OLBoolUnaryFunction>*)function
{
    OL_BEGIN_AUTO_CTOR(OLUnaryNegate)
        initWithUnaryFunction: function
    OL_END_AUTO_CTOR;
}

#if !defined(OL_NO_OPENSTEP)
- (id) initWithCoder: (NSCoder*)decoder
{
    [super initWithCoder: decoder];
    if ([decoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [decoder allowsKeyedCoding])
    {
        fn = OBJ_RETAIN_AUTO([decoder decodeObjectForKey: FUNCTION_KEY]);
    }
    else
    {
        fn = OBJ_RETAIN_AUTO([decoder decodeObject]);
    }
    return self;
}
#endif

- (id) initWithObjectInStream: (OLObjectInStream*)stream
{
    [super initWithObjectInStream: stream];
    fn = OBJ_RETAIN_AUTO([stream readObject]);
    return self;
}

- (id) initWithUnaryFunction: (OLStreamableFunctor<OLBoolUnaryFunction>*)function
{
    [super init];
    fn = OBJ_RETAIN(function);
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    OBJ_RELEASE(fn);
    SUPER_FREE;
}

#if !defined(OL_NO_OPENSTEP)
- (void) encodeWithCoder: (NSCoder*)encoder
{
    [super encodeWithCoder: encoder];
    if ([encoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [encoder allowsKeyedCoding])
    {
        [encoder encodeObject: fn forKey: FUNCTION_KEY];
    }
    else
    {
        [encoder encodeObject: fn];
    }
}
#endif

- (BOOL) performUnaryFunctionWithArg: (id)arg
{
    return [fn performUnaryFunctionWithArg: arg] ? NO : YES;
}

- (void) writeSelfToStream: (OLObjectOutStream*)stream
{
    [super writeSelfToStream: stream];
    [stream writeObject: fn];
}

@end
