//
// $Id: Algorithm.m,v 1.35 2007/03/28 03:16:51 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "Algorithm.h"
#import "Macros.h"
#import "RunTime.h"
#if defined(OL_NO_OPENSTEP)
#import "Text.h"
#else
#import <Foundation/NSString.h>
#import <Foundation/NSAutoreleasePool.h>
#endif
#if defined(OL_HAVE_RAND_S)
#define _CRT_RAND_S
#endif
#include <stdlib.h>

// Macros {{{
#define OL_SORT_THRESHOLD 16
#define OL_CHUNK_SIZE 7
// }}}

// Static data {{{
#if defined(OL_NO_OPENSTEP)
static OLConstantString* const __bufferJunk = @"will_mason's_mas_vale_cholo_buffer_junk";
#else
static NSString* const __bufferJunk = @"will_mason's_mas_vale_cholo_buffer_junk";
#endif
// }}}

// interface OLTempBuf {{{
@interface OLTempBuf :
#if defined(OL_NO_OPENSTEP)
    Object
#else
    NSObject
#endif
{
@private
    id*                 buffer;
    OLArrayIterator*    begin;
    OLArrayIterator*    end;
    unsigned            size;
}

- (id) initWithFirst: (OLForwardIterator*)first last: (OLForwardIterator*)last;
#if defined(OL_NO_OPENSTEP)
- (id) free;
#else
- (void) dealloc;
#endif

- (OLArrayIterator*) begin;
- (OLArrayIterator*) end;
- (unsigned) size;

@end
// }}}

// interface OLAlgorithm (PrivateMethods) {{{
@interface OLAlgorithm (PrivateMethods)

+ (void) adjustHeapFirst: (OLRandomAccessIterator*)first hole: (unsigned)hole len: (unsigned)len value: (id)object predicate: (id<OLBoolBinaryFunction>)pred;

+ (OLForwardIterator*) boundImplFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last value: (id)object predicate: (id<OLBoolBinaryFunction>)pred lower: (BOOL)lwr;

+ (void) chunkInsertionSortFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last chunkSize: (unsigned)size predicate: (id<OLBoolBinaryFunction>)pred;

+ (OLBidirectionalIterator*) copyBackwardImplFrom: (OLBidirectionalIterator*)first to: (OLBidirectionalIterator*)last destination: (OLBidirectionalIterator*)dest needItor: (BOOL)needItor;

+ (OLForwardIterator*) copyImplFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last destination: (OLForwardIterator*)dest needItor: (BOOL)needItor;

+ (void) finalInsertionSortFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last predicate: (id<OLBoolBinaryFunction>)pred;

+ (OLForwardIterator*) findIfImplFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last predicate: (id<OLBoolUnaryFunction>)pred;

+ (OLForwardIterator*) findImplFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last value: (id)object;

+ (void) insertionSortFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last predicate: (id<OLBoolBinaryFunction>)pred;

+ (void) introSortLoopFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last depthLimit: (unsigned)depthLimit predicate: (id<OLBoolBinaryFunction>)pred;

+ (void) linearInsertFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last value: (id)object predicate: (id<OLBoolBinaryFunction>)pred;

+ (id) medianFromOne: (id)one two: (id)two three: (id)three predicate: (id<OLBoolBinaryFunction>)pred;

+ (void) mergeAdaptiveFrom: (OLBidirectionalIterator*)first
                            middle: (OLBidirectionalIterator*)middle
                            last: (OLBidirectionalIterator*)last
                            len1: (unsigned)len1
                            len2: (unsigned)len2
                            buffer: (OLTempBuf*)buf
                            predicate: (id<OLBoolBinaryFunction>)pred;

+ (OLBidirectionalIterator*) mergeBackwardSeries1From: (OLBidirectionalIterator*)first1 series1To: (OLBidirectionalIterator*)last1 series2From: (OLBidirectionalIterator*)first2 series2To: (OLBidirectionalIterator*)last2 destination: (OLBidirectionalIterator*)result predicate: (id<OLBoolBinaryFunction>)pred needItor: (BOOL)needItor;

+ (OLForwardIterator*) mergeImplSeries1From: (OLForwardIterator*)first1 series1To: (OLForwardIterator*)last1 series2From: (OLForwardIterator*)first2 series2To: (OLForwardIterator*)last2 destination: (OLForwardIterator*)dest predicate: (id<OLBoolBinaryFunction>)pred needItor: (BOOL)needItor;

+ (void) mergeSortLoopFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last destination: (OLRandomAccessIterator*)dest stepSize: (unsigned)stepSize predicate: (id<OLBoolBinaryFunction>)pred;

+ (void) mergeSortWithBufferFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last buffer: (OLTempBuf*)buf predicate: (id<OLBoolBinaryFunction>)pred;

+ (void) popHeapImplFirst: (OLRandomAccessIterator*)first last: (OLRandomAccessIterator*)last result: (OLRandomAccessIterator*)result value: (id)object predicate: (id<OLBoolBinaryFunction>)pred;

+ (void) pushHeapImplFirst: (OLRandomAccessIterator*)first hole: (unsigned)hole top: (unsigned)top value: (id)object predicate: (id<OLBoolBinaryFunction>)pred;

+ (OLBidirectionalIterator*) rotateAdaptiveFirst: (OLBidirectionalIterator*)first
        middle: (OLBidirectionalIterator*)middle
        last: (OLBidirectionalIterator*)last
        len1: (unsigned)len1
        len2: (unsigned)len2
        buffer: (OLTempBuf*)buf;

+ (OLForwardIterator*) rotateImplFrom: (OLForwardIterator*)first middle: (OLForwardIterator*)mid to: (OLForwardIterator*)last;

+ (OLForwardIterator*) stablePartitionAdaptiveFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last predicate: (id<OLBoolUnaryFunction>)pred length: (unsigned)len buffer: (OLTempBuf*)buf;

+ (void) stableSortAdaptiveFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last buffer: (OLTempBuf*)buf predicate: (id<OLBoolBinaryFunction>)pred;

+ (void) unguardedLinearInsertAt: (OLRandomAccessIterator*)last value: (id)object predicate: (id<OLBoolBinaryFunction>)pred;

+ (void) unguardedInsertionSortFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last predicate: (id<OLBoolBinaryFunction>)pred;

+ (OLRandomAccessIterator*) unguardedPartitionFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last pivot: (id)object predicate: (id<OLBoolBinaryFunction>)pred;

@end
// }}}

// interface OLUnsignedNumber {{{
@interface OLUnsignedNumber :
#if defined(OL_NO_OPENSTEP)
    Object
#else
    NSObject
#endif
{
@private
    unsigned value;
}

- (id) initWithUnsignedInt: (unsigned)num;
- (unsigned) unsignedIntValue;

@end
// }}}

// implementation OLAlgorithm {{{
@implementation OLAlgorithm

// adjacentFindFrom:to: {{{
+ (OLForwardIterator*) adjacentFindFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    OLEqualTo* equal = [[OLEqualTo alloc] init];
    OLForwardIterator* result = [OLAlgorithm adjacentFindFrom: first to: last predicate: equal];

    OBJ_RELEASE(equal);
    return result;
}
// }}}

// adjacentFindFrom:to:predicate: {{{
+ (OLForwardIterator*) adjacentFindFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last predicate: (id<OLBoolBinaryFunction>) pred
{
    OLForwardIterator* result = nil;
    OLForwardIterator* cur;
    OLForwardIterator* next;

    if ([first isEqual: last])
    {
        result = [last copy];
    }
    else
    {
        cur = [first copy];
        next = [first copy];
        while (![[next advance] isEqual: last])
        {
            if ([pred performBinaryFunctionWithArg: [cur dereference]
                    andArg: [next dereference]])
            {
                result = cur;
                break;
            }
            [cur advance];
        }
        if (result == nil)
        {
            result = [last copy];
            OBJ_RELEASE(cur);
        }
        OBJ_RELEASE(next);
    }
    return OBJ_AUTORELEASE(result);
}
// }}}

// binarySearchFrom:to:value: {{{
+ (BOOL) binarySearchFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last value: (id)object
{
    OLLess* less = [[OLLess alloc] init];
    BOOL result = [OLAlgorithm binarySearchFrom: first to: last value: object predicate: less];

    OBJ_RELEASE(less);
    return result;
}
// }}}

// binarySearchFrom:to:value:predicate: {{{
+ (BOOL) binarySearchFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last value: (id)object predicate: (id<OLBoolBinaryFunction>)pred
{
    OLForwardIterator* itor =
        [OLAlgorithm boundImplFrom: first to: last value: object predicate: pred lower: YES];
    BOOL result = ![itor isEqual: last] &&
        ![pred performBinaryFunctionWithArg: object andArg: [itor dereference]];

    OBJ_RELEASE(itor);
    return result;
}
// }}}

// copyBackwardFrom:to:destination: {{{
+ (OLBidirectionalIterator*) copyBackwardFrom: (OLBidirectionalIterator*)first to: (OLBidirectionalIterator*)last destination: (OLBidirectionalIterator*)dest
{
    return OBJ_AUTORELEASE([OLAlgorithm copyBackwardImplFrom: first to: last destination: dest needItor: YES]);
}
// }}}

// copyFrom:to:destination: {{{
+ (OLForwardIterator*) copyFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last destination: (OLForwardIterator*)dest
{
    return OBJ_AUTORELEASE([OLAlgorithm copyImplFrom: first to: last destination: dest needItor: YES]);
}
// }}}

// countFrom:to:predicate: {{{
+ (unsigned) countFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last predicate: (id<OLBoolUnaryFunction>)pred
{
    OLForwardIterator* src = [first copy];
    unsigned result = 0;

    for ( ; ![src isEqual: last]; [src advance])
    {
        if ([pred performUnaryFunctionWithArg: [src dereference]])
            result++;
    }
    OBJ_RELEASE(src);
    return result;
}
// }}}

// countFrom:to:value: {{{
+ (unsigned) countFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last value: (id)object
{
    OLForwardIterator* src = [first copy];
    unsigned result = 0;

    for ( ; ![src isEqual: last]; [src advance])
    {
        if ([[src dereference] isEqual: object])
            result++;
    }
    OBJ_RELEASE(src);
    return result;
}
// }}}

// equalFrom:to:with: {{{
+ (BOOL) equalFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last with: (OLForwardIterator*)first2
{
    OLForwardIterator* one = [first copy];
    OLForwardIterator* two = [first2 copy];
    BOOL equal = YES;

    for ( ; ![one isEqual: last]; [one advance], [two advance])
    {
        if (![[one dereference] isEqual: [two dereference]])
        {
            equal = NO;
            break;
        }
    }
    OBJ_RELEASE(one);
    OBJ_RELEASE(two);
    return equal;
}
// }}}

// equalFrom:to:with:predicate: {{{
+ (BOOL) equalFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last with: (OLForwardIterator*)first2 predicate: (id<OLBoolBinaryFunction>)pred
{
    OLForwardIterator* one = [first copy];
    OLForwardIterator* two = [first2 copy];
    BOOL equal = YES;

    for ( ; ![one isEqual: last]; [one advance], [two advance])
    {
        if (![pred performBinaryFunctionWithArg: [one dereference] andArg: [two dereference]])
        {
            equal = NO;
            break;
        }
    }
    OBJ_RELEASE(one);
    OBJ_RELEASE(two);
    return equal;
}
// }}}

// equalRangeFrom:to:value: {{{
+ (OLPair*) equalRangeFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last value: (id)object
{
    OLLess* less = [[OLLess alloc] init];
    OLPair* result = [OLAlgorithm equalRangeFrom: first to: last value: object predicate: less];

    OBJ_RELEASE(less);
    return result;
}
// }}}

// equalRangeFrom:to:value:predicate: {{{
+ (OLPair*) equalRangeFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last value: (id)object predicate: (id<OLBoolBinaryFunction>)pred
{
    OLForwardIterator* cur = [first copy];
    OLForwardIterator* middle = [first copy];
    OLForwardIterator* left;
    OLForwardIterator* right;
    OLPair* result = nil;
    unsigned len = [OLIterator distanceFrom: first to: last];
    unsigned half;
    BOOL changeMiddle = NO;

    while (len > 0)
    {
        half = len >> 1;
        if (changeMiddle)
        {
            OBJ_RELEASE(middle);
            middle = [cur copy];
            changeMiddle = NO;
        }
        [OLIterator advanceIterator: middle distance: half];
        if ([pred performBinaryFunctionWithArg: [middle dereference] andArg: object])
        {
            OBJ_RELEASE(cur);
            cur = [middle copy];
            [cur advance];
            len = len - half - 1;
            changeMiddle = YES;
        }
        else if ([pred performBinaryFunctionWithArg: object andArg: [middle dereference]])
        {
            len = half;
        }
        else
        {
            left = [OLAlgorithm boundImplFrom: cur to: middle
                value: object predicate: pred lower: YES];
            [OLIterator advanceIterator: cur distance: len];
            right = [OLAlgorithm boundImplFrom: [middle advance] to: cur
                value: object predicate: pred lower: NO];
            result = [[OLPair alloc] initWithFirst: left second: right];
            OBJ_RELEASE(left);
            OBJ_RELEASE(right);
            break;
        }
    }
    if (result == nil)
        result = [[OLPair alloc] initWithFirst: cur second: cur];
    OBJ_RELEASE(cur);
    OBJ_RELEASE(middle);
    return OBJ_AUTORELEASE(result);
}
// }}}

// fillFrom:count:value: {{{
+ (void) fillFrom: (OLForwardIterator*)first count: (unsigned)num value: (id)object
{
    OLForwardIterator* dst = [first copy];
    unsigned i;

    for (i = 0; i < num; i++, [dst advance])
        [dst assign: object];
    OBJ_RELEASE(dst);
}
// }}}

// fillFrom:to:value: {{{
+ (void) fillFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last value: (id)object
{
    OLForwardIterator* dst = [first copy];

    for ( ; ![dst isEqual: last]; [dst advance])
        [dst assign: object];
    OBJ_RELEASE(dst);
}
// }}}

// findEndFrom:to:subFrom:subTo: {{{
+ (OLForwardIterator*) findEndFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last1 subFrom: (OLForwardIterator*)first2 subTo: (OLForwardIterator*)last2
{
    OLEqualTo* equal = [[OLEqualTo alloc] init];
    OLForwardIterator* result =
        [OLAlgorithm findEndFrom: first1 to: last1
        subFrom: first2 subTo: last2 predicate: equal];

    OBJ_RELEASE(equal);
    return result;
}
// }}}

// findEndFrom:to:subFrom:subTo:predicate: {{{
+ (OLForwardIterator*) findEndFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last1 subFrom: (OLForwardIterator*)first2 subTo: (OLForwardIterator*)last2 predicate: (id<OLBoolBinaryFunction>)pred
{
    OLForwardIterator* result;
    OLForwardIterator* newResult;
    OLForwardIterator* f1;

    if ([first2 isEqual: last2])
        return OBJ_AUTORELEASE([last1 copy]);
    result = nil;
    f1 = [first1 copy];
    while (YES)
    {
        newResult = [OLAlgorithm searchFrom: f1 to: last1
            subFrom: first2 subTo: last2 predicate: pred];
        if ([newResult isEqual: last1])
        {
            OBJ_FREE_AUTO(newResult);
            if (result == nil)
                result = OBJ_AUTORELEASE([last1 copy]);
            break;
        }
        else
        {
            OBJ_FREE_AUTO(result);
            result = newResult;
            OBJ_RELEASE(f1);
            f1 = [newResult copy];
            [f1 advance];
        }
    }
    OBJ_RELEASE(f1);
    return result;
}
// }}}

// findFirstFrom:to:ofFrom:ofTo: {{{
+ (OLForwardIterator*) findFirstFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last1 ofFrom: (OLForwardIterator*)first2 ofTo: (OLForwardIterator*)last2
{
    OLEqualTo* equal = [[OLEqualTo alloc] init];
    OLForwardIterator* result =
        [OLAlgorithm findFirstFrom: first1 to: last1
        ofFrom: first2 ofTo: last2 predicate: equal];

    OBJ_RELEASE(equal);
    return result;
}
// }}}

// findFirstFrom:to:ofFrom:ofTo:predicate: {{{
+ (OLForwardIterator*) findFirstFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last1 ofFrom: (OLForwardIterator*)first2 ofTo: (OLForwardIterator*)last2 predicate: (id<OLBoolBinaryFunction>)pred
{
    OLForwardIterator* f;
    id* series;
    unsigned count;
    unsigned i;

    if ([first1 isEqual: last1] || [first2 isEqual: last2])
        return OBJ_AUTORELEASE([last1 copy]);

    count = [OLIterator distanceFrom: first2 to: last2];
    series = objc_malloc(count * sizeof(id));
    for (f = [first2 copy], i = 0; i < count; [f advance], i++)
        series[i] = [f dereference];
    OBJ_RELEASE(f);
    for (f = [first1 copy]; ![f isEqual: last1]; [f advance])
    {
        for (i = 0; i < count; i++)
        {
            if ([pred performBinaryFunctionWithArg: [f dereference]
                    andArg: series[i]])
            {
                goto leaveFindFirstOf;
            }
        }
    }
leaveFindFirstOf:
    objc_free(series);
    return OBJ_AUTORELEASE(f);
}
// }}}

// findFrom:to:if: {{{
+ (OLForwardIterator*) findFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last if: (id<OLBoolUnaryFunction>)pred
{
    return OBJ_AUTORELEASE([OLAlgorithm findIfImplFrom: first to: last predicate: pred]);
}
// }}}

// findFrom:to:value: {{{
+ (OLForwardIterator*) findFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last value: (id)object
{
    return OBJ_AUTORELEASE([OLAlgorithm findImplFrom: first to: last value: object]);
}
// }}}

// forEachFrom:to:function: {{{
+ (void) forEachFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last function: (id<OLUnaryFunction>)func
{
    OLForwardIterator* src = [first copy];
    id result;
#if !defined(OL_NO_OPENSTEP)
    NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
    unsigned count = 0;
#endif

    for ( ; ![src isEqual: last]; [src advance])
    {
        result = [func performUnaryFunctionWithArg: [src dereference]];
#if defined(OL_NO_OPENSTEP)
        OBJ_RELEASE(result);
#else
        if (++count == 100)
        {
            OBJ_RELEASE(pool);
            pool = [[NSAutoreleasePool alloc] init];
            count = 0;
        }
#endif
    }
    OBJ_RELEASE(src);
#if !defined(OL_NO_OPENSTEP)
    OBJ_RELEASE(pool);
#endif
}
// }}}

// includesFrom:to:subFrom:subTo: {{{
+ (BOOL) includesFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last1 subFrom: (OLForwardIterator*)first2 subTo: (OLForwardIterator*)last2
{
    OLLess* less = [[OLLess alloc] init];
    BOOL result = [OLAlgorithm includesFrom: first1 to: last1
        subFrom: first2 subTo: last2 predicate: less];

    OBJ_RELEASE(less);
    return result;
}
// }}}

// includesFrom:to:subFrom:subTo:predicate: {{{
+ (BOOL) includesFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last1 subFrom: (OLForwardIterator*)first2 subTo: (OLForwardIterator*)last2 predicate: (id<OLBoolBinaryFunction>)pred
{
    OLForwardIterator* f1;
    OLForwardIterator* f2;
    BOOL returnFalse = NO;
    BOOL result;

    if ([first2 isEqual: last2])
        return YES;
    if ([first1 isEqual: last1])
        return NO;
    f1 = [first1 copy];
    f2 = [first2 copy];
    while (![f1 isEqual: last1] && ![f2 isEqual: last2])
    {
        if ([pred performBinaryFunctionWithArg: [f2 dereference]
                andArg: [f1 dereference]])
        {
            returnFalse = YES;
            break;
        }
        else if ([pred performBinaryFunctionWithArg: [f1 dereference]
                    andArg: [f2 dereference]])
        {
            [f1 advance];
        }
        else
        {
            [f1 advance];
            [f2 advance];
        }
    }
    result = returnFalse ? NO : [f2 isEqual: last2];
    OBJ_RELEASE(f1);
    OBJ_RELEASE(f2);
    return result;
}
// }}}

// inPlaceMergeFrom:middle:to: {{{
+ (void) inPlaceMergeFrom: (OLBidirectionalIterator*)first middle: (OLBidirectionalIterator*)middle to: (OLBidirectionalIterator*)last
{
    OLLess* less = [[OLLess alloc] init];

    [OLAlgorithm inPlaceMergeFrom: first middle: middle to: last predicate: less];
    OBJ_RELEASE(less);
}
// }}}

// inPlaceMergeFrom:middle:to:predicate: {{{
+ (void) inPlaceMergeFrom: (OLBidirectionalIterator*)first middle: (OLBidirectionalIterator*)middle to: (OLBidirectionalIterator*)last predicate: (id<OLBoolBinaryFunction>)pred
{
    OLTempBuf* buf;
    unsigned len1;
    unsigned len2;

    if (!([first isEqual: middle] || [middle isEqual: last]))
    {
        buf = [[OLTempBuf alloc] initWithFirst: first last: last];
        len1 = [OLIterator distanceFrom: first to: middle];
        len2 = [OLIterator distanceFrom: middle to: last];
        [OLAlgorithm mergeAdaptiveFrom: first middle: middle last: last
            len1: len1 len2: len2 buffer: buf predicate: pred];
        OBJ_RELEASE(buf);
    }
}
// }}}

// isHeapFrom:to: {{{
+ (BOOL) isHeapFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    OLLess* less = [[OLLess alloc] init];
    BOOL result = [OLAlgorithm isHeapFrom: first to: last predicate: less];

    OBJ_RELEASE(less);
    return result;
}
// }}}

// isHeapFrom:to:predicate: {{{
+ (BOOL) isHeapFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last predicate: (id<OLBoolBinaryFunction>)pred
{
    OLForwardIterator* parent = [first copy];
    OLForwardIterator* child = [first copy];
    unsigned counter;
    BOOL isHeap = YES;

    for ([child advance], counter = 1; ![child isEqual: last]; [child advance], counter++)
    {
        if ([pred performBinaryFunctionWithArg: [parent dereference]
                andArg: [child dereference]])
        {
            isHeap = NO;
            break;
        }
        if ((counter & 1) == 0)
            [parent advance];
    }
    OBJ_RELEASE(parent);
    OBJ_RELEASE(child);
    return isHeap;
}
// }}}

// isSortedFrom:to: {{{
+ (BOOL) isSortedFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    OLLess* less = [[OLLess alloc] init];
    BOOL result = [OLAlgorithm isSortedFrom: first to: last predicate: less];

    OBJ_RELEASE(less);
    return result;
}
// }}}

// isSortedFrom:to:predicate: {{{
+ (BOOL) isSortedFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last predicate: (id<OLBoolBinaryFunction>)pred
{
    OLForwardIterator* next;
    OLForwardIterator* cur;
    BOOL isSorted;

    if ([first isEqual: last])
        return YES;
    cur = [first copy];
    next = [first copy];
    isSorted = YES;
    for ([next advance]; ![next isEqual: last]; [cur advance], [next advance])
    {
        if ([pred performBinaryFunctionWithArg: [next dereference]
                andArg: [cur dereference]])
        {
            isSorted = NO;
            break;
        }
    }
    OBJ_RELEASE(cur);
    OBJ_RELEASE(next);
    return isSorted;
}
// }}}

// lexicographicalCompareFrom:to:andFrom:andTo: {{{
+ (BOOL) lexicographicalCompareFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last1 andFrom: (OLForwardIterator*)first2 andTo: (OLForwardIterator*)last2
{
    OLLess* less = [[OLLess alloc] init];
    BOOL result = [OLAlgorithm lexicographicalCompareFrom: first1 to: last1
        andFrom: first2 andTo: last2 predicate: less];

    OBJ_RELEASE(less);
    return result;
}
// }}}

// lexicographicalCompareFrom:to:andFrom:andTo:predicate: {{{
+ (BOOL) lexicographicalCompareFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last1 andFrom: (OLForwardIterator*)first2 andTo: (OLForwardIterator*)last2 predicate: (id<OLBoolBinaryFunction>)pred
{
    OLForwardIterator* f1 = [first1 copy];
    OLForwardIterator* f2 = [first2 copy];
    BOOL result = NO;
    BOOL gotResult = NO;

    for ( ; ![f1 isEqual: last1] && ![f2 isEqual: last2]; [f1 advance], [f2 advance])
    {
        if ([pred performBinaryFunctionWithArg: [f1 dereference]
                andArg: [f2 dereference]])
        {
            result = YES;
            gotResult = YES;
            break;
        }
        else if ([pred performBinaryFunctionWithArg: [f2 dereference]
                    andArg: [f1 dereference]])
        {
            gotResult = YES;
            break;
        }
    }
    if (!gotResult)
        result = [f1 isEqual: last1] && ![f2 isEqual: last2];
    OBJ_RELEASE(f1);
    OBJ_RELEASE(f2);
    return result;
}
// }}}

// lowerBoundFrom:to:value: {{{
+ (OLForwardIterator*) lowerBoundFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last value: (id)object
{
    OLLess* less = [[OLLess alloc] init];
    OLForwardIterator* result =
        [OLAlgorithm lowerBoundFrom: first to: last value: object predicate: less];

    OBJ_RELEASE(less);
    return result;
}
// }}}

// lowerBoundFrom:to:value:predicate: {{{
+ (OLForwardIterator*) lowerBoundFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last value: (id)object predicate: (id<OLBoolBinaryFunction>) pred
{
    return OBJ_AUTORELEASE([OLAlgorithm boundImplFrom: first to: last
        value: object predicate: pred lower: YES]);
}
// }}}

// makeHeapFrom:to: {{{
+ (void) makeHeapFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last
{
    OLLess* less = [[OLLess alloc] init];

    [OLAlgorithm makeHeapFrom: first to: last predicate: less];
    OBJ_RELEASE(less);
}
// }}}

// makeHeapFrom:to:predicate: {{{
+ (void) makeHeapFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last predicate: (id<OLBoolBinaryFunction>)pred
{
    OLRandomAccessIterator* f;
    unsigned len = [last difference: first];
    unsigned parent;

    if (len >= 2)
    {
        f = [first copy];
        parent = (len - 2) / 2;
        [f advanceBy: parent];
        while (YES)
        {
            [OLAlgorithm adjustHeapFirst: first hole: parent len: len
                value: [f dereference] predicate: pred];
            if (parent == 0)
                break;
            [f reverse];
            parent--;
        }
        OBJ_RELEASE(f);
    }
}
// }}}

// max:or: {{{
+ (id) max: (id)left or: (id)right
{
    return [left compare: right] < 0 ? right : left;
}
// }}}

// max:or:predicate: {{{
+ (id) max: (id)left or: (id)right predicate: (id<OLBoolBinaryFunction>)pred
{
    return [pred performBinaryFunctionWithArg: left andArg: right] ?
        right : left;
}
// }}}

// maxElementFrom:to: {{{
+ (OLForwardIterator*) maxElementFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    OLLess* less = [[OLLess alloc] init];
    OLForwardIterator* result =
        [OLAlgorithm maxElementFrom: first to: last predicate: less];

    OBJ_RELEASE(less);
    return result;
}
// }}}

// maxElementFrom:to:predicate: {{{
+ (OLForwardIterator*) maxElementFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last predicate: (id<OLBoolBinaryFunction>)pred
{
    OLForwardIterator* f;
    OLForwardIterator* result;
    id current;
    unsigned index = 0;
    unsigned currentIndex = 0;

    if (![first isEqual: last])
    {
        f = [first copy];
        current = [f dereference];
        while (![[f advance] isEqual: last])
        {
            index++;
            if ([pred performBinaryFunctionWithArg: current
                    andArg: [f dereference]])
            {
                current = [f dereference];
                currentIndex = index;
            }
        }
        OBJ_RELEASE(f);
        result = [first copy];
        [OLIterator advanceIterator: result distance: currentIndex];
    }
    else
    {
        result = [last copy];
    }
    return OBJ_AUTORELEASE(result);
}
// }}}

// mergeFrom:to:andFrom:andTo:destination: {{{
+ (OLForwardIterator*) mergeFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last1 andFrom: (OLForwardIterator*)first2 andTo: (OLForwardIterator*)last2 destination: (OLForwardIterator*)dest
{
    OLLess* less = [[OLLess alloc] init];
    OLForwardIterator* result =
        [OLAlgorithm mergeFrom: first1 to: last1
            andFrom: first2 andTo: last2 destination: dest predicate: less];

    OBJ_RELEASE(less);
    return result;
}
// }}}

// mergeFrom:to:andFrom:andTo:destination:predicate: {{{
+ (OLForwardIterator*) mergeFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last1 andFrom: (OLForwardIterator*)first2 andTo: (OLForwardIterator*)last2 destination: (OLForwardIterator*)dest predicate: (id<OLBoolBinaryFunction>)pred
{
    return OBJ_AUTORELEASE([OLAlgorithm mergeImplSeries1From: first1 series1To: last1
        series2From: first2 series2To: last2 destination: dest predicate: pred needItor: YES]);
}
// }}}

// min:or: {{{
+ (id) min: (id)left or: (id)right
{
    return [right compare: left] < 0 ? right : left;
}
// }}}

// min:or:predicate: {{{
+ (id) min: (id)left or: (id)right predicate: (id<OLBoolBinaryFunction>)pred
{
    return [pred performBinaryFunctionWithArg: right andArg: left] ?
        right : left;
}
// }}}

// minElementFrom:to: {{{
+ (OLForwardIterator*) minElementFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    OLLess* less = [[OLLess alloc] init];
    OLForwardIterator* result =
        [OLAlgorithm minElementFrom: first to: last predicate: less];

    OBJ_RELEASE(less);
    return result;
}
// }}}

// minElementFrom:to:predicate: {{{
+ (OLForwardIterator*) minElementFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last predicate: (id<OLBoolBinaryFunction>)pred
{
    OLForwardIterator* f;
    OLForwardIterator* result;
    id current;
    unsigned index = 0;
    unsigned currentIndex = 0;

    if (![first isEqual: last])
    {
        f = [first copy];
        current = [f dereference];
        while (![[f advance] isEqual: last])
        {
            index++;
            if ([pred performBinaryFunctionWithArg: [f dereference]
                    andArg: current])
            {
                current = [f dereference];
                currentIndex = index;
            }
        }
        OBJ_RELEASE(f);
        result = [first copy];
        [OLIterator advanceIterator: result distance: currentIndex];
    }
    else
    {
        result = [last copy];
    }
    return OBJ_AUTORELEASE(result);
}
// }}}

// mismatchFrom:to:with: {{{
+ (OLPair*) mismatchFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last1 with: (OLForwardIterator*)first2
{
    OLEqualTo* equal = [[OLEqualTo alloc] init];
    OLPair* result = [OLAlgorithm mismatchFrom: first1 to: last1 with: first2 predicate: equal];

    OBJ_RELEASE(equal);
    return result;
}
// }}}

// mismatchFrom:to:with:predicate: {{{
+ (OLPair*) mismatchFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last1 with: (OLForwardIterator*)first2 predicate: (id<OLBoolBinaryFunction>)pred
{
    OLForwardIterator* f1 = [first1 copy];
    OLForwardIterator* f2 = [first2 copy];
    OLPair* result;

    while (![f1 isEqual: last1] &&
           [pred performBinaryFunctionWithArg: [f1 dereference]
                andArg: [f2 dereference]])
    {
        [f1 advance];
        [f2 advance];
    }
    result = [[OLPair alloc] initWithFirst: f1 second: f2];
    OBJ_RELEASE(f1);
    OBJ_RELEASE(f2);
    return OBJ_AUTORELEASE(result);
}
// }}}

// nextPermutationFrom:to: {{{
+ (BOOL) nextPermutationFrom: (OLBidirectionalIterator*)first to: (OLBidirectionalIterator*)last
{
    OLLess* less = [[OLLess alloc] init];
    BOOL result = [OLAlgorithm nextPermutationFrom: first to: last predicate: less];

    OBJ_RELEASE(less);
    return result;
}
// }}}

// nextPermutationFrom:to:predicate: {{{
+ (BOOL) nextPermutationFrom: (OLBidirectionalIterator*)first to: (OLBidirectionalIterator*)last predicate: (id<OLBoolBinaryFunction>)pred
{
    OLBidirectionalIterator* i;
    OLBidirectionalIterator* ii;
    OLBidirectionalIterator* j = nil;
    BOOL result;

    if ([first isEqual: last] ||
        [OLIterator distanceFrom: first to: last] == 1)
    {
        return NO;
    }
    i = [last copy];
    [i reverse];
    ii = [i copy];
    while (YES)
    {
        [i reverse];
        if ([pred performBinaryFunctionWithArg: [i dereference]
                andArg: [ii dereference]])
        {
            j = [last copy];
            do
            {
                [j reverse];
            } while (![pred performBinaryFunctionWithArg: [i dereference]
                            andArg: [j dereference]]);
            [OLAlgorithm swapIterators: i and: j];
            [OLAlgorithm reverseFrom: ii to: last];
            result = YES;
            break;
        }
        if ([i isEqual: first])
        {
            [OLAlgorithm reverseFrom: first to: last];
            result = NO;
            break;
        }
        [ii reverse];
    }
    OBJ_RELEASE(i);
    OBJ_RELEASE(ii);
    OBJ_RELEASE(j);
    return result;
}
// }}}

// nthElementFrom:nth:to: {{{
+ (void) nthElementFrom: (OLRandomAccessIterator*)first nth: (OLRandomAccessIterator*)mid to: (OLRandomAccessIterator*)last
{
    OLLess* less = [[OLLess alloc] init];

    [OLAlgorithm nthElementFrom: first nth: mid to: last predicate: less];
    OBJ_RELEASE(less);
}
// }}}

// nthElementFrom:nth:to:predicate: {{{
+ (void) nthElementFrom: (OLRandomAccessIterator*)first nth: (OLRandomAccessIterator*)mid to: (OLRandomAccessIterator*)last predicate: (id<OLBoolBinaryFunction>)pred
{
    OLRandomAccessIterator* f = [first copy];
    OLRandomAccessIterator* l = [last copy];
    OLRandomAccessIterator* cut;
    id med;
    id fcur;
    int advance;

    while ([l difference: f] > 3)
    {
        advance = [l difference: f] / 2;
        fcur = [f dereference];
        med = [OLAlgorithm medianFromOne: fcur two: [[f advanceBy: advance] dereference]
            three: [[l reverse] dereference] predicate: pred];
        [f advanceBy: -advance];
        [l advance];
        cut = [OLAlgorithm unguardedPartitionFrom: f to: l pivot: med predicate: pred];
        if ([mid difference: cut] > 0)
        {
            OBJ_RELEASE(f);
            f = cut;
        }
        else
        {
            OBJ_RELEASE(l);
            l = cut;
        }
    }
    [OLAlgorithm insertionSortFrom: f to: l predicate: pred];
    OBJ_RELEASE(f);
    OBJ_RELEASE(l);
}
// }}}

// partialSortCopyFrom:to:destinationFrom:destinationTo: {{{
+ (OLRandomAccessIterator*) partialSortCopyFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last destinationFrom: (OLRandomAccessIterator*)rFirst destinationTo: (OLRandomAccessIterator*)rLast
{
    OLLess* less = [[OLLess alloc] init];
    OLRandomAccessIterator* result =
        [OLAlgorithm partialSortCopyFrom: first to: last
            destinationFrom: rFirst destinationTo: rLast predicate: less];

    OBJ_RELEASE(less);
    return result;
}
// }}}

// partialSortCopyFrom:to:destinationFrom:destinationTo:predicate: {{{
+ (OLRandomAccessIterator*) partialSortCopyFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last destinationFrom: (OLRandomAccessIterator*)rFirst destinationTo: (OLRandomAccessIterator*)rLast predicate: (id<OLBoolBinaryFunction>)pred
{
    OLForwardIterator* f;
    OLRandomAccessIterator* rlReal;

    if ([rFirst isEqual: rLast])
        return OBJ_AUTORELEASE([rLast copy]);
    f = [first copy];
    rlReal = [rFirst copy];
    while (![f isEqual: last] && ![rlReal isEqual: rLast])
    {
        [rlReal assign: [f dereference]];
        [rlReal advance];
        [f advance];
    }
    [OLAlgorithm makeHeapFrom: rFirst to: rlReal predicate: pred];
    while (![f isEqual: last])
    {
        if ([pred performBinaryFunctionWithArg: [f dereference]
                andArg: [rFirst dereference]])
        {
            [OLAlgorithm adjustHeapFirst: rFirst hole: 0 len: [rlReal difference: rFirst]
                value: [f dereference] predicate: pred];
        }
        [f advance];
    }
    OBJ_RELEASE(f);
    [OLAlgorithm sortHeapFrom: rFirst to: rlReal predicate: pred];
    return OBJ_AUTORELEASE(rlReal);
}
// }}}

// partialSortFrom:middle:to: {{{
+ (void) partialSortFrom: (OLRandomAccessIterator*)first middle: (OLRandomAccessIterator*)mid to: (OLRandomAccessIterator*)last
{
    OLLess* less = [[OLLess alloc] init];

    [OLAlgorithm partialSortFrom: first middle: mid to: last predicate: less];
    OBJ_RELEASE(less);
}
// }}}

// partialSortFrom:middle:to:predicate: {{{
+ (void) partialSortFrom: (OLRandomAccessIterator*)first middle: (OLRandomAccessIterator*)mid to: (OLRandomAccessIterator*)last predicate: (id<OLBoolBinaryFunction>)pred
{
    OLRandomAccessIterator* i = [mid copy];

    [OLAlgorithm makeHeapFrom: first to: mid predicate: pred];
    for ( ; ![i isEqual: last]; [i advance])
    {
        if ([pred performBinaryFunctionWithArg: [i dereference]
                andArg: [first dereference]])
        {
            [OLAlgorithm popHeapImplFirst: first last: mid result: i
                value: [i dereference] predicate: pred];
        }
    }
    [OLAlgorithm sortHeapFrom: first to: mid predicate: pred];
    OBJ_RELEASE(i);
}
// }}}

// partitionFrom:to:predicate: {{{
+ (OLBidirectionalIterator*) partitionFrom: (OLBidirectionalIterator*)first to: (OLBidirectionalIterator*)last predicate: (id<OLBoolUnaryFunction>)pred
{
    OLBidirectionalIterator* f = [first copy];
    OLBidirectionalIterator* l = [last copy];

    while (YES)
    {
        while (YES)
        {
            if ([f isEqual: l])
                goto leavePartitionFrom;
            else if ([pred performUnaryFunctionWithArg: [f dereference]])
                [f advance];
            else
                break;
        }
        [l reverse];
        while (YES)
        {
            if ([f isEqual: l])
                goto leavePartitionFrom;
            else if (![pred performUnaryFunctionWithArg: [l dereference]])
                [l reverse];
            else
                break;
        }
        [OLAlgorithm swapIterators: f and: l];
        [f advance];
    }
leavePartitionFrom:
    OBJ_RELEASE(l);
    return OBJ_AUTORELEASE(f);
}
// }}}

// popHeapFrom:to: {{{
+ (void) popHeapFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last
{
    OLLess* less = [[OLLess alloc] init];

    [OLAlgorithm popHeapFrom: first to: last predicate: less];
    OBJ_RELEASE(less);
}
// }}}

// popHeapFrom:to:predicate: {{{
+ (void) popHeapFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last predicate: (id<OLBoolBinaryFunction>)pred
{
    OLRandomAccessIterator* l = [last copy];

    [l reverse];
    [OLAlgorithm popHeapImplFirst: first last: l result: l value: [l dereference] predicate: pred];
    OBJ_RELEASE(l);
}
// }}}

// prevPermutationFrom:to: {{{
+ (BOOL) prevPermutationFrom: (OLBidirectionalIterator*)first to: (OLBidirectionalIterator*)last
{
    OLLess* less = [[OLLess alloc] init];
    BOOL result = [OLAlgorithm prevPermutationFrom: first to: last predicate: less];

    OBJ_RELEASE(less);
    return result;
}
// }}}

// prevPermutationFrom:to:predicate: {{{
+ (BOOL) prevPermutationFrom: (OLBidirectionalIterator*)first to: (OLBidirectionalIterator*)last predicate: (id<OLBoolBinaryFunction>)pred
{
    OLBidirectionalIterator* i;
    OLBidirectionalIterator* ii;
    OLBidirectionalIterator* j = nil;
    BOOL result;

    if ([first isEqual: last] ||
        [OLIterator distanceFrom: first to: last] == 1)
    {
        return NO;
    }
    i = [last copy];
    [i reverse];
    ii = [i copy];
    while (YES)
    {
        [i reverse];
        if ([pred performBinaryFunctionWithArg: [ii dereference]
                andArg: [i dereference]])
        {
            j = [last copy];
            do
            {
                [j reverse];
            } while (![pred performBinaryFunctionWithArg: [j dereference]
                            andArg: [i dereference]]);
            [OLAlgorithm swapIterators: i and: j];
            [OLAlgorithm reverseFrom: ii to: last];
            result = YES;
            break;
        }
        if ([i isEqual: first])
        {
            [OLAlgorithm reverseFrom: first to: last];
            result = NO;
            break;
        }
        [ii reverse];
    }
    OBJ_RELEASE(i);
    OBJ_RELEASE(ii);
    OBJ_RELEASE(j);
    return result;
}
// }}}

// pushHeapFrom:to: {{{
+ (void) pushHeapFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last
{
    OLLess* less = [[OLLess alloc] init];

    [OLAlgorithm pushHeapFrom: first to: last predicate: less];
    OBJ_RELEASE(less);
}
// }}}

// pushHeapFrom:to:predicate: {{{
+ (void) pushHeapFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last predicate: (id<OLBoolBinaryFunction>)pred
{
    OLRandomAccessIterator* l = [last copy];

    [OLAlgorithm pushHeapImplFirst: first hole: [last difference: first] - 1 top: 0
        value: [[l reverse] dereference] predicate: pred];
    OBJ_RELEASE(l);
}
// }}}

// randomShuffleFrom:to: {{{
+ (void) randomShuffleFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last
{
    OLRandomAccessIterator* i;
    OLRandomAccessIterator* ii;
    long int randomValue;
    long int advance;

    if (![first isEqual: last])
    {
        i = [first copy];
        ii = [first copy];
        [i advance];
        for ( ; ![i isEqual: last]; [i advance])
        {
            #if defined(OL_HAVE_RANDOM)
            randomValue = random();
            #elif defined(OL_HAVE_LRAND48)
            randomValue = lrand48();
            #elif defined(OL_HAVE_RAND_S)
            rand_s((unsigned int*)&randomValue);
            #elif defined(OL_HAVE_RAND)
            randomValue = rand();
            #else
            #error You do not have a suitable pseudo-random number generator
            #endif
            advance = randomValue % ([i difference: first] + 1);
            [ii advanceBy: advance];
            [OLAlgorithm swapIterators: i and: ii];
            [ii advanceBy: -advance];
        }
        OBJ_RELEASE(i);
        OBJ_RELEASE(ii);
    }
}
// }}}

// randomShuffleFrom:to:randGen: {{{
+ (void) randomShuffleFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last randGen: (id<OLUnaryFunction>)gen
{
    OLRandomAccessIterator* i;
    OLRandomAccessIterator* ii;
    OLUnsignedNumber* val;
    id result;
    unsigned advance;
#if !defined(OL_NO_OPENSTEP)
    NSAutoreleasePool* pool;
    unsigned count;
#endif

    if (![first isEqual: last])
    {
#if !defined(OL_NO_OPENSTEP)
        pool = [[NSAutoreleasePool alloc] init];
        count = 0;
#endif
        i = [first copy];
        ii = [first copy];
        [i advance];
        for ( ; ![i isEqual: last]; [i advance])
        {
            val = [[OLUnsignedNumber alloc] initWithUnsignedInt: [i difference: first] + 1];
            result = [gen performUnaryFunctionWithArg: val];
            OBJ_RELEASE(val);
            advance = [result unsignedIntValue];
#if defined(OL_NO_OPENSTEP)
            OBJ_RELEASE(result);
#else
            if (++count == 100)
            {
                OBJ_RELEASE(pool);
                pool = [[NSAutoreleasePool alloc] init];
                count = 0;
            }
#endif
            [ii advanceBy: advance];
            [OLAlgorithm swapIterators: i and: ii];
            [ii advanceBy: -advance];
        }
        OBJ_RELEASE(i);
        OBJ_RELEASE(ii);
#if !defined(OL_NO_OPENSTEP)
        OBJ_RELEASE(pool);
#endif
    }
}
// }}}

// removeCopyFrom:to:destination:if: {{{
+ (OLForwardIterator*) removeCopyFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last destination: (OLForwardIterator*)dest if: (id<OLBoolUnaryFunction>)pred
{
    OLForwardIterator* f = [first copy];
    OLForwardIterator* r = [dest copy];

    for ( ; ![f isEqual: last]; [f advance])
    {
        if (![pred performUnaryFunctionWithArg: [f dereference]])
        {
            [r assign: [f dereference]];
            [r advance];
        }
    }
    OBJ_RELEASE(f);
    return OBJ_AUTORELEASE(r);
}
// }}}

// removeCopyFrom:to:destination:value: {{{
+ (OLForwardIterator*) removeCopyFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last destination: (OLForwardIterator*)dest value: (id)object
{
    OLForwardIterator* f = [first copy];
    OLForwardIterator* r = [dest copy];

    for ( ; ![f isEqual: last]; [f advance])
    {
        if (![[f dereference] isEqual: object])
        {
            [r assign: [f dereference]];
            [r advance];
        }
    }
    OBJ_RELEASE(f);
    return OBJ_AUTORELEASE(r);
}
// }}}

// removeFrom:to:if: {{{
+ (OLForwardIterator*) removeFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last if: (id<OLBoolUnaryFunction>)pred
{
    OLForwardIterator* f =
        (OLForwardIterator*)[OLAlgorithm findIfImplFrom: first
            to: last predicate: pred];
    OLForwardIterator* next;
    OLForwardIterator* result;

    if ([f isEqual: last])
    {
        result = OBJ_AUTORELEASE(f);
    }
    else
    {
        next = [f copy];
        result = [OLAlgorithm removeCopyFrom: [next advance] to: last destination: f if: pred];
        OBJ_RELEASE(next);
        OBJ_RELEASE(f);
    }
    return result;
}
// }}}

// removeFrom:to:value: {{{
+ (OLForwardIterator*) removeFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last value: (id)object
{
    OLForwardIterator* f =
        (OLForwardIterator*)[OLAlgorithm findImplFrom: first
            to: last value: object];
    OLForwardIterator* next;
    OLForwardIterator* result;

    if ([f isEqual: last])
    {
        result = OBJ_AUTORELEASE(f);
    }
    else
    {
        next = [f copy];
        result = [OLAlgorithm removeCopyFrom: [next advance] to: last
            destination: f value: object];
        OBJ_RELEASE(next);
        OBJ_RELEASE(f);
    }
    return result;
}
// }}}

// replaceCopyFrom:to:destination:if:newValue: {{{
+ (OLForwardIterator*) replaceCopyFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last destination: (OLForwardIterator*)dest if: (id<OLBoolUnaryFunction>)pred newValue: (id)newv
{
    OLForwardIterator* f = [first copy];
    OLForwardIterator* r = [dest copy];

    for ( ; ![f isEqual: last]; [f advance], [r advance])
    {
        [r assign: [pred performUnaryFunctionWithArg: [f dereference]] ?
            newv : [f dereference]];
    }
    OBJ_RELEASE(f);
    return OBJ_AUTORELEASE(r);
}
// }}}

// replaceCopyFrom:to:destination:oldValue:newValue: {{{
+ (OLForwardIterator*) replaceCopyFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last destination: (OLForwardIterator*)dest oldValue: (id)oldv newValue: (id)newv
{
    OLForwardIterator* f = [first copy];
    OLForwardIterator* r = [dest copy];

    for ( ; ![f isEqual: last]; [f advance], [r advance])
        [r assign: ([[f dereference] isEqual: oldv]) ? newv : [f dereference]];
    OBJ_RELEASE(f);
    return OBJ_AUTORELEASE(r);
}
// }}}

// replaceFrom:to:if:newValue: {{{
+ (void) replaceFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last if: (id<OLBoolUnaryFunction>)pred newValue: (id)newv
{
    OLForwardIterator* f = [first copy];

    for ( ; ![f isEqual: last]; [f advance])
    {
        if ([pred performUnaryFunctionWithArg: [f dereference]])
            [f assign: newv];
    }
    OBJ_RELEASE(f);
}
// }}}

// replaceFrom:to:oldValue:newValue: {{{
+ (void) replaceFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last oldValue: (id)oldv newValue: (id)newv
{
    OLForwardIterator* f = [first copy];

    for ( ; ![f isEqual: last]; [f advance])
    {
        if ([[f dereference] isEqual: oldv])
            [f assign: newv];
    }
    OBJ_RELEASE(f);
}
// }}}

// reverseCopyFrom:to:destination: {{{
+ (OLForwardIterator*) reverseCopyFrom: (OLBidirectionalIterator*)first to: (OLBidirectionalIterator*)last destination: (OLForwardIterator*)dest
{
    OLBidirectionalIterator* l = [last copy];
    OLForwardIterator* r = [dest copy];

    while (![first isEqual: l])
    {
        [l reverse];
        [r assign: [l dereference]];
        [r advance];
    }
    OBJ_RELEASE(l);
    return OBJ_AUTORELEASE(r);
}
// }}}

// reverseFrom:to: {{{
+ (void) reverseFrom: (OLBidirectionalIterator*)first to: (OLBidirectionalIterator*)last
{
    OLBidirectionalIterator* f = [first copy];
    OLBidirectionalIterator* l = [last copy];

    for ( ; ![f isEqual: l] && ![f isEqual: [l reverse]]; [f advance])
        [OLAlgorithm swapIterators: f and: l];
    OBJ_RELEASE(f);
    OBJ_RELEASE(l);
}
// }}}

// rotateCopyFrom:middle:to:destination: {{{
+ (OLForwardIterator*) rotateCopyFrom: (OLForwardIterator*)first middle: (OLForwardIterator*)mid to: (OLForwardIterator*)last destination: (OLForwardIterator*)dest
{
    OLForwardIterator* tmp =
        [OLAlgorithm copyImplFrom: mid to: last destination: dest needItor: YES];
    OLForwardIterator* result = 
        [OLAlgorithm copyImplFrom: first to: mid destination: tmp needItor: YES];

    OBJ_RELEASE(tmp);
    return OBJ_AUTORELEASE(result);
}
// }}}

// rotateFrom:middle:to: {{{
+ (OLForwardIterator*) rotateFrom: (OLForwardIterator*)first middle: (OLForwardIterator*)mid to: (OLForwardIterator*)last
{
    return OBJ_AUTORELEASE([OLAlgorithm rotateImplFrom: first middle: mid to: last]);
}
// }}}

// searchFrom:to:count:value: {{{
+ (OLForwardIterator*) searchFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last count: (unsigned)num value: (id)object
{
    OLEqualTo* equal = [[OLEqualTo alloc] init];
    OLForwardIterator* result = [OLAlgorithm searchFrom: first to: last count: num
        value: object predicate: equal];

    OBJ_RELEASE(equal);
    return result;
}
// }}}

// searchFrom:to:count:value:predicate: {{{
+ (OLForwardIterator*) searchFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last count: (unsigned)num value: (id)object predicate: (id<OLBoolBinaryFunction>)pred
{
    OLForwardIterator* f;
    OLForwardIterator* i;
    OLForwardIterator* result = nil;
    unsigned n;

    if (num == 0)
    {
        result = [last copy];
    }
    else
    {
        f = [first copy];
        while (![f isEqual: last])
        {
            if ([pred performBinaryFunctionWithArg: [f dereference]
                    andArg: object])
            {
                break;
            }
            [f advance];
        }
        i = [f copy];
        while (![f isEqual: last])
        {
            n = num - 1;
            [i advance];
            while (![i isEqual: last] &&
                   n != 0 &&
                   [pred performBinaryFunctionWithArg: [i dereference]
                        andArg: object])
            {
                [i advance];
                n--;
            }
            if (n == 0)
            {
                result = f;
                break;
            }
            else
            {
                while (![i isEqual: last])
                {
                    if ([pred performBinaryFunctionWithArg: [i dereference]
                            andArg: object])
                    {
                        break;
                    }
                    [i advance];
                }
                OBJ_RELEASE(f);
                f = [i copy];
            }
            OBJ_RELEASE(i);
            i = [f copy];
        }
        if (result == nil)
            result = [last copy];
        OBJ_RELEASE(i);
        if (result != f)
            OBJ_RELEASE(f);
    }
    return OBJ_AUTORELEASE(result);
}
// }}}

// searchFrom:to:subFrom:subTo: {{{
+ (OLForwardIterator*) searchFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last1 subFrom: (OLForwardIterator*)first2 subTo: (OLForwardIterator*)last2
{
    OLEqualTo* equal = [[OLEqualTo alloc] init];
    OLForwardIterator* result =
        [OLAlgorithm searchFrom: first1 to: last1
        subFrom: first2 subTo: last2 predicate: equal];

    OBJ_RELEASE(equal);
    return result;
}
// }}}

// searchFrom:to:subFrom:subTo:predicate: {{{
+ (OLForwardIterator*) searchFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last1 subFrom: (OLForwardIterator*)first2 subTo: (OLForwardIterator*)last2 predicate: (id<OLBoolBinaryFunction>)pred
{
    OLForwardIterator* f1;
    OLForwardIterator* tmp;
    OLForwardIterator* tmp1;
    OLForwardIterator* cur;
    BOOL result;

    if ([first1 isEqual: last1] || [first2 isEqual: last2])
        return OBJ_AUTORELEASE([first1 copy]);
    tmp = [first2 copy];
    result = [[tmp advance] isEqual: last2];
    OBJ_RELEASE(tmp);
    f1 = [first1 copy];
    if (result)
    {
        while (![f1 isEqual: last1] &&
               ![pred performBinaryFunctionWithArg: [f1 dereference]
                    andArg: [first2 dereference]])
        {
            [f1 advance];
        }
        return OBJ_AUTORELEASE(f1);
    }
    tmp1 = [first2 copy];
    [tmp1 advance];
    while (![f1 isEqual: last1])
    {
        while (![f1 isEqual: last1])
        {
            if ([pred performBinaryFunctionWithArg: [f1 dereference]
                    andArg: [first2 dereference]])
            {
                break;
            }
            [f1 advance];
        }
        while (![f1 isEqual: last1] &&
               ![pred performBinaryFunctionWithArg: [f1 dereference]
                    andArg: [first2 dereference]])
        {
            [f1 advance];
        }
        if ([f1 isEqual: last1])
        {
            OBJ_RELEASE(tmp1);
            return OBJ_AUTORELEASE(f1);
        }
        cur = [f1 copy];
        if ([[cur advance] isEqual: last1])
        {
            OBJ_RELEASE(tmp1);
            OBJ_RELEASE(f1);
            return OBJ_AUTORELEASE(cur);
        }
        tmp = [tmp1 copy];
        while ([pred performBinaryFunctionWithArg: [cur dereference]
                    andArg: [tmp dereference]])
        {
            if ([[tmp advance] isEqual: last2])
            {
                OBJ_RELEASE(tmp);
                OBJ_RELEASE(tmp1);
                OBJ_RELEASE(cur);
                return OBJ_AUTORELEASE(f1);
            }
            if ([[cur advance] isEqual: last1])
            {
                OBJ_RELEASE(tmp);
                OBJ_RELEASE(tmp1);
                OBJ_RELEASE(f1);
                return OBJ_AUTORELEASE(cur);
            }
        }
        [f1 advance];
        OBJ_RELEASE(cur);
        OBJ_RELEASE(tmp);
    }
    OBJ_RELEASE(tmp1);
    return OBJ_AUTORELEASE(f1);
}
// }}}

// setDifferenceFrom:to:andFrom:andTo:destination: {{{
+ (OLForwardIterator*) setDifferenceFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last1 andFrom: (OLForwardIterator*)first2 andTo: (OLForwardIterator*)last2 destination: (OLForwardIterator*)dest
{
    OLLess* less = [[OLLess alloc] init];
    OLForwardIterator* result =
        [OLAlgorithm setDifferenceFrom: first1 to: last1
            andFrom: first2 andTo: last2 destination: dest predicate: less];

    OBJ_RELEASE(less);
    return result;
}
// }}}

// setDifferenceFrom:to:andFrom:andTo:destination:predicate: {{{
+ (OLForwardIterator*) setDifferenceFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last1 andFrom: (OLForwardIterator*)first2 andTo: (OLForwardIterator*)last2 destination: (OLForwardIterator*)dest predicate: (id<OLBoolBinaryFunction>)pred
{
    OLForwardIterator* f1 = [first1 copy];
    OLForwardIterator* f2 = [first2 copy];
    OLForwardIterator* d = [dest copy];
    OLForwardIterator* result;

    while (![f1 isEqual: last1] && ![f2 isEqual: last2])
    {
        if ([pred performBinaryFunctionWithArg: [f1 dereference]
                andArg: [f2 dereference]])
        {
            [d assign: [f1 dereference]];
            [f1 advance];
            [d advance];
        }
        else if ([pred performBinaryFunctionWithArg: [f2 dereference]
                    andArg: [f1 dereference]])
        {
            [f2 advance];
        }
        else
        {
            [f1 advance];
            [f2 advance];
        }
    }
    result = [OLAlgorithm copyImplFrom: f1 to: last1 destination: d needItor: YES];
    OBJ_RELEASE(f1);
    OBJ_RELEASE(f2);
    OBJ_RELEASE(d);
    return OBJ_AUTORELEASE(result);
}
// }}}

// setIntersectionFrom:to:andFrom:andTo:destination: {{{
+ (OLForwardIterator*) setIntersectionFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last1 andFrom: (OLForwardIterator*)first2 andTo: (OLForwardIterator*)last2 destination: (OLForwardIterator*)dest
{
    OLLess* less = [[OLLess alloc] init];
    OLForwardIterator* result =
        [OLAlgorithm setIntersectionFrom: first1 to: last1
            andFrom: first2 andTo: last2 destination: dest predicate: less];

    OBJ_RELEASE(less);
    return result;
}
// }}}

// setIntersectionFrom:to:andFrom:andTo:destination:predicate: {{{
+ (OLForwardIterator*) setIntersectionFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last1 andFrom: (OLForwardIterator*)first2 andTo: (OLForwardIterator*)last2 destination: (OLForwardIterator*)dest predicate: (id<OLBoolBinaryFunction>)pred
{
    OLForwardIterator* f1 = [first1 copy];
    OLForwardIterator* f2 = [first2 copy];
    OLForwardIterator* d = [dest copy];

    while (![f1 isEqual: last1] && ![f2 isEqual: last2])
    {
        if ([pred performBinaryFunctionWithArg: [f1 dereference]
                andArg: [f2 dereference]])
        {
            [f1 advance];
        }
        else if ([pred performBinaryFunctionWithArg: [f2 dereference]
                    andArg: [f1 dereference]])
        {
            [f2 advance];
        }
        else
        {
            [d assign: [f1 dereference]];
            [f1 advance];
            [f2 advance];
            [d advance];
        }
    }
    OBJ_RELEASE(f1);
    OBJ_RELEASE(f2);
    return OBJ_AUTORELEASE(d);
}
// }}}

// setSymmetricDifferenceFrom:to:andFrom:andTo:destination: {{{
+ (OLForwardIterator*) setSymmetricDifferenceFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last1 andFrom: (OLForwardIterator*)first2 andTo: (OLForwardIterator*)last2 destination: (OLForwardIterator*)dest
{
    OLLess* less = [[OLLess alloc] init];
    OLForwardIterator* result =
        [OLAlgorithm setSymmetricDifferenceFrom: first1 to: last1
            andFrom: first2 andTo: last2 destination: dest predicate: less];

    OBJ_RELEASE(less);
    return result;
}
// }}}

// setSymmetricDifferenceFrom:to:andFrom:andTo:destination:predicate: {{{
+ (OLForwardIterator*) setSymmetricDifferenceFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last1 andFrom: (OLForwardIterator*)first2 andTo: (OLForwardIterator*)last2 destination: (OLForwardIterator*)dest predicate: (id<OLBoolBinaryFunction>)pred
{
    OLForwardIterator* f1 = [first1 copy];
    OLForwardIterator* f2 = [first2 copy];
    OLForwardIterator* d = [dest copy];
    OLForwardIterator* tmp;
    OLForwardIterator* result;

    while (![f1 isEqual: last1] && ![f2 isEqual: last2])
    {
        if ([pred performBinaryFunctionWithArg: [f1 dereference]
                andArg: [f2 dereference]])
        {
            [d assign: [f1 dereference]];
            [f1 advance];
            [d advance];
        }
        else if ([pred performBinaryFunctionWithArg: [f2 dereference]
                    andArg: [f1 dereference]])
        {
            [d assign: [f2 dereference]];
            [f2 advance];
            [d advance];
        }
        else
        {
            [f1 advance];
            [f2 advance];
        }
    }
    tmp = [OLAlgorithm copyImplFrom: f1 to: last1 destination: d needItor: YES];
    result = [OLAlgorithm copyImplFrom: f2 to: last2 destination: tmp needItor: YES];
    OBJ_RELEASE(f1);
    OBJ_RELEASE(f2);
    OBJ_RELEASE(d);
    OBJ_RELEASE(tmp);
    return OBJ_AUTORELEASE(result);
}
// }}}

// setUnionFrom:to:andFrom:andTo:destination: {{{
+ (OLForwardIterator*) setUnionFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last1 andFrom: (OLForwardIterator*)first2 andTo: (OLForwardIterator*)last2 destination: (OLForwardIterator*)dest
{
    OLLess* less = [[OLLess alloc] init];
    OLForwardIterator* result =
        [OLAlgorithm setUnionFrom: first1 to: last1
            andFrom: first2 andTo: last2 destination: dest predicate: less];

    OBJ_RELEASE(less);
    return result;
}
// }}}

// setUnionFrom:to:andFrom:andTo:destination:predicate: {{{
+ (OLForwardIterator*) setUnionFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last1 andFrom: (OLForwardIterator*)first2 andTo: (OLForwardIterator*)last2 destination: (OLForwardIterator*)dest predicate: (id<OLBoolBinaryFunction>)pred
{
    OLForwardIterator* f1 = [first1 copy];
    OLForwardIterator* f2 = [first2 copy];
    OLForwardIterator* d = [dest copy];
    OLForwardIterator* tmp;
    OLForwardIterator* result;

    while (![f1 isEqual: last1] && ![f2 isEqual: last2])
    {
        if ([pred performBinaryFunctionWithArg: [f1 dereference]
                andArg: [f2 dereference]])
        {
            [d assign: [f1 dereference]];
            [f1 advance];
        }
        else if ([pred performBinaryFunctionWithArg: [f2 dereference]
                    andArg: [f1 dereference]])
        {
            [d assign: [f2 dereference]];
            [f2 advance];
        }
        else
        {
            [d assign: [f1 dereference]];
            [f1 advance];
            [f2 advance];
        }
        [d advance];
    }
    tmp = [OLAlgorithm copyImplFrom: f1 to: last1 destination: d needItor: YES];
    result = [OLAlgorithm copyImplFrom: f2 to: last2 destination: tmp needItor: YES];
    OBJ_RELEASE(f1);
    OBJ_RELEASE(f2);
    OBJ_RELEASE(d);
    OBJ_RELEASE(tmp);
    return OBJ_AUTORELEASE(result);
}
// }}}

// sortFrom:to: {{{
+ (void) sortFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last
{
    OLLess* less = [[OLLess alloc] init];

    [OLAlgorithm sortFrom: first to: last predicate: less];
    OBJ_RELEASE(less);
}
// }}}

// sortFrom:to:predicate: {{{
+ (void) sortFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last predicate: (id<OLBoolBinaryFunction>)pred
{
    unsigned depthLimit;
    unsigned len;

    if (![first isEqual: last])
    {
        len = [last difference: first];
        for (depthLimit = 0; len != 1; len >>= 1)
            depthLimit++;
        [OLAlgorithm introSortLoopFrom: first to: last
            depthLimit: depthLimit * 2 predicate: pred];
        [OLAlgorithm finalInsertionSortFrom: first to: last predicate: pred];
    }
}
// }}}

// sortHeapFrom:to: {{{
+ (void) sortHeapFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last
{
    OLLess* less = [[OLLess alloc] init];

    [OLAlgorithm sortHeapFrom: first to: last predicate: less];
    OBJ_RELEASE(less);
}
// }}}

// sortHeapFrom:to:predicate: {{{
+ (void) sortHeapFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last predicate: (id<OLBoolBinaryFunction>)pred
{
    OLRandomAccessIterator* l = [last copy];

    while ([l difference: first] > 1)
    {
        [OLAlgorithm popHeapFrom: first to: l predicate: pred];
        [l reverse];
    }
    OBJ_RELEASE(l);
}
// }}}

// stablePartitionFrom:to:predicate: {{{
+ (OLForwardIterator*) stablePartitionFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last predicate: (id<OLBoolUnaryFunction>)pred
{
    OLTempBuf* buf = [[OLTempBuf alloc] initWithFirst: first last: last];
    OLForwardIterator* result = [OLAlgorithm stablePartitionAdaptiveFrom: first to: last
        predicate: pred length: [buf size] buffer: buf];

    OBJ_RELEASE(buf);
    return OBJ_AUTORELEASE(result);
}
// }}}

// stableSortFrom:to: {{{
+ (void) stableSortFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last
{
    OLLess* less = [[OLLess alloc] init];

    [OLAlgorithm stableSortFrom: first to: last predicate: less];
    OBJ_RELEASE(less);
}
// }}}

// stableSortFrom:to:predicate: {{{
+ (void) stableSortFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last predicate: (id<OLBoolBinaryFunction>)pred
{
    OLTempBuf* buf = [[OLTempBuf alloc] initWithFirst: first last: last];

    [OLAlgorithm stableSortAdaptiveFrom: first to: last buffer: buf predicate: pred];
    OBJ_RELEASE(buf);
}
// }}}

// swap:and {{{
+ (void) swap: (id*)left and: (id*)right
{
    OL_PREPARE_SLOW_SWAP;

    if (left != right)
        OL_SLOW_SWAP(*left, *right);
}
// }}}

// swapIterators:and: {{{
+ (void) swapIterators: (OLIterator*)left and: (OLIterator*)right
{
    id tmp = OBJ_RETAIN([left dereference]);

    [left assign: [right dereference]];
    [right assign: tmp];
    OBJ_RELEASE(tmp);
}
// }}}

// swapRangesFrom:to:with: {{{
+ (OLForwardIterator*) swapRangesFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last1 with: (OLForwardIterator*)first2
{
    OLForwardIterator* f1 = [first1 copy];
    OLForwardIterator* f2 = [first2 copy];

    for ( ; ![f1 isEqual: last1]; [f1 advance], [f2 advance])
        [OLAlgorithm swapIterators: f1 and: f2];
    OBJ_RELEASE(f1);
    return OBJ_AUTORELEASE(f2);
}
// }}}

// transformFrom:to:destination:function: {{{
+ (OLForwardIterator*) transformFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last destination: (OLForwardIterator*)dest function: (id<OLUnaryFunction>)func
{
    OLForwardIterator* f = [first copy];
    OLForwardIterator* d = [dest copy];
    id result;
#if !defined(OL_NO_OPENSTEP)
    NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
    unsigned count = 0;
#endif

    for ( ; ![f isEqual: last]; [f advance], [d advance])
    {
        result = [func performUnaryFunctionWithArg: [f dereference]];
        [d assign: result];
#if defined(OL_NO_OPENSTEP)
        OBJ_RELEASE(result);
#else
        if (++count == 100)
        {
            OBJ_RELEASE(pool);
            pool = [[NSAutoreleasePool alloc] init];
            count = 0;
        }
#endif
    }
#if !defined(OL_NO_OPENSTEP)
    OBJ_RELEASE(pool);
#endif
    OBJ_RELEASE(f);
    return OBJ_AUTORELEASE(d);
}
// }}}

// transformFrom:to:withArgsFrom:destination:function: {{{
+ (OLForwardIterator*) transformFrom: (OLForwardIterator*)first1 to: (OLForwardIterator*)last withArgsFrom: (OLForwardIterator*)first2 destination: (OLForwardIterator*)dest function: (id<OLBinaryFunction>)func
{
    OLForwardIterator* f = [first1 copy];
    OLForwardIterator* f2 = [first2 copy];
    OLForwardIterator* d = [dest copy];
    id result;
#if !defined(OL_NO_OPENSTEP)
    NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
    unsigned count = 0;
#endif

    for ( ; ![f isEqual: last]; [f advance], [f2 advance], [d advance])
    {
        result = [func performBinaryFunctionWithArg: [f dereference]
            andArg: [f2 dereference]];
        [d assign: result];
#if defined(OL_NO_OPENSTEP)
        OBJ_RELEASE(result);
#else
        if (++count == 100)
        {
            OBJ_RELEASE(pool);
            pool = [[NSAutoreleasePool alloc] init];
            count = 0;
        }
#endif
    }
#if !defined(OL_NO_OPENSTEP)
    OBJ_RELEASE(pool);
#endif
    OBJ_RELEASE(f);
    OBJ_RELEASE(f2);
    return OBJ_AUTORELEASE(d);
}
// }}}

// uniqueCopyFrom:to:destination: {{{
+ (OLForwardIterator*) uniqueCopyFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last destination: (OLForwardIterator*)dest
{
    OLEqualTo* equal = [[OLEqualTo alloc] init];
    OLForwardIterator* result = [OLAlgorithm uniqueCopyFrom: first to: last
        destination: dest predicate: equal];

    OBJ_RELEASE(equal);
    return result;
}
// }}}

// uniqueCopyFrom:to:destination:predicate: {{{
+ (OLForwardIterator*) uniqueCopyFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last destination: (OLForwardIterator*)dest predicate: (id<OLBoolBinaryFunction>)pred
{
    OLForwardIterator* f = [first copy];
    OLForwardIterator* d = [dest copy];
    id val = [f dereference];

    [d assign: val];
    while (![[f advance] isEqual: last])
    {
        if (![pred performBinaryFunctionWithArg: val andArg: [f dereference]])
        {
            val = [f dereference];
            [[d advance] assign: val];
        }
    }
    OBJ_RELEASE(f);
    return OBJ_AUTORELEASE([d advance]);
}
// }}}

// uniqueFrom:to: {{{
+ (OLForwardIterator*) uniqueFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    return [OLAlgorithm uniqueCopyFrom: first to: last destination: first];
}
// }}}

// uniqueFrom:to:predicate: {{{
+ (OLForwardIterator*) uniqueFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last predicate: (id<OLBoolBinaryFunction>)pred
{
    return [OLAlgorithm uniqueCopyFrom: first to: last destination: first predicate: pred];
}
// }}}

// upperBoundFrom:to:value: {{{
+ (OLForwardIterator*) upperBoundFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last value: (id)object
{
    OLLess* less = [[OLLess alloc] init];
    OLForwardIterator* result =
        [OLAlgorithm upperBoundFrom: first to: last value: object predicate: less];

    OBJ_RELEASE(less);
    return result;
}
// }}}

// upperBoundFrom:to:value:predicate: {{{
+ (OLForwardIterator*) upperBoundFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last value: (id)object predicate: (id<OLBoolBinaryFunction>) pred
{
    return OBJ_AUTORELEASE([OLAlgorithm boundImplFrom: first to: last
        value: object predicate: pred lower: NO]);
}
// }}}

@end
// }}}

// implementation OLTempBuf {{{
@implementation OLTempBuf

// initWithFirst:last: {{{
- (id) initWithFirst: (OLForwardIterator*)first last: (OLForwardIterator*)last
{
    unsigned i;

    [super init];
    size = [OLIterator distanceFrom: first to: last];
    buffer = objc_malloc(size * sizeof(id));
    for (i = 0; i < size; i++)
        buffer[i] = OBJ_RETAIN(__bufferJunk);
    begin = [[OLArrayIterator alloc] initWithPointer: buffer];
    end = [[OLArrayIterator alloc] initWithPointer: buffer + size];
    return self;
}
// }}}

// dealloc {{{
#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    unsigned i;

    for (i = 0; i < size; i++)
        OBJ_RELEASE(buffer[i]);
    OBJ_RELEASE(begin);
    OBJ_RELEASE(end);
    objc_free(buffer);
    SUPER_FREE;
}
// }}}

// begin {{{
- (OLArrayIterator*) begin
{
    return begin;
}
// }}}

// end {{{
- (OLArrayIterator*) end
{
    return end;
}
// }}}

// size {{{
- (unsigned) size
{
    return size;
}
// }}}

@end
// }}}

// implementation OLAlgorithm (PrivateMethods) {{{
@implementation OLAlgorithm (PrivateMethods)

// adjustHeapFirst:hole:len:value:predicate: {{{
+ (void) adjustHeapFirst: (OLRandomAccessIterator*)first hole: (unsigned)hole len: (unsigned)len value: (id)object predicate: (id<OLBoolBinaryFunction>)pred
{
    OLRandomAccessIterator* f = [first copy];
    OLRandomAccessIterator* f2 = [first copy];
    unsigned top = hole;
    unsigned secondChild = 2 * hole + 2;
    BOOL result;
    id savedObject = OBJ_RETAIN(object);

    while (secondChild < len)
    {
        [f advanceBy: secondChild];
        [f2 advanceBy: secondChild - 1];
        result = [pred performBinaryFunctionWithArg: [f dereference]
                    andArg: [f2 dereference]];
        [f advanceBy: -secondChild];
        [f2 advanceBy: -(secondChild - 1)];
        if (result)
            secondChild--;
        [f advanceBy: hole];
        [f2 advanceBy: secondChild];
        [f assign: [f2 dereference]];
        [f advanceBy: -hole];
        [f2 advanceBy: -secondChild];
        hole = secondChild;
        secondChild = 2 * (secondChild + 1);
    }
    if (secondChild == len)
    {
        [[f advanceBy: hole] assign: [[f2 advanceBy: secondChild - 1] dereference]];
        hole = secondChild - 1;
    }
    OBJ_RELEASE(f);
    OBJ_RELEASE(f2);
    [OLAlgorithm pushHeapImplFirst: first hole: hole top: top
        value: savedObject predicate: pred];
    OBJ_RELEASE(savedObject);
}
// }}}

// boundImplFrom:to:value:predicate:lower: {{{
+ (OLForwardIterator*) boundImplFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last value: (id)object predicate: (id<OLBoolBinaryFunction>)pred lower: (BOOL)lwr
{
    OLForwardIterator* cur = [first copy];
    OLForwardIterator* middle = nil;
    unsigned len = [OLIterator distanceFrom: first to: last];
    unsigned half;

    while (len > 0)
    {
        half = len >> 1;
        OBJ_RELEASE(middle);
        middle = [cur copy];
        [OLIterator advanceIterator: middle distance: half];
        if (lwr)
        {
            if ([pred performBinaryFunctionWithArg: [middle dereference] andArg: object])
            {
                OBJ_RELEASE(cur);
                cur = [middle copy];
                [cur advance];
                len = len - half - 1;
            }
            else
            {
                len = half;
            }
        }
        else
        {
            if ([pred performBinaryFunctionWithArg: object andArg: [middle dereference]])
            {
                len = half;
            }
            else
            {
                OBJ_RELEASE(cur);
                cur = [middle copy];
                [cur advance];
                len = len - half - 1;
            }
        }
    }
    OBJ_RELEASE(middle);
    return cur;
}
// }}}

// chunkInsertionSortFrom:to:chunkSize:predicate: {{{
+ (void) chunkInsertionSortFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last chunkSize: (unsigned)size predicate: (id<OLBoolBinaryFunction>)pred
{
    OLRandomAccessIterator* f = [first copy];
    OLRandomAccessIterator* l = [first copy];

    [l advanceBy: size];
    while ([last difference: f] >= size)
    {
        [OLAlgorithm insertionSortFrom: f to: l predicate: pred];
        [f advanceBy: size];
        [l advanceBy: size];
    }
    [OLAlgorithm insertionSortFrom: f to: last predicate: pred];
    OBJ_RELEASE(l);
    OBJ_RELEASE(f);
}
// }}}

// copyBackwardImplFrom:to:destination:needItor: {{{
+ (OLBidirectionalIterator*) copyBackwardImplFrom: (OLBidirectionalIterator*)first to: (OLBidirectionalIterator*)last destination: (OLBidirectionalIterator*)dest needItor: (BOOL)needItor
{
    OLBidirectionalIterator* src = [last copy];
    OLBidirectionalIterator* dst = [dest copy];

    while (![src isEqual: first])
        [[dst reverse] assign: [[src reverse] dereference]];
    OBJ_RELEASE(src);
    if (needItor)
        return dst;
    OBJ_RELEASE(dst);
    return nil;
}
// }}}

// copyImplFrom:to:destination:needItor: {{{
+ (OLForwardIterator*) copyImplFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last destination: (OLForwardIterator*)dest needItor: (BOOL)needItor
{
    OLForwardIterator* src = [first copy];
    OLForwardIterator* dst = [dest copy];

    for ( ; ![src isEqual: last]; [src advance], [dst advance])
        [dst assign: [src dereference]];
    OBJ_RELEASE(src);
    if (needItor)
        return dst;
    OBJ_RELEASE(dst);
    return nil;
}
// }}}

// finalInsertionSortFrom:to:predicate: {{{
+ (void) finalInsertionSortFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last predicate: (id<OLBoolBinaryFunction>)pred
{
    OLRandomAccessIterator* tmp;

    if ([last difference: first] > OL_SORT_THRESHOLD)
    {
        tmp = [first copy];
        [tmp advanceBy: OL_SORT_THRESHOLD];
        [OLAlgorithm insertionSortFrom: first to: tmp predicate: pred];
        [OLAlgorithm unguardedInsertionSortFrom: tmp to: last predicate: pred];
        OBJ_RELEASE(tmp);
    }
    else
    {
        [OLAlgorithm insertionSortFrom: first to: last predicate: pred];
    }
}
// }}}

// findIfImplFrom:to:predicate: {{{
+ (OLForwardIterator*) findIfImplFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last predicate: (id<OLBoolUnaryFunction>)pred
{
    OLForwardIterator* src = [first copy];

    for ( ; ![src isEqual: last]; [src advance])
    {
        if ([pred performUnaryFunctionWithArg: [src dereference]])
            break;
    }
    return src;
}
// }}}

// findImplFrom:to:value: {{{
+ (OLForwardIterator*) findImplFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last value: (id)object
{
    OLForwardIterator* src = [first copy];

    for ( ; ![src isEqual: last]; [src advance])
    {
        if ([[src dereference] isEqual: object])
            break;
    }
    return src;
}
// }}}

// insertionSortFrom:to:predicate: {{{
+ (void) insertionSortFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last predicate: (id<OLBoolBinaryFunction>)pred
{
    OLRandomAccessIterator* i;

    if (![first isEqual: last])
    {
        i = [first copy];
        for ([i advance]; ![i isEqual: last]; [i advance])
            [OLAlgorithm linearInsertFrom: first to: i value: [i dereference] predicate: pred];
        OBJ_RELEASE(i);
    }
}
// }}}

// introSortLoopFrom:to:depthLimit:predicate: {{{
+ (void) introSortLoopFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last depthLimit: (unsigned)depthLimit predicate: (id<OLBoolBinaryFunction>)pred
{
    OLRandomAccessIterator* l = [last copy];
    OLRandomAccessIterator* cut;
    OLRandomAccessIterator* tmp = [first copy];
    id med;
    int advance;

    while ([l difference: first] > OL_SORT_THRESHOLD)
    {
        if (depthLimit == 0)
        {
            [OLAlgorithm partialSortFrom: first middle: l to: l predicate: pred];
            break;
        }
        depthLimit--;
        advance = [l difference: first] / 2;
        [tmp advanceBy: advance];
        [l reverse];
        med = [OLAlgorithm medianFromOne: [first dereference] two: [tmp dereference]
            three: [l dereference] predicate: pred];
        [tmp advanceBy: -advance];
        [l advance];
        cut = [OLAlgorithm unguardedPartitionFrom: first to: l pivot: med predicate: pred];
        [OLAlgorithm introSortLoopFrom: cut to: l depthLimit: depthLimit predicate: pred];
        OBJ_RELEASE(l);
        l = cut;
    }
    OBJ_RELEASE(l);
    OBJ_RELEASE(tmp);
}
// }}}

// linearInsertFrom:to:value:predicate: {{{
+ (void) linearInsertFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last value: (id)object predicate: (id<OLBoolBinaryFunction>)pred
{
    OLRandomAccessIterator* dest;
    id obj;

    if ([pred performBinaryFunctionWithArg: object andArg: [first dereference]])
    {
        obj = OBJ_RETAIN(object);
        dest = [last copy];
        [OLAlgorithm copyBackwardImplFrom: first to: last
            destination: [dest advance] needItor: NO];
        OBJ_RELEASE(dest);
        [first assign: obj];
        OBJ_RELEASE(obj);
    }
    else
    {
        [OLAlgorithm unguardedLinearInsertAt: last value: object predicate: pred];
    }
}
// }}}

// medianFromOne:two:three:predicate: {{{
+ (id) medianFromOne: (id)one two: (id)two three: (id)three predicate: (id<OLBoolBinaryFunction>)pred
{
    if ([pred performBinaryFunctionWithArg: one andArg: two])
    {
        if ([pred performBinaryFunctionWithArg: two andArg: three])
            return two;
        else if ([pred performBinaryFunctionWithArg: one andArg: three])
            return three;
        else
            return one;
    }
    else if ([pred performBinaryFunctionWithArg: one andArg: three])
    {
        return one;
    }
    else if ([pred performBinaryFunctionWithArg: two andArg: three])
    {
        return three;
    }
    return two;
}
// }}}

// mergeAdaptiveFrom:middle:last:len1:len2:buffer:predicate: {{{
+ (void) mergeAdaptiveFrom: (OLBidirectionalIterator*)first
                            middle: (OLBidirectionalIterator*)middle
                            last: (OLBidirectionalIterator*)last
                            len1: (unsigned)len1
                            len2: (unsigned)len2
                            buffer: (OLTempBuf*)buf
                            predicate: (id<OLBoolBinaryFunction>)pred
{
    OLBidirectionalIterator* firstCut;
    OLBidirectionalIterator* secondCut;
    OLBidirectionalIterator* newMiddle;
    OLBidirectionalIterator* bufEnd;
    unsigned len11;
    unsigned len22;

    if (len1 <= len2 && len1 <= [buf size])
    {
        bufEnd = (OLBidirectionalIterator*)[OLAlgorithm copyImplFrom: first to: middle
            destination: [buf begin] needItor: YES];
        [OLAlgorithm mergeImplSeries1From: [buf begin] series1To: bufEnd
            series2From: middle series2To: last destination: first
            predicate: pred needItor: NO];
        OBJ_RELEASE(bufEnd);
    }
    else if (len2 <= [buf size])
    {
        bufEnd = (OLBidirectionalIterator*)[OLAlgorithm copyImplFrom: middle to: last
            destination: [buf begin] needItor: YES];
        [OLAlgorithm mergeBackwardSeries1From: first series1To: middle
            series2From: [buf begin] series2To: bufEnd destination: last
            predicate: pred needItor: NO];
        OBJ_RELEASE(bufEnd);
    }
    else
    {
        if (len1 > len2)
        {
            firstCut = [first copy];
            len11 = len1 / 2;
            [OLIterator advanceIterator: firstCut distance: len11];
            secondCut = (OLBidirectionalIterator*)[OLAlgorithm boundImplFrom: middle
                to: last value: [firstCut dereference] predicate: pred lower: YES];
            len22 = [OLIterator distanceFrom: middle to: secondCut];
        }
        else
        {
            secondCut = [middle copy];
            len22 = len2 / 2;
            [OLIterator advanceIterator: secondCut distance: len22];
            firstCut = (OLBidirectionalIterator*)[OLAlgorithm boundImplFrom: first
                to: middle value: [secondCut dereference] predicate: pred lower: NO];
            len11 = [OLIterator distanceFrom: middle to: secondCut];
        }
        newMiddle = [OLAlgorithm rotateAdaptiveFirst: firstCut middle: middle last: secondCut
            len1: len1 - len11 len2: len22 buffer: buf];
        [OLAlgorithm mergeAdaptiveFrom: first middle: firstCut last: newMiddle
            len1: len11 len2: len22 buffer: buf predicate: pred];
        [OLAlgorithm mergeAdaptiveFrom: newMiddle middle: secondCut last: last
            len1: len1 - len11 len2: len2 - len22 buffer: buf predicate: pred];
        OBJ_RELEASE(firstCut);
        OBJ_RELEASE(secondCut);
        OBJ_RELEASE(newMiddle);
    }
}
// }}}

// mergeBackwardSeries1From:series1To:series2From:series2To:destination:predicate: {{{
+ (OLBidirectionalIterator*) mergeBackwardSeries1From: (OLBidirectionalIterator*)first1 series1To: (OLBidirectionalIterator*)last1 series2From: (OLBidirectionalIterator*)first2 series2To: (OLBidirectionalIterator*)last2 destination: (OLBidirectionalIterator*)result predicate: (id<OLBoolBinaryFunction>)pred needItor: (BOOL)needItor
{
    OLBidirectionalIterator* l1;
    OLBidirectionalIterator* l2;
    OLBidirectionalIterator* res;
    OLBidirectionalIterator* rc;

    if ([first1 isEqual: last1])
    {
        return [OLAlgorithm copyBackwardImplFrom: first2 to: last2
            destination: result needItor: needItor];
    }
    if ([first2 isEqual: last2])
    {
        return [OLAlgorithm copyBackwardImplFrom: first1 to: last1
            destination: result needItor: needItor];
    }
    l1 = [last1 copy];
    l2 = [last2 copy];
    res = [result copy];
    [l1 reverse];
    [l2 reverse];
    while (YES)
    {
        if ([pred performBinaryFunctionWithArg: [l2 dereference]
                andArg: [l1 dereference]])
        {
            [[res reverse] assign: [l1 dereference]];
            if ([first1 isEqual: l1])
            {
                rc = [OLAlgorithm copyBackwardImplFrom: first2 to: [l2 advance]
                    destination: res needItor: needItor];
                break;
            }
            [l1 reverse];
        }
        else
        {
            [[res reverse] assign: [l2 dereference]];
            if ([first2 isEqual: l2])
            {
                rc = [OLAlgorithm copyBackwardImplFrom: first1 to: [l1 advance]
                    destination: res needItor: needItor];
                break;
            }
            [l2 reverse];
        }
    }
    OBJ_RELEASE(l1);
    OBJ_RELEASE(l2);
    OBJ_RELEASE(res);
    return rc;
}
// }}}

// mergeImplSeries1From:series1To:series2From:series2To:destination:predicate:needItor: {{{
+ (OLForwardIterator*) mergeImplSeries1From: (OLForwardIterator*)first1 series1To: (OLForwardIterator*)last1 series2From: (OLForwardIterator*)first2 series2To: (OLForwardIterator*)last2 destination: (OLForwardIterator*)dest predicate: (id<OLBoolBinaryFunction>)pred needItor: (BOOL)needItor
{
    OLForwardIterator* f1 = [first1 copy];
    OLForwardIterator* f2 = [first2 copy];
    OLForwardIterator* result = [dest copy];
    OLForwardIterator* tmp1;
    OLForwardIterator* tmp2;

    while (![f1 isEqual: last1] && ![f2 isEqual: last2])
    {
        if ([pred performBinaryFunctionWithArg: [f2 dereference]
                andArg: [f1 dereference]])
        {
            [result assign: [f2 dereference]];
            [f2 advance];
        }
        else
        {
            [result assign: [f1 dereference]];
            [f1 advance];
        }
        [result advance];
    }
    tmp1 = [OLAlgorithm copyImplFrom: f1 to: last1 destination: result needItor: YES];
    tmp2 = [OLAlgorithm copyImplFrom: f2 to: last2 destination: tmp1 needItor: needItor];
    OBJ_RELEASE(tmp1);
    OBJ_RELEASE(f1);
    OBJ_RELEASE(f2);
    OBJ_RELEASE(result);
    return tmp2;
}
// }}}

// mergeSortLoopFrom:to:destination:stepSize:predicate: {{{
+ (void) mergeSortLoopFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last destination: (OLRandomAccessIterator*)dest stepSize: (unsigned)stepSize predicate: (id<OLBoolBinaryFunction>)pred
{
    OLRandomAccessIterator* f = [first copy];
    OLRandomAccessIterator* f2 = [first copy];
    OLRandomAccessIterator* f3 = [first copy];
    OLForwardIterator* dst = [dest copy];
    OLForwardIterator* tmp;
    unsigned twoStep = stepSize * 2;

    [f2 advanceBy: stepSize];
    [f3 advanceBy: twoStep];
    while ([last difference: f] >= twoStep)
    {
        tmp = [OLAlgorithm mergeImplSeries1From: f series1To: f2
            series2From: f2 series2To: f3 destination: dst
            predicate: pred needItor: YES];
        OBJ_RELEASE(dst);
        dst = tmp;
        [f advanceBy: twoStep];
        [f2 advanceBy: twoStep];
        [f3 advanceBy: twoStep];
    }
    [f2 advanceBy: -stepSize];
    stepSize = MIN([last difference: f], stepSize);
    [f2 advanceBy: stepSize];
    [OLAlgorithm mergeImplSeries1From: f series1To: f2
        series2From: f2 series2To: last destination: dst
        predicate: pred needItor: NO];
    OBJ_RELEASE(f);
    OBJ_RELEASE(f2);
    OBJ_RELEASE(f3);
    OBJ_RELEASE(dst);
}
// }}}

// mergeSortWithBufferFrom:to:buffer:predicate: {{{
+ (void) mergeSortWithBufferFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last buffer: (OLTempBuf*)buf predicate: (id<OLBoolBinaryFunction>)pred
{
    unsigned len = [last difference: first];
    unsigned stepSize = OL_CHUNK_SIZE;
    OLArrayIterator* bufEnd = [[buf begin] copy];

    [bufEnd advanceBy: len];
    [OLAlgorithm chunkInsertionSortFrom: first to: last chunkSize: stepSize predicate: pred];
    while (stepSize < len)
    {
        [OLAlgorithm mergeSortLoopFrom: first to: last destination: [buf begin]
            stepSize: stepSize predicate: pred];
        stepSize *= 2;
        [OLAlgorithm mergeSortLoopFrom: [buf begin] to: bufEnd destination: first
            stepSize: stepSize predicate: pred];
        stepSize *= 2;
    }
    OBJ_RELEASE(bufEnd);
}
// }}}

// popHeapImplFirst:last:result:value:predicate: {{{
+ (void) popHeapImplFirst: (OLRandomAccessIterator*)first last: (OLRandomAccessIterator*)last result: (OLRandomAccessIterator*)result value: (id)object predicate: (id<OLBoolBinaryFunction>)pred
{
    id savedObject = OBJ_RETAIN(object);

    [result assign: [first dereference]];
    [OLAlgorithm adjustHeapFirst: first hole: 0 len: [last difference: first]
        value: savedObject predicate: pred];
    OBJ_RELEASE(savedObject);
}
// }}}

// pushHeapImplFirst:hole:top:value:predicate: {{{
+ (void) pushHeapImplFirst: (OLRandomAccessIterator*)first hole: (unsigned)hole top: (unsigned)top value: (id)object predicate: (id<OLBoolBinaryFunction>)pred
{
    OLRandomAccessIterator* f = [first copy];
    OLRandomAccessIterator* f2 = [first copy];
    unsigned parent = (hole - 1) / 2;
    id savedObject = OBJ_RETAIN(object);

    while (hole > top)
    {
        [f advanceBy: parent];
        if (![pred performBinaryFunctionWithArg: [f dereference] andArg: savedObject])
        {
            [f advanceBy: -parent];
            break;
        }
        [f2 advanceBy: hole];
        [f2 assign: [f dereference]];
        [f advanceBy: -parent];
        [f2 advanceBy: -hole];
        hole = parent;
        parent = (hole - 1) / 2;
    }
    [[f advanceBy: hole] assign: savedObject];
    OBJ_RELEASE(savedObject);
    OBJ_RELEASE(f);
    OBJ_RELEASE(f2);
}
// }}}

// rotateAdaptiveFirst:middle:last:len1:len2:buffer: {{{
+ (OLBidirectionalIterator*) rotateAdaptiveFirst: (OLBidirectionalIterator*)first
        middle: (OLBidirectionalIterator*)middle
        last: (OLBidirectionalIterator*)last
        len1: (unsigned)len1
        len2: (unsigned)len2
        buffer: (OLTempBuf*)buf
{
    if (len1 > len2 && len2 <= [buf size])
    {
        [OLAlgorithm copyImplFrom: middle to: last destination: [buf begin] needItor: NO];
        [OLAlgorithm copyBackwardImplFrom: first to: middle destination: last needItor: NO];
        return [OLAlgorithm copyBackwardImplFrom: [buf begin] to: [buf end]
            destination: last needItor: YES];
    }
    else if (len1 <= [buf size])
    {
        [OLAlgorithm copyImplFrom: first to: middle destination: [buf begin] needItor: NO];
        [OLAlgorithm copyImplFrom: middle to: last destination: first needItor: NO];
        return [OLAlgorithm copyBackwardImplFrom: [buf begin] to: [buf end]
            destination: last needItor: YES];
    }
    else
    {
        return (OLBidirectionalIterator*)[OLAlgorithm rotateFrom: first
            middle: middle to: last];
    }
}
// }}}

// rotateImplFrom:middle:to:last: {{{
+ (OLForwardIterator*) rotateImplFrom: (OLForwardIterator*)first middle: (OLForwardIterator*)mid to: (OLForwardIterator*)last
{
    OLForwardIterator* f1;
    OLForwardIterator* f2;
    OLForwardIterator* mid1;
    OLForwardIterator* newMid;

    if ([first isEqual: mid])
        return last;
    if ([last isEqual: mid])
        return first;
    f1 = [first copy];
    f2 = [mid copy];
    mid1 = [mid copy];
    do
    {
        [OLAlgorithm swapIterators: f1 and: f2];
        [f1 advance];
        [f2 advance];
        if ([f1 isEqual: mid1])
        {
            OBJ_RELEASE(mid1);
            mid1 = [f2 copy];
        }
    } while (![f2 isEqual: last]);
    newMid = [f1 copy];
    OBJ_RELEASE(f2);
    f2 = [mid1 copy];
    while (![f2 isEqual: last])
    {
        [OLAlgorithm swapIterators: f1 and: f2];
        [f1 advance];
        [f2 advance];
        if ([f1 isEqual: mid1])
        {
            OBJ_RELEASE(mid1);
            mid1 = [f2 copy];
        }
        else if ([f2 isEqual: last])
        {
            OBJ_RELEASE(f2);
            f2 = [mid1 copy];
        }
    }
    OBJ_RELEASE(f1);
    OBJ_RELEASE(f2);
    OBJ_RELEASE(mid1);
    return newMid;
}
// }}}

// stablePartitionAdaptiveFrom:to:predicate:length:buffer: {{{
+ (OLForwardIterator*) stablePartitionAdaptiveFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last predicate: (id<OLBoolUnaryFunction>)pred length: (unsigned)len buffer: (OLTempBuf*)buf
{
    OLForwardIterator* middle;
    OLForwardIterator* rfirst;
    OLForwardIterator* rlast;
    OLForwardIterator* result1;
    OLArrayIterator* result2;
    OLForwardIterator* f;

    if (len <= [buf size])
    {
        result1 = [first copy];
        result2 = [[buf begin] copy];
        f = [first copy];
        for ( ; ![f isEqual: last]; [f advance])
        {
            if ([pred performUnaryFunctionWithArg: [f dereference]])
            {
                [result1 assign: [f dereference]];
                [result1 advance];
            }
            else
            {
                [result2 assign: [f dereference]];
                [result2 advance];
            }
        }
        [OLAlgorithm copyImplFrom: [buf begin] to: result2 destination: result1 needItor: NO];
        OBJ_RELEASE(f);
        OBJ_RELEASE(result2);
    }
    else
    {
        middle = [first copy];
        [OLIterator advanceIterator: middle distance: [buf size] / 2];
        rfirst = [OLAlgorithm stablePartitionAdaptiveFrom: first to: middle predicate: pred
            length: len / 2 buffer: buf];
        rlast = [OLAlgorithm stablePartitionAdaptiveFrom: middle to: last predicate: pred
            length: len - len / 2 buffer: buf];
        result1 = [OLAlgorithm rotateImplFrom: rfirst middle: middle to: rlast];
        OBJ_RELEASE(middle);
        OBJ_RELEASE(rfirst);
        OBJ_RELEASE(rlast);
    }
    return result1;
}
// }}}

// stableSortAdaptiveFrom:to:buffer:predicate: {{{
+ (void) stableSortAdaptiveFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last buffer: (OLTempBuf*)buf predicate: (id<OLBoolBinaryFunction>)pred
{
    OLRandomAccessIterator* middle = [first copy];
    unsigned len = ([last difference: first] + 1) / 2;

    [middle advanceBy: len];
    if (len > [buf size])
    {
        [OLAlgorithm stableSortAdaptiveFrom: first to: middle buffer: buf predicate: pred];
        [OLAlgorithm stableSortAdaptiveFrom: middle to: last buffer: buf predicate: pred];
    }
    else
    {
        [OLAlgorithm mergeSortWithBufferFrom: first to: middle buffer: buf predicate: pred];
        [OLAlgorithm mergeSortWithBufferFrom: middle to: last buffer: buf predicate: pred];
    }
    [OLAlgorithm mergeAdaptiveFrom: first middle: middle last: last
        len1: [middle difference: first] len2: [last difference: middle]
        buffer: buf predicate: pred];
    OBJ_RELEASE(middle);
}
// }}}

// unguardedLinearInsertAt:value:predicate: {{{
+ (void) unguardedLinearInsertAt: (OLRandomAccessIterator*)last value: (id)object predicate: (id<OLBoolBinaryFunction>)pred
{
    OLRandomAccessIterator* next = [last copy];
    OLRandomAccessIterator* l = [last copy];
    id savedObject = OBJ_RETAIN(object);

    [next reverse];
    while ([pred performBinaryFunctionWithArg: savedObject andArg: [next dereference]])
    {
        [l assign: [next dereference]];
        [l reverse];
        [next reverse];
    }
    [l assign: savedObject];
    OBJ_RELEASE(savedObject);
    OBJ_RELEASE(l);
    OBJ_RELEASE(next);
}
// }}}

// unguardedInsertionSortFrom:to:predicate: {{{
+ (void) unguardedInsertionSortFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last predicate: (id<OLBoolBinaryFunction>)pred
{
    OLRandomAccessIterator* i = [first copy];

    for ( ; ![i isEqual: last]; [i advance])
        [OLAlgorithm unguardedLinearInsertAt: i value: [i dereference] predicate: pred];
    OBJ_RELEASE(i);
}
// }}}

// unguardedPartitionFrom:to:pivot:predicate: {{{
+ (OLRandomAccessIterator*) unguardedPartitionFrom: (OLRandomAccessIterator*)first to: (OLRandomAccessIterator*)last pivot: (id)object predicate: (id<OLBoolBinaryFunction>)pred
{
    OLRandomAccessIterator* f = [first copy];
    OLRandomAccessIterator* l = [last copy];
    id savedObject = OBJ_RETAIN(object);

    while (YES)
    {
        while ([pred performBinaryFunctionWithArg: [f dereference]
                    andArg: savedObject])
        {
            [f advance];
        }
        [l reverse];
        while ([pred performBinaryFunctionWithArg: savedObject
                    andArg: [l dereference]])
        {
            [l reverse];
        }
        if ([f difference: l] >= 0)
        {
            OBJ_RELEASE(savedObject);
            OBJ_RELEASE(l);
            return f;
        }
        [OLAlgorithm swapIterators: f and: l];
        [f advance];
    }
}
// }}}

@end
// }}}

// implementation OLUnsignedNumber {{{
@implementation OLUnsignedNumber

- (id) initWithUnsignedInt: (unsigned)num
{
    [super init];
    value = num;
    return self;
}

- (unsigned) unsignedIntValue
{
    return value;
}

@end
// }}}
