//
// $Id: QueueTest.m,v 1.17 2007/03/28 03:16:53 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "QueueTest.h"
#import "Random.h"
#import <ObjectiveLib/Queue.h>
#import <ObjectiveLib/DataOutStream.h>
#import <ObjectiveLib/DataInStream.h>
#import <ObjectiveLib/ObjectOutStream.h>
#import <ObjectiveLib/ObjectInStream.h>
#import "Number.h"
#if defined(OL_NO_OPENSTEP)
#import <ObjectiveLib/Text.h>
#import <ObjectiveLib/Reaper.h>
#else
#import <Foundation/NSString.h>
#import <Foundation/NSData.h>
#import <Foundation/NSArchiver.h>
#if defined(HAVE_KEYED_ARCHIVES)
#import <Foundation/NSKeyedArchiver.h>
#endif
#import <Foundation/NSValue.h>
#endif
#include <stdlib.h>
#if defined(__NEXT_RUNTIME__)
#import <objc/objc-class.h>
#endif

@implementation QueueTest

- (void) testAccessors
{
    OLQueue* q;
    CONSTSTR* strs[] = { @"one", @"two", @"three" };
    int i;

    q = [[OLQueue alloc] init];
    for (i = 0; i < 3; i++)
        [q push: strs[i]];
    if (![[q front] isEqual: strs[0]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"%s\", but got \"%s\"",
            [strs[0] cString], [[q front] cString]];
    }
    if (![[q back] isEqual: strs[2]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"%s\", but got \"%s\"",
            [strs[2] cString], [[q front] cString]];
    }
    [q RELEASE];
}

#if !defined(OL_NO_OPENSTEP)
- (void) testCoding
{
    NSMutableData* data;
    NSArchiver* archiver;
    NSData* archData;
    OLQueue* q1;
    OLQueue* q2;
    OLNumber* num;
    int i;


    q1 = [[OLQueue alloc] init];
    for (i = 0; i< 5000; i++)
    {
        num = [[OLNumber alloc] initWithInt: OLRandom() % 10000];
        [q1 push: num];
        [num RELEASE];
    }
    data = [[NSMutableData alloc] initWithCapacity: 5000];
    archiver = [[NSArchiver alloc] initForWritingWithMutableData: data];
    [archiver encodeRootObject: q1];
    [archiver RELEASE];
    q2 = [NSUnarchiver unarchiveObjectWithData: data];
    [data RELEASE];
    if (![q1 isEqual: q2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The deques should be equal"];
    }
#if defined(HAVE_KEYED_ARCHIVES)
    archData = [NSKeyedArchiver archivedDataWithRootObject: q1];
    q2 = [NSKeyedUnarchiver unarchiveObjectWithData: archData];
    if (![q1 isEqual: q2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The deques should be equal"];
    }
#endif
    [q1 RELEASE];
}
#endif

- (void) testConvenienceAllocators
{
    OLQueue* q;
    OLQueue* q2;
    OLNumber* num;

    q = REAP([OLQueue queue]);
    if (![q IS_MEMBER_OF: [OLQueue class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"%s\", but got \"%s\"",
            ((Class)[OLQueue class])->name, ((Class)[q class])->name];
    }
    if (![q empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The queue should be empty"];
    }
    num = [[OLNumber alloc] initWithInt: 1];
    [q push: num];
    [num RELEASE];
    num = [[OLNumber alloc] initWithInt: 2];
    [q push: num];
    [num RELEASE];
    q2 = REAP([OLQueue queueWithQueue: q]);
    if (![q isEqual: q2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The queues should be equal"];
    }
}

- (void) testInitializers
{
    OLQueue* q;
    OLQueue* q2;
    CONSTSTR* strs[] = { @"one", @"two", @"three" };
    int i;

    q = [[OLQueue alloc] init];
    if ([q size] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %i", [q size]];
    }
    for (i = 0; i < 3; i++)
        [q push: strs[i]];
    q2 = [[OLQueue alloc] initWithQueue: q];
    if ([q2 size] != 3)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 3, but got %i", [q2 size]];
    }
    if (![q isEqual: q2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The queues are equal"];
    }
    [q RELEASE];
    [q2 RELEASE];
}

- (void) testPop
{
    OLQueue* q;
    CONSTSTR* strs[] = { @"one", @"two", @"three" };
    int i;

    q = [[OLQueue alloc] init];
    for (i = 0; i < 3; i++)
        [q push: strs[i]];
    for (i = 0; i < 3; i++)
    {
        if (![[q front] isEqual: strs[i]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"%s\", but got \"%s\"",
                [strs[i] cString], [[q front] cString]];
        }
        [q pop];
    }
    if ([q size] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %i", [q size]];
    }
    [q RELEASE];
}

- (void) testProperties
{
    OLQueue* q;
    OLQueue* q2;

    q = [[OLQueue alloc] init];
    if ([q size] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %i", [q size]];
    }
    if (![q empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The queue is empty"];
    }
    [q push: @"one"];
    q2 = [[OLQueue alloc] init];
    if ([q isEqual: q2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The queues are not equal"];
    }
    [q2 push: @"one"];
    if (![q isEqual: q2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The queues are equal"];
    }
    [q RELEASE];
    [q2 RELEASE];
}

- (void) testPush
{
    OLQueue* q;
    CONSTSTR* strs[] = { @"one", @"two", @"three" };
    int i;

    q = [[OLQueue alloc] init];
    for (i = 0; i < 3; i++)
    {
        [q push: strs[i]];
        if (![[q back] isEqual: strs[i]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected \"%s\", but got \"%s\"",
                [strs[i] cString], [[q back] cString]];
        }
    }
    [q RELEASE];
}

- (void) testStreaming
{
    OLDataOutStream* dout;
    OLObjectOutStream* oout;
    OLObjectInStream* oin;
    OLQueue* queue;
    OLQueue* readQueue;
    OLNumber* num;
    int i;

    dout = REAP([OLDataOutStream stream]);
    oout = REAP([OLObjectOutStream streamWithOutStream: dout]);
    queue = [[OLQueue alloc] init];
    for (i = 0; i< 10000; i++)
    {
        num = [[OLNumber alloc] initWithInt: OLRandom() % 10000];
        [queue push: num];
        [num RELEASE];
    }
    [oout writeObject: queue];
    oin = REAP([OLObjectInStream streamWithInStream:
        REAP([OLDataInStream streamWithBytes: [dout bytes] count: [dout count]])]);
    readQueue = REAP([oin readObject]);
    if (![readQueue IS_MEMBER_OF: [OLQueue class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected OLQueue class, but got %s", ((Class)[readQueue class])->name];
    }
    if (![readQueue isEqual: queue])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The queues should be equal"];
    }
    [queue RELEASE];
}

@end
