//
// $Id: Map.m,v 1.20 2007/03/06 20:42:19 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "Map.h"
#import "Tree.h"
#import "Macros.h"
#import "RunTime.h"
#import "ObjectInStream.h"
#import "ObjectOutStream.h"
#import "Utility.h"
#if !defined(OL_NO_OPENSTEP)
#import <Foundation/NSCoder.h>
#import <Foundation/NSAutoreleasePool.h>
#import <Foundation/NSString.h>
#endif
#import <stdlib.h>

#if !defined(OL_NO_OPENSTEP)
NSString* const COMPARATOR_KEY = @"OL_COMPARATOR_KEY";
#endif

@implementation OLMap

+ (id) map
{
    OL_BEGIN_AUTO_CTOR(OLMap)
        init
    OL_END_AUTO_CTOR;
}

+ (id) mapFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    OL_BEGIN_AUTO_CTOR(OLMap)
        initFrom: first to: last
    OL_END_AUTO_CTOR;
}

+ (id) mapWithCompare: (OLStreamableFunctor<OLBoolBinaryFunction>*)comp
{
    OL_BEGIN_AUTO_CTOR(OLMap)
        initWithCompare: comp
    OL_END_AUTO_CTOR;
}

+ (id) mapWithMap: (OLMap*)right
{
    OL_BEGIN_AUTO_CTOR(OLMap)
        initWithMap: right
    OL_END_AUTO_CTOR;
}

- (id) init
{
    [super init];
    tree = [[OLTreeMap alloc] init];
    return self;
}

- (id) initFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    [self init];
    [self insertFrom: first to: last];
    return self;
}

- (id) initFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last compare: (OLStreamableFunctor<OLBoolBinaryFunction>*)comp
{
    [self initWithCompare: comp];
    [self insertFrom: first to: last];
    return self;
}

#if !defined(OL_NO_OPENSTEP)
- (id) initWithCoder: (NSCoder*)decoder
{
    OLStreamableFunctor<OLBoolBinaryFunction>* comp;

    if ([decoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [decoder allowsKeyedCoding])
    {
        comp = [decoder decodeObjectForKey: COMPARATOR_KEY];
    }
    else
    {
        comp = [decoder decodeObject];
    }
    [self initWithCompare: comp];
    readContainerWithInsertRange(self, decoder, @selector(decodeObject));
    return self;
}
#endif

- (id) initWithCompare: (OLStreamableFunctor<OLBoolBinaryFunction>*)comp
{
    [super init];
    tree = [[OLTreeMap alloc] initWithCompare: comp];
    return self;
}

- (id) initWithMap: (OLMap*)right
{
    [super init];
    tree = [[OLTreeMap alloc] initWithTree: right->tree];
    return self;
}

- (id) initWithObjectInStream: (OLObjectInStream*)stream
{
    OLStreamableFunctor<OLBoolBinaryFunction>* comp = [stream readObject];
    [self initWithCompare: comp];
    OBJ_FREE_AUTO(comp);
    readContainerWithInsertRange(self, stream, @selector(readObject));
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    OBJ_RELEASE(tree);
    SUPER_FREE;
}

- (void) assignKey: (id)key value: (id)value
{
    [tree assignUniqueKey: key value: value];
}

- (OLAssociativeIterator*) begin
{
    return [tree begin];
}

- (void) clear
{
    [tree clear];
}

- (int) compare: (id)other
{
    if (IS_KIND_OF(other, OLMap))
        return [tree compare: ((OLMap*)other)->tree];
    return -1;
}

#if defined(OL_NO_OPENSTEP)

- (id) copy
{
    return [[OLMap alloc] initWithMap: self];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [[OLMap allocWithZone: zone] initWithMap: self];
}

#endif

- (unsigned) count: (id)key
{
    return [tree count: key];
}

- (BOOL) empty
{
    return [tree empty];
}

#if !defined(OL_NO_OPENSTEP)
- (void) encodeWithCoder: (NSCoder*)encoder
{
    if ([encoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [encoder allowsKeyedCoding])
    {
        [encoder encodeObject: [tree keyComp] forKey: COMPARATOR_KEY];
    }
    else
    {
        [encoder encodeObject: [tree keyComp]];
    }
    writeContainer(tree, @selector(beginImpl), @selector(endImpl),
        encoder, @selector(encodeObject:));
}
#endif

- (OLAssociativeIterator*) end
{
    return [tree end];
}

- (OLPair*) equalRange: (id)key
{
    return [tree equalRange: key];
}

- (void) erase: (OLAssociativeIterator*)where
{
    [tree erase: where];
}

- (void) eraseFrom: (OLAssociativeIterator*)first to: (OLAssociativeIterator*)last
{
    [tree eraseFrom: first to: last];
}

- (unsigned) eraseKey: (id)key
{
    return [tree eraseKey: key];
}

- (OLAssociativeIterator*) find: (id)key
{
    return [tree find: key];
}

- (id) insert: (OLPair*)keyValue
{
    return [tree insertUnique: keyValue];
}

- (OLAssociativeIterator*) insertAt: (OLAssociativeIterator*)where value: (OLPair*)keyVal
{
    return [tree insertUniqueAt: where value: keyVal];
}

- (void) insertFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    [tree insertUniqueFrom: first to: last];
}

- (id) insertKey: (id)key value: (id)val
{
    OLPair* p = [[OLPair alloc] initWithFirst: key second: val];
    id r = [self insert: p];

    OBJ_RELEASE(p);
    return r;
}

- (BOOL) isEqual: (id)object
{
    return IS_KIND_OF(object, OLMap) &&
           [tree isEqualUnique: ((OLMap*)object)->tree];
}

- (OLStreamableFunctor<OLBoolBinaryFunction>*) keyComp
{
    return [tree keyComp];
}

- (OLAssociativeIterator*) lowerBound: (id)key
{
    return [tree lowerBound: key];
}

- (unsigned) maxSize
{
    return [tree maxSize];
}

- (OLReverseBidiIterator*) rbegin
{
    return [tree rbegin];
}

- (OLReverseBidiIterator*) rend
{
    return [tree rend];
}

- (unsigned) size
{
    return [tree size];
}

- (void) swap: (OLMap*)right
{
    [tree swap: right->tree];
}

- (OLAssociativeIterator*) upperBound: (id)key
{
    return [tree upperBound: key];
}

- (OLStreamableFunctor<OLBoolBinaryFunction>*) valueComp
{
    return [tree keyComp];
}

- (id) valueForKey: (id)key
{
    return [tree findValue: key];
}

- (void) writeSelfToStream: (OLObjectOutStream*)stream
{
    [stream writeObject: [tree keyComp]];
    writeContainer(tree, @selector(beginImpl), @selector(endImpl),
        stream, @selector(writeObject:));
}

@end

@implementation OLMultiMap

+ (id) multiMap
{
    OL_BEGIN_AUTO_CTOR(OLMultiMap)
        init
    OL_END_AUTO_CTOR;
}

+ (id) multiMapFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    OL_BEGIN_AUTO_CTOR(OLMultiMap)
        initFrom: first to: last
    OL_END_AUTO_CTOR;
}

+ (id) multiMapWithCompare: (OLStreamableFunctor<OLBoolBinaryFunction>*)comp
{
    OL_BEGIN_AUTO_CTOR(OLMultiMap)
        initWithCompare: comp
    OL_END_AUTO_CTOR;
}

+ (id) multiMapWithMap: (OLMap*)right
{
    OL_BEGIN_AUTO_CTOR(OLMultiMap)
        initWithMap: right
    OL_END_AUTO_CTOR;
}

- (void) assignKey: (id)key value: (id)value
{
    [tree assignEqualKey: key value: value];
}

- (id) insert: (OLPair*)keyValue
{
    return [tree insertEqual: keyValue];
}

- (OLAssociativeIterator*) insertAt: (OLAssociativeIterator*)where value: (OLPair*)keyVal
{
    return [tree insertEqualAt: where value: keyVal];
}

- (void) insertFrom: (OLForwardIterator*)first to: (OLForwardIterator*)last
{
    [tree insertEqualFrom: first to: last];
}

- (BOOL) isEqual: (id)object
{
    return IS_KIND_OF(object, OLMultiMap) &&
           [tree isEqualNonUnique: ((OLMultiMap*)object)->tree];
}

- (id) valueForKey: (id)key
{
    return nil;
}

@end
