//
// $Id: BzlibOutStream.h,v 1.11 2007/03/06 20:42:19 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#if !defined(BZLIBOUTSTREAM_OL_GUARD)
#define BZLIBOUTSTREAM_OL_GUARD

#include <ObjectiveLib/LayeredOutStream.h>

/**
 * @class OLBzlibOutStream BzlibOutStream.h ObjectiveLib/BzlibOutStream.h
 *
 * A stream for bzlib compression. This is a concrete layered stream that compresses all
 * data written using the library bzlib. The bzlib library is very similar to the zlib
 * compression library, except that it compresses much better and usually more slowly
 * than zlib.
 *
 * @note Data written through OLBzlibOutStream is suitable for reading using the
 * command-line tool @c bzip2.
 *
 * @sa OLBzlibInStream, OLZlibOutStream
 *
 * @ingroup Streams
 */
@interface OLBzlibOutStream : OLLayeredOutStream
{
@protected
    /**
     * The opaque data type used by bzlib to represent the stream
     */
    void*               bzstream;

    /**
     * The buffer required by bzlib
     */
    uint8_t*            buffer;

    /**
     * The size of the zlib buffer
     */
    unsigned            bufferSize;
}

/**
 * Create and return a new compression stream. The stream uses 
 * a block size of 9 and a buffer size of 1,024. The given out stream
 * @a underStream is the next lower stream in the stream stack.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @exception OLInputOutputException if the stream cannot be created for compression
 * @param underStream the next lower stream in the stack
 * @return a reference to the newly created stream
 */
+ (id) streamWithOutStream: (OLOutStream*)underStream;

/**
 * Create and return a new compression stream. The stream uses
 * the given block size and a buffer size of 1,024. The given out stream
 * @a underStream is the next lower stream in the stream stack.
 *
 * @pre The indicated block must be in the range [1,9].
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @if OL_NO_OPENSTEP
 * @exception OLInvalidArgumentException if the block size is not in the range [1,9]
 * @else
 * @exception NSInvalidArgumentException if the block size is not in the range [1,9]
 * @endif
 * @exception OLInputOutputException if the stream cannot be created for compression
 * @param underStream the next lower stream in the stack
 * @param block the block size which must be in the range [1,9]
 * @return a reference to the newly created stream
 */
+ (id) streamWithOutStream: (OLOutStream*)underStream blockSize: (int)block;

/**
 * @name Initializers and Deallocators
 */
/* @{ */
/**
 * Initialize the compression stream. The stream uses a block size of 9 and
 * a buffer size of 1,024. The given out stream @a underStream is the next
 * lower stream in the stream stack.
 *
 * @exception OLInputOutputException if the stream cannot be created for compression
 * @param underStream the next lower stream in the stack
 * @return a reference to this stream
 */
- (id) initWithOutStream: (OLOutStream*)underStream;

/**
 * Initialize the compression stream. The stream uses the given block size and
 * a buffer size of 1,024. The given out stream @a underStream is the next
 * lower stream in the stream stack.
 *
 * @pre The indicated block must be in the range [1,9].
 *
 * @exception OLInputOutputException if the stream cannot be created for compression
 * @param underStream the next lower stream in the stack
 * @param block the block size which must be in the range [1,9]
 * @return a reference to this stream
 */
- (id) initWithOutStream: (OLOutStream*)underStream blockSize: (int)block;

/**
 * Initialize the compression stream. The stream uses the given block size
 * and buffer size. The given out stream @a underStream is the next lower
 * stream in the stream stack.
 *
 * @pre The indicated block must be in the range [1,9].
 *
 * @exception OLInputOutputException if the stream cannot be created for compression
 * @param underStream the next lower stream in the stack
 * @param block the block size which must be in the range [1,9]
 * @param size the buffer size
 * @return a reference to this stream
 */
- (id) initWithOutStream: (OLOutStream*)underStream blockSize: (int)block bufferSize: (unsigned)size;
#if !defined(OL_NO_OPENSTEP)
- (void) dealloc;
#endif
/* @} */

- (void) close;
#if defined(OL_NO_OPENSTEP)
- (void) freeStreamResources;
#endif

/**
 * Write a sequence of bytes to the stream. Compress up to @a num bytes and return the
 * number actually written.
 *
 * @exception OLInputOutputException if there is an error writing to the stream
 * @param bytes the bytes to write
 * @param num the number of bytes to write
 * @return the number of bytes written
 */
- (unsigned) writeBytes: (const uint8_t*)bytes count: (unsigned)num;

@end

#endif
