//
// nono
// Copyright (C) 2022 nono project
// Licensed under nono-license.txt
//

//
// ローダ
//

#pragma once

#include "object.h"

class LoadInfo;
class MainRAMDevice;

// RAM から呼ばれてファイルを読み込むだけのヘルパー。
class BootLoader : public Object
{
	using inherited = Object;
 public:
	explicit BootLoader(MainRAMDevice *mainram_);
	~BootLoader() override;

	// info で指定された実行ファイルを mainram にロードする。
	// data, size が指定されていればバッファから読み込む。
	// 指定されていなければホストの path から読み込む。
	// (path はいずれにしてもエラー出力の際に使う)
	static bool LoadExec(MainRAMDevice *mainram_, LoadInfo *info);

	// info で指定されたデータファイルを mainram にロードする。
	// data, size が指定されていればバッファから読み込む。
	// 指定されていなければホストの path から読み込む。
	// (path はいずれにしてもエラー出力の際に使う)
	static bool LoadData(MainRAMDevice *mainram_, LoadInfo *info);

	bool LoadExec(LoadInfo *);
	bool LoadData(LoadInfo *);

 private:
	bool LoadExecFromFile(LoadInfo *);
	bool LoadExecFromBuf(LoadInfo *);
	bool LoadGzFile(LoadInfo *);
	bool Load_aout(LoadInfo *);
	bool Load_elf32(LoadInfo *);
	bool Load_elf32_exec(LoadInfo *);
	bool Load_elf32_rel(LoadInfo *);
	bool Load_elf32_sym(LoadInfo *, std::string&);

	MainRAMDevice *mainram {};
};

// 実行ファイルを読み込む際の受け渡しパラメータ
class LoadInfo {
 public:
	LoadInfo();
	explicit LoadInfo(const char *path_);
	LoadInfo(const char *path_, const uint8 *data_, size_t size_);

	// ファイルから読み込む時はファイルパスを指定する。
	// バッファから読み込む時はエラーの時に表示される名前を指定する。
	const char *path {};

	// バッファから読み込む際は、データ本体とそのサイズを指定する。
	// ファイルから読み込む時は 0 を指定しておくこと (ただし内部で使用する)。
	const uint8 *data {};
	size_t size {};

	// シンボルテーブルを読み込まない場合は false をセットする。
	// デフォルトは true。
	bool loadsym {};

	// RAM にロードした先頭アドレスを返す。
	uint32 start {};

	// RAM にロードしたファイルのエントリポイントを返す。
	uint32 entry {};

	// RAM にロードしたカーネル本体の後のシンボル開始アドレスを返す。
	uint32 sym {};

	// RAM にロードした最終アドレス(の次のアドレス)を返す。
	uint32 end {};
};
