//
// nono
// Copyright (C) 2024 nono project
// Licensed under nono-license.txt
//

//
// SSG (YM2149)
//

#pragma once

#include "device.h"

// レジスタ
struct SSG
{
	//      7   6   5   4   3   2   1   0
	//    +---+---+---+---+---+---+---+---+
	// R0 |              Low              |
	//    +---------------+---------------+ Frequency of channel A
	// R1 |       -       |      High     |
	//    +---+---+---+---+---+---+---+---+
	//
	//    +---+---+---+---+---+---+---+---+
	// R2 |              Low              |
	//    +---------------+---------------+ Frequency of channel B
	// R3 |       -       |      High     |
	//    +---+---+---+---+---+---+---+---+
	//
	//    +---+---+---+---+---+---+---+---+
	// R4 |              Low              |
	//    +---------------+---------------+ Frequency of channel C
	// R5 |       -       |      High     |
	//    +---+---+---+---+---+---+---+---+
	//
	//    +-----------+---+---+---+---+---+
	// R6 |     -     |       freq        | Frequency of noise
	//    +-----------+---+---+---+---+---+
	//
	//    +---+---+---+---+---+---+---+---+
	// R7 |  I/O  |   Noise   |   Tone    | I/O port and mixer settings
	//    | B | A | C | B | A | C | B | A |
	//    +---+---+---+---+---+---+---+---+
	//
	//    +-----------+---+---+---+---+---+
	// R8 |     -     | M |     level     | Level of channel A
	//    +-----------+---+---+---+---+---+
	//    +-----------+---+---+---+---+---+
	// R9 |     -     | M |     level     | Level of channel B
	//    +-----------+---+---+---+---+---+
	//    +-----------+---+---+---+---+---+
	// RA |     -     | M |     level     | Level of channel C
	//    +-----------+---+---+---+---+---+
	//
	//    +-------------------------------+
	// RB |           freq (Low)          | Frequency of envelope
	// RC |           freq (High)         |
	//    +-------------------------------+
	//
	//    +---------------+---+---+---+---+
	// RD |       -       |CNT|ATT|ALT|HLD| Shape of envelope
	//    +---------------+---+---+---+---+
	//
	//    +-------------------------------+
	// RE |           8 bit data          | Data of I/O port A
	//    +-------------------------------+
	// RF |           8 bit data          | Data of I/O port B
	//    +-------------------------------+

	uint freq[3];
	uint noise_freq;
	uint settings;
	bool ampmode[3];
	uint amplevel[3];
	uint envelope_freq;
	uint envelope_shape;
	uint dataA;
	uint dataB;

	// Rn の値を返す
	uint32 Get(uint n) const;

	// レベルレジスタの値を返す
	uint32 GetLevelReg(int n) const {
		return (ampmode[n] ? 0x10 : 0) | amplevel[n];
	}
};

class SSGDevice : public IODevice
{
	using inherited = IODevice;

	// マスタークロック (1.536MHz)
	static const uint fMaster = 1536000;

 public:
	SSGDevice();
	~SSGDevice() override;

	void ResetHard(bool poweron) override;

	busdata Read1(uint32 addr) override;
	busdata Write1(uint32 addr, uint32 data) override;
	busdata Peek1(uint32 addr) override;

 private:
	DECLARE_MONITOR_SCREEN(MonitorScreen);

	uint32 GetReg(uint32 n) const;

	uint32 regno {};
	SSG reg {};

	Monitor *monitor {};
};

inline SSGDevice *GetSSGDevice() {
	return Object::GetObject<SSGDevice>(OBJ_SSG);
}
