//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

#include "m680x0disasm.h"
#include "debugger_memory.h"
#include "mystring.h"
#include <vector>

struct {
	std::vector<uint16> inst;
	std::string expected;
} table[] = {
	{ { 0x0000, 0x89ab },				"ori.b   #$ab,d0" },
	{ { 0x003c, 0x89ab },				"ori.b   #$ab,ccr" },
	{ { 0x0040, 0x89ab },				"ori.w   #$89ab,d0" },
	{ { 0x007c, 0x89ab },				"ori.w   #$89ab,sr" },
	{ { 0x0080, 0x89ab, 0xcdef },		"ori.l   #$89abcdef,d0" },
	{ { 0x00d0, 0x9000 },				"cmp2.b  (a0),a1" },
	{ { 0x00f8, 0x9000, 0x89ab },		"cmp2.b  $89ab,a1" },
	{ { 0x00d0, 0x9800 },				"chk2.b  (a0),a1" },
	{ { 0x00f8, 0x9800, 0x89ab },		"chk2.b  $89ab,a1" },
	{ { 0x0300 },						"btst.l  d1,d0" },
	{ { 0x0310 },						"btst.b  d1,(a0)" },
	{ { 0x030a, 0x89ab },				"movep.w ($89ab,a2),d1" },
	{ { 0x0340 },						"bchg.l  d1,d0" },
	{ { 0x0350 },						"bchg.b  d1,(a0)" },
	{ { 0x034a, 0x89ab },				"movep.l ($89ab,a2),d1" },
	{ { 0x0380 },						"bclr.l  d1,d0" },
	{ { 0x0390 },						"bclr.b  d1,(a0)" },
	{ { 0x038a, 0x89ab },				"movep.w d1,($89ab,a2)" },
	{ { 0x03c0 },						"bset.l  d1,d0" },
	{ { 0x03d0 },						"bset.b  d1,(a0)" },
	{ { 0x03ca, 0x89ab },				"movep.l d1,($89ab,a2)" },
	{ { 0x0200, 0x89ab },				"andi.b  #$ab,d0" },
	{ { 0x023c, 0x89ab },				"andi.b  #$ab,ccr" },
	{ { 0x0240, 0x89ab },				"andi.w  #$89ab,d0" },
	{ { 0x027c, 0x89ab },				"andi.w  #$89ab,sr" },
	{ { 0x0280, 0x89ab, 0xcdef },		"andi.l  #$89abcdef,d0" },
	{ { 0x02d0, 0x9000 },				"cmp2.w  (a0),a1" },
	{ { 0x02f8, 0x9000, 0x89ab },		"cmp2.w  $89ab,a1" },
	{ { 0x02d0, 0x9800 },				"chk2.w  (a0),a1" },
	{ { 0x02f8, 0x9800, 0x89ab },		"chk2.w  $89ab,a1" },
	{ { 0x0400, 0x89ab },				"subi.b  #$ab,d0" },
	{ { 0x0440, 0x89ab },				"subi.w  #$89ab,d0" },
	{ { 0x0480, 0x89ab, 0xcdef },		"subi.l  #$89abcdef,d0" },
	{ { 0x04d0, 0x9000 },				"cmp2.l  (a0),a1" },
	{ { 0x04f8, 0x9000, 0x89ab },		"cmp2.l  $89ab,a1" },
	{ { 0x04d0, 0x9800 },				"chk2.l  (a0),a1" },
	{ { 0x04f8, 0x9800, 0x89ab },		"chk2.l  $89ab,a1" },
	{ { 0x0600, 0x89ab },				"addi.b  #$ab,d0" },
	{ { 0x0640, 0x89ab },				"addi.w  #$89ab,d0" },
	{ { 0x0680, 0x89ab, 0xcdef },		"addi.l  #$89abcdef,d0" },
	{ { 0x0800, 0x00ff },				"btst.l  #31,d0" },
	{ { 0x0838, 0x00ff, 0x89ab },		"btst.b  #7,$89ab" },
	{ { 0x0840, 0x00ff },				"bchg.l  #31,d0" },
	{ { 0x0878, 0x00ff, 0x89ab },		"bchg.b  #7,$89ab" },
	{ { 0x0880, 0x00ff },				"bclr.l  #31,d0" },
	{ { 0x08b8, 0x00ff, 0x89ab },		"bclr.b  #7,$89ab" },
	{ { 0x08c0, 0x00ff },				"bset.l  #31,d0" },
	{ { 0x08f8, 0x00ff, 0x89ab },		"bset.b  #7,$89ab" },
	{ { 0x0a00, 0x89ab },				"eori.b  #$ab,d0" },
	{ { 0x0a3c, 0x89ab },				"eori.b  #$ab,ccr" },
	{ { 0x0a40, 0x89ab },				"eori.w  #$89ab,d0" },
	{ { 0x0a7c, 0x89ab },				"eori.w  #$89ab,sr" },
	{ { 0x0a80, 0x89ab, 0xcdef },		"eori.l  #$89abcdef,d0" },
	{ { 0x0ad0, 0x0081 },				"cas.b   d1,d2,(a0)" },
	{ { 0x0af8, 0x0081, 0x89ab },		"cas.b   d1,d2,$89ab" },
	{ { 0x0c00, 0x89ab },				"cmpi.b  #$ab,d0" },
	{ { 0x0c40, 0x89ab },				"cmpi.w  #$89ab,d0" },
	{ { 0x0c80, 0x89ab, 0xcdef },		"cmpi.l  #$89abcdef,d0" },
	{ { 0x0cd0, 0x0081 },				"cas.w   d1,d2,(a0)" },
	{ { 0x0cf8, 0x0081, 0x89ab },		"cas.w   d1,d2,$89ab" },
	{ { 0x0cfc, 0xb081, 0xe144 },		"cas2.w  d1:d4,d2:d5,(a3):(a6)" },
	{ { 0x0e38, 0x9000, 0x89ab },		"moves.b $89ab,a1" },
	{ { 0x0e38, 0x9800, 0x89ab },		"moves.b a1,$89ab" },
	{ { 0x0e78, 0x9000, 0x89ab },		"moves.w $89ab,a1" },
	{ { 0x0e78, 0x9800, 0x89ab },		"moves.w a1,$89ab" },
	{ { 0x0eb8, 0x9000, 0x89ab },		"moves.l $89ab,a1" },
	{ { 0x0eb8, 0x9800, 0x89ab },		"moves.l a1,$89ab" },
	{ { 0x0ed0, 0x0081 },				"cas.l   d1,d2,(a0)" },
	{ { 0x0ef8, 0x0081, 0x89ab },		"cas.l   d1,d2,$89ab" },
	{ { 0x0efc, 0xb081, 0xe144 },		"cas2.l  d1:d4,d2:d5,(a3):(a6)" },

	{ { 0x1200 },						"move.b  d0,d1" },
	{ { 0x1280 },						"move.b  d0,(a1)" },
	{ { 0x12c0 },						"move.b  d0,(a1)+" },
	{ { 0x1300 },						"move.b  d0,-(a1)" },
	{ { 0x1340, 0x89ab },				"move.b  d0,-$7655(a1)" },
	{ { 0x1380, 0x20fe },				"move.b  d0,-$02(a1,d2.w)" },
	{ { 0x11c0, 0x89ab },				"move.b  d0,$89ab" },
	{ { 0x13c0, 0x89ab, 0xcdef },		"move.b  d0,$89abcdef" },
	{ { 0x2200 },						"move.l  d0,d1" },
	{ { 0x2240 },						"movea.l d0,a1" },
	{ { 0x2280 },						"move.l  d0,(a1)" },
	{ { 0x22c0 },						"move.l  d0,(a1)+" },
	{ { 0x2300 },						"move.l  d0,-(a1)" },
	{ { 0x2340, 0x89ab },				"move.l  d0,-$7655(a1)" },
	{ { 0x2380, 0x20fe },				"move.l  d0,-$02(a1,d2.w)" },
	{ { 0x21c0, 0x89ab },				"move.l  d0,$89ab" },
	{ { 0x23c0, 0x89ab, 0xcdef },		"move.l  d0,$89abcdef" },
	{ { 0x3200 },						"move.w  d0,d1" },
	{ { 0x3240 },						"movea.w d0,a1" },
	// movea.w $ffff89ab,a1 のように表示したい気もするけど、今の所
	// 構造上面倒なのでちょっとおいておく。
	{ { 0x327c, 0x89ab },				"movea.w #$89ab,a1" },
	{ { 0x3280 },						"move.w  d0,(a1)" },
	{ { 0x32c0 },						"move.w  d0,(a1)+" },
	{ { 0x3300 },						"move.w  d0,-(a1)" },
	{ { 0x3340, 0x89ab },				"move.w  d0,-$7655(a1)" },
	{ { 0x3380, 0x20fe },				"move.w  d0,-$02(a1,d2.w)" },
	{ { 0x31c0, 0x89ab },				"move.w  d0,$89ab" },
	{ { 0x33c0, 0x89ab, 0xcdef },		"move.w  d0,$89abcdef" },
	{ { 0x1f7c, 0x0012, 0x89ab },		"move.b  #$12,-$7655(a7)" },

	{ { 0x4000 },						"negx.b  d0" },
	{ { 0x4040 },						"negx.w  d0" },
	{ { 0x4080 },						"negx.l  d0" },
	{ { 0x40c0 },						"move.w  sr,d0" },
	{ { 0x4300 },						"chk.l   d0,d1" },
	{ { 0x4380 },						"chk.w   d0,d1" },
	{ { 0x43d0 },						"lea.l   (a0),a1" },
	{ { 0x49c0 },						"extb.l  d0" },
	{ { 0x4200 },						"clr.b   d0" },
	{ { 0x4240 },						"clr.w   d0" },
	{ { 0x4280 },						"clr.l   d0" },
	{ { 0x42c0 },						"move.w  ccr,d0" },
	{ { 0x4400 },						"neg.b   d0" },
	{ { 0x4440 },						"neg.w   d0" },
	{ { 0x4480 },						"neg.l   d0" },
	{ { 0x44c0 },						"move.w  d0,ccr" },
	{ { 0x4600 },						"not.b   d0" },
	{ { 0x4640 },						"not.w   d0" },
	{ { 0x4680 },						"not.l   d0" },
	{ { 0x46c0 },						"move.w  d0,sr" },
	{ { 0x4800 },						"nbcd.b  d0" },
	{ { 0x4808, 0x89ab, 0xcdef },		"link.l  a0,#-$76543211" },
	{ { 0x4840 },						"swap.w  d0" },
	{ { 0x4849 },						"bkpt    #1" },
	{ { 0x4850 },						"pea.l   (a0)" },
	{ { 0x4880 },						"ext.w   d0" },
	{ { 0x4890, 0x0101 },				"movem.w d0/a0,(a0)" },
	{ { 0x48b8, 0x0101, 0x89ab },		"movem.w d0/a0,$89ab" },
	{ { 0x48c0 },						"ext.l   d0" },
	{ { 0x48d0, 0x0101 },				"movem.l d0/a0,(a0)" },
	{ { 0x48f8, 0x0101, 0x89ab },		"movem.l d0/a0,$89ab" },
	{ { 0x4a00 },						"tst.b   d0" },
	{ { 0x4a40 },						"tst.w   d0" },
	{ { 0x4a80 },						"tst.l   d0" },
	{ { 0x4ac0 },						"tas.b   d0" },
	{ { 0x4afc },						"illegal" },
	// ※ mul[us].l <ea>,dh:dl (h==l) はニーモニックとしては正しい
	{ { 0x4c38, 0x1000, 0x89ab },		"mulu.l  $89ab,d1" },
	{ { 0x4c38, 0x1401, 0x89ab },		"mulu.l  $89ab,d1:d1" }, // ※
	{ { 0x4c38, 0x1402, 0x89ab },		"mulu.l  $89ab,d2:d1" },
	{ { 0x4c38, 0x1800, 0x89ab },		"muls.l  $89ab,d1" },
	{ { 0x4c38, 0x1c01, 0x89ab },		"muls.l  $89ab,d1:d1" }, // ※
	{ { 0x4c38, 0x1c02, 0x89ab },		"muls.l  $89ab,d2:d1" },
	{ { 0x4c78, 0x1001, 0x89ab },		"divu.l  $89ab,d1" },
	{ { 0x4c78, 0x1002, 0x89ab },		"divu.l  $89ab,d2:d1" },
	{ { 0x4c78, 0x1402, 0x89ab },		"divul.l $89ab,d2:d1" },
	{ { 0x4c78, 0x1801, 0x89ab },		"divs.l  $89ab,d1" },
	{ { 0x4c78, 0x1802, 0x89ab },		"divs.l  $89ab,d2:d1" },
	{ { 0x4c78, 0x1c02, 0x89ab },		"divsl.l $89ab,d2:d1" },
	{ { 0x4c90, 0x0101 },				"movem.w (a0),d0/a0" },
	{ { 0x4cb8, 0x0101, 0x89ab },		"movem.w $89ab,d0/a0" },
	{ { 0x4cd0, 0x0101 },				"movem.l (a0),d0/a0" },
	{ { 0x4cf8, 0x0101, 0x89ab },		"movem.l $89ab,d0/a0" },
	{ { 0x4e4f },						"trap    #15" },
	{ { 0x4e50, 0x89ab },				"link.w  a0,#-$7655" },
	{ { 0x4e58 },						"unlk    a0" },
	{ { 0x4e60 },						"move.l  a0,usp" },
	{ { 0x4e68 },						"move.l  usp,a0" },
	{ { 0x4e70 },						"reset" },
	{ { 0x4e71 },						"nop" },
	{ { 0x4e72, 0x89ab },				"stop    #$89ab" },
	{ { 0x4e73 },						"rte" },
	{ { 0x4e74, 0x89ab },				"rtd     #$89ab" },
	{ { 0x4e75 },						"rts" },
	{ { 0x4e76 },						"trapv" },
	{ { 0x4e77 },						"rtr" },
	{ { 0x4e7a, 0x8801 },				"movec.l vbr,a0" },
	{ { 0x4e7b, 0x8801 },				"movec.l a0,vbr" },
	{ { 0x4e90 },						"jsr     (a0)" },
	{ { 0x4ed0 },						"jmp     (a0)" },

	{ { 0x5000 },						"addq.b  #8,d0" },
	{ { 0x5040 },						"addq.w  #8,d0" },
	{ { 0x5048 },						"addq.w  #8,a0" },	// .l にする?
	{ { 0x5080 },						"addq.l  #8,d0" },
	{ { 0x5088 },						"addq.l  #8,a0" },
	{ { 0x50c0 },						"st.b    d0" },
	{ { 0x50c8, 0xfffe },				"dbt.w   d0,$00000100" },
	{ { 0x51c8, 0xfffe },				"dbra.w  d0,$00000100" },
	{ { 0x50fa, 0x89ab },				"trapt.w #$89ab" },
	{ { 0x50fb, 0x89ab, 0xcdef },		"trapt.l #$89abcdef" },
	{ { 0x50fc },						"trapt" },
	{ { 0x5100 },						"subq.b  #8,d0" },
	{ { 0x5140 },						"subq.w  #8,d0" },
	{ { 0x5148 },						"subq.w  #8,a0" },	// .l にする?
	{ { 0x5180 },						"subq.l  #8,d0" },
	{ { 0x5188 },						"subq.l  #8,a0" },

	{ { 0x6000, 0xfffe },				"bra.w   $00000100" },
	{ { 0x60fe },						"bra.b   $00000100" },
	{ { 0x60ff, 0xffff, 0xfffe },		"bra.l   $00000100" },
	{ { 0x6100, 0xfffe },				"bsr.w   $00000100" },
	{ { 0x61fe },						"bsr.b   $00000100" },
	{ { 0x61ff, 0xffff, 0xfffe },		"bsr.l   $00000100" },
	{ { 0x6200, 0xfffe },				"bhi.w   $00000100" },
	{ { 0x62fe },						"bhi.b   $00000100" },
	{ { 0x62ff, 0xffff, 0xfffe },		"bhi.l   $00000100" },

	{ { 0x727f },						"moveq.l #$7f,d1" },
	{ { 0x7280 },						"moveq.l #$ffffff80,d1" },

	{ { 0x8200 },						"or.b    d0,d1" },
	{ { 0x8240 },						"or.w    d0,d1" },
	{ { 0x8280 },						"or.l    d0,d1" },
	{ { 0x82c0 },						"divu.w  d0,d1" },
	{ { 0x8300 },						"sbcd.b  d0,d1" },
	{ { 0x8308 },						"sbcd.b  -(a0),-(a1)" },
	{ { 0x8310 },						"or.b    d1,(a0)" },
	{ { 0x8340, 0x89ab },				"pack    d0,d1,#$89ab" },
	{ { 0x8348, 0x89ab },				"pack    -(a0),-(a1),#$89ab" },
	{ { 0x8350 },						"or.w    d1,(a0)" },
	{ { 0x8380, 0x89ab },				"unpk    d0,d1,#$89ab" },
	{ { 0x8388, 0x89ab },				"unpk    -(a0),-(a1),#$89ab" },
	{ { 0x8390 },						"or.l    d1,(a0)" },
	{ { 0x83c0 },						"divs.w  d0,d1" },

	{ { 0x9200 },						"sub.b   d0,d1" },
	{ { 0x9240 },						"sub.w   d0,d1" },
	{ { 0x9280 },						"sub.l   d0,d1" },
	{ { 0x92c0 },						"suba.w  d0,a1" },
	{ { 0x9300 },						"subx.b  d0,d1" },
	{ { 0x9308 },						"subx.b  -(a0),-(a1)" },
	{ { 0x9310 },						"sub.b   d1,(a0)" },
	{ { 0x9340 },						"subx.w  d0,d1" },
	{ { 0x9348 },						"subx.w  -(a0),-(a1)" },
	{ { 0x9350 },						"sub.w   d1,(a0)" },
	{ { 0x9380 },						"subx.l  d0,d1" },
	{ { 0x9388 },						"subx.l  -(a0),-(a1)" },
	{ { 0x9390 },						"sub.l   d1,(a0)" },
	{ { 0x93c0 },						"suba.l  d0,a1" },

	// A-Line

	{ { 0xb200 },						"cmp.b   d0,d1" },
	{ { 0xb240 },						"cmp.w   d0,d1" },
	{ { 0xb280 },						"cmp.l   d0,d1" },
	{ { 0xb2c0 },						"cmpa.w  d0,a1" },
	{ { 0xb300 },						"eor.b   d1,d0" },
	{ { 0xb308 },						"cmpm.b  (a0)+,(a1)+" },
	{ { 0xb340 },						"eor.w   d1,d0" },
	{ { 0xb348 },						"cmpm.w  (a0)+,(a1)+" },
	{ { 0xb380 },						"eor.l   d1,d0" },
	{ { 0xb388 },						"cmpm.l  (a0)+,(a1)+" },
	{ { 0xb3c0 },						"cmpa.l  d0,a1" },

	{ { 0xc200 },						"and.b   d0,d1" },
	{ { 0xc240 },						"and.w   d0,d1" },
	{ { 0xc280 },						"and.l   d0,d1" },
	{ { 0xc2c0 },						"mulu.w  d0,d1" },
	{ { 0xc300 },						"abcd.b  d0,d1" },
	{ { 0xc308 },						"abcd.b  -(a0),-(a1)" },
	{ { 0xc310 },						"and.b   d1,(a0)" },
	{ { 0xc340 },						"exg.l   d1,d0" },
	{ { 0xc348 },						"exg.l   a1,a0" },
	{ { 0xc350 },						"and.w   d1,(a0)" },
	{ { 0xc388 },						"exg.l   d1,a0" },
	{ { 0xc390 },						"and.l   d1,(a0)" },
	{ { 0xc3c0 },						"muls.w  d0,d1" },

	{ { 0xd200 },						"add.b   d0,d1" },
	{ { 0xd240 },						"add.w   d0,d1" },
	{ { 0xd280 },						"add.l   d0,d1" },
	{ { 0xd2c0 },						"adda.w  d0,a1" },
	{ { 0xd300 },						"addx.b  d0,d1" },
	{ { 0xd308 },						"addx.b  -(a0),-(a1)" },
	{ { 0xd310 },						"add.b   d1,(a0)" },
	{ { 0xd340 },						"addx.w  d0,d1" },
	{ { 0xd348 },						"addx.w  -(a0),-(a1)" },
	{ { 0xd350 },						"add.w   d1,(a0)" },
	{ { 0xd380 },						"addx.l  d0,d1" },
	{ { 0xd388 },						"addx.l  -(a0),-(a1)" },
	{ { 0xd390 },						"add.l   d1,(a0)" },
	{ { 0xd3c0 },						"adda.l  d0,a1" },

	{ { 0xe000 },						"asr.b   #8,d0" },
	{ { 0xe008 },						"lsr.b   #8,d0" },
	{ { 0xe010 },						"roxr.b  #8,d0" },
	{ { 0xe018 },						"ror.b   #8,d0" },
	{ { 0xe220 },						"asr.b   d1,d0" },
	{ { 0xe228 },						"lsr.b   d1,d0" },
	{ { 0xe230 },						"roxr.b  d1,d0" },
	{ { 0xe238 },						"ror.b   d1,d0" },

	{ { 0xe040 },						"asr.w   #8,d0" },
	{ { 0xe048 },						"lsr.w   #8,d0" },
	{ { 0xe050 },						"roxr.w  #8,d0" },
	{ { 0xe058 },						"ror.w   #8,d0" },
	{ { 0xe260 },						"asr.w   d1,d0" },
	{ { 0xe268 },						"lsr.w   d1,d0" },
	{ { 0xe270 },						"roxr.w  d1,d0" },
	{ { 0xe278 },						"ror.w   d1,d0" },

	{ { 0xe080 },						"asr.l   #8,d0" },
	{ { 0xe088 },						"lsr.l   #8,d0" },
	{ { 0xe090 },						"roxr.l  #8,d0" },
	{ { 0xe098 },						"ror.l   #8,d0" },
	{ { 0xe2a0 },						"asr.l   d1,d0" },
	{ { 0xe2a8 },						"lsr.l   d1,d0" },
	{ { 0xe2b0 },						"roxr.l  d1,d0" },
	{ { 0xe2b8 },						"ror.l   d1,d0" },

	{ { 0xe100 },						"asl.b   #8,d0" },
	{ { 0xe108 },						"lsl.b   #8,d0" },
	{ { 0xe110 },						"roxl.b  #8,d0" },
	{ { 0xe118 },						"rol.b   #8,d0" },
	{ { 0xe320 },						"asl.b   d1,d0" },
	{ { 0xe328 },						"lsl.b   d1,d0" },
	{ { 0xe330 },						"roxl.b  d1,d0" },
	{ { 0xe338 },						"rol.b   d1,d0" },

	{ { 0xe140 },						"asl.w   #8,d0" },
	{ { 0xe148 },						"lsl.w   #8,d0" },
	{ { 0xe150 },						"roxl.w  #8,d0" },
	{ { 0xe158 },						"rol.w   #8,d0" },
	{ { 0xe360 },						"asl.w   d1,d0" },
	{ { 0xe368 },						"lsl.w   d1,d0" },
	{ { 0xe370 },						"roxl.w  d1,d0" },
	{ { 0xe378 },						"rol.w   d1,d0" },

	{ { 0xe180 },						"asl.l   #8,d0" },
	{ { 0xe188 },						"lsl.l   #8,d0" },
	{ { 0xe190 },						"roxl.l  #8,d0" },
	{ { 0xe198 },						"rol.l   #8,d0" },
	{ { 0xe3a0 },						"asl.l   d1,d0" },
	{ { 0xe3a8 },						"lsl.l   d1,d0" },
	{ { 0xe3b0 },						"roxl.l  d1,d0" },
	{ { 0xe3b8 },						"rol.l   d1,d0" },

	{ { 0xe0d0 },						"asr.w   (a0)" },
	{ { 0xe1d0 },						"asl.w   (a0)" },
	{ { 0xe2d0 },						"lsr.w   (a0)" },
	{ { 0xe3d0 },						"lsl.w   (a0)" },
	{ { 0xe4d0 },						"roxr.w  (a0)" },
	{ { 0xe5d0 },						"roxl.w  (a0)" },
	{ { 0xe6d0 },						"ror.w   (a0)" },
	{ { 0xe7d0 },						"rol.w   (a0)" },

	{ { 0xe8c0, 0x0862 },				"bftst   d0{d1:d2}" },
	{ { 0xe9c0, 0x3862 },				"bfextu  d0{d1:d2},d3" },
	{ { 0xeac0, 0x0862 },				"bfchg   d0{d1:d2}" },
	{ { 0xebc0, 0x3862 },				"bfexts  d0{d1:d2},d3" },
	{ { 0xecc0, 0x0862 },				"bfclr   d0{d1:d2}" },
	{ { 0xedc0, 0x3862 },				"bfffo   d0{d1:d2},d3" },
	{ { 0xeec0, 0x0862 },				"bfset   d0{d1:d2}" },
	{ { 0xefc0, 0x3862 },				"bfins   d3,d0{d1:d2}" },
	{ { 0xe8f8, 0x0862, 0x89ab },		"bftst   $89ab{d1:d2}" },
	{ { 0xe9f8, 0x3862, 0x89ab },		"bfextu  $89ab{d1:d2},d3" },
	{ { 0xeaf8, 0x0862, 0x89ab },		"bfchg   $89ab{d1:d2}" },
	{ { 0xebf8, 0x3862, 0x89ab },		"bfexts  $89ab{d1:d2},d3" },
	{ { 0xecf8, 0x0862, 0x89ab },		"bfclr   $89ab{d1:d2}" },
	{ { 0xedf8, 0x3862, 0x89ab },		"bfffo   $89ab{d1:d2},d3" },
	{ { 0xeef8, 0x0862, 0x89ab },		"bfset   $89ab{d1:d2}" },
	{ { 0xeff8, 0x3862, 0x89ab },		"bfins   d3,$89ab{d1:d2}" },
	{ { 0xe8c0, 0x0022 }, "bftst   d0{#0:d2}" },	// offset 0
	{ { 0xe8c0, 0x07e2 }, "bftst   d0{#31:d2}" },	// offset 31
	{ { 0xe8c0, 0x0841 }, "bftst   d0{d1:#1}" },	// width 1
	{ { 0xe8c0, 0x0840 }, "bftst   d0{d1:#32}" },	// width 32
	{ { 0xe8c0, 0x0000 }, "bftst   d0{#0:#32}" },	// oもwも即値

	// EA
	{ { 0x4a07 },						"tst.b   d7" },			// dn
	{ { 0x4a0f },						"tst.b   a7" },			// an
	{ { 0x4a17 },						"tst.b   (a7)" },		// (an)
	{ { 0x4a1f },						"tst.b   (a7)+" },		// (an)+
	{ { 0x4a27 },						"tst.b   -(a7)" },		// -(an)
	{ { 0x4a2f, 0x89ab },				"tst.b   -$7655(a7)" },	// d16(an)
	{ { 0x4a38, 0x89ab },				"tst.b   $89ab" },		// abs.w
	{ { 0x4a39, 0x89ab, 0xcdef },		"tst.b   $89abcdef" },	// abs.l
	{ { 0x4a3a, 0xfffe },				"tst.b   $00000100(pc)" }, // d16(pc)
	{ { 0x4a3c, 0x89ab },				"tst.b   #$ab" },		// #imm
	// (an,ix) 短縮フォーマット
	{ { 0x4a37, 0x9000 },				"tst.b   (a7,a1.w)" },
	{ { 0x4a37, 0x9201 },				"tst.b   $01(a7,a1.w*2)" },
	{ { 0x4a37, 0x947f },				"tst.b   $7f(a7,a1.w*4)" },
	{ { 0x4a37, 0x9680 },				"tst.b   -$80(a7,a1.w*8)" },
	{ { 0x4a37, 0x98ff },				"tst.b   -$01(a7,a1.l)" },
	{ { 0x4a37, 0x9a00 },				"tst.b   (a7,a1.l*2)" },
	{ { 0x4a37, 0x9c00 },				"tst.b   (a7,a1.l*4)" },
	{ { 0x4a37, 0x9e00 },				"tst.b   (a7,a1.l*8)" },
	// (an,ix) フルフォーマット
#define BW	0x89ab
#define BL	0x89ab, 0xcdef
#define OW  0xcdef
#define OL  0xcdef, 0x0123
#define MA(x, ...)	{ 0x4a37, x, __VA_ARGS__ }
#define T			"tst.b   "
	// an,bs=0, bdsize=1
	{ MA(0x1110),			T "(a7,d1.w)" },
	{ MA(0x1111),			T "([a7,d1.w])" },
	{ MA(0x1112, OW),		T "([a7,d1.w],-$3211)" },
	{ MA(0x1113, OL),		T "([a7,d1.w],-$3210fedd)" },
	{ MA(0x1115),			T "([a7],d1.w)" },
	{ MA(0x1116, OW),		T "([a7],d1.w,-$3211)" },
	{ MA(0x1117, OL),		T "([a7],d1.w,-$3210fedd)" },
	{ MA(0x1150),			T "(a7)" },
	{ MA(0x1151),			T "([a7])" },
	{ MA(0x1152, OW),		T "([a7],-$3211)" },
	{ MA(0x1153, OL),		T "([a7],-$3210fedd)" },
	// an,bs=0, bdsize=2
	{ MA(0x1120, BW),		T "(-$7655,a7,d1.w)" },
	{ MA(0x1121, BW),		T "([-$7655,a7,d1.w])" },
	{ MA(0x1122, BW, OW),	T "([-$7655,a7,d1.w],-$3211)" },
	{ MA(0x1123, BW, OL),	T "([-$7655,a7,d1.w],-$3210fedd)" },
	{ MA(0x1125, BW),		T "([-$7655,a7],d1.w)" },
	{ MA(0x1126, BW, OW),	T "([-$7655,a7],d1.w,-$3211)" },
	{ MA(0x1127, BW, OL),	T "([-$7655,a7],d1.w,-$3210fedd)" },
	{ MA(0x1160, BW),		T "(-$7655,a7)" },
	{ MA(0x1161, BW),		T "([-$7655,a7])" },
	{ MA(0x1162, BW, OW),	T "([-$7655,a7],-$3211)" },
	{ MA(0x1163, BW, OL),	T "([-$7655,a7],-$3210fedd)" },
	// an,bs=0, bdsize=3
	{ MA(0x1130, BL),		T "(-$76543211,a7,d1.w)" },
	{ MA(0x1131, BL),		T "([-$76543211,a7,d1.w])" },
	{ MA(0x1132, BL, OW),	T "([-$76543211,a7,d1.w],-$3211)" },
	{ MA(0x1133, BL, OL),	T "([-$76543211,a7,d1.w],-$3210fedd)" },
	{ MA(0x1135, BL),		T "([-$76543211,a7],d1.w)" },
	{ MA(0x1136, BL, OW),	T "([-$76543211,a7],d1.w,-$3211)" },
	{ MA(0x1137, BL, OL),	T "([-$76543211,a7],d1.w,-$3210fedd)" },
	{ MA(0x1170, BL),		T "(-$76543211,a7)" },
	{ MA(0x1171, BL),		T "([-$76543211,a7])" },
	{ MA(0x1172, BL, OW),	T "([-$76543211,a7],-$3211)" },
	{ MA(0x1173, BL, OL),	T "([-$76543211,a7],-$3210fedd)" },
	// an,bs=1, bdsize=1
	{ MA(0x1190),			T "(za7,d1.w)" },
	{ MA(0x1191),			T "([za7,d1.w])" },
	{ MA(0x1192, OW),		T "([za7,d1.w],-$3211)" },
	{ MA(0x1193, OL),		T "([za7,d1.w],-$3210fedd)" },
	{ MA(0x1195),			T "([za7],d1.w)" },
	{ MA(0x1196, OW),		T "([za7],d1.w,-$3211)" },
	{ MA(0x1197, OL),		T "([za7],d1.w,-$3210fedd)" },
	{ MA(0x11d0),			T "(za7)" },
	{ MA(0x11d1),			T "([za7])" },
	{ MA(0x11d2, OW),		T "([za7],-$3211)" },
	{ MA(0x11d3, OL),		T "([za7],-$3210fedd)" },
	// an,bs=1, bdsize=2
	{ MA(0x11a0, BW),		T "(-$7655,za7,d1.w)" },
	{ MA(0x11a1, BW),		T "([-$7655,za7,d1.w])" },
	{ MA(0x11a2, BW, OW),	T "([-$7655,za7,d1.w],-$3211)" },
	{ MA(0x11a3, BW, OL),	T "([-$7655,za7,d1.w],-$3210fedd)" },
	{ MA(0x11a5, BW),		T "([-$7655,za7],d1.w)" },
	{ MA(0x11a6, BW, OW),	T "([-$7655,za7],d1.w,-$3211)" },
	{ MA(0x11a7, BW, OL),	T "([-$7655,za7],d1.w,-$3210fedd)" },
	{ MA(0x11e0, BW),		T "(-$7655,za7)" },
	{ MA(0x11e1, BW),		T "([-$7655,za7])" },
	{ MA(0x11e2, BW, OW),	T "([-$7655,za7],-$3211)" },
	{ MA(0x11e3, BW, OL),	T "([-$7655,za7],-$3210fedd)" },
	// an,bs=1, bdsize=3
	{ MA(0x11b0, BL),		T "(-$76543211,za7,d1.w)" },
	{ MA(0x11b1, BL),		T "([-$76543211,za7,d1.w])" },
	{ MA(0x11b2, BL, OW),	T "([-$76543211,za7,d1.w],-$3211)" },
	{ MA(0x11b3, BL, OL),	T "([-$76543211,za7,d1.w],-$3210fedd)" },
	{ MA(0x11b5, BL),		T "([-$76543211,za7],d1.w)" },
	{ MA(0x11b6, BL, OW),	T "([-$76543211,za7],d1.w,-$3211)" },
	{ MA(0x11b7, BL, OL),	T "([-$76543211,za7],d1.w,-$3210fedd)" },
	{ MA(0x11f0, BL),		T "(-$76543211,za7)" },
	{ MA(0x11f1, BL),		T "([-$76543211,za7])" },
	{ MA(0x11f2, BL, OW),	T "([-$76543211,za7],-$3211)" },
	{ MA(0x11f3, BL, OL),	T "([-$76543211,za7],-$3210fedd)" },
#undef MA
#undef T
#define MA(x, ...)	{ 0x4a3b, x, __VA_ARGS__ }
#define T			"tst.b   "
	// pc,bs=0, bdsize=1
	{ MA(0x1110),			T "(pc,d1.w)" },
	{ MA(0x1111),			T "([pc,d1.w])" },
	{ MA(0x1112, OW),		T "([pc,d1.w],-$3211)" },
	{ MA(0x1113, OL),		T "([pc,d1.w],-$3210fedd)" },
	{ MA(0x1115),			T "([pc],d1.w)" },
	{ MA(0x1116, OW),		T "([pc],d1.w,-$3211)" },
	{ MA(0x1117, OL),		T "([pc],d1.w,-$3210fedd)" },
	{ MA(0x1150),			T "(pc)" },
	{ MA(0x1151),			T "([pc])" },
	{ MA(0x1152, OW),		T "([pc],-$3211)" },
	{ MA(0x1153, OL),		T "([pc],-$3210fedd)" },
	// pc,bs=0, bdsize=2
	{ MA(0x1120, BW),		T "(-$7655,pc,d1.w)" },
	{ MA(0x1121, BW),		T "([-$7655,pc,d1.w])" },
	{ MA(0x1122, BW, OW),	T "([-$7655,pc,d1.w],-$3211)" },
	{ MA(0x1123, BW, OL),	T "([-$7655,pc,d1.w],-$3210fedd)" },
	{ MA(0x1125, BW),		T "([-$7655,pc],d1.w)" },
	{ MA(0x1126, BW, OW),	T "([-$7655,pc],d1.w,-$3211)" },
	{ MA(0x1127, BW, OL),	T "([-$7655,pc],d1.w,-$3210fedd)" },
	{ MA(0x1160, BW),		T "(-$7655,pc)" },
	{ MA(0x1161, BW),		T "([-$7655,pc])" },
	{ MA(0x1162, BW, OW),	T "([-$7655,pc],-$3211)" },
	{ MA(0x1163, BW, OL),	T "([-$7655,pc],-$3210fedd)" },
	// an,bs=0, bdsize=3
	{ MA(0x1130, BL),		T "(-$76543211,pc,d1.w)" },
	{ MA(0x1131, BL),		T "([-$76543211,pc,d1.w])" },
	{ MA(0x1132, BL, OW),	T "([-$76543211,pc,d1.w],-$3211)" },
	{ MA(0x1133, BL, OL),	T "([-$76543211,pc,d1.w],-$3210fedd)" },
	{ MA(0x1135, BL),		T "([-$76543211,pc],d1.w)" },
	{ MA(0x1136, BL, OW),	T "([-$76543211,pc],d1.w,-$3211)" },
	{ MA(0x1137, BL, OL),	T "([-$76543211,pc],d1.w,-$3210fedd)" },
	{ MA(0x1170, BL),		T "(-$76543211,pc)" },
	{ MA(0x1171, BL),		T "([-$76543211,pc])" },
	{ MA(0x1172, BL, OW),	T "([-$76543211,pc],-$3211)" },
	{ MA(0x1173, BL, OL),	T "([-$76543211,pc],-$3210fedd)" },
	// an,bs=1, bdsize=1
	{ MA(0x1190),			T "(zpc,d1.w)" },
	{ MA(0x1191),			T "([zpc,d1.w])" },
	{ MA(0x1192, OW),		T "([zpc,d1.w],-$3211)" },
	{ MA(0x1193, OL),		T "([zpc,d1.w],-$3210fedd)" },
	{ MA(0x1195),			T "([zpc],d1.w)" },
	{ MA(0x1196, OW),		T "([zpc],d1.w,-$3211)" },
	{ MA(0x1197, OL),		T "([zpc],d1.w,-$3210fedd)" },
	{ MA(0x11d0),			T "(zpc)" },
	{ MA(0x11d1),			T "([zpc])" },
	{ MA(0x11d2, OW),		T "([zpc],-$3211)" },
	{ MA(0x11d3, OL),		T "([zpc],-$3210fedd)" },
	// an,bs=1, bdsize=2
	{ MA(0x11a0, BW),		T "(-$7655,zpc,d1.w)" },
	{ MA(0x11a1, BW),		T "([-$7655,zpc,d1.w])" },
	{ MA(0x11a2, BW, OW),	T "([-$7655,zpc,d1.w],-$3211)" },
	{ MA(0x11a3, BW, OL),	T "([-$7655,zpc,d1.w],-$3210fedd)" },
	{ MA(0x11a5, BW),		T "([-$7655,zpc],d1.w)" },
	{ MA(0x11a6, BW, OW),	T "([-$7655,zpc],d1.w,-$3211)" },
	{ MA(0x11a7, BW, OL),	T "([-$7655,zpc],d1.w,-$3210fedd)" },
	{ MA(0x11e0, BW),		T "(-$7655,zpc)" },
	{ MA(0x11e1, BW),		T "([-$7655,zpc])" },
	{ MA(0x11e2, BW, OW),	T "([-$7655,zpc],-$3211)" },
	{ MA(0x11e3, BW, OL),	T "([-$7655,zpc],-$3210fedd)" },
	// an,bs=1, bdsize=3
	{ MA(0x11b0, BL),		T "(-$76543211,zpc,d1.w)" },
	{ MA(0x11b1, BL),		T "([-$76543211,zpc,d1.w])" },
	{ MA(0x11b2, BL, OW),	T "([-$76543211,zpc,d1.w],-$3211)" },
	{ MA(0x11b3, BL, OL),	T "([-$76543211,zpc,d1.w],-$3210fedd)" },
	{ MA(0x11b5, BL),		T "([-$76543211,zpc],d1.w)" },
	{ MA(0x11b6, BL, OW),	T "([-$76543211,zpc],d1.w,-$3211)" },
	{ MA(0x11b7, BL, OL),	T "([-$76543211,zpc],d1.w,-$3210fedd)" },
	{ MA(0x11f0, BL),		T "(-$76543211,zpc)" },
	{ MA(0x11f1, BL),		T "([-$76543211,zpc])" },
	{ MA(0x11f2, BL, OW),	T "([-$76543211,zpc],-$3211)" },
	{ MA(0x11f3, BL, OL),	T "([-$76543211,zpc],-$3210fedd)" },
#undef MA
#undef T

	// EA のないところが illegal になるか。
	{ { 0x4207 },						"clr.b   d7" },			// dn
	{ { 0x420f },						"" },					// an
	{ { 0x4217 },						"clr.b   (a7)" },		// (an)
	{ { 0x421f },						"clr.b   (a7)+" },		// (an)+
	{ { 0x4227 },						"clr.b   -(a7)" },		// -(an)
	{ { 0x4238, 0x89ab },				"clr.b   $89ab" },		// abs.w
	{ { 0x4239, 0x89ab, 0xcdef },		"clr.b   $89abcdef" },	// abs.l
	{ { 0x423a },						"" },					// d16(pc)
	{ { 0x423b },						"" },					// d8(pc,ix)
	{ { 0x423c },						"" },					// #imm
	{ { 0x423d },						"" },					// invalid
	{ { 0x423e },						"" },					// invalid
	{ { 0x423f },						"" },					// invalid

	// movem (制御・ポストインクリメント)
	{ { 0x48d0, 0x0000 }, "movem.l noregs,(a0)" },		// リストなし
	{ { 0x48d0, 0x0001 }, "movem.l d0,(a0)" },			// 単体
	{ { 0x48d0, 0x0006 }, "movem.l d1-d2,(a0)" },		// dで連続
	{ { 0x48d0, 0x00f6 }, "movem.l d1-d2/d4-d7,(a0)" },	// 区間が複数
	{ { 0x48d0, 0x01ff }, "movem.l d0-d7/a0,(a0)" },	// d/a をまたぐ
	{ { 0x48d0, 0x03ff }, "movem.l d0-d7/a0-a1,(a0)" },	// d/a をまたいで連続
	{ { 0x48d0, 0xff00 }, "movem.l a0-a7,(a0)" },		// aで連続
	{ { 0x48d0, 0x6186 }, "movem.l d1-d2/d7/a0/a5-a6,(a0)" }, // 複雑なの
	// movem (プリデクリメント)
	{ { 0x48e0, 0x0000 }, "movem.l noregs,-(a0)" },		// リストなし
	{ { 0x48e0, 0x8000 }, "movem.l d0,-(a0)" },			// 単体
	{ { 0x48e0, 0x6000 }, "movem.l d1-d2,-(a0)" },		// dで連続
	{ { 0x48e0, 0x6f00 }, "movem.l d1-d2/d4-d7,-(a0)" },// 区間が複数
	{ { 0x48e0, 0xff80 }, "movem.l d0-d7/a0,-(a0)" },	// d/a をまたぐ
	{ { 0x48e0, 0xffc0 }, "movem.l d0-d7/a0-a1,-(a0)" },// d/a をまたいで連続
	{ { 0x48e0, 0x00ff }, "movem.l a0-a7,-(a0)" },		// aで連続
	{ { 0x48e0, 0x6186 }, "movem.l d1-d2/d7/a0/a5-a6,-(a0)" }, // 複雑なの
	// movec (制御レジスタだけ確認、向きはもういいだろう)
	{ { 0x4e7b, 0x0000 }, "movec.l d0,sfc" },
	{ { 0x4e7b, 0x0001 }, "movec.l d0,dfc" },
	{ { 0x4e7b, 0x0002 }, "movec.l d0,cacr" },
	{ { 0x4e7b, 0x8800 }, "movec.l a0,usp" },
	{ { 0x4e7b, 0x8801 }, "movec.l a0,vbr" },
	{ { 0x4e7b, 0x8802 }, "movec.l a0,caar" },
	{ { 0x4e7b, 0x8803 }, "movec.l a0,msp" },
	{ { 0x4e7b, 0x8804 }, "movec.l a0,isp" },
	// condition
	{ { 0x50fc },			"trapt" },
	{ { 0x51fc },			"trapf" },
	{ { 0x52fc },			"traphi" },
	{ { 0x53fc },			"trapls" },
	{ { 0x54fc },			"trapcc" },
	{ { 0x55fc },			"trapcs" },
	{ { 0x56fc },			"trapne" },
	{ { 0x57fc },			"trapeq" },
	{ { 0x58fc },			"trapvc" },
	{ { 0x59fc },			"trapvs" },
	{ { 0x5afc },			"trappl" },
	{ { 0x5bfc },			"trapmi" },
	{ { 0x5cfc },			"trapge" },
	{ { 0x5dfc },			"traplt" },
	{ { 0x5efc },			"trapgt" },
	{ { 0x5ffc },			"traple" },

	// MMU
	{ { 0xf010, 0x0800 },				"pmove.l (a0),tt0" },
	{ { 0xf010, 0x0900 },				"pmovefd.l (a0),tt0" },
	{ { 0xf010, 0x0a00 },				"pmove.l tt0,(a0)" },
	{ { 0xf010, 0x0c00 },				"pmove.l (a0),tt1" },
	{ { 0xf010, 0x0d00 },				"pmovefd.l (a0),tt1" },
	{ { 0xf010, 0x0e00 },				"pmove.l tt1,(a0)" },
	{ { 0xf010, 0x2000 },				"ploadw  sfc,(a0)" },
	{ { 0xf010, 0x2008 },				"ploadw  d0,(a0)" },
	{ { 0xf010, 0x2017 },				"ploadw  #7,(a0)" },
	{ { 0xf010, 0x2201 },				"ploadr  dfc,(a0)" },
	{ { 0xf010, 0x2208 },				"ploadr  d0,(a0)" },
	{ { 0xf010, 0x2210 },				"ploadr  #0,(a0)" },
	{ { 0xf000, 0x2400 },				"pflusha" },
	{ { 0xf000, 0x30e0 },				"pflush  sfc,#7" },
	{ { 0xf000, 0x3011 },				"pflush  #1,#0" },
	{ { 0xf010, 0x38e0 },				"pflush  sfc,#7,(a0)" },
	{ { 0xf010, 0x3811 },				"pflush  #1,#0,(a0)" },
	{ { 0xf010, 0x4000 },				"pmove.l (a0),tc" },
	{ { 0xf010, 0x4100 },				"pmovefd.l (a0),tc" },
	{ { 0xf010, 0x4200 },				"pmove.l tc,(a0)" },
	{ { 0xf010, 0x4800 },				"pmove.q (a0),srp" },
	{ { 0xf010, 0x4900 },				"pmovefd.q (a0),srp" },
	{ { 0xf010, 0x4a00 },				"pmove.q srp,(a0)" },
	{ { 0xf010, 0x4c00 },				"pmove.q (a0),crp" },
	{ { 0xf010, 0x4d00 },				"pmovefd.q (a0),crp" },
	{ { 0xf010, 0x4e00 },				"pmove.q crp,(a0)" },
	{ { 0xf010, 0x6000 },				"pmove.w (a0),mmusr" },
	{ { 0xf010, 0x6200 },				"pmove.w mmusr,(a0)" },
	{ { 0xf010, 0x8000 },				"ptestw  sfc,(a0),#0" },
	{ { 0xf010, 0x9d21 },				"ptestw  dfc,(a0),#7,a1" },
	{ { 0xf010, 0x8208 },				"ptestr  d0,(a0),#0" },
	{ { 0xf010, 0x9ff7 },				"ptestr  #7,(a0),#7,a7" },

	// FPU
	{ { 0xf200, 0x0000 },				"fmove.x fp0,fp0" },
	{ { 0xf200, 0x0500 },				"fmove.x fp1,fp2" },
	{ { 0xf200, 0x0001 },				"fint.x  fp0" },
	{ { 0xf200, 0x0501 },				"fint.x  fp1,fp2" },
	{ { 0xf200, 0x0002 },				"fsinh.x fp0" },
	{ { 0xf200, 0x0502 },				"fsinh.x fp1,fp2" },
	{ { 0xf200, 0x0003 },				"fintrz.x fp0" },
	{ { 0xf200, 0x0503 },				"fintrz.x fp1,fp2" },
	{ { 0xf200, 0x0004 },				"fsqrt.x fp0" },
	{ { 0xf200, 0x0504 },				"fsqrt.x fp1,fp2" },
	{ { 0xf200, 0x0006 },				"flognp1.x fp0" },
	{ { 0xf200, 0x0506 },				"flognp1.x fp1,fp2" },
	{ { 0xf200, 0x0008 },				"fetoxm1.x fp0" },
	{ { 0xf200, 0x0508 },				"fetoxm1.x fp1,fp2" },
	{ { 0xf200, 0x0009 },				"ftanh.x fp0" },
	{ { 0xf200, 0x0509 },				"ftanh.x fp1,fp2" },
	{ { 0xf200, 0x000a },				"fatan.x fp0" },
	{ { 0xf200, 0x050a },				"fatan.x fp1,fp2" },
	{ { 0xf200, 0x000c },				"fasin.x fp0" },
	{ { 0xf200, 0x050c },				"fasin.x fp1,fp2" },
	{ { 0xf200, 0x000d },				"fatanh.x fp0" },
	{ { 0xf200, 0x050d },				"fatanh.x fp1,fp2" },
	{ { 0xf200, 0x000e },				"fsin.x  fp0" },
	{ { 0xf200, 0x050e },				"fsin.x  fp1,fp2" },
	{ { 0xf200, 0x000f },				"ftan.x  fp0" },
	{ { 0xf200, 0x050f },				"ftan.x  fp1,fp2" },
	{ { 0xf200, 0x0010 },				"fetox.x fp0" },
	{ { 0xf200, 0x0510 },				"fetox.x fp1,fp2" },
	{ { 0xf200, 0x0011 },				"ftwotox.x fp0" },
	{ { 0xf200, 0x0511 },				"ftwotox.x fp1,fp2" },
	{ { 0xf200, 0x0012 },				"ftentox.x fp0" },
	{ { 0xf200, 0x0512 },				"ftentox.x fp1,fp2" },
	{ { 0xf200, 0x0014 },				"flogn.x fp0" },
	{ { 0xf200, 0x0514 },				"flogn.x fp1,fp2" },
	{ { 0xf200, 0x0015 },				"flog10.x fp0" },
	{ { 0xf200, 0x0515 },				"flog10.x fp1,fp2" },
	{ { 0xf200, 0x0016 },				"flog2.x fp0" },
	{ { 0xf200, 0x0516 },				"flog2.x fp1,fp2" },
	{ { 0xf200, 0x0018 },				"fabs.x  fp0" },
	{ { 0xf200, 0x0518 },				"fabs.x  fp1,fp2" },
	{ { 0xf200, 0x0019 },				"fcosh.x fp0" },
	{ { 0xf200, 0x0519 },				"fcosh.x fp1,fp2" },
	{ { 0xf200, 0x001a },				"fneg.x  fp0" },
	{ { 0xf200, 0x051a },				"fneg.x  fp1,fp2" },
	{ { 0xf200, 0x001c },				"facos.x fp0" },
	{ { 0xf200, 0x051c },				"facos.x fp1,fp2" },
	{ { 0xf200, 0x001d },				"fcos.x  fp0" },
	{ { 0xf200, 0x051d },				"fcos.x  fp1,fp2" },
	{ { 0xf200, 0x001e },				"fgetexp.x fp0" },
	{ { 0xf200, 0x051e },				"fgetexp.x fp1,fp2" },
	{ { 0xf200, 0x001f },				"fgetman.x fp0" },
	{ { 0xf200, 0x051f },				"fgetman.x fp1,fp2" },
	{ { 0xf200, 0x0020 },				"fdiv.x  fp0" },
	{ { 0xf200, 0x0520 },				"fdiv.x  fp1,fp2" },
	{ { 0xf200, 0x0021 },				"fmod.x  fp0" },
	{ { 0xf200, 0x0521 },				"fmod.x  fp1,fp2" },
	{ { 0xf200, 0x0022 },				"fadd.x  fp0" },
	{ { 0xf200, 0x0522 },				"fadd.x  fp1,fp2" },
	{ { 0xf200, 0x0023 },				"fmul.x  fp0" },
	{ { 0xf200, 0x0523 },				"fmul.x  fp1,fp2" },
	{ { 0xf200, 0x0024 },				"fsgldiv.x fp0" },
	{ { 0xf200, 0x0524 },				"fsgldiv.x fp1,fp2" },
	{ { 0xf200, 0x0025 },				"frem.x  fp0" },
	{ { 0xf200, 0x0525 },				"frem.x  fp1,fp2" },
	{ { 0xf200, 0x0026 },				"fscale.x fp0" },
	{ { 0xf200, 0x0526 },				"fscale.x fp1,fp2" },
	{ { 0xf200, 0x0027 },				"fsglmul.x fp0" },
	{ { 0xf200, 0x0527 },				"fsglmul.x fp1,fp2" },
	{ { 0xf200, 0x0028 },				"fsub.x  fp0" },
	{ { 0xf200, 0x0528 },				"fsub.x  fp1,fp2" },
	{ { 0xf200, 0x05b2 },				"fsincos.x fp1,fp2:fp3" },
	{ { 0xf200, 0x04b1 },				"fsincos.x fp1,fp1:fp1" },
	{ { 0xf200, 0x0038 },				"fcmp.x  fp0" },
	{ { 0xf200, 0x0538 },				"fcmp.x  fp1,fp2" },
	{ { 0xf200, 0x003a },				"ftst.x  fp0" },
	{ { 0xf200, 0x053a },				"ftst.x  fp1,fp2" },

	{ { 0xf210, 0x4080 },				"fmove.l (a0),fp1" },
	{ { 0xf210, 0x4480 },				"fmove.s (a0),fp1" },
	{ { 0xf210, 0x4880 },				"fmove.x (a0),fp1" },
	{ { 0xf210, 0x4c80 },				"fmove.p (a0),fp1" },
	{ { 0xf210, 0x5080 },				"fmove.w (a0),fp1" },
	{ { 0xf210, 0x5480 },				"fmove.d (a0),fp1" },
	{ { 0xf210, 0x5880 },				"fmove.b (a0),fp1" },
	{ { 0xf23c, 0x4080, 0x89ab,0xcdef},	"fmove.l #$89abcdef,fp1" },
	{ { 0xf23c, 0x4480, 0x89ab,0xcdef},	"fmove.s #$89abcdef,fp1" },
	{ { 0xf23c, 0x4880, 0x89ab,0xcdef, 0x1122,0x3344, 0x5566, 0x7788 },
							"fmove.x #$89abcdef_11223344_55667788,fp1" },
	{ { 0xf23c, 0x4c80, 0x89ab,0xcdef, 0x1122,0x3344, 0x5566, 0x7788 },
							"fmove.p #$89abcdef_11223344_55667788,fp1" },
	{ { 0xf23c, 0x5080, 0x89ab },		"fmove.w #$89ab,fp1" },
	{ { 0xf23c, 0x5480, 0x89ab,0xcdef, 0x1122,0x3344 },
							"fmove.d #$89abcdef_11223344,fp1" },
	{ { 0xf23c, 0x5880, 0x89ab },		"fmove.b #$ab,fp1" },
	// 代表で fmove 以外のを1つだけ
	{ { 0xf210, 0x4081 },				"fint.l  (a0),fp1" },
	{ { 0xf210, 0x4481 },				"fint.s  (a0),fp1" },
	{ { 0xf210, 0x4881 },				"fint.x  (a0),fp1" },
	{ { 0xf210, 0x4c81 },				"fint.p  (a0),fp1" },
	{ { 0xf210, 0x5081 },				"fint.w  (a0),fp1" },
	{ { 0xf210, 0x5481 },				"fint.d  (a0),fp1" },
	{ { 0xf210, 0x5881 },				"fint.b  (a0),fp1" },

	{ { 0xf200, 0x5c81 },				"fmovecr.x #$01,fp1" },	// 不正定数
	{ { 0xf200, 0x5cff },				"fmovecr.x #$7f,fp1" },
	{ { 0xf200, 0x5c80 },				"fmovecr.x #$00<pi>,fp1" },
	{ { 0xf200, 0x5c8b },				"fmovecr.x #$0b<log10(2)>,fp1" },
	{ { 0xf200, 0x5c8c },				"fmovecr.x #$0c<e>,fp1" },
	{ { 0xf200, 0x5c8d },				"fmovecr.x #$0d<log2(e)>,fp1" },
	{ { 0xf200, 0x5c8e },				"fmovecr.x #$0e<log10(e)>,fp1" },
	{ { 0xf200, 0x5c8f },				"fmovecr.x #$0f<0>,fp1" },
	{ { 0xf200, 0x5cb0 },				"fmovecr.x #$30<ln(2)>,fp1" },
	{ { 0xf200, 0x5cb1 },				"fmovecr.x #$31<ln(10)>,fp1" },
	{ { 0xf200, 0x5cb2 },				"fmovecr.x #$32<1>,fp1" },
	{ { 0xf200, 0x5cb3 },				"fmovecr.x #$33<10>,fp1" },
	{ { 0xf200, 0x5cb4 },				"fmovecr.x #$34<1e2>,fp1" },
	{ { 0xf200, 0x5cb5 },				"fmovecr.x #$35<1e4>,fp1" },
	{ { 0xf200, 0x5cb6 },				"fmovecr.x #$36<1e8>,fp1" },
	{ { 0xf200, 0x5cb7 },				"fmovecr.x #$37<1e16>,fp1" },
	{ { 0xf200, 0x5cb8 },				"fmovecr.x #$38<1e32>,fp1" },
	{ { 0xf200, 0x5cb9 },				"fmovecr.x #$39<1e64>,fp1" },
	{ { 0xf200, 0x5cba },				"fmovecr.x #$3a<1e128>,fp1" },
	{ { 0xf200, 0x5cbb },				"fmovecr.x #$3b<1e256>,fp1" },
	{ { 0xf200, 0x5cbc },				"fmovecr.x #$3c<1e512>,fp1" },
	{ { 0xf200, 0x5cbd },				"fmovecr.x #$3d<1e1024>,fp1" },
	{ { 0xf200, 0x5cbe },				"fmovecr.x #$3e<1e2048>,fp1" },
	{ { 0xf200, 0x5cbf },				"fmovecr.x #$3f<1e4096>,fp1" },

	{ { 0xf210, 0x6080 },				"fmove.l fp1,(a0)" },
	{ { 0xf210, 0x6480 },				"fmove.s fp1,(a0)" },
	{ { 0xf210, 0x6880 },				"fmove.x fp1,(a0)" },
	{ { 0xf210, 0x6c80 },				"fmove.p fp1,(a0){#0}" },
	{ { 0xf210, 0x6cbf },				"fmove.p fp1,(a0){#63}" },
	{ { 0xf210, 0x6cc0 },				"fmove.p fp1,(a0){#-64}" },
	{ { 0xf210, 0x7080 },				"fmove.w fp1,(a0)" },
	{ { 0xf210, 0x7480 },				"fmove.d fp1,(a0)" },
	{ { 0xf210, 0x7880 },				"fmove.b fp1,(a0)" },
	{ { 0xf210, 0x7cf0 },				"fmove.p fp1,(a0){d7}" },

	{ { 0xf210, 0x8400 },				"fmove.l (a0),fpiar" },
	{ { 0xf210, 0x8800 },				"fmove.l (a0),fpsr" },
	{ { 0xf210, 0x8c00 },				"fmovem.l (a0),fpsr/fpiar" },
	{ { 0xf210, 0x9000 },				"fmove.l (a0),fpcr" },
	{ { 0xf210, 0x9400 },				"fmovem.l (a0),fpcr/fpiar" },
	{ { 0xf210, 0x9800 },				"fmovem.l (a0),fpcr/fpsr" },
	{ { 0xf210, 0x9c00 },				"fmovem.l (a0),fpcr/fpsr/fpiar" },
	{ { 0xf23c, 0x8400, 0x89ab,0xcdef },"fmove.l #$89abcdef,fpiar" },
	{ { 0xf23c, 0x8800, 0x89ab,0xcdef },"fmove.l #$89abcdef,fpsr" },
	{ { 0xf23c, 0x8c00, 0x89ab,0xcdef, 0xabcd,0xef01 },
				"fmovem.l #$89abcdef/#$abcdef01,fpsr/fpiar" },
	{ { 0xf23c, 0x9000, 0x89ab,0xcdef },"fmove.l #$89abcdef,fpcr" },
	{ { 0xf23c, 0x9400, 0x89ab,0xcdef, 0xabcd,0xef01 },
				"fmovem.l #$89abcdef/#$abcdef01,fpcr/fpiar" },
	{ { 0xf23c, 0x9800, 0x89ab,0xcdef, 0xabcd,0xef01 },
				"fmovem.l #$89abcdef/#$abcdef01,fpcr/fpsr" },
	{ { 0xf23c, 0x9c00, 0x89ab,0xcdef, 0xabcd,0xef01, 0xcdef,0x1234 },
				"fmovem.l #$89abcdef/#$abcdef01/#$cdef1234,fpcr/fpsr/fpiar" },
	{ { 0xf210, 0xa400 },				"fmove.l fpiar,(a0)" },
	{ { 0xf210, 0xa800 },				"fmove.l fpsr,(a0)" },
	{ { 0xf210, 0xac00 },				"fmovem.l fpsr/fpiar,(a0)" },
	{ { 0xf210, 0xb000 },				"fmove.l fpcr,(a0)" },
	{ { 0xf210, 0xb400 },				"fmovem.l fpcr/fpiar,(a0)" },
	{ { 0xf210, 0xb800 },				"fmovem.l fpcr/fpsr,(a0)" },
	{ { 0xf210, 0xbc00 },				"fmovem.l fpcr/fpsr/fpiar,(a0)" },

	{ { 0xf210, 0xd080 },				"fmovem.x (a0),fp0" },
	{ { 0xf210, 0xd810 },				"fmovem.x (a0),d1" },
	{ { 0xf220, 0xe801 },				"fmovem.x fp0,-(a0)" },
	{ { 0xf210, 0xf080 },				"fmovem.x fp0,(a0)" },
	{ { 0xf210, 0xf810 },				"fmovem.x d1,(a0)" },
	// fmovem (制御・ポストインクリメント)
	{ { 0xf210, 0xf000 }, "fmovem.x nofpn,(a0)" },			// リストなし
	{ { 0xf210, 0xf0c0 }, "fmovem.x fp0-fp1,(a0)" },		// 連続
	{ { 0xf210, 0xf0ec }, "fmovem.x fp0-fp2/fp4-fp5,(a0)" },// 区間が複数
	{ { 0xf210, 0xf00f }, "fmovem.x fp4-fp7,(a0)" },		// fp7で終わる
	// fmovem (プリデクリメント)
	{ { 0xf220, 0xe800 }, "fmovem.x nofpn,-(a0)" },			// リストなし
	{ { 0xf220, 0xe803 }, "fmovem.x fp0-fp1,-(a0)" },		// 連続
	{ { 0xf220, 0xe837 }, "fmovem.x fp0-fp2/fp4-fp5,-(a0)"},// 区間が複数
	{ { 0xf220, 0xe8f0 }, "fmovem.x fp4-fp7,-(a0)" },		// fp7で終わる

	{ { 0xf240, 0x0001 },				"fseq.b  d0" },
	{ { 0xf249, 0x0001, 0xfffe },		"fdbeq   d1,$00000100" },
	{ { 0xf27a, 0x0001, 0x89ab },		"ftrapeq.w #$89ab" },
	{ { 0xf27b, 0x0001, 0x89ab, 0xcdef},"ftrapeq.l #$89abcdef" },
	{ { 0xf27c, 0x0001 },				"ftrapeq" },
	{ { 0xf281, 0xfffe },				"fbeq.w  $00000100" },
	{ { 0xf2c1, 0xffff, 0xfffe },		"fbeq.l  $00000100" },
	{ { 0xf280, 0x0000 },				"fnop" },
	{ { 0xf280, 0xfffe },				"fbf.w   $00000100" },
	{ { 0xf310 },						"fsave   (a0)" },
	{ { 0xf350 },						"frestore (a0)" },

	// 040
	{ { 0xf409 },						"cinvl   nc,(a1)" },
	{ { 0xf449 },						"cinvl   dc,(a1)" },
	{ { 0xf489 },						"cinvl   ic,(a1)" },
	{ { 0xf4c9 },						"cinvl   bc,(a1)" },
	{ { 0xf451 },						"cinvp   dc,(a1)" },
	{ { 0xf458 },						"cinva   dc" },
	{ { 0xf469 },						"cpushl  dc,(a1)" },
	{ { 0xf471 },						"cpushp  dc,(a1)" },
	{ { 0xf478 },						"cpusha  dc" },
	{ { 0xf501 },						"pflushn (a1)" },
	{ { 0xf509 },						"pflush  (a1)" },
	{ { 0xf510 },						"pflushan" },
	{ { 0xf518 },						"pflusha_040" },
	{ { 0xf549 },						"ptestw  (a1)" },
	{ { 0xf569 },						"ptestr  (a1)" },
	{ { 0xf601, 0x1234, 0x5678 },		"move16  (a1)+,$12345678" },
	{ { 0xf609, 0x1234, 0x5678 },		"move16  $12345678,(a1)+" },
	{ { 0xf611, 0x1234, 0x5678 },		"move16  (a1),$12345678" },
	{ { 0xf619, 0x1234, 0x5678 },		"move16  $12345678,(a1)" },
	{ { 0xf621, 0x2000 },				"move16  (a1)+,(a2)+" },

	// cp*
	{ { 0xf511 },						"cpsave  (a1)" },	// cpid=2
	{ { 0xf551 },						"cprestore (a1)" },
	{ { 0xf711 },						"cpsave  (a1)" },	// cpid=3
	{ { 0xf751 },						"cprestore (a1)" },
	{ { 0xf911 },						"cpsave  (a1)" },	// cpid=4
	{ { 0xf951 },						"cprestore (a1)" },
	{ { 0xfb11 },						"cpsave  (a1)" },	// cpid=5
	{ { 0xfb51 },						"cprestore (a1)" },
	{ { 0xfd11 },						"cpsave  (a1)" },	// cpid=6
	{ { 0xfd51 },						"cprestore (a1)" },
	// cpid=7 の cpSAVE/RESTORE だが無条件に DOS コールを優先している。
	{ { 0xff11 },						"DOS     _PRNSNS" },	// cpid=7
	{ { 0xff51 },						"DOS     _GETPDB(v2)" },
};

static uint8 ram[1024];
static int failed;

#include "test_debugger_memory.cpp"

// リンカを通すためのダミー
DisasmLine::~DisasmLine() { }
DebuggerMD::~DebuggerMD() { }
uint32 DebuggerMD::GetNextPC() { return 0; }

int
main(int ac, char *av[])
{
	busaddr saddr;

	// DebuggerMD (の継承クラス)を用意するのは芋づる式に大変なので、
	// 中でアクセスする変数だけ強引に用意する。
	DebuggerMD *md = (DebuggerMD *)calloc(1, sizeof(DebuggerMD));
	md->inst_bytes = 2;

	for (int i = 0; i < countof(table); i++) {
		std::string expected = table[i].expected;
		std::vector<uint16> inst = table[i].inst;

		if (expected.empty()) {
			expected = "<illegal instruction>";
		}

		// エラー表示用
		std::string inst_str;
		for (const auto j : inst) {
			inst_str += string_format(" %04x", j);
		}

		// 全体を初期化
		memset(ram, 0xdd, sizeof(ram));
		// 100番地から命令列を書き込む
		for (int j = 0; j < inst.size(); j++) {
			*(uint16 *)(ram + 0x100 + j * 2) = inst[j];
		}
		// 検査
		saddr = busaddr(0x100);
		m680x0disasm dis;
		DebuggerMemoryStream mem(NULL, saddr);
		dis.Exec(&mem);
		// 照合
		if (dis.text != expected) {
			printf(">%s expects \"%s\" but \"%s\"\n", inst_str.c_str(),
				expected.c_str(), dis.text.c_str());
			failed++;
			continue;
		}
		uint32 expected_len = inst.size() * 2;
		if (dis.bin.size() != expected_len) {
			printf(">%s expects len=$%x but $%x\n", inst_str.c_str(),
				expected_len, (uint)dis.bin.size());
			failed++;
			continue;
		}
	}

	int total = countof(table);
	printf("%d tests, %d success", total, total - failed);
	if (failed > 0) {
		printf(", %d failed", failed);
	}
	printf("\n");
	return 0;
}
