/*	$NetBSD: fpu_implode.c,v 1.15 2013/03/26 11:30:21 isaki Exp $ */

/*
 * Copyright (c) 1992, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This software was developed by the Computer Systems Engineering group
 * at Lawrence Berkeley Laboratory under DARPA contract BG 91-66 and
 * contributed to Berkeley.
 *
 * All advertising materials mentioning features or use of this software
 * must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Lawrence Berkeley Laboratory.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)fpu_implode.c	8.1 (Berkeley) 6/11/93
 */

/*
 * FPU subroutines: `implode' internal format numbers into the machine's
 * `packed binary' format.
 */

#include "fpu_emulate.h"

/* Conversion from internal format -- note asymmetry. */
static uint32_t	fpu_ftoi(struct fpemu *fe, struct fpn *fp);
static uint32_t	fpu_ftos(struct fpemu *fe, struct fpn *fp);
static uint32_t	fpu_ftod(struct fpemu *fe, struct fpn *fp, uint32_t *);
static uint32_t	fpu_ftox(struct fpemu *fe, struct fpn *fp, uint32_t *);
static void fpu_round_chkmin(struct fpemu *fe, struct fpn *fp, int, int);
static void fpu_round_chkinf(struct fpemu *fe, struct fpn *fp, int);

/*
 * Round a number (algorithm from Motorola MC68882 manual, modified for
 * our internal format).  Set inexact exception if rounding is required.
 * Return true iff we rounded up.
 *
 * After rounding, we discard the guard and round bits by shifting right
 * 2 bits (a la fpu_shr(), but we do not bother with fp->fp_sticky).
 * This saves effort later.
 *
 * Note that we may leave the value 2.0 in fp->fp_mant; it is the caller's
 * responsibility to fix this if necessary.
 */
// FPCR_ROUND (NR, RZ, RP, RM) によって fp (拡張精度) をラウンディングする。
// 戻り値はラウンドアップしたら 1(true)。
// ラウンディングが発生すると内部 FPSR:EXCP に INEX2 を立てる。
// 任意精度のラウンディングは fpu_round_prec() 参照のこと。XXX どうしたもんか
//
// 実行後の fp_mant[2] は GR ビットを捨ててあり、つまり fp_mant 全体が当初
// に比べて2ビット右シフトされた状態となる。
// ラウンドアップによって生じた整数部 2 は何もせずそのまま返されるので、
// これを必要に応じてどうにかするのは呼び出し側の責任。
// (ただし整数部(小数点位置)も2ビットずれてるはず、でいいのかな?)
// fp_sticky は常にゼロクリアされる。
// fp_exp は変更しない。
int
fpu_round(struct fpemu *fe, struct fpn *fp)
{
	uint32_t m0, m1, m2;
	int gr, s;

	m0 = fp->fp_mant[0];
	m1 = fp->fp_mant[1];
	m2 = fp->fp_mant[2];
	gr = m2 & 3;
	s = fp->fp_sticky;

	/* mant >>= FP_NG */
	m2 = (m2 >> FP_NG) | (m1 << (32 - FP_NG));
	m1 = (m1 >> FP_NG) | (m0 << (32 - FP_NG));
	m0 >>= FP_NG;

	if ((gr | s) == 0)	/* result is exact: no rounding needed */
		goto rounddown;

	fe->fe_fpsr |= FPSR_INEX2;	/* inexact */

	/* Go to rounddown to round down; break to round up. */
	switch (fe->fe_fpcr & FPCR_ROUND) {

	case FPCR_NEAR:
	default:
		/*
		 * Round only if guard is set (gr & 2).  If guard is set,
		 * but round & sticky both clear, then we want to round
		 * but have a tie, so round to even, i.e., add 1 iff odd.
		 */
		if ((gr & 2) == 0)
			goto rounddown;
		if ((gr & 1) || fp->fp_sticky || (m2 & 1))
			break;
		goto rounddown;

	case FPCR_ZERO:
		/* Round towards zero, i.e., down. */
		goto rounddown;

	case FPCR_MINF:
		/* Round towards -Inf: up if negative, down if positive. */
		if (fp->fp_sign)
			break;
		goto rounddown;

	case FPCR_PINF:
		/* Round towards +Inf: up if positive, down otherwise. */
		if (!fp->fp_sign)
			break;
		goto rounddown;
	}

	/* Bump low bit of mantissa, with carry. */
	if (++m2 == 0 && ++m1 == 0)
		m0++;
	fp->fp_sticky = 0;
	fp->fp_mant[0] = m0;
	fp->fp_mant[1] = m1;
	fp->fp_mant[2] = m2;
	return (1);

rounddown:
	fp->fp_sticky = 0;
	fp->fp_mant[0] = m0;
	fp->fp_mant[1] = m1;
	fp->fp_mant[2] = m2;
	return (0);
}

/*
 * For overflow: return true if overflow is to go to +/-Inf, according
 * to the sign of the overflowing result.  If false, overflow is to go
 * to the largest magnitude value instead.
 */
static int
toinf(struct fpemu *fe, int sign)
{
	int inf;

	/* look at rounding direction */
	switch (fe->fe_fpcr & FPCR_ROUND) {

	default:
	case FPCR_NEAR:		/* the nearest value is always Inf */
		inf = 1;
		break;

	case FPCR_ZERO:		/* toward 0 => never towards Inf */
		inf = 0;
		break;

	case FPCR_PINF:		/* toward +Inf iff positive */
		inf = (sign == 0);
		break;

	case FPCR_MINF:		/* toward -Inf iff negative */
		inf = sign;
		break;
	}
	return (inf);
}

/*
 * fpn -> int (int value returned as return value).
 *
 * N.B.: this conversion always rounds towards zero (this is a peculiarity
 * of the SPARC instruction set).
 */
static uint32_t
fpu_ftoi(struct fpemu *fe, struct fpn *fp)
{
	uint32_t i;
	int sign, exp;

	sign = fp->fp_sign;
	switch (fp->fp_class) {
	case FPC_ZERO:
		return (0);

	case FPC_NUM:
		/*
		 * If exp >= 2^32, overflow.  Otherwise shift value right
		 * into last mantissa word (this will not exceed 0xffffffff),
		 * shifting any guard and round bits out into the sticky
		 * bit.  Then ``round'' towards zero, i.e., just set an
		 * inexact exception if sticky is set (see fpu_round()).
		 * If the result is > 0x80000000, or is positive and equals
		 * 0x80000000, overflow; otherwise the last fraction word
		 * is the result.
		 */
		if ((exp = fp->fp_exp) >= 32)
			break;
		/* NB: the following includes exp < 0 cases */
		if (fpu_shr(fp, FP_NMANT - 1 - FP_NG - exp) != 0) {
			/*
			 * m68881/2 do not underflow when
			 * converting to integer
			 */
			;
		}
		fpu_round(fe, fp);
		i = fp->fp_mant[2];
		if (i >= ((uint32_t)0x80000000 + sign))
			break;
		return (sign ? -i : i);

	default:		/* Inf, qNaN, sNaN */
		break;
	}
	/* overflow: replace any inexact exception with invalid */
	fe->fe_fpsr = (fe->fe_fpsr & ~FPSR_INEX2) | FPSR_OPERR;
	return (0x7fffffff + sign);
}

/*
 * fpn -> single (32 bit single returned as return value).
 * We assume <= 29 bits in a single-precision fraction (1.f part).
 */
static uint32_t
fpu_ftos(struct fpemu *fe, struct fpn *fp)
{
	uint32_t sign = fp->fp_sign << 31;
	int exp;

#define	SNG_EXP(e)	((e) << SNG_FRACBITS)	/* makes e an exponent */
#define	SNG_MASK	(SNG_EXP(1) - 1)	/* mask for fraction */

	/* Take care of non-numbers first. */
	if (ISNAN(fp)) {
		/*
		 * Preserve upper bits of NaN, per SPARC V8 appendix N.
		 * Note that fp->fp_mant[0] has the quiet bit set,
		 * even if it is classified as a signalling NaN.
		 */
		(void) fpu_shr(fp, FP_NMANT - 1 - SNG_FRACBITS);
		exp = SNG_EXP_INFNAN;
		goto done;
	}
	if (ISINF(fp))
		return (sign | SNG_EXP(SNG_EXP_INFNAN));
	if (ISZERO(fp))
		return (sign);

	/*
	 * Normals (including subnormals).  Drop all the fraction bits
	 * (including the explicit ``implied'' 1 bit) down into the
	 * single-precision range.  If the number is subnormal, move
	 * the ``implied'' 1 into the explicit range as well, and shift
	 * right to introduce leading zeroes.  Rounding then acts
	 * differently for normals and subnormals: the largest subnormal
	 * may round to the smallest normal (1.0 x 2^minexp), or may
	 * remain subnormal.  In the latter case, signal an underflow
	 * if the result was inexact or if underflow traps are enabled.
	 *
	 * Rounding a normal, on the other hand, always produces another
	 * normal (although either way the result might be too big for
	 * single precision, and cause an overflow).  If rounding a
	 * normal produces 2.0 in the fraction, we need not adjust that
	 * fraction at all, since both 1.0 and 2.0 are zero under the
	 * fraction mask.
	 *
	 * Note that the guard and round bits vanish from the number after
	 * rounding.
	 */
	if ((exp = fp->fp_exp + SNG_EXP_BIAS) <= 0) {	/* subnormal */
		fe->fe_fpsr |= FPSR_UNFL;
		/* -NG for g,r; -SNG_FRACBITS-exp for fraction */
		(void) fpu_shr(fp, FP_NMANT - FP_NG - SNG_FRACBITS - exp);
		if (fpu_round(fe, fp) && fp->fp_mant[2] == SNG_EXP(1))
			return (sign | SNG_EXP(1) | 0);
		if (fe->fe_fpsr & FPSR_INEX2) {
			/* mc68881/2 don't underflow when converting */
			fe->fe_fpsr |= FPSR_UNFL;
		}
		return (sign | SNG_EXP(0) | fp->fp_mant[2]);
	}
	/* -FP_NG for g,r; -1 for implied 1; -SNG_FRACBITS for fraction */
	(void) fpu_shr(fp, FP_NMANT - FP_NG - 1 - SNG_FRACBITS);
#ifdef DIAGNOSTIC
	if ((fp->fp_mant[2] & SNG_EXP(1 << FP_NG)) == 0)
		panic("fpu_ftos");
#endif
	if (fpu_round(fe, fp) && fp->fp_mant[2] == SNG_EXP(2))
		exp++;
	if (exp >= SNG_EXP_INFNAN) {
		/* overflow to inf or to max single */
		fe->fe_fpsr |= FPSR_OVFL;
		if (toinf(fe, sign)) {
			fp->fp_class = FPC_INF;
			return (sign | SNG_EXP(SNG_EXP_INFNAN));
		}
		fe->fe_fpsr |= FPSR_OPERR;
		return (sign | SNG_EXP(SNG_EXP_INFNAN - 1) | SNG_MASK);
	}
done:
	if ((fp->fp_mant[2] & SNG_MASK) == 0)
		fp->fp_class = FPC_ZERO;
	/* phew, made it */
	return (sign | SNG_EXP(exp) | (fp->fp_mant[2] & SNG_MASK));
}

/*
 * fpn -> double (32 bit high-order result returned; 32-bit low order result
 * left in res[1]).  Assumes <= 61 bits in double precision fraction.
 *
 * This code mimics fpu_ftos; see it for comments.
 */
static uint32_t
fpu_ftod(struct fpemu *fe, struct fpn *fp, uint32_t *res)
{
	uint32_t sign = fp->fp_sign << 31;
	int exp;

#define	DBL_EXP(e)	((e) << (DBL_FRACBITS & 31))
#define	DBL_MASK	(DBL_EXP(1) - 1)

	if (ISNAN(fp)) {
		(void) fpu_shr(fp, FP_NMANT - 1 - DBL_FRACBITS);
		exp = DBL_EXP_INFNAN;
		goto done;
	}
	if (ISINF(fp)) {
		sign |= DBL_EXP(DBL_EXP_INFNAN);
		res[1] = 0;
		return (sign);
	}
	if (ISZERO(fp)) {
		res[1] = 0;
		return (sign);
	}

	if ((exp = fp->fp_exp + DBL_EXP_BIAS) <= 0) {
		fe->fe_fpsr |= FPSR_UNFL;
		(void) fpu_shr(fp, FP_NMANT - FP_NG - DBL_FRACBITS - exp);
		if (fpu_round(fe, fp) && fp->fp_mant[1] == DBL_EXP(1)) {
			res[1] = 0;
			return (sign | DBL_EXP(1) | 0);
		}
		if (fe->fe_fpsr & FPSR_INEX2) {
			/* mc68881/2 don't underflow when converting */
			fe->fe_fpsr |= FPSR_UNFL;
		}
		exp = 0;
		goto done;
	}
	(void) fpu_shr(fp, FP_NMANT - FP_NG - 1 - DBL_FRACBITS);
	if (fpu_round(fe, fp) && fp->fp_mant[1] == DBL_EXP(2))
		exp++;
	if (exp >= DBL_EXP_INFNAN) {
		fe->fe_fpsr |= FPSR_OVFL;
		if (toinf(fe, sign)) {
			fp->fp_class = FPC_INF;
			res[1] = 0;
			return (sign | DBL_EXP(DBL_EXP_INFNAN) | 0);
		}
		fe->fe_fpsr |= FPSR_OPERR;
		res[1] = ~0;
		return (sign | DBL_EXP(DBL_EXP_INFNAN) | DBL_MASK);
	}
done:
	res[1] = fp->fp_mant[2];
	if (((fp->fp_mant[1] & DBL_MASK) | res[1]) == 0)
		fp->fp_class = FPC_ZERO;
	return (sign | DBL_EXP(exp) | (fp->fp_mant[1] & DBL_MASK));
}

/*
 * fpn -> 68k extended (32 bit high-order result returned; two 32-bit low
 * order result left in res[1] & res[2]).  Assumes == 64 bits in extended
 * precision fraction.
 *
 * This code mimics fpu_ftos; see it for comments.
 */
static uint32_t
fpu_ftox(struct fpemu *fe, struct fpn *fp, uint32_t *res)
{
	uint32_t sign = fp->fp_sign << 31;
	int exp;

#define	EXT_EXP(e)	((e) << 16)
/*
 * on m68k extended prec, significand does not share the same long
 * word with exponent
 */
#define	EXT_MASK	0
#define EXT_EXPLICIT1	(1UL << (63 & 31))
#define EXT_EXPLICIT2	(1UL << (64 & 31))

	if (ISNAN(fp)) {
		(void) fpu_shr(fp, FP_NMANT - EXT_FRACBITS);
		exp = EXT_EXP_INFNAN;
		goto done;
	}
	if (ISINF(fp)) {
		sign |= EXT_EXP(EXT_EXP_INFNAN);
		res[1] = res[2] = 0;
		return (sign);
	}
	if (ISZERO(fp)) {
		res[1] = res[2] = 0;
		return (sign);
	}

	if ((exp = fp->fp_exp + EXT_EXP_BIAS) < 0) {
		fe->fe_fpsr |= FPSR_UNFL;
		/*
		 * I'm not sure about this <=... exp==0 doesn't mean
		 * it's a denormal in extended format
		 */
		(void) fpu_shr(fp, FP_NMANT - FP_NG - EXT_FRACBITS - exp);
		if (fpu_round(fe, fp) && fp->fp_mant[1] == EXT_EXPLICIT1) {
			res[1] = res[2] = 0;
			return (sign | EXT_EXP(1) | 0);
		}
		if (fe->fe_fpsr & FPSR_INEX2) {
			/* mc68881/2 don't underflow */
			fe->fe_fpsr |= FPSR_UNFL;
		}
		exp = 0;
		goto done;
	}
#if (FP_NMANT - FP_NG - EXT_FRACBITS) > 0
	(void) fpu_shr(fp, FP_NMANT - FP_NG - EXT_FRACBITS);
#endif
	if (fpu_round(fe, fp) && fp->fp_mant[0] == EXT_EXPLICIT2) {
		exp++;
		fpu_shr(fp, 1);
	}
	if (exp >= EXT_EXP_INFNAN) {
		fe->fe_fpsr |= FPSR_OVFL;
		if (toinf(fe, sign)) {
			fp->fp_class = FPC_INF;
			res[1] = res[2] = 0;
			return (sign | EXT_EXP(EXT_EXP_INFNAN) | 0);
		}
		fe->fe_fpsr |= FPSR_OPERR;
		res[1] = res[2] = ~0;
		return (sign | EXT_EXP(EXT_EXP_INFNAN) | EXT_MASK);
	}
done:
	res[1] = fp->fp_mant[1];
	res[2] = fp->fp_mant[2];
	if ((res[1] | res[2]) == 0)
		fp->fp_class = FPC_ZERO;
	return (sign | EXT_EXP(exp));
}

/*
 * Implode an fpn, writing the result into the given space.
 */
void
fpu_implode(struct fpemu *fe, struct fpn *fp, int type, uint32_t *space)
{
	/* XXX Dont delete exceptions set here: fe->fe_fpsr &= ~FPSR_EXCP; */

	switch (type) {
	case FTYPE_LNG:
		space[0] = fpu_ftoi(fe, fp);
		break;

	case FTYPE_SNG:
		space[0] = fpu_ftos(fe, fp);
		break;

	case FTYPE_DBL:
		space[0] = fpu_ftod(fe, fp, space);
		break;

	case FTYPE_EXT:
		/* funky rounding precision options ?? */
		space[0] = fpu_ftox(fe, fp, space);
		break;

	default:
		/* 何も出来ることがない */
		break;
	}
}

#if defined(XM6i_FPE)
/*
 * Shift the given number left lsh bits.
 * Note that the sticky filed is cleared.
 */
static void
fpu_shl(struct fpn *fp, int lsh)
{
	uint32_t m0, m1, m2;
	int rsh;

	m0 = fp->fp_mant[0];
	m1 = fp->fp_mant[1];
	m2 = fp->fp_mant[2];

	while (lsh >= 32) {
		m0 = m1;
		m1 = m2;
		m2 = (fp->fp_sticky != 0) ? 0x80000000 : 0;
		fp->fp_sticky = 0;
		lsh -= 32;
	}
	if (lsh != 0) {
		rsh = 32 - lsh;
		m0 = (m0 << lsh) | (m1 >> rsh);
		m1 = (m1 << lsh) | (m2 >> rsh);
		m2 = (m2 << lsh);
		m2 |= (fp->fp_sticky != 0) ? (1 << (lsh - 1)) : 0;
		fp->fp_sticky = 0;
	}
	fp->fp_mant[0] = m0 & (FP_2 - 1);
	fp->fp_mant[1] = m1;
	fp->fp_mant[2] = m2;
}

/*
 * Round a number according to FPCR_MODE.
 * (fpu_round() rounds according to FPCR_ROUND as FPCR_MODE = FPCR_EXTD)
 */
// fp を FPCR_PREC(精度)/ FPCR_MODE(RN,RZ,RP,RM) によってラウンディングする。
// FPSR:EXCP に UNFL、INEX2 を立てる場合がある。
// 元からある fpu_round() は拡張精度限定のラウンディング。どうしたもんか。
void
fpu_round_prec(struct fpemu *fe, struct fpn *fp)
{
	if (fp->fp_class != FPC_NUM) {
		PRINTF("fpu_round_prec class != NUM\n");
		return;
	}

	switch ((fe->fe_fpcr & FPCR_PREC)) {
	 case FPCR_SNGL:
		fpu_round_chkmin(fe, fp, SNG_EXP_BIAS, SNG_FRACBITS);
		fpu_round_chkinf(fe, fp, SNG_EXP_BIAS);
		break;

	 case FPCR_DBL:
		fpu_round_chkmin(fe, fp, DBL_EXP_BIAS, DBL_FRACBITS);
		fpu_round_chkinf(fe, fp, DBL_EXP_BIAS);
		break;

	 case FPCR_EXTD:
	 default:
		//
		// 拡張精度だけ fpu_round_chk{min,inf}() と微妙に処理が異なる…。
		//
		if (fp->fp_exp < -EXT_EXP_BIAS - EXT_FRACBITS) {
			// 拡張精度にすると指数が小さすぎて表現できない場合
			fe->fe_fpsr |= FPSR_UNFL;
			DUMPFP("e1:start", fp);

			// sticky だけの状態から FPCR_MODE によるラウンディングで
			// fp_mant だけ作る。ここで仮数部ゼロならゼロ。
			fp->fp_mant[0] = 0;
			fp->fp_mant[1] = 0;
			fp->fp_mant[2] = 0;
			fp->fp_sticky = 1;
			fpu_round(fe, fp);
			if (fp->fp_mant[2] == 0)
				fp->fp_class = FPC_ZERO;
			// fp_mant の LSB に立ってるはずのビットを整数部になるよう正規化。
			fp->fp_exp = FP_NMANT - EXT_EXP_BIAS - EXT_FRACBITS;
			fpu_norm(fp);
		} else if (fp->fp_exp < -EXT_EXP_BIAS) {
			// 拡張精度にすると非正規化数になる場合
			fe->fe_fpsr |= FPSR_UNFL;
			DUMPFP("e2:start", fp);

			int shift = FP_NMANT - EXT_FRACBITS - 0/*integer bit included*/;
			int effbits = EXT_EXP_BIAS + EXT_FRACBITS + fp->fp_exp;
			PRINTF("effbits=%d\n", effbits);
			shift += EXT_FRACBITS - effbits;
			PRINTF("shift=%d\n", shift);
			fpu_shr(fp, shift - FP_NG);
			DUMPFP("e2:shr  ", fp);

			int rup = fpu_round(fe, fp);
			uint64_t m = (((uint64_t)fp->fp_mant[1]) << 32)
			             | (uint64_t)fp->fp_mant[2];
			if (rup && m == (1ULL << effbits)) {
				fp->fp_exp++;
				shift--;
			}
			DUMPFP("e2:round", fp);

			fpu_shl(fp, shift);
			DUMPFP("e2:shl  ", fp);

			if (fp->fp_exp <= -EXT_EXP_BIAS - EXT_FRACBITS) {
				PRINTF("e2:zero\n");
				fp->fp_class = FPC_ZERO;
			}
		} else {
			int shift = FP_NMANT - EXT_FRACBITS - 0/*Integer bit included*/;
			PRINTF("shift=%d\n", shift);
			fpu_shr(fp, shift - FP_NG);
			DUMPFP("e3:shr  ", fp);

			int rup = fpu_round(fe, fp);
			uint32_t m = fp->fp_mant[1] | fp->fp_mant[2];
			if (rup && fp->fp_mant[0] == 1 && m == 0) {	/* mant == 2.0 */
				fp->fp_exp++;
				shift--;
				DUMPFP("e3:rup  ", fp);
			} else
				DUMPFP("e3:round", fp);

			fpu_shl(fp, shift);
			DUMPFP("e3:shl  ", fp);
		}

		if (fp->fp_exp > EXT_EXP_BIAS) {
			PRINTF("fpu_round_prec EXTD\n");
			fe->fe_fpsr |= FPSR_OVFL;

			// 拡張精度で表現できない大きい値の場合は
			// 正で RN/RP か、負で RN/RM なら Inf、
			// そうでなければ拡張精度で表現できる最大値。
			if (fp->fp_sign == 0) {
				switch ((fe->fe_fpcr & FPCR_ROUND)) {
				 case FPCR_NEAR:
				 case FPCR_PINF:
					fp->fp_class = FPC_INF;
					break;
				}
			} else {
				switch ((fe->fe_fpcr & FPCR_ROUND)) {
				 case FPCR_NEAR:
				 case FPCR_MINF:
					fp->fp_class = FPC_INF;
					break;
				}
			}
			if (fp->fp_class != FPC_INF) {
				PRINTF("fpu_round_prec make INF\n");
				// 最大値
				fp->fp_exp = EXT_EXP_BIAS;
				fp->fp_mant[0] = FP_2 -1;
				fp->fp_mant[1] = 0xffffffff;
				fp->fp_mant[2] = 0xfff80000;
			}
		}
		break;
	}
}

void
fpu_round_chkmin(struct fpemu *fe, struct fpn *fp, int EXP_BIAS, int FRACBITS)
{
	int shift;
	int effbits;
	int rup;

	if (fp->fp_exp < -EXP_BIAS - FRACBITS) {
		// 指定精度にすると指数が小さすぎて表現できない場合
		fe->fe_fpsr |= FPSR_UNFL;
		DUMPFP("s1:start", fp);

		// sticky だけの状態から FPCR_MODE によるラウンディングで
		// fp_mant だけ作る。ここで仮数部ゼロならゼロ。
		fp->fp_mant[0] = 0;
		fp->fp_mant[1] = 0;
		fp->fp_mant[2] = 0;
		fp->fp_sticky = 1;
		fpu_round(fe, fp);
		DUMPFP("s1:round", fp);
		if (fp->fp_mant[2] == 0)
			fp->fp_class = FPC_ZERO;
		// fp_mant の LSB に立ってるはずのビットを整数部になるよう正規化。
		fp->fp_exp = FP_NMANT - EXP_BIAS - FRACBITS;
		fpu_norm(fp);
		DUMPFP("s1:norm ", fp);
		return;
	}
	if (fp->fp_exp <= -EXP_BIAS) {
		// 指定精度にすると非正規化数になる場合
		fe->fe_fpsr |= FPSR_UNFL;
		DUMPFP("s2:start", fp);

		shift = FP_NMANT - FRACBITS - 0/*No integer bit*/;
		/*
		 * 1.XX~XX * 2^-127 => effbits = 22
		 * 1.X     * 2^-148 => effbits = 2
		 * 1.      * 2^-149 => effbits = 1
		 */
		effbits = EXP_BIAS + FRACBITS + fp->fp_exp;
		PRINTF("effbits=%d\n", effbits);
		shift += FRACBITS - effbits;
		PRINTF("shift=%d\n", shift);
		fpu_shr(fp, shift - FP_NG);
		DUMPFP("s2:shr  ", fp);

		rup = fpu_round(fe, fp);
		uint64_t m = (((uint64_t)fp->fp_mant[1]) << 32)
		             | (uint64_t)fp->fp_mant[2];
		if (rup && m == (1ULL << effbits)) {
			fp->fp_exp++;
			shift--;
		}
		DUMPFP("s2:round", fp);

		fpu_shl(fp, shift);
		DUMPFP("s2:shl  ", fp);

		if (fp->fp_exp <= -EXP_BIAS - FRACBITS) {
			PRINTF("s2:zero\n");
			fp->fp_class = FPC_ZERO;
		}
		return;
	}

	// 指定精度の正規化数で表現できる場合
	DUMPFP("s3:start", fp);

	// [0]                 [1]  [2]
	// 8765432109876543210       0         0         0         0
	// IMMMMMMMMMMMMMMMMMM m..m MMMMMMMMMMMMMMMMMMMMMMMMMMMMMMGR : origin
	//                                IMMMMMMMMMMMMMMMMMMmmmmmmm : shifted
	//                                 21098765432109876543210GR
	shift = FP_NMANT - FRACBITS - 1/*Integer bit*/;
	fpu_shr(fp, shift - FP_NG);
	DUMPFP("s3:shr  ", fp);

	rup = fpu_round(fe, fp);
	uint64_t m = (((uint64_t)fp->fp_mant[1]) << 32)
	             | (uint64_t)fp->fp_mant[2];
	if (rup && (m == (2ULL << FRACBITS))) {
		fp->fp_exp++;
		shift--;
	}
	DUMPFP("s3:round", fp);

	fpu_shl(fp, shift);
	DUMPFP("s3:shl  ", fp);
}

void
fpu_round_chkinf(struct fpemu *fe, struct fpn *fp, int EXP_BIAS)
{
	if (fp->fp_exp > EXP_BIAS) {
		fe->fe_fpsr |= FPSR_OVFL;

		// 指定精度で表現できない大きい値の場合は
		// 正で RN/RP か、負で RN/RM なら Inf、
		// そうでなければ指定精度で表現できる最大値。
		if (fp->fp_sign == 0) {
			switch ((fe->fe_fpcr & FPCR_ROUND)) {
			 case FPCR_NEAR:
			 case FPCR_PINF:
				fp->fp_class = FPC_INF;
				break;
			}
		} else {
			switch ((fe->fe_fpcr & FPCR_ROUND)) {
			 case FPCR_NEAR:
			 case FPCR_MINF:
				fp->fp_class = FPC_INF;
				break;
			}
		}
		if (fp->fp_class != FPC_INF) {
			// 最大値
			fp->fp_exp = EXP_BIAS;
			fp->fp_mant[0] = FP_2 -1;
			switch ((fe->fe_fpcr & FPCR_PREC)) {
			 case FPCR_SNGL:
				fp->fp_mant[1] = 0xf8000000;
				fp->fp_mant[2] = 0;
				break;
			 case FPCR_DBL:
				fp->fp_mant[1] = 0xffffffff;
				fp->fp_mant[2] = 0xc0000000;
				break;
			 case FPCR_EXTD:
				fp->fp_mant[1] = 0xffffffff;
				fp->fp_mant[2] = 0xfff80000;
				break;
			}
		}
	}
}
#endif /* XM6i_FPE */
