//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// バージョンダイアログ
//

#include "wxversion.h"
#include "hostnet.h"
#include "monitor.h"
#include <sys/utsname.h>

// グローバル参照用
HostInfoMonitor *gHostInfo;

// コンストラクタ
WXVersionDlg::WXVersionDlg(wxWindow *parent)
	: inherited(parent, wxID_ANY, _("About nono"))
{
	wxBoxSizer *vbox = new wxBoxSizer(wxVERTICAL);

	wxString title = wxString::Format("nono %d.%d.%d (%s)",
		NONO_MAJOR_VER, NONO_MINOR_VER, NONO_PATCH_VER, NONO_DATE);

	// タイトルだけ少し大きく
	// XXX ..したいのだが wxOSX だと色々おかしいのでただのテキストにする。
	auto titlectrl = new wxStaticText(this, wxID_ANY, title);
#if !defined(__WXOSX__)
	titlectrl->SetLabelMarkup("<span size=\"x-large\">" + title + "</span>");
#endif
	vbox->Add(titlectrl, 0, wxALIGN_CENTER);

	// 空行
	vbox->Add(new wxStaticText(this, wxID_ANY, ""));

	// Copyright
	vbox->Add(new wxStaticText(this, wxID_ANY,
		"Copyright (C) 2020-2025 nono project"));

	// 空行
	vbox->Add(new wxStaticText(this, wxID_ANY, ""));

	// OK
	vbox->Add(CreateButtonSizer(wxOK), 0, wxALIGN_CENTER);

	// 周囲に 12px の余白をつける
	wxBoxSizer *topsizer = new wxBoxSizer(wxVERTICAL);
	topsizer->Add(vbox, 0, wxALL, 12);
	SetSizer(topsizer);
	topsizer->SetSizeHints(this);
}

// デストラクタ
WXVersionDlg::~WXVersionDlg()
{
}


//
// ホスト情報モニタ
//

// コンストラクタ
HostInfoMonitor::HostInfoMonitor()
	: inherited(OBJ_HOSTINFO)
{
	// ログは不要
	ClearAlias();

	// この時点で必要な情報は全部集めておく
	int width = 0;
	int height = 4;

	// OS 名
	const auto info = wxPlatformInfo::Get();
#if defined(__WXOSX__)
	// "Apple Mac OS X 10.14" みたいに表示される
	osname = info.GetOperatingSystemIdName();
	osname += wxString::Format(" %d.%d",
		info.GetOSMajorVersion(),
		info.GetOSMinorVersion());
#else
	// Linux ならディストリビューション
	if (info.GetOperatingSystemId() == wxOS_UNIX_LINUX) {
		auto distro = info.GetLinuxDistributionInfo();
		osname = distro.Description;
	}
	// そうでなければ
	if (osname.IsEmpty()) {
		// GetOperatingSystemDescription() はほぼ uname -srm のようなので
		// 前の2つだけにして uname -sr 相当で表示。
		osname = info.GetOperatingSystemDescription();
		auto pos = osname.rfind(' ');
		if (pos != std::string::npos) {
			osname = osname.substr(0, pos);
		}
	}
#endif
	width = std::max(width, (int)osname.Length());

	// Machine (アーキテクチャと Endian を表示したい)
	struct utsname ut;
	if (uname(&ut) == 0) {
		machinename = ut.machine;
		machinename += " ";
	}
	// GetEndiannessName() は endian が小文字なので大文字にしたい
	switch (info.GetEndianness()) {
	 case wxENDIAN_BIG:
		machinename += "(Big Endian)";
		break;
	 case wxENDIAN_LITTLE:
		machinename += "(Little Endian)";
		break;
	 case wxENDIAN_PDP:
		machinename += "(PDP Endian)";
		break;
	 default:
		break;
	}
	width = std::max(width, (int)machinename.Length());

	// CPU Features
#if defined(__x86_64__)
	height++;
# if defined(HAVE_AVX2)
	if (gMainApp.detect_avx2) {
		if (gMainApp.enable_avx2) {
			avx2str = "enabled";
		} else {
			avx2str = "disabled";
		}
	} else {
		avx2str = "not detected";
	}
# else
	avx2str = "not compiled";
# endif
#endif

#if defined(__aarch64__)
	height++;
# if defined(HAVE_NEON)
	if (gMainApp.detect_neon) {
		if (gMainApp.enable_neon) {
			neonstr = "enabled";
		} else {
			neonstr = "disabled";
		}
	} else {
		neonstr = "not detected";
	}
# else
	neonstr = "not compiled";
# endif
#endif
	width = std::max(width, (int)avx2str.Length());
	width = std::max(width, (int)neonstr.Length());

	// wxWidgets バージョン
	wxname = wxVERSION_NUM_DOT_STRING_T;
	// wxWidgets ポート名
	// wxGTK なら wxGTK{2,3} の区別をしておきたい
	auto wxport = info.GetPortIdName();
	if (info.GetPortId() == wxPORT_GTK) {
		wxport += wxString::Format("%d", info.GetToolkitMajorVersion());
	}
	wxname = wxname + " (" + wxport + ")";
	width = std::max(width, (int)wxname.Length());

	// libslirp バージョン
	slirpver = HostNetDevice::GetSlirpVersion();
	if (slirpver == NULL) {
		slirpver = "not compiled";
	}
	width = std::max(width, (int)strlen(slirpver));

	monitor = gMonitorManager->Regist(ID_MONITOR_WXHOSTINFO, this);
	monitor->SetCallback(&HostInfoMonitor::MonitorScreen);
	monitor->SetSize(11 + width, height);
}

// デストラクタ
HostInfoMonitor::~HostInfoMonitor()
{
	gHostInfo = NULL;
}

// モニタを更新
void
HostInfoMonitor::MonitorScreen(Monitor *, TextScreen& screen)
{
	int x = 11;
	int y;

	screen.Clear();
	y = 0;

	screen.Puts(0, y, "OS:");
	screen.Puts(x, y, (const char *)osname.mb_str());
	y++;

	screen.Puts(0, y, "Machine:");
	screen.Puts(x, y, (const char *)machinename.mb_str());
	y++;

#if defined(__x86_64__)
	screen.Puts(0, y, "AVX2:");
	screen.Puts(x, y, (const char *)avx2str.mb_str());
	y++;
#endif
#if defined(__aarch64__)
	screen.Puts(0, y, "NEON:");
	screen.Puts(x, y, (const char *)neonstr.mb_str());
	y++;
#endif

	screen.Puts(0, y, "wxWidgets:");
	screen.Puts(x, y, (const char *)wxname.mb_str());
	y++;

	screen.Puts(0, y, "libslirp:");
	screen.Puts(x, y, slirpver);
	y++;
}
