//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// テキストフォントを描画するパネル
//

#include "wxtextpanel.h"
#include "fontmanager.h"
#include "sjis.h"

// コンストラクタ
WXTextPanel::WXTextPanel(wxWindow *parent, wxWindowID id,
	const wxPoint& position, const wxSize& size, long style)
	: inherited(parent, id, position, size, style)
{
	// デフォルト色
	ResetTextColor();
}

// デストラクタ
WXTextPanel::~WXTextPanel()
{
}

void
WXTextPanel::FontChanged()
{
	font_width = gFontManager->GetFontWidth();
	font_height = gFontManager->GetFontHeight();
}

// テキスト色をデフォルトに設定する(戻す)
void
WXTextPanel::ResetTextColor()
{
	SetTextColor(UD_BLACK, BGPANEL);
}

// テキスト色を指定する
void
WXTextPanel::SetTextColor(Color fg, Color bg)
{
	palette[FG] = fg;
	palette[BG] = bg;
}

// (px, py) を左上とする座標から Shift_JIS 文字列を、前景色 c、属性 attr で
// 描画する。
// ただし attr のうちここで参照するのは EM (ボールド) かそうでないかのみ。
// フォントサイズは現行のフォント。
// キャンバスをはみ出さないこと。
void
WXTextPanel::DrawStringSJIS(Color c, int px, int py, const char *sjis,
	uint attr)
{
	Color backup = palette[FG];

	palette[FG] = c;
	DrawStringSJIS(px, py, sjis, attr);
	palette[FG] = backup;
}

// (px, py) を左上とする座標から Shift_JIS 文字列を属性 attr で描画する。
// ただし attr のうちここで参照するのは EM (ボールド) かそうでないかのみ。
// 描画色、背景色は事前に SetTextColor() で設定したもの。
// フォントサイズは現行のフォント。
// キャンバスをはみ出さないこと。
void
WXTextPanel::DrawStringSJIS(int px, int py, const char *sjis, uint attr)
{
	const uint8 *s;

	for (s = (const uint8 *)sjis; *s; ) {
		if (__predict_true(SJIS::IsHankaku(*s))) {
			// 半角
			DrawChar1(px, py, *s++, attr);
			px += font_width;
		} else {
			// 全角
			uint code;
			code = (*s++ << 8);
			code |= *s++;
			DrawChar2(px, py, code, attr);
			px += font_width * 2;
		}
	}
}

// (px, py) を左上とする座標から半角文字 code を属性 attr で描画する。
// code は ASCII または半角カナ。attr のうち反応するのは Bold のみ。
// フォントサイズは現行のフォント。
void
WXTextPanel::DrawChar1(int px, int py, uint code, uint attr)
{
	assertmsg(code < 0x100, "code=0x%x", code);

	const uint8 *glyph = gFontManager->AsciiGlyph(code);
	BitmapI1 src(glyph, font_width, font_height);
	if (__predict_false((attr & TA::Bold))) {
		src = src.ConvertToBold();
	}
	bitmap.DrawBitmapI1(px, py, src, palette);
}

// (px, py) を左上とする座標から漢字1文字を属性 attr で描画する。
// sjiscode は Shift_JIS。attr のうち反応するのは Bold のみ。
// フォントサイズは現行のフォント。
void
WXTextPanel::DrawChar2(int px, int py, uint sjiscode, uint attr)
{
	const uint8 *glyph = gFontManager->KanjiGlyph(sjiscode);
	BitmapI1 src(glyph, font_width * 2, font_height);
	if (__predict_false((attr & TA::Bold))) {
		src = src.ConvertToBold();
	}
	bitmap.DrawBitmapI1(px, py, src, palette);
}
