//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// イベント
//

#pragma once

#include "device.h"

template <typename T>
EventCallback_t ToEventCallback(T f) {
	return static_cast<EventCallback_t>(f);
}


// イベント
//
// イベントは必要なデバイスごとに用意する。おそらく最初から個数が決まって
// いることが多いはずなので、デバイスクラス内のメンバ変数(実体) として
// 宣言し、デバイスのコンストラクタ等で初期化するという流れになるはず。
//
// VM リセットによるイベントタイマーの停止処理は各デバイスの責任で行うこと。
class Event
{
 public:
	Event(Device *, int code_, EventCallback_t, const std::string&);
	~Event();

	// このイベントが動作中なら true を返す
	bool IsRunning() const {
		return active;
	}

	Device *GetOwner() const noexcept	{ return dev; }
	EventCallback_t GetCallback() const noexcept	{ return func; }

	// 以下はイベントを呼び出す側がセットする

	// コールバック関数をセットする。
	template <typename T>
	void SetCallback(void (T::*func_)(Event *)) {
		func = static_cast<EventCallback_t>(func_);
	}

	// イベントを起こすまでの仮想時刻での相対時間 [nsec]
	// 例えば今から 1msec 後にイベントを起こしたい場合は 1000000 となる。
	uint64 time {};

	int code {};				// コールバック関数への引数
	const std::string GetName() const { return name; }
	void SetName(const std::string& val);

	// スケジューラが内部で使う
	bool active {};				// スケジューラにセットされていれば真
	uint64 vtime {};			// イベントを起こす仮想時刻

	uint64 count {};			// コールバックを呼び出した回数
	uint64 last_count {};		// (表示用)前回表示したときの count

 private:
	Device *dev {};				// デバイス
	EventCallback_t func {};	// コールバック関数

	// イベント名(表示用)
	// '\0' 含まず25文字まで。デバイス名も必要なら含めること。
	std::string name {};
};

//
// イベントマネージャ
//
class EventManager : public Object
{
	using inherited = Object;
	friend class Scheduler;
 public:
	EventManager();
	~EventManager() override;

	// イベントを登録する。
	Event *Regist(Device *d) {
		return Regist(d, 0);
	}
	Event *Regist(Device *d, EventCallback_t f) {
		return Regist(d, 0, f);
	}
	Event *Regist(Device *d, EventCallback_t f, const std::string& n) {
		return Regist(d, 0, f, n);
	}
	Event *Regist(Device *d, int code) {
		return Regist(d, code, NULL);
	}
	Event *Regist(Device *d, int code, EventCallback_t f) {
		return Regist(d, code, f, "");
	}
	Event *Regist(Device *d, int code, EventCallback_t f, const std::string& n);

 private:
	std::vector<Event *> all {};
};

static inline EventManager* GetEventManager() {
	return Object::GetObject<EventManager>(OBJ_EVENT_MANAGER);
}
