/**
 *    Copyright (C) 2018-present MongoDB, Inc.
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the Server Side Public License, version 1,
 *    as published by MongoDB, Inc.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    Server Side Public License for more details.
 *
 *    You should have received a copy of the Server Side Public License
 *    along with this program. If not, see
 *    <http://www.mongodb.com/licensing/server-side-public-license>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the Server Side Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */

#pragma once

#include <string>

#include "mongo/db/jsobj.h"
#include "mongo/db/namespace_string.h"
#include "mongo/db/pipeline/runtime_constants_gen.h"
#include "mongo/db/query/plan_executor.h"

namespace mongo {

class DeleteRequest {
    DeleteRequest(const DeleteRequest&) = delete;
    DeleteRequest& operator=(const DeleteRequest&) = delete;

public:
    explicit DeleteRequest(const NamespaceString& nsString)
        : _nsString(nsString),
          _multi(false),
          _god(false),
          _fromMigrate(false),
          _isExplain(false),
          _returnDeleted(false),
          _yieldPolicy(PlanExecutor::NO_YIELD) {}

    void setQuery(const BSONObj& query) {
        _query = query;
    }
    void setProj(const BSONObj& proj) {
        _proj = proj;
    }
    void setSort(const BSONObj& sort) {
        _sort = sort;
    }
    void setRuntimeConstants(RuntimeConstants runtimeConstants) {
        _runtimeConstants = std::move(runtimeConstants);
    }
    void setCollation(const BSONObj& collation) {
        _collation = collation;
    }
    void setMulti(bool multi = true) {
        _multi = multi;
    }
    void setGod(bool god = true) {
        _god = god;
    }
    void setFromMigrate(bool fromMigrate = true) {
        _fromMigrate = fromMigrate;
    }
    void setExplain(bool isExplain = true) {
        _isExplain = isExplain;
    }
    void setReturnDeleted(bool returnDeleted = true) {
        _returnDeleted = returnDeleted;
    }
    void setYieldPolicy(PlanExecutor::YieldPolicy yieldPolicy) {
        _yieldPolicy = yieldPolicy;
    }

    const NamespaceString& getNamespaceString() const {
        return _nsString;
    }
    const BSONObj& getQuery() const {
        return _query;
    }
    const BSONObj& getProj() const {
        return _proj;
    }
    const BSONObj& getSort() const {
        return _sort;
    }
    const boost::optional<RuntimeConstants>& getRuntimeConstants() const {
        return _runtimeConstants;
    }
    const BSONObj& getCollation() const {
        return _collation;
    }
    bool isMulti() const {
        return _multi;
    }
    bool isGod() const {
        return _god;
    }
    bool isFromMigrate() const {
        return _fromMigrate;
    }
    bool isExplain() const {
        return _isExplain;
    }
    void setHint(const BSONObj& hint) {
        _hint = hint;
    }
    BSONObj getHint() const {
        return _hint;
    }
    bool shouldReturnDeleted() const {
        return _returnDeleted;
    }
    PlanExecutor::YieldPolicy getYieldPolicy() const {
        return _yieldPolicy;
    }

    void setStmtId(StmtId stmtId) {
        _stmtId = std::move(stmtId);
    }

    StmtId getStmtId() const {
        return _stmtId;
    }

private:
    const NamespaceString& _nsString;
    BSONObj _hint;
    BSONObj _query;
    BSONObj _proj;
    BSONObj _sort;
    BSONObj _collation;
    boost::optional<RuntimeConstants> _runtimeConstants;
    // The statement id of this request.
    StmtId _stmtId = kUninitializedStmtId;
    bool _multi;
    bool _god;
    bool _fromMigrate;
    bool _isExplain;
    bool _returnDeleted;
    PlanExecutor::YieldPolicy _yieldPolicy;
};

}  // namespace mongo
