/*
 * Copyright (c) 2017, Emmanuel Dreyfus <manu@netbsd.org
 *               2003, WebThing Ltd, Nick Kew <nick@webthing.com>
 */

/* 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#define APR_WANT_MEMFUNC
#include <apr_want.h>
#include <apr_time.h>
#include <apr_lib.h>
#include <apr_strings.h>
#include <apr_atomic.h>
#include <apr_file_io.h>
#include <apr_thread_proc.h>
#include <apr_thread_mutex.h>
#include <apr_thread_cond.h>
#include <httpd.h>
#include <http_request.h>
#include <http_core.h>
#include <util_filter.h>
#include <http_config.h>
#include <http_log.h>

#ifdef HAVE_PHP
#include <SAPI.h>
#include <php_variables.h>
#endif

#define UPLOAD_DEFAULT_FORM_SIZE 16
#define UPLOAD_DEFAULT_BLOCK_SIZE (1024*1024)
#define UPLOAD_MAX_HEADER_LEN 998
#define UPLOAD_DEFAULT_TIMEOUT 60
#define UPLOAD_FILE_INDEX 10

/* from http://php.net/manual/en/features.file-upload.errors.php */
#define UPLOAD_ERR_OK		0
#define UPLOAD_ERR_INI_SIZE	1
#define UPLOAD_ERR_FORM_SIZE	2
#define UPLOAD_ERR_PARTIAL	3
#define UPLOAD_ERR_NO_FILE	4
#define UPLOAD_ERR_NO_TMP_DIR	5
#define UPLOAD_ERR_CANT_WRITE	6
#define UPLOAD_ERR_EXTENSION	7

typedef enum { start, header, value, file, end } upload_state_t;

typedef struct {
  const char *key;
  const char *val;
  const char *ctype;
} upload_formdata_t;

typedef struct {
  apr_array_header_t *file_formdata;
  apr_array_header_t *filenames;
  int done;
} upload_note_t;

typedef struct {
  apr_file_t *fh;
  char *key;
  char *xkey;
  int index;
  char *tmpname;
  char *name;
  int error;
  char *buffer;
  apr_size_t written;
  apr_off_t size;
  struct {
    apr_time_t start;
    apr_time_t written;
    apr_time_t closed;
    apr_time_t moved;
  } timer;
  int empty;
} upload_file_ctx_t;

typedef struct {
  apr_pool_t *pool;
  upload_file_ctx_t *file;
  apr_array_header_t *formdata;
  char *boundary;
  apr_size_t boundary_len;
  char *boundary_part;
  apr_size_t boundary_part_len;
  char *boundary_end;
  apr_size_t boundary_end_len;
  char saved[UPLOAD_MAX_HEADER_LEN + 1];
  apr_size_t saved_len;
  char *key;
  char *val;
  char *filename;
  char *ctype;
  int error;
  upload_state_t state;
  apr_table_t *file_index;
  apr_thread_mutex_t *mtx;
  apr_thread_cond_t *cond;
  apr_uint32_t workers;
  ap_filter_t *f;
  upload_note_t *note;
} upload_ctx_t;

typedef struct {
  upload_ctx_t *ctx;
  upload_file_ctx_t *file;
} file_worker_args_t;

typedef struct {
  int enable;
  const char *dir;
  const char *prefix;
  enum { off, on, force } commit;
  int form_size;
  size_t block_size;
  int timeout;
  int compat_php;
} upload_conf_t;

/*
 * Helper macro, -1 is for trailing \0 which is counted by sizeof
 */
#define UPLOAD_VALUE_AFTER(str, prefix, len)              \
  ((len >= sizeof(prefix) - 1) &&                         \
   (strncasecmp(str, prefix, sizeof(prefix) - 1) == 0)) ? \
      str + sizeof(prefix) - 1 : NULL

module AP_MODULE_DECLARE_DATA upload_module;

static void *
upload_create_dir_config(apr_pool_t *p, char *x)
{
  upload_conf_t *conf = apr_pcalloc(p, sizeof(upload_conf_t));
  apr_status_t res;

  if ((res = apr_temp_dir_get(&conf->dir, p)) != APR_SUCCESS) {
    ap_log_perror(APLOG_MARK, APLOG_ERR, res, p,
                  "Cannot get temporary directory");
    return NULL;
  }
  conf->enable = 0;
  conf->commit = off; 
  conf->form_size = UPLOAD_DEFAULT_FORM_SIZE;
  conf->block_size = UPLOAD_DEFAULT_BLOCK_SIZE;
  conf->timeout = UPLOAD_DEFAULT_TIMEOUT;
  conf->compat_php = 0;
  return conf;
}

static const char *
set_commit(cmd_parms *cmd, void *cfg, const char *mode)
{
  upload_conf_t *conf = (upload_conf_t *)cfg;

  if (!strcasecmp(mode, "on"))
    conf->commit = on;
  else if (!strcasecmp(mode, "off"))
    conf->commit = off;
  else if (!strcasecmp(mode, "force"))
    conf->commit = force;
  else
    return "Unexpected commit mode";

  return NULL;
}

static char *
unquote(upload_ctx_t *ctx, const char *str) {
  char *s;
  char *qp;

  s = apr_pstrdup(ctx->pool, str);
  if (*s == '"')
   s++;

  if ((qp = strrchr(s, '"')) != NULL)
    *qp = '\0';

  return s;
}

static apr_status_t
get_boundary(upload_ctx_t *ctx, const char *ctype)
{
  char *val;
  char *last;
  char *attr;
  char sep[] = ";\r";
  char skip[] = " \t";
  apr_status_t res = APR_EINVAL;

  val = apr_pstrdup(ctx->pool, ctype);

  for (attr = apr_strtok(val, sep, &last);
       attr != NULL;
       attr = apr_strtok(NULL, sep, &last)) {
    char *aval;

    attr += strspn(attr, skip);

    if ((aval = UPLOAD_VALUE_AFTER(attr, "boundary=", attr - val)) != NULL) {
      ctx->boundary = unquote(ctx, aval);
      ctx->boundary_len = strlen(ctx->boundary);
      ctx->boundary_part = apr_pstrcat(ctx->pool, "--", ctx->boundary, NULL);
      ctx->boundary_part_len = ctx->boundary_len + 2;
      ctx->boundary_end = apr_pstrcat(ctx->pool, 
                                      ctx->boundary_part, "--", NULL);
      ctx->boundary_end_len = ctx->boundary_part_len + 2;
  
      res = APR_SUCCESS;
    }
  }

  return res;
}

static void
copy_file_formdata(upload_ctx_t *ctx, upload_formdata_t *ufd)
{
  upload_formdata_t *fufd;

  fufd = (upload_formdata_t *)apr_array_push(ctx->note->file_formdata);
  fufd->key = apr_pstrdup(ctx->pool, ufd->key);
  fufd->val = apr_pstrdup(ctx->pool, ufd->val);
  fufd->ctype = NULL;

  return;
}

static void
upload_filter_insert(request_rec *r)
{
  upload_conf_t *conf = 
    ap_get_module_config(r->per_dir_config, &upload_module);

  if (conf->enable)
    ap_add_input_filter("UPLOAD", NULL, r, r->connection);
}

static void
upload_filter_insert_compatphp(request_rec *r)
{
  upload_conf_t *conf = 
    ap_get_module_config(r->per_dir_config, &upload_module);

  if (conf->enable && conf->compat_php) {
#ifdef HAVE_PHP
    ap_add_input_filter("UPLOAD_COMPATPHP", NULL, r, r->connection);
#else /* HAVE_PHP */
    ap_log_rerror(APLOG_MARK,APLOG_ERR, 0, r,
                  "UploadCompatPHP set but PHP support is not built-in");
#endif /* HAVE_PHP */
  }
}

#ifdef HAVE_PHP
static apr_status_t 
upload_filter_compatphp(ap_filter_t *f, apr_bucket_brigade *bb_in,
                        ap_input_mode_t mode, apr_read_type_e block,
                        apr_off_t nbytes) 
{
  request_rec *r = f->r;
  upload_conf_t *conf = ap_get_module_config(r->per_dir_config, &upload_module);
  upload_note_t *note;
  apr_bucket *b;
  apr_array_header_t *filenames;
  char *filename;
  char **filenames_array;
  zend_string *tmp_name;
  upload_formdata_t *ufd;
  int i;
  int res = APR_SUCCESS;

  res = ap_get_brigade(f->next, bb_in, mode, block, nbytes);
  if (res != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK,APLOG_ERR, res, f->r,
		  "ap_get_brigade failed in %s. Possible cause is "
		  "LimitRequestBody = %d", __func__,
		  ap_get_limit_req_body(f->r));
    goto out;
  }

  note = (upload_note_t *)apr_table_get(r->notes, "mod_upload");
  if (note == NULL) {
    res = APR_EINVAL;
    ap_log_rerror(APLOG_MARK,APLOG_ERR, res, f->r, "missing mod_upload note");
    goto out;
  }

  for (b = APR_BRIGADE_FIRST(bb_in);
        b != APR_BRIGADE_SENTINEL(bb_in) && !APR_BUCKET_IS_EOS(b);
        b = APR_BUCKET_NEXT(b));

  if (!note->done)
    goto out;

  filenames_array = (char **)note->filenames->elts;
  for (i = 0; i < note->filenames->nelts; i++) {
    filename = filenames_array[i];
    tmp_name = zend_string_init(filename, strlen(filename), 0);
    zend_hash_add_ptr(SG(rfc1867_uploaded_files), tmp_name, tmp_name);
  }

  apr_array_clear(note->filenames);

  if (Z_TYPE(PG(http_globals)[TRACK_VARS_FILES]) != IS_ARRAY)
    array_init(&PG(http_globals)[TRACK_VARS_FILES]);
  
  for (i = 0; i < note->file_formdata->nelts; i++) {
    ufd = (upload_formdata_t *)(note->file_formdata->elts) + i;
    php_register_variable_safe((char *)ufd->key, (char *)ufd->val, 
                               strlen(ufd->val), 
                               &PG(http_globals)[TRACK_VARS_FILES]);
  }

  apr_array_clear(note->file_formdata);

out:
  if (res != APR_SUCCESS || note->done) {
    ap_remove_input_filter(f);
  }

  return res;
}
#endif /* HAVE_PHP */

static apr_status_t
upload_filter_init(ap_filter_t *f)
{
  upload_conf_t *conf = 
    ap_get_module_config(f->r->per_dir_config, &upload_module);
  upload_ctx_t *ctx = apr_palloc(f->r->pool, sizeof(upload_ctx_t));
  const char *ctype = apr_table_get(f->r->headers_in, "Content-Type");
  apr_status_t res = APR_SUCCESS;


  ctx->note = apr_palloc(f->r->pool, sizeof(upload_note_t));
  ctx->note->filenames = apr_array_make(f->r->pool, conf->form_size,
                                        sizeof(char **));
  ctx->note->file_formdata = apr_array_make(f->r->pool, conf->form_size, 
				            sizeof(upload_formdata_t));
  ctx->note->done = 0;
  apr_table_setn(f->r->notes, "mod_upload", (char *)ctx->note);
  
  if (!ctype || strstr(ctype , "multipart/form-data") != ctype) {
    ap_remove_input_filter(f);
    return APR_SUCCESS;
  }

  if ((res = apr_atomic_init(f->r->pool)) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, res, f->r,
                  "apr_atomic_init failed");
    goto out;
  }

  ctx->f = f;
  ctx->pool = f->r->pool;

  if ((res = get_boundary(ctx, ctype)) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, res, f->r,
                  "MIME multipart boundary not found");
    goto out;
  }

  ctx->formdata = apr_array_make(ctx->pool, conf->form_size, 
				 sizeof(upload_formdata_t));
  ctx->state = start;
  ctx->key = NULL;
  ctx->val = NULL;
  ctx->filename = NULL;
  ctx->ctype = NULL;
  ctx->file = NULL;
  ctx->error = UPLOAD_ERR_OK;
  ctx->saved_len = 0;
  apr_atomic_set32(&ctx->workers, 0);

  ctx->file_index = apr_table_make(ctx->pool, UPLOAD_FILE_INDEX);

  res = apr_thread_mutex_create(&ctx->mtx, APR_THREAD_MUTEX_DEFAULT, ctx->pool);
  if (res != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, res, f->r,
                  "apr_thread_mutex_create failed");
    goto out;
  }

  res = apr_thread_cond_create(&ctx->cond, ctx->pool);
  if (res != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, res, f->r,
                  "apr_thread_cond_create failed");
    goto out;
  }

  f->ctx = ctx;

out:
  if (res != APR_SUCCESS)
    ap_remove_input_filter(f);

  return res;
}


static void *APR_THREAD_FUNC  
file_worker(apr_thread_t *thread, void *v)
{ 
  file_worker_args_t *args = v;
  upload_ctx_t *ctx = args->ctx;
  upload_file_ctx_t *file = args->file;
  request_rec *r = ctx->f->r;
  upload_conf_t *conf = ap_get_module_config(r->per_dir_config,&upload_module);
  upload_formdata_t *ufd;
  apr_status_t res = APR_SUCCESS;
  float throughput;

  if ((res = apr_file_close(file->fh)) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, res, ctx->f->r,
                  "close \"%s\" failed", file->tmpname);
    file->error = UPLOAD_ERR_CANT_WRITE;
  }

  file->timer.closed = apr_time_now();
  file->fh = NULL;

  if (conf->commit != off) {
    apr_finfo_t st;
    int exists = (apr_stat(&st, file->name, 0, ctx->pool) == APR_SUCCESS);

    if (exists && conf->commit != force) {
      ap_log_rerror(APLOG_MARK, APLOG_ERR, res, ctx->f->r,
                    "filename \"%s\" already exists", ctx->file->name);
      res = APR_EEXIST;
      file->error = UPLOAD_ERR_CANT_WRITE;
      goto giveup;
    }

    res = apr_file_rename(file->tmpname, file->name, ctx->pool);
    if (res != APR_SUCCESS) {
      ap_log_rerror(APLOG_MARK, APLOG_ERR, res, ctx->f->r,
                    "rename \"%s\" to \"%s\" failed",
                    file->tmpname, file->name);
      file->error = UPLOAD_ERR_CANT_WRITE;
      goto giveup;
    }
  }

  file->timer.moved = apr_time_now();
  throughput = file->size / (float)(file->timer.written - file->timer.start);
  throughput = throughput * APR_USEC_PER_SEC / (1024 * 1024);

  ap_log_rerror(APLOG_MARK, APLOG_INFO, 0, ctx->f->r, 
                "file \"%s\" write %llds (%3.3fMB/s), close %llds move %llds",
                file->name, 
                apr_time_sec(file->timer.written - file->timer.start),
                throughput,
                apr_time_sec(file->timer.closed - file->timer.written),
                apr_time_sec(file->timer.moved - file->timer.closed));

  if ((res = apr_thread_mutex_lock(ctx->mtx)) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, res, ctx->f->r,
                  "apr_thread_mutex_lock failed");
  }

  ufd = (upload_formdata_t *)apr_array_push(ctx->formdata);
  ufd->key = apr_pstrcat(ctx->pool, file->key, "[error]", file->xkey, NULL);
  ufd->val = apr_psprintf(ctx->pool, "%d", file->error);
  ufd->ctype = NULL;
  copy_file_formdata(ctx, ufd);

  ufd =(upload_formdata_t *)apr_array_push(ctx->formdata);
  ufd->key = apr_pstrcat(ctx->pool, file->key, "[size]", file->xkey, NULL);
  ufd->val = apr_psprintf(ctx->pool, "%qu", (intmax_t)file->size);
  ufd->ctype = NULL;
  copy_file_formdata(ctx, ufd);
  
#ifdef DEBUG
  ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, ctx->f->r, 
                "key=\"%s\", filename=\"%s\" tmpname=\"%s\"",
                file->key, file->name, file->tmpname);
#endif

  if ((res = apr_thread_mutex_unlock(ctx->mtx)) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, res, ctx->f->r,
                  "apr_thread_mutex_unlock failed");
  }

giveup:
  apr_atomic_dec32(&ctx->workers);

  if ((res = apr_thread_cond_signal(ctx->cond)) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, res, ctx->f->r,
                  "apr_thread_cond_signal failed");
  }

  return NULL;
}

static void
set_file_key(upload_ctx_t *ctx, upload_file_ctx_t *file)
{
  request_rec *r = ctx->f->r;
  upload_conf_t *conf = ap_get_module_config(r->per_dir_config,&upload_module);
  char *file_key;
  char *ap;
  const char *index;

  if ((ap = strstr(ctx->key, "[]")) != NULL) {
    index = apr_table_get(ctx->file_index, ctx->key);
    if (index == NULL)
      index = "0";

    file->index = apr_atoi64(index);

    apr_table_set(ctx->file_index, ctx->key,
                  apr_itoa(ctx->pool, file->index + 1));

    file_key = apr_pstrndup(ctx->pool, ctx->key, ap - ctx->key);
    file->xkey = apr_pstrcat(ctx->pool, "[", index, "]", NULL);
  } else {
    file_key = apr_pstrdup(ctx->pool, ctx->key);
    file->xkey = NULL;
  }

  if (conf->prefix)
    file->key = apr_pstrcat(ctx->pool, conf->prefix, "[", file_key, "]", NULL);
  else
    file->key = file_key;

  return;
}

static apr_status_t
new_file(upload_ctx_t *ctx)
{
  request_rec *r = ctx->f->r;
  upload_conf_t *conf = ap_get_module_config(r->per_dir_config,&upload_module);
  char *filename;
  upload_formdata_t *ufd;
  upload_file_ctx_t *file;
  int mktemp_flags = APR_FOPEN_CREATE | APR_FOPEN_READ | APR_FOPEN_WRITE |
                     APR_FOPEN_EXCL;
  apr_status_t res = APR_SUCCESS;
  int empty = 0;

  if (ctx->ctype == NULL)
    ctx->ctype = "application/octet-stream";

  if (strchr(ctx->filename, '/') != NULL) {
    res = APR_EINVAL;
    ap_log_rerror(APLOG_MARK, APLOG_ERR, res, r,
                  "upload filename \"%s\" contains a slash", ctx->filename);
    goto giveup;
  }

  file = apr_palloc(ctx->pool, sizeof(*file));

  set_file_key(ctx, file);

  file->name = apr_pstrcat(ctx->pool, conf->dir, "/", ctx->filename, NULL);
  file->tmpname = apr_pstrcat(ctx->pool, file->name, ".XXXXXX", NULL);
  file->error = UPLOAD_ERR_OK;
  file->buffer = apr_palloc(ctx->pool, conf->block_size);
  file->written = 0; 
  file->size = 0; 
  file->timer.start = apr_time_now();
  file->empty = (*ctx->filename == '\0');

  if (!file->empty) {
#ifdef DEBUG
    ap_log_rerror(APLOG_MARK, APLOG_INFO, res, ctx->f->r,
                  "file \"%s\" upload starts", file->name);
#endif

    res = apr_file_mktemp(&file->fh, file->tmpname, mktemp_flags, r->pool);
    if (res != APR_SUCCESS) {
      ap_log_rerror(APLOG_MARK, APLOG_ERR, res, r,
                    "Cannot create temporary file %s", file->name);
      goto giveup;
    }
  }

  ctx->file = file;

  if (conf->commit != off) {
    filename = file->name;
  } else {
    filename = file->tmpname;
  }

  if ((res = apr_thread_mutex_lock(ctx->mtx)) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, res, ctx->f->r,
                  "apr_thread_mutex_lock failed");
  }

  ufd =(upload_formdata_t *)apr_array_push(ctx->formdata);
  ufd->key = apr_pstrcat(ctx->pool, file->key, "[name]", file->xkey, NULL);
  ufd->val = file->empty ? "" : apr_filepath_name_get(file->name);
  ufd->ctype = NULL;
  copy_file_formdata(ctx, ufd);

  ufd =(upload_formdata_t *)apr_array_push(ctx->formdata);
  ufd->key = apr_pstrcat(ctx->pool, file->key, "[type]", file->xkey, NULL);
  ufd->val = file->empty ? "" : ctx->ctype;
  ufd->ctype = NULL;
  copy_file_formdata(ctx, ufd);
  
  ufd =(upload_formdata_t *)apr_array_push(ctx->formdata);
  ufd->key = apr_pstrcat(ctx->pool, file->key, "[tmp_name]", file->xkey, NULL);
  ufd->val = file->empty ? "" : filename;
  ufd->ctype = NULL;
  copy_file_formdata(ctx, ufd);

  if ((res = apr_thread_mutex_unlock(ctx->mtx)) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, res, ctx->f->r,
                  "apr_thread_mutex_unlock failed");
  }

  if (!file->empty) {
    char **filenamep;

     filenamep = (char **)apr_array_push(ctx->note->filenames);
     *filenamep = filename;
  }

giveup:
  if (res != APR_SUCCESS)
    ap_remove_input_filter(ctx->f);

  return res;
}

static apr_status_t
write_file(upload_ctx_t *ctx, const char *data, apr_size_t data_len, int eof)
{
  apr_status_t res = APR_SUCCESS;
  request_rec *r = ctx->f->r;
  upload_conf_t *conf = ap_get_module_config(r->per_dir_config,&upload_module);
  upload_file_ctx_t *file = ctx->file;
  apr_size_t nbytes;

  if (eof || file->written + data_len > conf->block_size) {
    nbytes = file->written;
    res = apr_file_write(file->fh, file->buffer, &nbytes);
    if (res != APR_SUCCESS) {
      ap_log_rerror(APLOG_MARK, APLOG_ERR, res, ctx->f->r,
                    "writing to \"%s\" failed (write)",
                    file->tmpname);
      file->error = UPLOAD_ERR_CANT_WRITE;
      goto out;
    }

    if (nbytes != file->written) {
      res = EINVAL;
      ap_log_rerror(APLOG_MARK, APLOG_ERR, res, ctx->f->r,
                    "writing to \"%s\" failed (short write %d/%d)",
                    file->tmpname, nbytes, file->written);
      file->error = UPLOAD_ERR_CANT_WRITE;
      goto out;
    }
   
    file->written = 0;
    file->size += nbytes;
  }

  if (!eof) {
    (void)memcpy(file->buffer + file->written, data, data_len);
    file->written += data_len;
  }

out:
  return res;
}

static apr_status_t
set_header(upload_ctx_t *ctx, const char *data, apr_size_t data_len) 
{
  apr_status_t res = APR_SUCCESS;
  char *line;
  char *val;

  line = apr_pstrmemdup(ctx->pool, data, data_len);

  val = UPLOAD_VALUE_AFTER(line, "Content-Disposition:", data_len);
  if (val != NULL) {
    char *last;
    char *attr;
    char sep[] = ";\r";
    char skip[] = " \t";
    
    for (attr = apr_strtok(val, sep, &last);
         attr != NULL;
         attr = apr_strtok(NULL, sep, &last)) {
      char *aval;

      attr += strspn(attr, skip);

      if ((aval = UPLOAD_VALUE_AFTER(attr, "name=", attr - line)) != NULL)
        ctx->key = unquote(ctx, aval);

      if ((aval = UPLOAD_VALUE_AFTER(attr, "filename=", attr - line)) != NULL)
        ctx->filename = unquote(ctx, aval);
    }
  }

  val = UPLOAD_VALUE_AFTER(line, "Content-Type:", data_len);
  if (val != NULL) {
    char *ch;

    while (*val && apr_isspace(*val))
      val++;

    ch = strchr(val, APR_ASCII_CR);
    *ch = '\0';

    ctx->ctype = apr_pstrdup(ctx->pool, val);
  }

  return res;
}

static apr_status_t
set_value(upload_ctx_t *ctx, const char *data, apr_size_t data_len) {
  const char *cr;
  char *tmp;
  apr_status_t res = APR_SUCCESS;

  cr = memchr(data, APR_ASCII_LF, data_len);
  if (cr == NULL) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, res, ctx->f->r,
                  "parse error in header \"%s\"", data);
    res = APR_EINVAL;
    goto out;
  }

  tmp = apr_pstrndup(ctx->pool, data, cr - data);

  /* Strip trailing \r */
  if (cr - data > 0 && tmp[cr - data - 1] == '\r')
    tmp[cr - data - 1] = '\0';

  if (ctx->val)
    ctx->val = apr_pstrcat(ctx->pool, ctx->val, tmp, NULL);
  else
    ctx->val = tmp;

out:
  return res;
}


static void
end_empty_file(upload_ctx_t *ctx)
{
  request_rec *r = ctx->f->r;
  upload_conf_t *conf = ap_get_module_config(r->per_dir_config, &upload_module);
  upload_formdata_t *ufd;
  apr_status_t res;
  upload_file_ctx_t *file = ctx->file;

  if ((res = apr_thread_mutex_lock(ctx->mtx)) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, res, ctx->f->r,
                  "apr_thread_mutex_lock failed");
  }

  ufd = (upload_formdata_t *)apr_array_push(ctx->formdata);
  ufd->key = apr_pstrcat(ctx->pool, file->key, "[error]", file->xkey, NULL);
  ufd->val = apr_psprintf(ctx->pool, "%d", UPLOAD_ERR_NO_FILE);
  ufd->ctype = NULL;
  copy_file_formdata(ctx, ufd);

  ufd =(upload_formdata_t *)apr_array_push(ctx->formdata);
  ufd->key = apr_pstrcat(ctx->pool, file->key, "[size]", file->xkey, NULL);
  ufd->val = "0";
  ufd->ctype = NULL;
  copy_file_formdata(ctx, ufd);
  
  if ((res = apr_thread_mutex_unlock(ctx->mtx)) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, res, ctx->f->r,
                  "apr_thread_mutex_unlock failed");
  }

  return;
}

static void
end_file(upload_ctx_t *ctx) 
{
  apr_status_t res = APR_SUCCESS;
  file_worker_args_t *args;
  apr_thread_t *thread;

  if (ctx->file->empty) {
    end_empty_file(ctx);
    goto out;
  }

  if ((res = write_file(ctx, NULL, 0, 1)) != APR_SUCCESS)
    goto out;

  ctx->file->timer.written = apr_time_now();

  args = apr_palloc(ctx->pool, sizeof(*args));
  args->ctx = ctx;
  args->file = ctx->file;

  apr_atomic_inc32(&ctx->workers);

  res = apr_thread_create(&thread, NULL, *file_worker, args, ctx->pool);
  if (res != APR_SUCCESS) {
    apr_atomic_dec32(&ctx->workers);
    ap_log_rerror(APLOG_MARK, APLOG_ERR, res, ctx->f->r,
                  "apr_thread_create failed");
    goto out;
  }

out:
  ctx->val = NULL;
  ctx->filename = NULL;
  ctx->key = NULL;
  ctx->ctype = NULL;
  ctx->file = NULL;
  return;
}

static void
end_value(upload_ctx_t *ctx) 
{
  upload_formdata_t *ufd;
  apr_status_t res = APR_SUCCESS;;

  if (ctx->key == NULL)
    goto out;

  if ((res = apr_thread_mutex_lock(ctx->mtx)) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, res, ctx->f->r,
                  "apr_thread_mutex_lock failed");
  }

  ufd = (upload_formdata_t *)apr_array_push(ctx->formdata);
  ufd->key = ctx->key;
  ufd->val = ctx->val;
  ufd->ctype = ctx->ctype;

  if ((res = apr_thread_mutex_unlock(ctx->mtx)) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, res, ctx->f->r,
                  "apr_thread_mutex_unlock failed");
  }

  
#ifdef DEBUG
  ap_log_rerror(APLOG_MARK, APLOG_DEBUG, 0, ctx->f->r, 
                "key=\"%s\", val=\"%s\" ctype=\"%s\"",
                ctx->key, ctx->val, ctx->ctype);
#endif

out:
  ctx->key = NULL;
  ctx->val = NULL;
  ctx->filename = NULL;
  ctx->ctype = NULL;
  ctx->file = NULL;

  return;
}

static apr_status_t
wait_workers(upload_ctx_t *ctx) 
{
  request_rec *r = ctx->f->r;
  upload_conf_t *conf = ap_get_module_config(r->per_dir_config, &upload_module);
  apr_time_t timeout = apr_time_from_sec(conf->timeout);
  apr_status_t res = APR_SUCCESS;
  int finished;

  if (ctx->workers == 0)
    return res;

  if ((res = apr_thread_mutex_lock(ctx->mtx)) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, res, ctx->f->r,
                  "apr_thread_mutex_lock failed");
    goto out;
  }
  
  do {                    
    res = apr_thread_cond_timedwait(ctx->cond, ctx->mtx, timeout);
    if (res != APR_SUCCESS && res != APR_TIMEUP) {
      ap_log_rerror(APLOG_MARK, APLOG_ERR, res, ctx->f->r,
                    "apr_thread_cond_wait failed");
      goto out;
    }
                          
    finished = (ctx->workers == 0);
                   
    if (!finished) {      
      ap_log_rerror(APLOG_MARK, APLOG_WARNING, res, ctx->f->r,
                    "%d file workers left", ctx->workers);
      if (res == APR_TIMEUP) 
        goto out;
    }                 
  } while (!finished);
                          
  if ((res = apr_thread_mutex_unlock(ctx->mtx)) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, res, ctx->f->r,
                  "apr_thread_mutex_unlock failed");
    goto out;             
  }    

#ifdef DEBUG
  if (finished)
    ap_log_rerror(APLOG_MARK, APLOG_DEBUG, res, ctx->f->r,
                  "file workers done");
#endif

out:
  return res;
}

static char *
_replace_content(upload_ctx_t *ctx)
{
  char *buf = "";
  upload_formdata_t *ufd;
  apr_status_t res = APR_SUCCESS;;
  int i;

  if ((res = apr_thread_mutex_lock(ctx->mtx)) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, res, ctx->f->r,
                  "apr_thread_mutex_lock failed");
  }

  /*
   * We cannot afford an empty brigade, because the PHP POST
   * handler considers this as end of transaction. If there 
   * is nothing to output, which happens during big files, 
   * send padding. This will require the removal of
   * mod_security's CRS rule 200004 for unmatched boundary.
   */
  if (ctx->formdata->nelts == 0) {
    buf = apr_pstrcat(ctx->pool, "--", ctx->boundary, "\r\n", NULL);
  }
 
  for (i = 0; i < ctx->formdata->nelts; i++) {
    char *cdisp;
    char *ctype;

    ufd = (upload_formdata_t *)(ctx->formdata->elts) + i;

    cdisp = apr_pstrcat(ctx->pool, "Content-Disposition: form-data; name=\"", 
                        ufd->key, "\"", NULL);

    if (ufd->ctype != NULL)
      ctype = apr_pstrcat(ctx->pool, "Content-Type: ",
                          ufd->ctype, "\r\n", NULL);
    else
      ctype = "";

    buf = apr_pstrcat(ctx->pool, buf,
                "--", ctx->boundary, "\r\n",
                cdisp, "\r\n",
                ctype, 
                "\r\n",
                ufd->val, "\r\n",
                NULL);
  }

  apr_array_clear(ctx->formdata);

  if ((res = apr_thread_mutex_unlock(ctx->mtx)) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, res, ctx->f->r,
                  "apr_thread_mutex_unlock failed");
  }

  return buf;
}


static apr_status_t
replace_content_final(apr_bucket_brigade *bb, upload_ctx_t *ctx)
{
  char *buf;
  apr_bucket *b;
  apr_size_t len;
  apr_status_t res = APR_SUCCESS;

  if ((res = wait_workers(ctx)) != APR_SUCCESS)
    goto out;

  buf = _replace_content(ctx);
  buf = apr_pstrcat(ctx->pool, buf, "--", ctx->boundary, "--\r\n", NULL);
  len = strlen(buf);

  b = apr_bucket_pool_create(buf, len, ctx->pool, bb->bucket_alloc);
  APR_BRIGADE_INSERT_TAIL(bb, b);
  APR_BRIGADE_INSERT_TAIL(bb, apr_bucket_eos_create(bb->bucket_alloc));

out:
  ap_remove_input_filter(ctx->f);
  return res;
}

static apr_status_t
replace_content(apr_bucket_brigade *bb, upload_ctx_t *ctx)
{
  char *buf = "";
  apr_bucket *b;
  apr_size_t len;
  apr_status_t res = APR_SUCCESS;
  static int i = 0;

  res = apr_brigade_cleanup(bb);
  if (res != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK,APLOG_ERR, res, ctx->f->r,
                  "apr_brigade_cleanup failed");
    goto out;
  } 

  buf = _replace_content(ctx);
  len = strlen(buf);

  b = apr_bucket_pool_create(buf, len, ctx->pool, bb->bucket_alloc);
  APR_BRIGADE_INSERT_TAIL(bb, b);

out:
  if (res != APR_SUCCESS)
    ap_remove_input_filter(ctx->f);

  return res;
}

static inline int
is_crlf(const char *data, apr_size_t data_len)
{
  return (data_len == 2 && data[0] == '\r' && data[1] == '\n');
}


static const char *
print_state(upload_state_t state)
{
  switch (state) {
  case start: return "start";
  case header: return "header";
  case value: return "value";
  case file: return "file";
  case end: return "end";
  default: return "unknown";
  }
  return "error";
}

static void
search_boundary(upload_ctx_t *ctx, apr_bucket *b,
                const char *data, apr_size_t *data_len)
{
  char *nl;
  const char *d = data;
  apr_size_t dlen = *data_len;

  while ((nl = memchr(d, APR_ASCII_LF, dlen)) != NULL) {
    const char *nextline = nl + 1;
    apr_ssize_t nextline_len;

    nextline_len = dlen - (d - nl);

    if (nextline_len >= ctx->boundary_end_len
        && strncmp(nextline, ctx->boundary_end,
                     ctx->boundary_end_len) == 0) {

      apr_bucket_split(b, nl - data + 1);
      *data_len = nl - data + 1 - 2; /* -2 to remove trailing CRLF */
      ctx->state = start;
      break;
    }
  
    if (nextline_len >= ctx->boundary_part_len
        && strncmp(nextline, ctx->boundary_part,
                   ctx->boundary_part_len) == 0) {
      apr_bucket_split(b, nl - data + 1);
      *data_len = nl - data + 1 - 2; /* -2 to remove trailing CRLF */
      ctx->state = start;
      break;
    }

    d = nl + 1;
    dlen = *data_len - (d - data);
  }

  return; 
}

static apr_status_t
fetch_header_line(upload_ctx_t *ctx, apr_bucket *b,
                  const char **data, apr_size_t *data_len)
{
  const char *nl;
  apr_off_t newdata_offset = ctx->saved_len;
  apr_size_t newdata_len = *data_len;
  apr_status_t res = APR_SUCCESS;

  if (ctx->saved_len + newdata_len > UPLOAD_MAX_HEADER_LEN)
    newdata_len = UPLOAD_MAX_HEADER_LEN - ctx->saved_len;

  if (newdata_len == 0) {
    res = APR_EINVAL;
    ap_log_rerror(APLOG_MARK,APLOG_ERR, res, ctx->f->r,
		  "Parse error: header line too long");
    goto out;
  }

  memcpy(ctx->saved + ctx->saved_len, *data, newdata_len);
  ctx->saved_len += newdata_len;
  if ((nl = memchr(ctx->saved, APR_ASCII_LF, ctx->saved_len)) == NULL) {
    res = APR_NOTFOUND;
    goto out;
  }

  *data_len = nl - ctx->saved + 1;
  *data = ctx->saved;

  apr_bucket_split(b, *data_len - newdata_offset);

  ctx->saved_len = 0;
out:
  return res;
}

static apr_status_t 
upload_filter(ap_filter_t *f, apr_bucket_brigade *bb_in,
              ap_input_mode_t mode, apr_read_type_e block, apr_off_t nbytes) 
{
  upload_ctx_t *ctx = (upload_ctx_t *) f->ctx;
  apr_bucket *b = NULL;
  int res = APR_SUCCESS;

  res = ap_get_brigade(f->next, bb_in, mode, block, nbytes);
  if (res != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK,APLOG_ERR, res, f->r,
		  "ap_get_brigade failed in %s. Possible cause is "
		  "LimitRequestBody = %d", __func__,
		  ap_get_limit_req_body(f->r));
    goto out;
  }

  for (b = APR_BRIGADE_FIRST(bb_in);
        b != APR_BRIGADE_SENTINEL(bb_in) && !APR_BUCKET_IS_EOS(b);
        b = APR_BUCKET_NEXT(b)) {
    const char *data;
    apr_size_t data_len;
    apr_size_t boundary_len;
    apr_off_t brigade_resid;

    if ((res = apr_brigade_length(bb_in, 0, &brigade_resid)) != APR_SUCCESS)
      ap_log_rerror(APLOG_MARK,APLOG_ERR, res, f->r,
                    "apr_brigade_length failed");

    if (brigade_resid == 0 || ctx->state == end)
      break;
   
    res = apr_bucket_read(b, &data, &data_len, APR_BLOCK_READ);
    if (res != APR_SUCCESS) {
      ap_log_rerror(APLOG_MARK, APLOG_ERR, res, f->r,
                    "apr_bucket_read failed");
      goto out;
    }

    if (data == NULL || data_len == 0)
      break;

#if DEBUG2
    ap_log_rdata(APLOG_MARK,APLOG_ERR, f->r, print_state(ctx->state),
                 data, data_len < 512 ? data_len : 512, 0);
#endif

    if (ctx->state != file) {
      switch (res = fetch_header_line(ctx, b, &data, &data_len)) {
      case APR_SUCCESS:
	break;
      case APR_NOTFOUND:
	/*
	 * Not really an error, do not give up for it
	 */
	res = APR_SUCCESS;
	continue;
      default:
	goto out;
      }
    }

    switch (ctx->state) {
    case start:
      if (data_len < ctx->boundary_part_len ||
          strncmp(data, ctx->boundary_part, ctx->boundary_part_len) != 0) {
        res = EINVAL;
        ap_log_rerror(APLOG_MARK, APLOG_ERR, res, f->r,
                      "parse error: no first header");
        goto out;
      }

      if (data_len >= ctx->boundary_end_len && 
          strncmp(data, ctx->boundary_end, ctx->boundary_end_len) == 0) {
        ctx->state = end;
      } else {
        ctx->state = header;
      }
      break;

    case header:
      if (is_crlf(data, data_len)) {
        if (ctx->filename) {
          if ((res = new_file(ctx)) != APR_SUCCESS)
            goto out;
        }
        ctx->state = (ctx->file) ? file : value;
        break;
      }

      if ((res = set_header(ctx, data, data_len)) != APR_SUCCESS)
          goto out;
      break;

    case value:
      boundary_len = ctx->boundary_end_len;
      if (data_len >= boundary_len && 
          strncmp(data, ctx->boundary_end, boundary_len) == 0) {
        end_value(ctx);
        ctx->state = end;
        break;
      }

      boundary_len = ctx->boundary_part_len;
      if (data_len >= boundary_len &&
          strncmp(data, ctx->boundary_part, boundary_len) == 0) {
        end_value(ctx);
        ctx->state = header;
        break;
      }

      if ((res = set_value(ctx, data, data_len)) != APR_SUCCESS) 
        goto out;
     break;

    case file: 
     search_boundary(ctx, b, data, &data_len);

     if ((res = write_file(ctx, data, data_len, 0)) != APR_SUCCESS)
        goto out;

     if (ctx->state != file)
       end_file(ctx);

     break;
 
    case end:
      break;

    default:
        res = EINVAL;
        ap_log_rerror(APLOG_MARK, APLOG_ERR, res, f->r,
                      "parse error: unexpected ctx->state %d (%s)", 
                      ctx->state, print_state(ctx->state));
        goto out;
        break;
    }
  }

out:

  if (res == APR_SUCCESS)
    res = replace_content(bb_in, ctx);

  if (res != APR_SUCCESS)
    ap_remove_input_filter(f);

  if (ctx->state == end || (b != NULL && APR_BUCKET_IS_EOS(b))) {
    if ((res = replace_content_final(bb_in, ctx)) == APR_SUCCESS)
      ctx->note->done = 1;
  }

  return res;
}


static const command_rec upload_cmds[] = {
        AP_INIT_FLAG("UploadEnable", ap_set_flag_slot,
                      (void *)APR_OFFSETOF(upload_conf_t, enable), OR_ALL,
                      "Enable upload filter") ,
        AP_INIT_TAKE1("UploadDir", ap_set_string_slot,
                      (void *)APR_OFFSETOF(upload_conf_t, dir), OR_ALL,
                      "Set upload directory") ,
        AP_INIT_TAKE1("UploadPrefix", ap_set_string_slot,
                      (void *)APR_OFFSETOF(upload_conf_t, prefix), OR_ALL,
                      "Set upload form field prefix for files") ,
        AP_INIT_TAKE1("UploadCommit", set_commit, NULL, OR_ALL,
                      "Set upload commit mode (on, off, force)") ,
        AP_INIT_TAKE1("UploadFormSize", ap_set_int_slot,
                      (void *)APR_OFFSETOF(upload_conf_t, form_size), OR_ALL,
                      "Provide hint number of form fields") ,
        AP_INIT_TAKE1("UploadBlockSize", ap_set_int_slot,
                      (void *)APR_OFFSETOF(upload_conf_t, block_size), OR_ALL,
                      "Set file write block size in bytes") ,
        AP_INIT_TAKE1("UploadTimeout", ap_set_int_slot,
                      (void *)APR_OFFSETOF(upload_conf_t, timeout), OR_ALL,
                      "Set file write timeout in seconds") ,
        AP_INIT_FLAG("UploadCompatPHP", ap_set_flag_slot,
                     (void *)APR_OFFSETOF(upload_conf_t, compat_php), OR_ALL,
                     "PHP compatibility") ,
        {NULL}
};

static void 
upload_hooks(apr_pool_t *p) {
  ap_hook_insert_filter(upload_filter_insert, NULL, NULL, APR_HOOK_FIRST);
  ap_register_input_filter("UPLOAD", upload_filter,
                           upload_filter_init, AP_FTYPE_CONTENT_SET);

  ap_hook_insert_filter(upload_filter_insert_compatphp,
                        NULL, NULL, APR_HOOK_LAST);
#ifdef HAVE_PHP
  ap_register_input_filter("UPLOAD_COMPATPHP", upload_filter_compatphp,
                           NULL, AP_FTYPE_CONTENT_SET);
#endif /* HAVE_PHP */
  return;
}

module AP_MODULE_DECLARE_DATA upload_module = {
        STANDARD20_MODULE_STUFF,
        upload_create_dir_config,
        NULL,
        NULL,
        NULL,
        upload_cmds,
        upload_hooks
};
