/*
 *
 * j-chkmail - filtre de messagerie pour sendmail - MILTER
 *
 * Copyright (c) 2001, 2002 Ecole des Mines de Paris
 *
 *  Auteur     : Jose Marcio Martins da Cruz
 *               martins@paris.ensmp.fr
 *
 *  Historique :
 *  Creation     : janvier 2002
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <j-sys.h>

#if HAVE_SEARCH_H
#include <search.h>
#endif


#include "libmilter/mfapi.h"

#include "j-chkmail.h"

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */

#define      SZ_IP         64

#define STATIC_DIM_CACHE   1

#if STATIC_DIM_CACHE == 0
#define         DIM_HIST_T     0x4000
#define         DIM_RES_T  0x1000
#else
static size_t   DIM_HIST_T = 0x4000;
static size_t   DIM_RES_T = 0x1000;
#endif

typedef struct Hist_T {
  uint32_t        signature;
  time_t          date;
  char            host[SZ_IP];
  int             nb;
} Hist_T;


typedef struct Res_T {
  char            host[SZ_IP];
  int             nb;
  int             nbrt;
  int             conn_1m;
  int             conn_10m;
  int             conn_1h;
  int             conn_6h;
  int             rcpt_1m;
  int             rcpt_10m;
  int             rcpt_1h;
  int             rcpt_6h;
} Res_T;

typedef struct HStat_T {
  time_t          date;
  int             nb;
} HStat_T;


/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */

static unsigned int current_short_throttle = 0;

unsigned int    throttle_interval = 60; /* 1 minute */
unsigned int    throttle_window = 600;  /* 10 minutes */


/*
  DIM_HI    number of seconds of connection counting
  DIM_SHFT  log2(DIM_RAP)
  DIM_LO    number of normalised minutes of counting
  DIM_RAP   number of seconds in a normalised minute
*/
#define DIM_HI            1024
#define DIM_SHFT             6
#define DIM_LO            (DIM_HI >> DIM_SHFT)
#define DIM_RAP           (DIM_HI / DIM_LO)


static struct {
  Hist_T         *conn;
  long            cptr;
  Res_T          *conn_res;
  size_t          conn_nb;
  int             conn_rate;

  Hist_T         *rcpt;
  long            rptr;
  Res_T          *rcpt_res;
  size_t          rcpt_nb;
  int             rcpt_rate;

  time_t          last;

  pthread_mutex_t mutex;

  HStat_T         sec[DIM_HI];
  HStat_T         min[DIM_LO];

} hdata = {
NULL, 0, NULL, 0, 0, NULL, 0, NULL, 0, 0, 0, PTHREAD_MUTEX_INITIALIZER};

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
static void     add_global_throttle_entry (time_t);
void            update_global_throttle (time_t);

double          poisson_upper_bound (double, double);

static int      res_t_cmp_by_value (const void *, const void *);
static int      res_t_cmp_by_addr (const void *, const void *);

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
unsigned int
throttle_get_short ()
{
  return current_short_throttle;
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
unsigned int
throttle_get_long ()
{
  return hdata.conn_rate;
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
bool
throttle_init (t_max)
     int             t_max;
{
  pthread_mutex_lock (&hdata.mutex);

  /* Initialisation of connection data */
  if (hdata.conn == NULL)
    hdata.conn = (Hist_T *) malloc (DIM_HIST_T * sizeof (Hist_T));
  if (hdata.conn != NULL) {
    memset (hdata.conn, 0, DIM_HIST_T * sizeof (Hist_T));
    hdata.cptr = 0;
  } else {
    syslog (LOG_ERR, "%s : Error malloc conn array : %s", J_FUNCTION, strerror (errno));

    pthread_mutex_unlock (&hdata.mutex);
    return FALSE;
  }

  /* Initialisation of host throttle results array */
  if (hdata.conn_res == NULL)
    hdata.conn_res = (Res_T *) malloc (DIM_RES_T * sizeof (Res_T));
  if (hdata.conn_res != NULL) {
    memset (hdata.conn_res, 0, DIM_RES_T * sizeof (Res_T));
    hdata.conn_nb = 0;
  } else {
    syslog (LOG_ERR, "%s : Error malloc host_throttle array : %s",
            J_FUNCTION, strerror (errno));

    pthread_mutex_unlock (&hdata.mutex);
    return FALSE;
  }

  /* Initialisation of recipients data */
  if (hdata.rcpt == NULL)
    hdata.rcpt = (Hist_T *) malloc (DIM_HIST_T * sizeof (Hist_T));

  if (hdata.rcpt != NULL) {
    memset (hdata.rcpt, 0, DIM_HIST_T * sizeof (Hist_T));
    hdata.rptr = 0;
  } else {
    syslog (LOG_ERR, "%s : Error malloc rcpt array : %s", J_FUNCTION, strerror (errno));

    pthread_mutex_unlock (&hdata.mutex);
    return FALSE;
  }

  /* Initialisation of rcpt throttle results array */
  if (hdata.rcpt_res == NULL)
    hdata.rcpt_res = (Res_T *) malloc (DIM_RES_T * sizeof (Res_T));
  if (hdata.rcpt_res != NULL) {
    memset (hdata.rcpt_res, 0, DIM_RES_T * sizeof (Res_T));
    hdata.rcpt_nb = 0;
  } else {
    syslog (LOG_ERR, "%s : Error malloc rcpt_throttle array : %s",
            J_FUNCTION, strerror (errno));

    pthread_mutex_unlock (&hdata.mutex);
    return FALSE;
  }

  /* Initialisation of connection rate statistics */
  memset (hdata.sec, 0, sizeof (hdata.sec));
  memset (hdata.min, 0, sizeof (hdata.min));

  pthread_mutex_unlock (&hdata.mutex);

  return TRUE;
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
void
throttle_free ()
{
  if (hdata.conn != NULL)
    free (hdata.conn);
  hdata.conn = NULL;

  if (hdata.rcpt != NULL)
    free (hdata.rcpt);
  hdata.rcpt = NULL;

  if (hdata.conn_res != NULL)
    free (hdata.conn_res);
  hdata.conn_res = NULL;

  if (hdata.rcpt_res != NULL)
    free (hdata.rcpt_res);
  hdata.rcpt_res = NULL;
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
int
throttle_resize (sza, szb)
     size_t          sza;
     size_t          szb;
{
  throttle_free ();
  return throttle_init (10000);
}



/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
int
throttle_add_host_entry (host, t)
     char           *host;
     time_t          t;
{
  Res_T           p;
  Res_T          *ptr = NULL;
  int             lock_res;

  if (hdata.conn == NULL || hdata.rcpt == NULL) {
    if (!throttle_init (10000)) {
      syslog (LOG_ERR, "%s : Can't continue : connection cache null ptr", J_FUNCTION);
      return 0;
    }
  }

  if ((lock_res = pthread_mutex_lock (&hdata.mutex)) != 0)
    syslog (LOG_ERR, "%s : pthread_mutex_lock : %s", J_FUNCTION, strerror (errno));

  hdata.conn[hdata.cptr].signature = SIGNATURE;
  snprintf (hdata.conn[hdata.cptr].host, sizeof (hdata.conn[hdata.cptr].host), "%s",
            host);
  hdata.conn[hdata.cptr].date = t;
  hdata.conn[hdata.cptr].nb = 0;
  hdata.cptr++;
  hdata.cptr %= DIM_HIST_T;

  /* update throttle table... */
  memset (&p, 0, sizeof (p));
  strcpy (p.host, host);

  ptr = bsearch (&p, hdata.conn_res, hdata.conn_nb, sizeof (Res_T), res_t_cmp_by_addr);
  if (ptr != NULL) {
    ptr->nbrt++;
  } else {
    strcpy (p.host, host);
    p.nb = 0;
    p.nbrt = 1;

    if (hdata.conn_nb < DIM_RES_T) {
      hdata.conn_res[hdata.conn_nb++] = p;

      qsort (hdata.conn_res, hdata.conn_nb, sizeof (Res_T), res_t_cmp_by_addr);
    }
  }

  current_short_throttle++;

  add_global_throttle_entry (t);

  if (lock_res == 0)
    pthread_mutex_unlock (&hdata.mutex);

  return (hdata.conn_rate + current_short_throttle);
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
int
throttle_check_host (host)
     char           *host;
{
  Res_T           p;
  Res_T          *t = NULL;
  int             lock_res = 0;

  if (host == NULL)
    return 0;
  memset (&p, 0, sizeof (p));
  strcpy (p.host, host);

  if (hdata.conn == NULL || hdata.rcpt == NULL) {
    if (!throttle_init (10000)) {
      syslog (LOG_ERR, "%s : Can't continue : connection cache null ptr", J_FUNCTION);
      return 0;
    }
  }

  if ((lock_res = pthread_mutex_lock (&hdata.mutex)) != 0)
    syslog (LOG_ERR, "%s : pthread_mutex_lock : %s", J_FUNCTION, strerror (errno));

  t = bsearch (&p, hdata.conn_res, hdata.conn_nb, sizeof (Res_T), res_t_cmp_by_addr);
  if (lock_res == 0)
    pthread_mutex_unlock (&hdata.mutex);

  if (t != NULL)
    return t->nb + t->nbrt;

  return 0;
}


/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
int
throttle_add_rcpt_entry (host, nb, t)
     char           *host;
     int             nb;
     time_t          t;
{
  int             lock_res = 0;

  if (hdata.conn == NULL || hdata.rcpt == NULL) {
    if (!throttle_init (10000)) {
      syslog (LOG_ERR, "%s : Can't continue : connection cache null ptr", J_FUNCTION);
      return 0;
    }
  }

  if ((lock_res = pthread_mutex_lock (&hdata.mutex)) != 0)
    syslog (LOG_ERR, "%s : pthread_mutex_lock : %s", J_FUNCTION, strerror (errno));

  hdata.rcpt[hdata.rptr].signature = SIGNATURE;
  snprintf (hdata.rcpt[hdata.rptr].host, sizeof (hdata.rcpt[hdata.rptr].host), "%s",
            host);
  hdata.rcpt[hdata.rptr].date = t;
  hdata.rcpt[hdata.rptr].nb = nb;

  hdata.rptr++;
  hdata.rptr %= DIM_HIST_T;
  if (lock_res == 0)
    pthread_mutex_unlock (&hdata.mutex);
  return 0;
}



/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
int
throttle_check_rcpt (host)
     char           *host;
{
  Res_T           p;
  Res_T          *t = NULL;
  int             lock_res = 0;

  if (host == NULL)
    return 0;
  memset (&p, 0, sizeof (p));
  strcpy (p.host, host);

  if (hdata.conn == NULL || hdata.rcpt == NULL) {
    if (!throttle_init (10000)) {
      syslog (LOG_ERR, "%s : Can't continue : connection cache null ptr", J_FUNCTION);
      return 0;
    }
  }

  if ((lock_res = pthread_mutex_lock (&hdata.mutex)) != 0)
    syslog (LOG_ERR, "%s : pthread_mutex_lock : %s", J_FUNCTION, strerror (errno));

  t = bsearch (&p, hdata.rcpt_res, hdata.rcpt_nb, sizeof (Res_T), res_t_cmp_by_addr);
  if (lock_res == 0)
    pthread_mutex_unlock (&hdata.mutex);
  if (t != NULL)
    return t->nb;
  return 0;
}



/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
int
throttle_update_table (w_width)
     time_t          w_width;
{
  int             i;
  time_t          now = time (NULL);
  Res_T           p;
  Res_T          *t;
  int             current = 0;
  int             lock_res = 0;

  static int      save_it = 0;

  if (hdata.conn == NULL || hdata.rcpt == NULL) {
    if (!throttle_init (10000)) {
      syslog (LOG_ERR, "%s : Can't continue : connection cache null ptr", J_FUNCTION);
      return 0;
    }
  }

  if ((lock_res = pthread_mutex_lock (&hdata.mutex)) != 0)
    syslog (LOG_ERR, "%s : pthread_mutex_lock : %s", J_FUNCTION, strerror (errno));

  /* Let's update rcpt throttle */
  current = 0;
  hdata.rcpt_nb = 0;
  memset (hdata.rcpt_res, 0, DIM_RES_T * sizeof (Res_T));

  for (i = 0; i < DIM_HIST_T; i++) {
    if ((strlen (hdata.rcpt[i].host) > 0) && (hdata.rcpt[i].date + w_width > now)) {
      memset (&p, 0, sizeof (p));
      strcpy (p.host, hdata.rcpt[i].host);

      if ((t = lfind (&p, hdata.rcpt_res, &hdata.rcpt_nb,
                      sizeof (Res_T), res_t_cmp_by_addr)) != NULL) {
        if (hdata.rcpt[i].date + 600 > now)
          t->nb += hdata.rcpt[i].nb;
        if (hdata.rcpt[i].date + 60 > now)
          t->rcpt_1m += hdata.rcpt[i].nb;
        if (hdata.rcpt[i].date + 600 > now)
          t->rcpt_10m += hdata.rcpt[i].nb;
        if (hdata.rcpt[i].date + 3600 > now)
          t->rcpt_1h += hdata.rcpt[i].nb;
        if (hdata.rcpt[i].date + 21600 > now)
          t->rcpt_6h += hdata.rcpt[i].nb;
      } else {
        if (hdata.rcpt_nb < DIM_RES_T) {

          if (hdata.rcpt[i].date + 600 > now)
            p.nb = hdata.rcpt[i].nb;
          if (hdata.rcpt[i].date + 60 > now)
            p.rcpt_1m = hdata.rcpt[i].nb;
          if (hdata.rcpt[i].date + 600 > now)
            p.rcpt_10m = hdata.rcpt[i].nb;
          if (hdata.rcpt[i].date + 3600 > now)
            p.rcpt_1h = hdata.rcpt[i].nb;
          if (hdata.rcpt[i].date + 21600 > now)
            p.rcpt_6h = hdata.rcpt[i].nb;

          hdata.rcpt_res[hdata.rcpt_nb++] = p;
        }
      }
    }
  }
  qsort (hdata.rcpt_res, hdata.rcpt_nb, sizeof (Res_T), res_t_cmp_by_addr);

  for (i = 0; i < hdata.rcpt_nb; i++) {
    current += hdata.rcpt_res[i].nb;
  }
  if (log_level > 15)
    syslog (LOG_INFO, "RCPT %d %d", hdata.rcpt_nb, current);
  hdata.rcpt_rate = current;

  /* Let's update connection throttle */
  current = 0;
  hdata.conn_nb = 0;
  memset (hdata.conn_res, 0, DIM_RES_T * sizeof (Res_T));

  for (i = 0; i < DIM_HIST_T; i++) {
    if ((strlen (hdata.conn[i].host) > 0) && (hdata.conn[i].date + w_width > now)) {
      memset (&p, 0, sizeof (p));
      strcpy (p.host, hdata.conn[i].host);

      if ((t = lfind (&p, hdata.conn_res, &hdata.conn_nb,
                      sizeof (Res_T), res_t_cmp_by_addr)) != NULL) {
        if (hdata.conn[i].date + 600 > now)
          t->nb++;
        if (hdata.conn[i].date + 60 > now)
          t->conn_1m++;
        if (hdata.conn[i].date + 600 > now)
          t->conn_10m++;
        if (hdata.conn[i].date + 3600 > now)
          t->conn_1h++;
        if (hdata.conn[i].date + 21600 > now)
          t->conn_6h++;
      } else {
        if (hdata.conn_nb < DIM_RES_T) {
          if (hdata.conn[i].date + 600 > now)
            p.nb = 1;
          if (hdata.conn[i].date + 60 > now)
            p.conn_1m = 1;
          if (hdata.conn[i].date + 600 > now)
            p.conn_10m = 1;
          if (hdata.conn[i].date + 3600 > now)
            p.conn_1h = 1;
          if (hdata.conn[i].date + 21600 > now)
            p.conn_6h = 1;

          hdata.conn_res[hdata.conn_nb++] = p;
        }
      }
    }
  }
  qsort (hdata.conn_res, hdata.conn_nb, sizeof (Res_T), res_t_cmp_by_addr);

  for (i = 0; i < DIM_HIST_T; i++) {
    if ((strlen (hdata.rcpt[i].host) > 0) && (hdata.rcpt[i].date + w_width > now)) {
      memset (&p, 0, sizeof (p));
      strcpy (p.host, hdata.rcpt[i].host);

      if ((t = lfind (&p, hdata.conn_res, &hdata.conn_nb,
                      sizeof (Res_T), res_t_cmp_by_addr)) != NULL) {
        if (hdata.rcpt[i].date + 60 > now)
          t->rcpt_1m += hdata.rcpt[i].nb;
        if (hdata.rcpt[i].date + 600 > now)
          t->rcpt_10m += hdata.rcpt[i].nb;
        if (hdata.rcpt[i].date + 3600 > now)
          t->rcpt_1h += hdata.rcpt[i].nb;
        if (hdata.rcpt[i].date + 21600 > now)
          t->rcpt_6h += hdata.rcpt[i].nb;

      } else {
#if 0
        if (hdata.conn_nb < DIM_RES_T) {
          if (hdata.rcpt[i].date + 60 > now)
            p.rcpt_1m = hdata.rcpt[i].nb;
          if (hdata.rcpt[i].date + 600 > now)
            p.rcpt_10m = hdata.rcpt[i].nb;
          if (hdata.rcpt[i].date + 3600 > now)
            p.rcpt_1h = hdata.rcpt[i].nb;
          if (hdata.rcpt[i].date + 21600 > now)
            p.rcpt_6h = hdata.rcpt[i].nb;

          hdata.conn_res[hdata.conn_nb++] = p;
        }
#endif
      }
    }
  }
  qsort (hdata.conn_res, hdata.conn_nb, sizeof (Res_T), res_t_cmp_by_addr);

  current = 0;
  for (i = 0; i < hdata.conn_nb; i++) {
    current += hdata.conn_res[i].conn_10m;
  }

  hdata.conn_rate = current;
  current_short_throttle = 0;

  if (lock_res == 0)
    pthread_mutex_unlock (&hdata.mutex);

  update_global_throttle (now);

  if (log_level > 15)
    syslog (LOG_INFO, "HOST %d %d", hdata.conn_nb, current);

  if (save_it) {
    throttle_save_table (NULL);
  }
  save_it = !save_it;

  return current;
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
#define      DT_SLEEP    60

static void    *
throttle_loop (data)
     void           *data;
{
  time_t          t_now, t_last;
  char            s[256];

  t_last = t_now = time (NULL);
  ctime_r (&t_now, s);
  syslog (LOG_WARNING, " throttle_loop : starting at %s", s);

  for (;;) {
    sleep (DT_SLEEP);
    t_now = time (NULL);

    if (t_last + DT_SLEEP > t_now)
      continue;

    throttle_update_table (throttle_window);

    if (log_level > 10) {
      t_now = time (NULL);
      ctime_r (&t_now, s);
      syslog (LOG_WARNING, " throttle_loop : %s", s);
    }
    t_last = t_now;
  }
}

void
throttle_update_thread ()
{
  pthread_t       tid;
  int             r;

  if ((r = pthread_create (&tid, NULL, throttle_loop, (void *) NULL)) != 0) {
    syslog (LOG_WARNING, "Couldn't launch throttle_loop thread");
  }

}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
void
throttle_log_table ()
{
  int             i;

  if (log_level >= 10)
    syslog (LOG_INFO, "*** THROTTLE TABLE");
  for (i = 0; i < hdata.conn_nb; i++) {
    if (log_level >= 10 && hdata.conn_res[i].nb > 5)
      syslog (LOG_INFO, " CONN THROTTLE : %-16s %5d",
              hdata.conn_res[i].host, hdata.conn_res[i].nb);
  }

  for (i = 0; i < hdata.rcpt_nb; i++) {
    if (log_level >= 10 && hdata.rcpt_res[i].nb > 10)
      syslog (LOG_INFO, " RCPT THROTTLE : %-16s %5d",
              hdata.rcpt_res[i].host, hdata.rcpt_res[i].nb);
  }
}


/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
void
throttle_save_table (filename)
     char           *filename;
{
#if 1
  char           *work_dir = cf_get_str (CF_WORKDIR);
#else
  char           *work_dir = J_WORKDIR;
#endif
  char            fname[256];
  int             fd;

  if (hdata.conn == NULL || hdata.rcpt == NULL) {
    if (!throttle_init (10000)) {
      syslog (LOG_ERR, "%s : Can't continue : connection cache null ptr", J_FUNCTION);
      return;
    }
  }

  if (filename == NULL)
    snprintf (fname, sizeof (fname), "%s/%s", work_dir, "j-conndata");
  else
    strlcpy (fname, filename, sizeof (fname));
  if ((fd = open (fname, O_WRONLY | O_CREAT | O_TRUNC, 00644)) >= 0) {
    if (write (fd, hdata.conn, DIM_HIST_T * sizeof (Hist_T)) !=
        DIM_HIST_T * sizeof (Hist_T)) {
      ;
    }
    close (fd);
  } else {
    ;
  }

  snprintf (fname, sizeof (fname), "%s/%s", work_dir, "j-rcptdata");
  if ((fd = open (fname, O_WRONLY | O_CREAT | O_TRUNC, 00644)) >= 0) {
    if (write (fd, hdata.rcpt, DIM_HIST_T * sizeof (Hist_T)) !=
        DIM_HIST_T * sizeof (Hist_T)) {
      ;
    }
    close (fd);
  } else {
    ;
  }
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
int
throttle_read_table (filename)
     char           *filename;
{
#if 1
  char           *work_dir = cf_get_str (CF_WORKDIR);
#else
  char           *work_dir = J_WORKDIR;
#endif
  char            fname[256];
  int             fd;
  int             i, imax;
  time_t          tmax;

#ifndef INT_MAX
#define INT_MAX    2147483647
#endif

  if (work_dir == NULL)
    work_dir = J_WORKDIR;

  if (hdata.conn == NULL || hdata.rcpt == NULL) {
    if (!throttle_init (10000)) {
      syslog (LOG_ERR, "%s : Can't continue : connection cache null ptr", J_FUNCTION);
      return 0;
    }
  }

  if (filename == NULL)
    snprintf (fname, sizeof (fname), "%s/%s", work_dir, "j-conndata");
  else
    strlcpy (fname, filename, sizeof (fname));

  if ((fd = open (fname, O_RDONLY)) >= 0) {
    if (read (fd, hdata.conn, DIM_HIST_T * sizeof (Hist_T)) !=
        DIM_HIST_T * sizeof (Hist_T)) {
      ;
    }
    close (fd);
  } else {
    ;
  }

  for (i = 0, imax = 0, tmax = 0; i < DIM_HIST_T; i++); {
    if (tmax == 0)
      tmax = hdata.conn[i].date;
    if (hdata.conn[i].date > tmax) {
      tmax = hdata.conn[i].date;
      imax = i;
    }
    add_global_throttle_entry (hdata.conn[i].date);
  }
  hdata.cptr = 0;
  if (imax > 0)
    hdata.cptr = imax + 1;
  hdata.cptr %= DIM_HIST_T;

  snprintf (fname, sizeof (fname), "%s/%s", work_dir, "j-rcptdata");
  if ((fd = open (fname, O_RDONLY)) >= 0) {
    if (read (fd, hdata.rcpt, DIM_HIST_T * sizeof (Hist_T)) !=
        DIM_HIST_T * sizeof (Hist_T)) {
      ;
    }
    close (fd);
  } else {
    ;
  }
  for (i = 0, imax = 0; i < DIM_HIST_T; i++); {
    if (tmax == 0)
      tmax = hdata.rcpt[i].date;
    if (hdata.rcpt[i].date > tmax) {
      tmax = hdata.rcpt[i].date;
      imax = i;
    }
  }
  hdata.rptr = 0;
  if (imax > 0)
    hdata.rptr = imax + 1;
  hdata.rptr %= DIM_HIST_T;

#if 0
  throttle_update_table ();
#endif

  return 0;
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
void
throttle_print_table (allhosts, verbose, hostnames)
     int             allhosts;
     int             verbose;
     int             hostnames;
{
  int             i;
  int             v;

  char            nodename[128];

  unsigned long   tmin = 0, tmax = 0;
  unsigned long   hh, mm, ss;

  time_t          now;
  char            s[256];

  if (hdata.conn == NULL || hdata.rcpt == NULL) {
    if (!throttle_init (10000)) {
      syslog (LOG_ERR, "%s : Can't continue : connection cache null ptr", J_FUNCTION);
      return;
    }
  }

  printf ("%-30s : %s\n", "Version", PACKAGE);
  now = time (NULL);
  ctime_r (&now, s);
  printf ("*** THROTTLE TABLE (units each 10 minutes) at %s\n", s);

  /* Connection cache processing */
  qsort (hdata.conn_res, hdata.conn_nb, sizeof (Res_T), res_t_cmp_by_value);
  for (v = 0, i = 0; i < hdata.conn_nb; i++)
    v += hdata.conn_res[i].nb;

  printf ("*** CONNECTIONS : %6d / 10 min (%d/%d entries)\n",
          v, hdata.conn_nb, DIM_RES_T);

  tmin = tmax = hdata.conn[0].date;
  for (v = 0, i = 0; i < DIM_HIST_T; i++) {
    if (hdata.conn[i].date != 0) {
      if (tmin == 0)
        tmin = hdata.conn[i].date;
      if (tmax == 0)
        tmax = hdata.conn[i].date;
      if (hdata.conn[i].date < tmin)
        tmin = hdata.conn[i].date;
      if (hdata.conn[i].date > tmax)
        tmax = hdata.conn[i].date;
      v++;
    }
  }

  ss = tmax - tmin;
  /* printf ("--> %ld %ld %ld \n", tmax, tmin, ss); */
  hh = ss / 3600;
  ss -= hh * 3600;
  mm = ss / 60;
  ss -= mm * 60;

  printf ("    HISTORY     : %3ld:%02ld:%02ld (%d/%d entries)\n",
          hh, mm, ss, v, DIM_HIST_T);

  /* Recipient connection cache processing */
  qsort (hdata.rcpt_res, hdata.rcpt_nb, sizeof (Res_T), res_t_cmp_by_value);
  for (v = 0, i = 0; i < hdata.rcpt_nb; i++)
    v += hdata.rcpt_res[i].nb;
  printf ("*** RECIPIENTS  : %6d / 10 min (%d/%d entries)\n",
          v, hdata.rcpt_nb, DIM_RES_T);

  tmin = tmax = hdata.rcpt[0].date;
  for (v = 0, i = 0; i < DIM_HIST_T; i++) {
    if (hdata.rcpt[i].date != 0) {
      if (tmin == 0)
        tmin = hdata.rcpt[i].date;
      if (tmax == 0)
        tmax = hdata.rcpt[i].date;
      if (hdata.rcpt[i].date < tmin)
        tmin = hdata.rcpt[i].date;
      if (hdata.rcpt[i].date > tmax)
        tmax = hdata.rcpt[i].date;
      v++;
    }
  }

  ss = tmax - tmin;
  /* printf ("--> %ld %ld %ld \n", tmax, tmin, ss); */
  hh = ss / 3600;
  ss -= hh * 3600;
  mm = ss / 60;
  ss -= mm * 60;

  printf ("    HISTORY     : %3ld:%02ld:%02ld (%d/%d entries)\n\n",
          hh, mm, ss, v, DIM_HIST_T);

  /* Let's print data... */
  if (allhosts) {
    printf ("                    -    CONNECTIONS   :    RECIPIENTS\n");
    printf ("   HOST ADDRESS     - 01m 10m 01h  06h : 01m 10m 01h  06h : HOST NAME\n");
    for (i = 0; i < hdata.conn_nb; i++) {
      if (hdata.conn_res[i].nb >= 1) {
        memset (nodename, 0, sizeof (nodename));
        if (hostnames)
          j_gethostbyipaddr (hdata.conn_res[i].host, nodename, sizeof (nodename));
        printf (" . %-16s - %3d %3d %3d %4d : %3d %3d %3d %4d : %s\n",
                hdata.conn_res[i].host,
                hdata.conn_res[i].conn_1m,
                hdata.conn_res[i].conn_10m,
                hdata.conn_res[i].conn_1h,
                hdata.conn_res[i].conn_6h,
                hdata.conn_res[i].rcpt_1m,
                hdata.conn_res[i].rcpt_10m,
                hdata.conn_res[i].rcpt_1h, hdata.conn_res[i].rcpt_6h, nodename);
      }
    }
  }
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */

static void
add_global_throttle_entry (t)
     time_t          t;
{
  int             i;

  i = t % DIM_HI;

  if (hdata.sec[i].date != t) {
    hdata.sec[i].date = t;
    hdata.sec[i].nb = 0;
  }
  hdata.sec[i].nb++;

  t >>= DIM_SHFT;
  i = t % DIM_LO;

  if (hdata.min[i].date != t) {
    hdata.min[i].date = t;
    hdata.min[i].nb = 0;
  }
  hdata.min[i].nb++;
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
static STATS    st_sec;
static STATS    st_min;
static time_t   st_last = 0;

void
update_global_throttle (t)
     time_t          t;
{
  int             i, j;
  time_t          t0, ti;

  if (t < DIM_HI)
    return;

  st_last = t;

  memset (hdata.min, 0, sizeof (hdata.min));

  t0 = t + 1 - DIM_HI;

  for (ti = t0; ti < t + 1; ti++) {
    i = ti % DIM_HI;
    if (hdata.sec[i].date != ti) {
      hdata.sec[i].date = ti;
      hdata.sec[i].nb = 0;
    }
    j = (ti - t0) >> DIM_SHFT;

    hdata.min[j].nb += hdata.sec[i].nb;
  }
  kstats_reset (&st_sec);
  kstats_reset (&st_min);

  for (i = 0; i < DIM_HI; i++)
    kstats_update (&st_sec, (double) hdata.sec[i].nb);

  for (i = 0; i < DIM_LO; i++)
    kstats_update (&st_min, (double) hdata.min[i].nb);

  if (log_level > 10)
    syslog (LOG_INFO,
            "THROTTLE : short=[%7.3f/%7.3f] long=[%7.3f/%7.3f] (mean/std dev)",
            kmean (&st_sec), kstddev (&st_sec), kmean (&st_min), kstddev (&st_min));

  update_throttle_dos ();
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
static bool     dos_current = FALSE;

bool
check_throttle_dos ()
{
  return dos_current;
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
#define        HOLD_TIME       120

static time_t   dos_start_time = 0;
static time_t   dos_last_log = 0;

#define  DOS_COEF      5.0

bool
update_throttle_dos ()
{
  double          mean, gmean;
  double          stddev;
  double          last_mean = (double) hdata.min[DIM_LO - 1].nb;
  int             i;
  STATS           stats;

  kstats_reset (&stats);

  pthread_mutex_lock (&hdata.mutex);
  for (i = 0; i < (DIM_LO - 1); i++)
    kstats_update (&stats, (double) hdata.min[i].nb);
  last_mean = (double) hdata.min[DIM_LO - 1].nb;
  pthread_mutex_unlock (&hdata.mutex);

  gmean = mean = kmean (&stats);
  stddev = kstddev (&stats);

  /* shall see this again later */
  dos_current = FALSE;

  if (gmean < 10)
    gmean = 10;

  if (last_mean > gmean + DOS_COEF * sqrt (gmean)) {
    time_t          now = time (NULL);

    if (!dos_current)
      dos_start_time = now;

    if ((dos_last_log == (time_t) 0) && (dos_last_log + HOLD_TIME < now)) {
      syslog (LOG_INFO, "*** DoS - THROTTLE : %7.3f %7.3f %7.3f",
              mean, stddev, last_mean);
      dos_last_log = now;
    }
    dos_current = TRUE;
    return TRUE;
  } else {
    if (dos_last_log != (time_t) 0) {
      syslog (LOG_INFO, "*** DoS - THROTTLE : END");
      dos_last_log = (time_t) 0;
    }
    dos_start_time = (time_t) 0;
  }

  return FALSE;
}


/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
void
log_throttle_stats ()
{
  double          mean = kmean (&st_min);
  double          stddev = kstddev (&st_min);

  double          last_mean = (double) hdata.min[DIM_LO - 1].nb;

  syslog (LOG_INFO, "THROTTLE STAT : %7.3f %7.3f %7.3f", mean, stddev, last_mean);
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
double
poisson_upper_bound (lambda, prob)
     double          lambda;
     double          prob;
{
  double          sum, tmp;
  int             i = 0;

  sum = tmp = exp (-lambda);

  while (sum < prob) {
    i++;
    tmp *= lambda / i;
    sum += tmp;
  }
  printf (" i : %3d - sum : %7.5f\n", i, sum);
  return i;
}


/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
static int
res_t_cmp_by_addr (a, b)
     const void     *a;
     const void     *b;
{
  Res_T          *ta = (Res_T *) a;
  Res_T          *tb = (Res_T *) b;

  return (strcmp (ta->host, tb->host));
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
static int
res_t_cmp_by_value (a, b)
     const void     *a;
     const void     *b;
{
  Res_T          *ta = (Res_T *) a;
  Res_T          *tb = (Res_T *) b;

  return (tb->nb - ta->nb);
}
