/*
 *
 * j-chkmail - filtre de messagerie pour sendmail - MILTER
 *
 * Copyright (c) 2001, 2002 Ecole des Mines de Paris
 *
 *  Auteur     : Jose Marcio Martins da Cruz
 *               martins@paris.ensmp.fr
 *
 *  Historique :
 *  Creation     : janvier 2002
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <j-sys.h>

#include "j-chkmail.h"

static char        *j_state_file = J_STATE_FILE;

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
j_stats             m_stats;

static pthread_mutex_t st_mutex = PTHREAD_MUTEX_INITIALIZER;


/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
void
log_stats (log, dump)
     int                 log;
     int                 dump;
{
  /* lock record */
  int                 lock_res = 0;
  time_t              now = time (NULL);

  char               *stats_file = cf_get_str (CF_STATS_FILE);

  if (log) {
    syslog (LOG_INFO, "STATS :  CONN  MSGS    TO  EXES");
    if ((lock_res = pthread_mutex_lock (&st_mutex)) != 0)
      syslog (LOG_ERR, "%s : pthread_mutex_lock : %s", J_FUNCTION,
              strerror (errno));
    syslog (LOG_INFO,
            "STATS : %5ld %5ld %5ld %5ld %5ld %5ld %5ld %5ld %5ld %5ld %5ld",
            m_stats.glob.value[STAT_CONNECT], m_stats.glob.value[STAT_MSGS],
            m_stats.glob.value[STAT_ENVTO], m_stats.glob.value[STAT_XFILES],
            m_stats.glob.value[STAT_FILES], m_stats.glob.value[STAT_LUSERS],
            m_stats.glob.value[STAT_NO_FROM_HEADERS],
            m_stats.glob.value[STAT_NO_TO_HEADERS],
            m_stats.glob.value[STAT_OUTLOOK], m_stats.glob.value[STAT_ABORT],
            m_stats.glob.value[STAT_CLOSE]);
    if (lock_res == 0)
      pthread_mutex_unlock (&st_mutex);
  }

  if (dump && stats_file && strcasecmp (stats_file, "NONE") != 0) {
    char                line[1024];

    int                 fd;

    if ((fd = open (stats_file, O_WRONLY | O_CREAT | O_APPEND, 0644)) >= 0) {
      if ((lock_res = pthread_mutex_lock (&st_mutex)) != 0)
        syslog (LOG_ERR, "%s : pthread_mutex_lock : %s", J_FUNCTION,
                strerror (errno));
      sprintf (line,
               "------ %12ld %5ld %5ld %5ld %5ld %5ld %5ld %5ld %5ld %5ld %5ld %5ld %5ld %5ld %5ld %5ld %5ld %5ld\n",
               (long) now, m_stats.glob.value[STAT_CONNECT],
               m_stats.glob.value[STAT_MSGS], m_stats.glob.value[STAT_ENVTO],
               m_stats.glob.value[STAT_XFILES],
               m_stats.glob.value[STAT_FILES],
               m_stats.glob.value[STAT_LUSERS],
               m_stats.glob.value[STAT_NO_FROM_HEADERS],
               m_stats.glob.value[STAT_NO_TO_HEADERS],
               m_stats.glob.value[STAT_OUTLOOK],
               m_stats.glob.value[STAT_ABORT], m_stats.glob.value[STAT_CLOSE],
               m_stats.glob.value[STAT_RESOLVE_FAIL],
               m_stats.glob.value[STAT_RESOLVE_FORGED],
               m_stats.glob.value[STAT_MAX_RCPT],
               m_stats.glob.value[STAT_THROTTLE],
               m_stats.glob.value[STAT_NO_SUBJECT_HEADER],
               m_stats.glob.value[STAT_NO_HEADERS]);
      if (lock_res == 0)
        pthread_mutex_unlock (&st_mutex);
      if (write (fd, line, strlen (line)) != strlen (line)) {
        syslog (LOG_ERR, "Error writing stats file : %s %s",
                stats_file, strerror (errno));
      }
      close (fd);
    } else
      syslog (LOG_ERR, " log_stats : %s : %s", stats_file, strerror (errno));
  }
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
void
stats_reset ()
{
  time_t              now = time (NULL);
  int                 lock_res = 0;

  if ((lock_res = pthread_mutex_lock (&st_mutex)) != 0)
    syslog (LOG_ERR, "%s : pthread_mutex_lock : %s", J_FUNCTION,
            strerror (errno));

  memset (&m_stats, 0, sizeof (m_stats));
  snprintf (m_stats.version, sizeof (m_stats.version), "%s", VERSION);

  m_stats.glob.start = now;
  m_stats.proc.start = now;

  if (lock_res == 0)
    pthread_mutex_unlock (&st_mutex);
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
void
stats_inc (which, n)
     int                 which;
     long                n;
{
  int                 lock_res = 0;

  if (which < 0 || which > DIM_STATS)
    return;

  if ((lock_res = pthread_mutex_lock (&st_mutex)) != 0)
    syslog (LOG_ERR, "%s : pthread_mutex_lock : %s", J_FUNCTION,
            strerror (errno));

  switch (which) {
    case STAT_BYTES:
      m_stats.glob.value[STAT_BYTES] += (n >> 6);
      m_stats.proc.value[STAT_BYTES] += (n >> 6);
      break;
    default:
      m_stats.glob.value[which] += n;
      m_stats.proc.value[which] += n;
      break;
  }

  if (lock_res == 0)
    pthread_mutex_unlock (&st_mutex);
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
void
save_state ()
{
  int                 fd;
  time_t              now = time (NULL);
  char               *state_file = cf_get_str (CF_STATE_FILE);
  int                 lock_res = 0;

  if (state_file != NULL)
    j_state_file = state_file;

  if (j_state_file == NULL || strlen (j_state_file) == 0) {
    syslog (LOG_ERR, "undefined state file");
    return;
  }

  if ((fd = open (j_state_file, O_WRONLY | O_CREAT, 00644)) >= 0) {
    if ((lock_res = pthread_mutex_lock (&st_mutex)) != 0)
      syslog (LOG_ERR, "%s : pthread_mutex_lock : %s", J_FUNCTION,
              strerror (errno));
    m_stats.last_save = now;
    if (lock_res == 0)
      pthread_mutex_unlock (&st_mutex);
    if (write (fd, &m_stats, sizeof (m_stats)) != sizeof (m_stats))
      syslog (LOG_ERR, "error writing %s file : %s", j_state_file,
              strerror (errno));
    close (fd);
  } else
    syslog (LOG_ERR, "Error opening %s file : %s", j_state_file,
            strerror (errno));
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
void
read_state ()
{
  int                 fd;
  int                 lock_res = 0;
  char               *state_file = cf_get_str (CF_STATE_FILE);

  if (state_file != NULL)
    j_state_file = state_file;

  if (j_state_file == NULL || strlen (j_state_file) == 0) {
    syslog (LOG_ERR, "undefined state file");
    return;
  }
  if ((fd = open (j_state_file, O_RDONLY)) >= 0) {
    if ((lock_res = pthread_mutex_lock (&st_mutex)) != 0)
      syslog (LOG_ERR, "%s : pthread_mutex_lock : %s", J_FUNCTION,
              strerror (errno));
    if (read (fd, &m_stats, sizeof (m_stats)) != sizeof (m_stats))
      syslog (LOG_ERR, "error reading %s file : %s", j_state_file,
              strerror (errno));
    if (lock_res == 0)
      pthread_mutex_unlock (&st_mutex);
    close (fd);
  } else
    syslog (LOG_ERR, "Error opening %s file : %s", j_state_file,
            strerror (errno));
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
void
init_proc_state ()
{
  time_t              now = time (NULL);
  int                 lock_res = 0;

  read_state ();
  /* save_state (); */

  if ((lock_res = pthread_mutex_lock (&st_mutex)) != 0)
    syslog (LOG_ERR, "%s : pthread_mutex_lock : %s", J_FUNCTION,
            strerror (errno));

  memset (&m_stats.proc, 0, sizeof (m_stats.proc));
  m_stats.proc.start = now;
  if (m_stats.glob.start == 0)
    m_stats.glob.start = now;

  if (lock_res == 0)
    pthread_mutex_unlock (&st_mutex);

  save_state ();
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
void
reset_state ()
{
  time_t              now = time (NULL);
  int                 lock_res = 0;

  if ((lock_res = pthread_mutex_lock (&st_mutex)) != 0)
    syslog (LOG_ERR, "%s : pthread_mutex_lock : %s", J_FUNCTION,
            strerror (errno));

  memset (&m_stats, 0, sizeof (m_stats));

  snprintf (m_stats.version, sizeof (m_stats.version), "%s", VERSION);
  m_stats.glob.start = now;
  m_stats.proc.start = now;

  if (lock_res == 0)
    pthread_mutex_unlock (&st_mutex);
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
int
print_state ()
{
  int                 fd;
  j_stats             st;

  char               *state_file = cf_get_str (CF_STATE_FILE);

  if (state_file != NULL)
    j_state_file = state_file;

  if (j_state_file == NULL || strlen (j_state_file) == 0) {
    printf ("undefined state file\n");
    return -1;
  }
  if ((fd = open (j_state_file, O_RDONLY)) >= 0) {
    if (read (fd, &st, sizeof (st)) == sizeof (st)) {
      char                out[256];

      close (fd);

      printf ("%-30s : %s\n", "Version", PACKAGE);

#if 0
      cftime (out, NULL, &st.glob.start);
#endif
      printf ("%-30s : %s\n", "Start", out);
      printf ("%-30s : %ld\n", "# Start-up", st.glob.value[STAT_RESTART]);
      printf ("%-30s : %ld\n", "# Messages", st.glob.value[STAT_MSGS]);
      printf ("%-30s : %ld\n", "# Connect", st.glob.value[STAT_CONNECT]);
      printf ("%-30s : %ld\n", "# Abort", st.glob.value[STAT_ABORT]);
      printf ("%-30s : %ld\n", "# Close", st.glob.value[STAT_CLOSE]);
      printf ("%-30s : %ld\n", "# ENV RCPT", st.glob.value[STAT_ENVTO]);
      printf ("%-30s : %ld\n", "# Attached files", st.glob.value[STAT_FILES]);
      printf ("%-30s : %ld\n", "# X-Files", st.glob.value[STAT_XFILES]);
      printf ("%-30s : %ld\n", "# Virus", st.glob.value[STAT_VIRUS]);
      printf ("%-30s : %ld\n", "# Reject Local Users",
              st.glob.value[STAT_LUSERS]);
      printf ("%-30s : %ld\n", "# Reject Outlook",
              st.glob.value[STAT_OUTLOOK]);
      printf ("%-30s : %ld\n", "# Reject No RCPT Headers",
              st.glob.value[STAT_NO_TO_HEADERS]);
      printf ("%-30s : %ld\n", "# Reject No Senders ",
              st.glob.value[STAT_NO_FROM_HEADERS]);
      printf ("%-30s : %ld\n", "# Reject Exceed Max RCPT",
              st.glob.value[STAT_MAX_RCPT]);

      printf ("\n");
#if 0
      cftime (out, NULL, &st.proc.start);
#endif
      printf ("%-30s : %s\n", "Start", out);
      printf ("%-30s : %ld\n", "# Start-up", st.proc.value[STAT_RESTART]);
      printf ("%-30s : %ld\n", "# Messages", st.proc.value[STAT_MSGS]);
      printf ("%-30s : %ld\n", "# Connect", st.proc.value[STAT_CONNECT]);
      printf ("%-30s : %ld\n", "# Abort", st.proc.value[STAT_ABORT]);
      printf ("%-30s : %ld\n", "# Close", st.proc.value[STAT_CLOSE]);
      printf ("%-30s : %ld\n", "# ENV RCPT", st.proc.value[STAT_ENVTO]);
      printf ("%-30s : %ld\n", "# Attached files", st.proc.value[STAT_FILES]);
      printf ("%-30s : %ld\n", "# X-Files", st.proc.value[STAT_XFILES]);
      printf ("%-30s : %ld\n", "# Virus", st.proc.value[STAT_VIRUS]);
      printf ("%-30s : %ld\n", "# Reject Local Users",
              st.proc.value[STAT_LUSERS]);
      printf ("%-30s : %ld\n", "# Reject Outlook",
              st.proc.value[STAT_OUTLOOK]);
      printf ("%-30s : %ld\n", "# Reject No RCPT Headers",
              st.proc.value[STAT_NO_TO_HEADERS]);
      printf ("%-30s : %ld\n", "# Reject No Senders ",
              st.proc.value[STAT_NO_FROM_HEADERS]);
      printf ("%-30s : %ld\n", "# Reject Exceed Max RCPT",
              st.proc.value[STAT_MAX_RCPT]);
    } else {
      close (fd);
      printf ("Error reading %s file : %s\n", j_state_file, strerror (errno));
      return 1;
    }
  } else {
    printf ("Error opening %s file : %s\n", j_state_file, strerror (errno));
    return 1;
  }
  return 0;
}





/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */

typedef struct _code {
  int                 c_val;
  int                 order;
  char               *c_name;
} CODE;

static CODE         stat_names[] = {
  {STAT_RESTART, 0, "Filter launches"},
  {STAT_BYTES, 0, "KBytes transfered"},
  {STAT_MSGS, 0, "Messages"},
  {STAT_CONNECT, 0, "Connect"},
  {STAT_ABORT, 0, "Abort"},
  {STAT_CLOSE, 0, "Close"},
  {STAT_ENVTO, 0, "Cumulative # RCPT"},
  {STAT_FILES, 0, "Attached files"},
  {STAT_XFILES, 0, "Attached unsafe files"},
  {STAT_VIRUS, 0, "Attached virus"},
  {STAT_LUSERS, 0, "Msgs sent to intranet users"},
  {STAT_OUTLOOK, 0, "Msgs sent by Outlook"},
  {STAT_NO_TO_HEADERS, 0, "No To or CC headers"},
  {STAT_NO_FROM_HEADERS, 0, "No From headers"},
  {STAT_NO_SUBJECT_HEADER, 0, "No Subject header"},
  {STAT_NO_HEADERS, 0, "No headers at all"},
  {STAT_RESOLVE_FAIL, 0, "Gateways without DNS resolution"},
  {STAT_RESOLVE_FORGED, 0, "Gateways with forged DNS names"},
  {STAT_MAX_RCPT, 0, "Msgs exceeding allowed # of recipients"},
  {STAT_THROTTLE, 0, "Hosts exceeding allowed connection rate"},
  {STAT_SUBJECT_CONTENTS, 0, "Forbidden subject contents (regex)"},
  {STAT_BODY_CONTENTS, 0, "Forbiden body contents (regex)"},
  {STAT_BINARY, 0, "Binary encoded message body"},
  {STAT_BASE64, 0, "BASE 64 encoded message body"},
  {STAT_QUOTED_PRINTABLE, 0, "Quoted-printable message body"},
  {STAT_POLICY, 0, "Message violating site policy"},
  {-1, 0, NULL}
};


char               *
stats_title (code)
     int                 code;
{
  CODE               *p = stat_names;

  if (code < 0)
    return NULL;

  while (p->c_name) {
    if (code == p->c_val)
      return p->c_name;
    p++;
  }
  return NULL;
}
