/*
 *
 * j-chkmail - filtre de messagerie pour sendmail - MILTER
 *
 * Copyright (c) 2001, 2002 Ecole des Mines de Paris
 *
 *  Auteur     : Jose Marcio Martins da Cruz
 *               martins@paris.ensmp.fr
 *
 *  Historique :
 *  Creation     : janvier 2002
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <j-sys.h>

#include <libmilter/mfapi.h>

#include "j-chkmail.h"

#include "j-history.h"

#define  USE_SMFI_STOP    1

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */

static bool     flag_save = FALSE;
static bool     flag_reset = FALSE;
static bool     flag_exit = FALSE;
static bool     flag_log = FALSE;
static bool     flag_conf = FALSE;

static time_t   time2exit = 0;

RETSIGTYPE
j_filter_sig_handler (signo)
     int             signo;
{
  time_t          now = time (NULL);

  signal (signo, j_filter_sig_handler);
  if (log_level >= 20)
    syslog (LOG_DEBUG, "*** Received SIGNAL %d : %s", signo, ctime (&now));

  switch (signo) {
    case SIGHUP:
      syslog (LOG_NOTICE, " *** SIG_HUP ");
      break;
    case SIGUSR1:
      flag_log = TRUE;
      flag_save = TRUE;
      syslog (LOG_NOTICE, " *** Dump state command received");
      break;
    case SIGUSR2:
      flag_reset = TRUE;
      syslog (LOG_NOTICE, " *** Dump state command received");
      break;
    case SIGALRM:
#if 0
      alarm (DT_SIGALRM);
#endif
      break;
    default:
      syslog (LOG_WARNING, "%s : Undefined behavior for signal %d !",
              J_FUNCTION, signo);
      break;
  }
}


/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
static void
do_proc_save ()
{
  save_state ();
  throttle_save_table (NULL);
  resolve_tab_save (NULL);

  flag_save = FALSE;
}

static void
do_proc_log ()
{
  log_stats (cf_get_int (CF_LOG_COUNTERS), cf_get_int (CF_DUMP_COUNTERS));
  do_proc_save ();

  flag_log = FALSE;
}

static void
do_proc_reset ()
{
  reset_state ();
  do_proc_save ();

  flag_reset = FALSE;
}

static void
do_proc_exit ()
{
  do_proc_save ();

#if (USE_SMFI_STOP == 1) && defined(HAVE_SMFI_STOP)
  if (log_level > 10)
    syslog (LOG_WARNING, "%s : USING smfi_stop()", J_FUNCTION);
  smfi_stop ();
  remove_milter_unix_sock ();
#else
  if (log_level > 10)
    syslog (LOG_WARNING, "%s : USING kill(0, SIGTERM)", J_FUNCTION);
  remove_milter_unix_sock ();
  kill (0, SIGHUP);
#endif

  flag_exit = FALSE;
#if 0
  remove_milter_unix_sock ();
  exit (0);
#endif
}

static void
do_proc_conf ()
{
  do_proc_save ();
  kill (getpid (), SIGHUP);
  flag_conf = FALSE;
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
#define        DT_LOOP        DT_SIGALRM
#define        DT_SAVE        300
#define        DT_UPDATE       30
#define        DT_GUPDATE       5
#define        DT_COUNT_FD      5

#define        MAX_NB_INT      16

static void    *
periodic_tasks_loop (data)
     void           *data;
{
  char            s[256];

  time_t          t_now, t_last;
  time_t          t_loop, t_save, t_updt, t_stats, t_gt_updt, t_fd, t_conf;
  int             msg_ok = 0;

  int             nb_int = 0;

  int             DT_RELOAD = cf_get_int (CF_AUTO_RELOAD_TABLES);

  t_loop = t_save = t_updt = t_stats = t_gt_updt = t_fd = t_conf = 0;
  t_last = t_now = time (NULL);
  ctime_r (&t_now, s);
  if (log_level > 9)
    syslog (LOG_INFO, "*** Starting %s at %s", J_FUNCTION, s);

  for (;;) {
    t_last = t_now;
    sleep (DT_LOOP);
    t_now = time (NULL);

    if ((time2exit != 0) && ((t_now - time2exit) > 10)) {
      syslog (LOG_WARNING,
              "%s : why system continues to run ? Surely one Linux box...", J_FUNCTION);
      remove_milter_unix_sock ();
      kill (0, SIGTERM);
      exit (0);
    }

    if (t_now < (t_last + DT_LOOP)) {
      nb_int++;
      if ((nb_int > MAX_NB_INT) && ((nb_int % MAX_NB_INT) == 0))
        syslog (LOG_WARNING, "%s : Too much interrupts : %d", J_FUNCTION, nb_int);
      continue;
    }
    nb_int = 0;

    t_loop += t_now - t_last;
    t_save += t_now - t_last;
    t_updt += t_now - t_last;
    t_gt_updt += t_now - t_last;
    t_stats += t_now - t_last;
    t_fd += t_now - t_last;
    t_conf += t_now - t_last;

    if (log_level >= 15) {
      ctime_r (&t_now, s);
      syslog (LOG_DEBUG, "%s : %s", J_FUNCTION, s);
    }

    if (flag_conf) {
      do_proc_conf ();
    }

    if (flag_log) {
      do_proc_log ();
      t_stats = 0;
    }

    if (flag_save || ((DT_SAVE > 0) && (t_save >= DT_SAVE))) {
      do_proc_save ();
      t_save = 0;
    }

    if (flag_reset) {
      do_proc_reset ();
    }

    if (flag_exit) {
      do_proc_exit ();
      t_save = 0;
    }

    if (statistics_interval > 0 && t_stats >= statistics_interval) {
      do_proc_log ();
      log_throttle_stats ();
      t_stats = 0;
    }

    if ((DT_RELOAD > 0) && (t_conf >= DT_RELOAD)) {
      char           *fname;

      if (log_level >= 9)
        syslog (LOG_INFO, " %s : Reloading configuration tables...", J_FUNCTION);

      fname = cf_get_str (CF_REGEX_FILE);
      if (log_level > 10)
        syslog (LOG_INFO, " %s : Reloading %s file...", J_FUNCTION, fname);
      if (load_regex_table (fname) != 0)
        syslog (LOG_ERR, "%s : Unable to reload regex table", J_FUNCTION);

      fname = cf_get_str (CF_NETS_FILE);
      if (log_level > 10)
        syslog (LOG_INFO, " %s : Reloading %s file...", J_FUNCTION, fname);
      if (load_networks_table (fname) != 0)
        syslog (LOG_ERR, "%s : Unable to reload networks table", J_FUNCTION);

      fname = cf_get_str (CF_HOST_ACCESS_FILE);
      if (log_level > 10)
        syslog (LOG_INFO, " %s : Reloading %s file...", J_FUNCTION, fname);
      if (load_host_access_table (fname) != 0)
        syslog (LOG_ERR, "%s : Unable to reload host access table", J_FUNCTION);

      fname = cf_get_str (CF_USER_ACCESS_FILE);
      if (log_level > 10)
        syslog (LOG_INFO, " %s : Reloading %s file...", J_FUNCTION, fname);
      if (load_user_access_table (fname) != 0)
        syslog (LOG_ERR, "%s : Unable to reload user access table", J_FUNCTION);

      fname = cf_get_str (CF_USERS_FILE);
      if (log_level > 10)
        syslog (LOG_INFO, " %s : Reloading %s file...", J_FUNCTION, fname);
      if (load_local_users_table (fname) != 0)
        syslog (LOG_ERR, "%s : Unable to reload local users table", J_FUNCTION);

      fname = cf_get_str (CF_CW_FILE);
      if (log_level > 10)
        syslog (LOG_INFO, " %s : Reloading %s file...", J_FUNCTION, fname);
      if (load_classw_table (fname) != 0)
        syslog (LOG_ERR, "%s : Unable to reload class W table", J_FUNCTION);

      t_conf = 0;
    }

    if (DT_COUNT_FD > 0 && t_fd > DT_COUNT_FD) {
      int             nb = count_file_descriptors ();

      check_file_descriptors ();

      if (log_level >= 15)
        syslog (LOG_INFO, "%08lX: Nb of open files : %d", t_now, nb);
    }

    if (DT_UPDATE > 0 && t_updt >= DT_UPDATE) {
      int             current = 0;

      res_history_update (NULL, NULL, t_now, 600);

      resolve_tab_update (TRUE);
      current = throttle_update_table (throttle_window);
      if (log_level > 10)
        syslog (LOG_DEBUG, "%s : Current throttle = %d", J_FUNCTION, current);
      if (log_level > 15)
        throttle_log_table ();
      t_updt = 0;
    }

    if (DT_GUPDATE > 0 && t_gt_updt >= DT_GUPDATE) {
      update_global_throttle (t_now);
      check_throttle_dos ();
      t_gt_updt = 0;
    }

    if (1) {
      int             nb = 0;
      int             msg;

      nb = 0;
      while (!recv_msg_channel (pipe_filter, &msg, CHAN_CHILD)) {
        nb++;
        msg_ok = 0;
        if (log_level >= 20)
          syslog (LOG_DEBUG, "%s : FILTER - MSG RECV : %d", J_FUNCTION, msg);
        switch (msg) {
          case MSG_OK:
            break;
          case MSG_TERM:
            syslog (LOG_INFO, "SUPERVISOR said : QUIT !");
            flag_exit = TRUE;
#if (USE_SMFI_STOP == 1) && defined(HAVE_SMFI_STOP)
            if (log_level >= 10)
              syslog (LOG_WARNING, "%s : USING smfi_stop()", J_FUNCTION);
            smfi_stop ();
#else
            if (log_level >= 10)
              syslog (LOG_WARNING, "%s : USING kill(0, SIGTERM)", J_FUNCTION);
            kill (0, SIGTERM);
#endif
            if (time2exit == 0)
              time2exit = time (NULL);
            break;
          case MSG_CONF:
            flag_conf = TRUE;
            break;
          case MSG_RESET:
            syslog (LOG_NOTICE, " *** Reset state command received");
            flag_reset = TRUE;
            break;
          case MSG_DUMP:
            flag_log = TRUE;
            syslog (LOG_NOTICE, " *** Dump state command received");
            break;
          default:
            ;
        }
      }
      if (nb == 0) {
        msg_ok++;
        if (log_level >= 15)
          syslog (LOG_DEBUG, "%s : FILTER - MSG RECV : NULL : %d", J_FUNCTION, msg_ok);
      }

      if (!flag_exit && ((msg_ok > 3) || (getppid () == 1))) {
        syslog (LOG_WARNING, "%s : FILTER - SUPERVISOR DIED ???", J_FUNCTION);
        if (log_level >= 10)
          syslog (LOG_WARNING, "%s : NOW EXIT !", J_FUNCTION);
        flag_exit = TRUE;
        save_state ();
        throttle_save_table (NULL);
        resolve_tab_save (NULL);
#if (USE_SMFI_STOP == 1) && defined(HAVE_SMFI_STOP)
        if (log_level > 10)
          syslog (LOG_WARNING, "%s : USING smfi_stop()", J_FUNCTION);
        smfi_stop ();
#else
        if (log_level > 10)
          syslog (LOG_WARNING, "%s : USING kill(0, SIGTERM)", J_FUNCTION);
        kill (0, SIGHUP);
#endif
        if (time2exit == 0)
          time2exit = time (NULL);
      }
    }

  }
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
void
launch_periodic_tasks_thread ()
{
  pthread_t       tid;
  int             r;

  if (log_level > 9)
    syslog (LOG_INFO, "*** Starting %s ...", J_FUNCTION);

  if (log_level > 9)
    syslog (LOG_INFO, "    Tables will be updated by thread");
  if ((r = pthread_create (&tid, NULL, periodic_tasks_loop, (void *) NULL)) != 0) {
    syslog (LOG_WARNING,
            "%s : Couldn't launch periodic_tasks_loop thread : %s",
            J_FUNCTION, strerror (r));
  }
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
void
remove_milter_unix_sock (void)
{
  char           *sock_file;

  sock_file = cf_get_str (CF_UNIX_SOCK);

  if (sock_file != NULL && strlen (sock_file) > 0) {
    struct stat     buf;

    if (lstat (sock_file, &buf) == 0) {
      syslog (LOG_WARNING, "Removing SOCK_FILE : %s", sock_file);
      remove (sock_file);
    }
  }
}
