/*
 *
 * j-chkmail - filtre de messagerie pour sendmail - MILTER
 *
 * Copyright (c) 2001, 2002 Ecole des Mines de Paris
 *
 *  Auteur     : Jose Marcio Martins da Cruz
 *               martins@paris.ensmp.fr
 *
 *  Historique :
 *  Creation     : janvier 2002
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <j-sys.h>

#include "j-chkmail.h"

/*****************************************************************************
 *                                                                           * 
 *                                                                           *
 *****************************************************************************/
#define        SZ_NEW       0x10000
#define        SZ_OLD       0x2000
#define        SZ_WORK      (SZ_NEW + SZ_OLD + 1)

#if !defined(CRLF)
#define CRLF      "\r\n"
#endif

#define USE_DYNBUF            1

/*****************************************************************************
 *                                                                           *
 *                                                                           *
 *****************************************************************************/
static int      scan_text_block (unsigned long, char *);
static int      get_max_line_length (char *);

static int      decode_base64_parts(char *, char *, uint32_t, char *);

static int      do_print = 0;
static int      nblines = 0;

/*****************************************************************************
 *                                                                           *
 *                                                                           *
 *****************************************************************************/
int
check_body_contents (id, fname, state, maxsize, encoding, boundary)
     uint32_t        id;
     char           *fname;
     int            *state;
     uint32_t        maxsize;
     int             encoding;
     char           *boundary;
{
  struct stat     fstat;
  int             result = 0;
  int             linelen = 0;

  *state = 0;

  if (fname == NULL)
    return -1;

  if (stat (fname, &fstat) == 0) {
    uint32_t        fsize = fstat.st_size;
    char           *bufin = NULL, *bufout = NULL;
    FILE           *fin;
    char           *p;
    char            line[MAX_LINE_LEN];

    if ((maxsize > 0) && (fsize > maxsize))
      return 0;

    if ((bufin = malloc (fsize + 1)) == NULL) {
      syslog (LOG_WARNING, "%s : malloc error %s", J_FUNCTION, strerror (errno));
      return -2;
    }
    memset (bufin, 0, fsize + 1);

    if ((bufout = malloc (fsize + 1)) == NULL) {
      syslog (LOG_WARNING, "%s : malloc error %s", J_FUNCTION, strerror (errno));
      if (bufin != NULL)
        free (bufin);
      return -2;
    }
    memset (bufout, 0, fsize + 1);

    if ((fin = fopen (fname, "r")) == NULL) {
      syslog (LOG_WARNING, "%s : fopen error %s", J_FUNCTION, strerror (errno));
      if (bufin != NULL)
        free (bufin);
      if (bufout != NULL)
        free (bufout);
      return -3;
    }
    while ((p = fgets (line, sizeof (line), fin)) == line) {
      if (strspn (line, " \t\r\n") == strlen (line))
        break;
    }

    if (p != NULL) {
      if ((fsize = fread (bufin, 1, fsize, fin)) > 0) {
        bufin[fsize] = '\0';
        strchknull (bufin, fsize);

        linelen = get_max_line_length (bufin);
        if (log_level > 12)
          syslog (LOG_INFO, "%s : max line length = %d", J_FUNCTION, linelen);

        switch (encoding) {
          case MIME_ENCODE_BASE64:
            base64_decode (bufout, bufin, NULL, NULL);
            break;
          default:
            (void ) decode_base64_parts(bufout, bufin, fsize + 1, boundary);
            break;
        }
        *state = check_regex (id, bufout, MAIL_BODY);
      }

    }
    if (bufin)
      free (bufin);
    if (bufout)
      free (bufout);

    fclose (fin);
  } else {
    syslog (LOG_WARNING, "%s : stat error %s", J_FUNCTION, strerror (errno));
    result = -2;
  }

  return result;
}

/*****************************************************************************
 *                                                                           *
 *                                                                           *
 *****************************************************************************/
int
scan_msg_file_regex (id, fname, state, maxsize)
     unsigned long   id;
     char           *fname;
     int            *state;
     long            maxsize;
{
  struct stat     fstat;
  int             result = 0;

  int             linelen = 0;

  *state = 0;

  if (fname == NULL)
    return -1;

  if (stat (fname, &fstat) == 0) {
    long            fsize = fstat.st_size;
    char           *bufin = NULL;
    FILE           *fin;
    char           *p;
    char            line[MAX_LINE_LEN];

    if ((maxsize > 0) && (fsize > maxsize))
      return 0;

#if USE_DYNBUF == 1
    if ((bufin = malloc (fsize + 1)) == NULL) {
      syslog (LOG_WARNING, "%s : malloc error %s", J_FUNCTION, strerror (errno));
      return -2;
    }
    memset (bufin, 0, fsize + 1);
#endif

    if ((fin = fopen (fname, "r")) == NULL) {
      syslog (LOG_WARNING, "%s : fopen error %s", J_FUNCTION, strerror (errno));
      free (bufin);
      return -3;
    }
    while ((p = fgets (line, sizeof (line), fin)) == line) {
      if (strspn (line, " \t\r\n") == strlen (line))
        break;
    }

    if (p != NULL) {
#if USE_DYNBUF == 1
      if ((fsize = fread (bufin, 1, fsize, fin)) > 0) {
        bufin[fsize] = '\0';
        strchknull (bufin, fsize);

        linelen = get_max_line_length (bufin);
        if (log_level > 12)
          syslog (LOG_INFO, "%s : max line length = %d", J_FUNCTION, linelen);

        *state = check_regex (id, bufin, MAIL_BODY);
      }
#else
      char            obuf[SZ_OLD];
      char            rbuf[SZ_NEW];
      scanbuf_rec     old, new;

      old.buffer = obuf;
      memset (obuf, 0, sizeof (obuf));
      old.len = 0;
      old.size = sizeof (obuf);

      new.buffer = rbuf;
      new.len = 0;
      new.size = sizeof (rbuf);

      while ((new.len = fread (new.buffer, 1, new.size, fin)) > 0) {
        scan_body_regex (id, &old, new.buffer, new.len, state);
      }
#endif

    }
    if (bufin)
      free (bufin);

    fclose (fin);
  } else {
    syslog (LOG_WARNING, "%s : stat error %s", J_FUNCTION, strerror (errno));
    result = -2;
  }

  return result;
}

/*****************************************************************************
 *                                                                           *
 *                                                                           *
 *****************************************************************************/
int
scan_msg_file_regex_base64 (id, fname, state, maxsize)
     unsigned long   id;
     char           *fname;
     int            *state;
     long            maxsize;
{
  struct stat     fstat;
  int             result = 0;

  *state = 0;

  if (fname == NULL)
    return -1;

  if (stat (fname, &fstat) == 0) {
    long            fsize = fstat.st_size;
    char           *bufin = NULL, *bufout = NULL;
    FILE           *fin;
    char           *p;
    char            line[MAX_LINE_LEN];

    if ((maxsize > 0) && (fsize > maxsize))
      return 0;

    if ((bufin = malloc (fsize + 1)) == NULL) {
      syslog (LOG_WARNING, "%s : malloc error %s", J_FUNCTION, strerror (errno));
      return -2;
    }
    memset (bufin, 0, fsize + 1);

    if ((bufout = malloc (fsize + 1)) == NULL) {
      syslog (LOG_WARNING, "%s : malloc error %s", J_FUNCTION, strerror (errno));
      if (bufin != NULL)
        free (bufin);
      return -2;
    }
    memset (bufout, 0, fsize + 1);

    if ((fin = fopen (fname, "r")) == NULL) {
      syslog (LOG_WARNING, "%s : fopen error %s", J_FUNCTION, strerror (errno));
      if (bufin != NULL)
        free (bufin);
      if (bufout != NULL)
        free (bufout);
      return -3;
    }
    while ((p = fgets (line, sizeof (line), fin)) == line) {
      if (strspn (line, " \t\r\n") == strlen (line))
        break;
    }

    if (p != NULL) {

      if ((fsize = fread (bufin, 1, fsize, fin)) > 0) {
        bufin[fsize] = '\0';
        strchknull (bufin, fsize);
        base64_decode (bufout, bufin, NULL, NULL);
        *state = check_regex (id, bufout, MAIL_BODY);
      }

    }
    if (bufin)
      free (bufin);
    if (bufout)
      free (bufout);

    fclose (fin);
  } else {
    syslog (LOG_WARNING, "%s : stat error %s", J_FUNCTION, strerror (errno));
    result = -2;
  }

  return result;
}

/*****************************************************************************
 *                                                                           *
 *                                                                           *
 *****************************************************************************/
int
scan_body_regex (id, old, new, sznew, state)
     unsigned long   id;
     scanbuf_rec    *old;
     char           *new;
     long            sznew;
     int            *state;
{
  char           *p, *q;
  char            work[SZ_WORK];
  int             i;
  int             nbnext = 0;

  if (new == NULL || sznew == 0)
    return 0;

  if (old != NULL && old->buffer == NULL)
    return 0;

  memset (work, 0, sizeof (work));

  strchknull (new, sznew);

  q = work;

  if (old != NULL) {
    long            szold = old->len;

    /* strip out bad chars from old and new buffers */
    strchknull (old->buffer, szold);

    memcpy (q, old->buffer, szold);
    q += szold;
    *q = '\0';
    *old->buffer = '\0';
    old->len = 0;
  }

  for (i = sznew, p = new + sznew - 1; i > 0; i--, p--) {
    if (*p == '\r' || *p == '\n') {
      *p = '\0';
      if ((sznew - i > 0) && (old != NULL) && (old->buffer != NULL)) {
        nbnext = sznew - i;
        memcpy (old->buffer, p + 1, nbnext);
        old->buffer[nbnext] = '\0';
        old->len = nbnext;
      }
      break;
    }
  }

  if (sznew - nbnext > 0) {
    if (sznew - nbnext < SZ_WORK) {
      memcpy (q, new, sznew - nbnext);
      q[sznew - nbnext] = '\0';
    } else {
      syslog (LOG_WARNING, "%s : buffer size isn't enough large", J_FUNCTION);
    }
  }

  *state += scan_text_block (id, work);

  return 0;
}

/*****************************************************************************
 *                                                                           *
 *                                                                           *
 *****************************************************************************/
static int
scan_text_block (conn_id, line)
     unsigned long   conn_id;
     char           *line;
{
  int             result = 0;

  nblines++;

  result = check_regex (conn_id, line, MAIL_BODY);

  if (do_print && result > 0)
    printf ("--> Found at line %5d : %s\n", nblines, line);

  return result;
}


/*****************************************************************************
 *                                                                           *
 *                                                                           *
 *****************************************************************************/
static int
get_max_line_length (s)
     char           *s;
{
  char           *p = s;
  int             lmax = 0;

  if ((s == NULL) || (strlen (s) == 0))
    return 0;

  while (*p != '\0') {
    int             n = strcspn (p, "\r\n");

    if (n > lmax)
      lmax = n;
    p += n;
    p += strspn (p, "\r\n");
  }

  return lmax;
}


/*****************************************************************************
 *                                                                           *
 *                                                                           *
 *****************************************************************************/
#define   STR_CT   "content-type.*text/(html|plain)"
#define   STR_CE  "content-transfer-encoding.* base64"

#define   REGCOMP_FLAGS  (REG_ICASE | REG_NEWLINE | REG_EXTENDED)

static int
decode_base64_parts(out, in, sz, boundary)
     char *out;
     char *in;
     uint32_t sz;
     char     *boundary;
{
  regex_t         re_b, re_ce, re_ct;
  int             r;
  bool            ok;

  if ((boundary == NULL) || (strlen(boundary) == 0)) {
    memcpy (out, in, sz);
    return 0;
  }

  memset(out, 0, sz);

  ok = TRUE;
  if ((r = regcomp (&re_ct, STR_CT, REGCOMP_FLAGS)) != 0) {
    char            sout[256];

    regerror (r, &re_ct, sout, sizeof (sout));
    syslog(LOG_ERR,"%s regcomp error : %s", J_FUNCTION, sout);
    ok = FALSE;
  }
  if ((r = regcomp (&re_ce, STR_CE, REGCOMP_FLAGS)) != 0) {
    char            sout[256];

    regerror (r, &re_ce, sout, sizeof (sout));
    syslog(LOG_ERR,"%s regcomp error : %s", J_FUNCTION, sout);
    ok = FALSE;
  }
  if ((r = regcomp (&re_b, boundary, REGCOMP_FLAGS)) != 0) {
    char            sout[256];

    regerror (r, &re_b, sout, sizeof (sout));
    syslog(LOG_ERR,"%s regcomp error : %s", J_FUNCTION, sout);
    ok = FALSE;
  }

  if (ok) {
    regmatch_t      pm_ce, pm_ct, pm_b;
    int             ok_ce, ok_ct, ok_b;

    char *org = in, *dst = out, *p;

    ok_ce = ok_ct = ok_b = FALSE;

    while ((org != NULL) && (strlen(org) > 0)) {
      uint32_t pmin, pmax;

      ok_ct = !regexec (&re_ct, org, 1, &pm_ct, REG_NOTBOL | REG_NOTEOL);
      ok_ce = !regexec (&re_ce, org, 1, &pm_ce, REG_NOTBOL | REG_NOTEOL);
      ok_b  = !regexec (&re_b,  org, 1, &pm_b, REG_NOTBOL | REG_NOTEOL);

      if (log_level > 15) {
        if (ok_ct)
          syslog(LOG_DEBUG,"CT : %3d %3d", pm_ct.rm_so, pm_ct.rm_eo);
        if (ok_ce)
          syslog(LOG_DEBUG,"CE : %3d %3d", pm_ce.rm_so, pm_ce.rm_eo);
        if (ok_b)
          syslog(LOG_DEBUG,"B  : %3d %3d", pm_b.rm_so, pm_b.rm_eo);
      }

      if (!ok_ct || !ok_ce) {
        strcpy(dst, org);
        dst += strlen(dst);
        org += strlen(org);
        continue;
      }

      pmin = min(pm_ct.rm_so, pm_ce.rm_so);
      pmax = max(pm_ct.rm_eo, pm_ce.rm_eo);

      if (ok_b && (pm_b.rm_eo < pmin)) {
        int n = pm_b.rm_eo;
        strncpy(dst, org, n);
        dst[n] = '\0';
        dst += n;
        org += n;
        continue;
      }

      if (!ok_b || ((pm_b.rm_eo < pmin) || (pm_b.rm_so > pmax))) {
        uint32_t ni, no, n;

        memcpy(dst, org, pmax + 1);
        dst += (pmax + 1);
        org += (pmax + 1);
        *dst = '\0';
        n = base64_decode(dst, org, &no, &ni);
        dst += no;
        org += ni;
      }
    }
    *dst = '\0';
    for (p = out; p != dst; p++) {
      if (*p == '\0')
        *p = ' ';
    }
  }

  regfree(&re_ct);
  regfree(&re_ce);
  regfree(&re_b);

  return 0;
}
