/*
 *
 * j-chkmail - filtre de messagerie pour sendmail - MILTER
 *
 * Copyright (c) 2001, 2002 Ecole des Mines de Paris
 *
 *  Auteur     : Jose Marcio Martins da Cruz
 *               martins@paris.ensmp.fr
 *
 *  Historique :
 *  Creation     : janvier 2002
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <j-sys.h>

#include "j-chkmail.h"


/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */

#define    DT_CHECK      10

#define    J_FD_MAX      -1
#define    J_FD_DFL      0

#define    J_NOFILE      256

static long     fd_last = 0;
static long     fd_open = 0;
static long     fd_idle = 0;

static rlim_t   fd_max = 0;

static rlim_t   fd_soft = 0;
static rlim_t   fd_hard = 0;

static long     fd_conf = J_FD_DFL;
static rlim_t   fd_cur = 0;

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
int
count_file_descriptors ()
{
  int             i = 0;
  struct rlimit   rlp;
  int             nb = 0;

  time_t          now = time (NULL);

#if HAVE_SETRLIMIT
  if (fd_max == 0) {
    if (getrlimit (RLIMIT_NOFILE, &rlp) == 0) {
      fd_max = rlp.rlim_cur;
    } else {
      fd_max = sysconf (_SC_OPEN_MAX);
    }
  }
#endif

  if (fd_last + DT_CHECK > now)
    return fd_open;

  nb = fd_idle = 0;
  for (i = 0; i < fd_max; i++) {
    struct stat     buf;

    if (fstat (i, &buf) == 0)
      nb++;
    else
      fd_idle++;
  }
  fd_last = now;
  fd_open = nb;

  return nb;
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */

int
check_file_descriptors ()
{
  long            lsoft, lhard;

  lsoft = cf_get_int (CF_FD_FREE_SOFT);
  lhard = cf_get_int (CF_FD_FREE_HARD);

  count_file_descriptors ();

  if (fd_max == 0)
    return FD_LEVEL_OK;

  if (fd_open + lhard > fd_max) {
    syslog (LOG_WARNING,
            "%s : file descriptors being short...(soft limit) %ld/%ld",
            J_FUNCTION, (long) fd_open, (long) fd_max);
    return FD_LEVEL_HI;
  }

  if (fd_open + lsoft > fd_max) {
    syslog (LOG_WARNING,
            "%s : file descriptors being short... (hard limit) %ld/%ld",
            J_FUNCTION, (long) fd_open, (long) fd_max);
    return FD_LEVEL_SHORT;
  }

  return FD_LEVEL_OK;
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
int
setup_file_descriptors ()
{
#if HAVE_SETRLIMIT
  struct rlimit   rlp;
  char           *strlim = cf_get_str (CF_FILE_DESCRIPTORS);
  long            lsoft, lhard;

  lsoft = cf_get_int (CF_FD_FREE_SOFT);
  lhard = cf_get_int (CF_FD_FREE_HARD);

  if (log_level > 9 && strlim != NULL)
    syslog (LOG_INFO, "CF_FILE_DESCRIPTORS = %s", strlim);

  if (getrlimit (RLIMIT_NOFILE, &rlp) == 0) {
    if (fd_soft == 0)
      fd_soft = rlp.rlim_cur;
    if (fd_hard == 0)
      fd_hard = rlp.rlim_max;
  } else {
    syslog (LOG_ERR, "%s : getrlimit error : %s", J_FUNCTION, strerror (errno));
    return 0;
  }

  if (strlim != NULL) {
    if (strcasecmp (strlim, "DEFAULT") == 0)
      fd_conf = J_FD_DFL;
    else {
      if (strcasecmp (strlim, "MAX") == 0) {
#if 0
        fd_conf = J_FD_MAX;
#else
        fd_conf = fd_hard;
#endif
      } else {
        long            n;

#if HAVE_STRTOL
        n = strtol (strlim, (char **) NULL, 10);
        if (errno == ERANGE || errno == EINVAL || n <= 0)
          n = J_FD_DFL;
#else
        n = atoi (strlim);
        if ((n <= 0) || (n > fd_hard))
          n = J_FD_DFL;
#endif
        fd_conf = n;
      }
    }
  }

  if (fd_conf > fd_hard) {
    syslog (LOG_WARNING,
            "%s : Can't set file descriptor limit greater than %ld",
            J_FUNCTION, (long) fd_hard);
    fd_conf = fd_hard;
  }

  if ((fd_conf > 1) && (fd_conf < J_NOFILE)) {
    fd_conf = J_NOFILE;
  }

  switch (fd_conf) {
    case J_FD_DFL:
      fd_cur = fd_soft;
      break;
    case J_FD_MAX:
      fd_cur = fd_hard;
      break;
    default:
      fd_cur = (fd_conf >= J_NOFILE ? fd_conf : J_NOFILE);
  }

  syslog (LOG_INFO, "%s : Will set file descriptor limit to %ld [%ld,%ld]",
          J_FUNCTION, (long) fd_cur, (long) fd_soft, (long) fd_hard);

  if ((cf_get_int (CF_USE_SELECT_LIMIT) == OPT_YES) && (fd_cur > FD_SETSIZE)) {
    syslog (LOG_INFO,
            "%s : May not set file descriptor limit to more than FD_SETSIZE",
            J_FUNCTION);
    fd_cur = FD_SETSIZE;
  }

  rlp.rlim_cur = fd_cur;
  if (setrlimit (RLIMIT_NOFILE, &rlp) != 0) {
    syslog (LOG_ERR, "%s : setrlimit(RLIMIT_NOFILE) error : %s\n",
            J_FUNCTION, strerror (errno));
  } else
    syslog (LOG_INFO, "%s : RLIMIT_NOFILE set to %ld files",
            J_FUNCTION, (long) rlp.rlim_cur);

  syslog (LOG_INFO, "%s : Thresholds are : soft=[%ld] - hard=[%ld]",
          J_FUNCTION, (long) (fd_cur - lsoft), (long) (fd_cur - lhard));

#endif

  return 0;
}
