/*
 *
 * j-chkmail - filtre de messagerie pour sendmail - MILTER
 *
 * Copyright (c) 2001, 2002 Ecole des Mines de Paris
 *
 *  Auteur     : Jose Marcio Martins da Cruz
 *               martins@paris.ensmp.fr
 *
 *  Historique :
 *  Creation     : janvier 2002
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <j-sys.h>

#if HAVE_SEARCH_H
#include <search.h>
#endif

#include "j-chkmail.h"


#define UPDATE_TABLE       1
#define FIXED_ACCEPT       0

#define FUNC_UPDATE        1
#define FUNC_CHECK         1

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */

#define             _01M            60

#define             _01H          3600
#define             _06H         21600
#define             _12H         43200
#define             _18H         64800
#define             _24H         86400
#define             _48H        172800
#define             _72H        259200

static void         set_threshold ();

#if FIXED_ACCEPT == 1
#define             _S06        2
#define             _S12        4
#define             _S18        6
#define             _S24        6
#else
static int          _S06 = 4;
static int          _S12 = 4;
static int          _S18 = 6;
static int          _S24 = 6;
#endif

#define      SZ_IP         64
#define      DIM_CONN_DATA 0x4000

typedef struct conn_data {
#if SIZEOF_LONG == 4
  unsigned long       signature;
#else
  unsigned int        signature;
#endif
  time_t              date;
  char                host[SZ_IP];
  int                 resolve;
  int                 granted;
} conn_data;

static conn_data    conn[DIM_CONN_DATA];
static int          i_conn = 0;

typedef struct resolve_rec {
  char                host[SZ_IP];
  int                 resolve;
  int                 granted[6];
  int                 connect[6];
} resolve_rec;

static resolve_rec  synth[DIM_CONN_DATA];
static size_t       i_synth = 0;

unsigned int        resolve_interval = _01M;  /* 1 minute */
unsigned int        resolve_window = _48H;  /* 24 heures */

static pthread_mutex_t st_mutex = PTHREAD_MUTEX_INITIALIZER;

static void         update_resolve_rec (resolve_rec *, time_t, time_t, int);
static int          check_grant (resolve_rec *);

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
void
resolve_tab_init ()
{
  set_threshold ();
  memset (conn, 0, sizeof (conn));
  i_conn = 0;
  memset (synth, 0, sizeof (synth));
  i_synth = 0;
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
static int
resolve_compare (a, b)
     const void         *a;
     const void         *b;
{
  resolve_rec        *ta = (resolve_rec *) a;
  resolve_rec        *tb = (resolve_rec *) b;

  return (strcmp (ta->host, tb->host));
}



/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
int
resolve_tab_add_entry (host, t, resolve_result, update)
     char               *host;
     time_t              t;
     int                 resolve_result;
     int                 update;
{
  resolve_rec        *ptr = NULL, p;
  int                 grant = TRUE;
  int                 lock_res = 0;

  set_threshold ();

  if ((lock_res = pthread_mutex_lock (&st_mutex)) != 0)
    syslog (LOG_ERR, "%s : pthread_mutex_lock : %s", J_FUNCTION,
            strerror (errno));

  conn[i_conn].signature = 0x5A5A5A5A;
  snprintf (conn[i_conn].host, sizeof (conn[i_conn].host), "%s", host);
  conn[i_conn].date = t;
  conn[i_conn].resolve = resolve_result;

  /* decider de la valeur du champs granted */
  memset (&p, 0, sizeof (p));
  strcpy (p.host, host);
  p.resolve = resolve_result;
  if ((ptr = bsearch (&p, synth, i_synth, sizeof (resolve_rec),
                      resolve_compare)) != NULL) {
    /* FOUND - access may be granted */
    grant = check_grant (ptr);
    conn[i_conn].granted = grant;

    if (update)
      update_resolve_rec (ptr, t, t, grant);

  } else {
    /* NOT FOUND - will grant access */
    grant = TRUE;
    conn[i_conn].granted = grant;

    if (update) {
      memset (&p, 0, sizeof (p));
      strcpy (p.host, host);
      p.resolve = resolve_result;

      update_resolve_rec (&p, t, t, grant);

      synth[i_synth++] = p;
      qsort (synth, i_synth, sizeof (resolve_rec), resolve_compare);
    }
  }

  i_conn++;
  i_conn %= DIM_CONN_DATA;

  if (lock_res == 0)
    pthread_mutex_unlock (&st_mutex);

  return grant;
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
int
resolve_tab_check_host (host, update)
     char               *host;
     int                 update;
{
  resolve_rec         p;
  resolve_rec        *t = NULL;
  int                 res = TRUE;
  int                 lock_res = 0;
  time_t              now = time (NULL);

  if (host == NULL)
    return res;

  set_threshold ();
  memset (&p, 0, sizeof (p));
  strcpy (p.host, host);

  if ((lock_res = pthread_mutex_lock (&st_mutex)) != 0)
    syslog (LOG_ERR, "%s : pthread_mutex_lock : %s", J_FUNCTION,
            strerror (errno));

  if ((t =
       bsearch (&p, synth, i_synth, sizeof (resolve_rec),
                resolve_compare)) != NULL) {
    /* FOUND - access may be granted */
    res = check_grant (t);

    if (update)
      update_resolve_rec (t, now, now, res);
  }

  if (lock_res == 0)
    pthread_mutex_unlock (&st_mutex);

  return res;
}


/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
int
resolve_tab_update (up2now)
     int                 up2now;
{
  int                 i;
  time_t              now = time (NULL);
  resolve_rec         p;
  resolve_rec        *t;
  int                 lock_res = 0;
  static int          save_it = 0;

  if (log_level > 10)
    syslog (LOG_INFO, "Entering resolv_tab_update...");

  if ((lock_res = pthread_mutex_lock (&st_mutex)) != 0)
    syslog (LOG_ERR, "%s : pthread_mutex_lock : %s", J_FUNCTION,
            strerror (errno));

  /* Let's update connection throttle */
  i_synth = 0;
  memset (synth, 0, sizeof (synth));
  memset (&p, 0, sizeof (p));

  if (!up2now) {
    now = 0;
    for (i = 0; i < DIM_CONN_DATA; i++) {
      if (now < conn[i].date)
        now = conn[i].date;
    }
  }

  for (i = 0; i < DIM_CONN_DATA; i++) {

    if ((strlen (conn[i].host) > 0) && (conn[i].date + resolve_window > now)) {
      memset (&p, 0, sizeof (p));
      strcpy (p.host, conn[i].host);

      if ((t = lfind (&p, synth, &i_synth,
                      sizeof (resolve_rec), resolve_compare)) != NULL) {
        update_resolve_rec (t, conn[i].date, now, conn[i].granted);

      } else {
        if (i_synth < DIM_CONN_DATA) {
          memset (&p, 0, sizeof (p));
          strcpy (p.host, conn[i].host);
          p.resolve = conn[i].resolve;

          update_resolve_rec (&p, conn[i].date, now, conn[i].granted);

          synth[i_synth++] = p;
        }
      }
    }
  }
  qsort (synth, i_synth, sizeof (resolve_rec), resolve_compare);

  if (lock_res == 0)
    pthread_mutex_unlock (&st_mutex);

  if (log_level > 15)
    syslog (LOG_INFO, "HOST %d", i_synth);

  if (save_it) {
    resolve_tab_save (NULL);
  }
  save_it = !save_it;

  return 0;
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
#define      DT_SLEEP    60

static void        *
resolve_loop (data)
     void               *data;
{
  time_t              t_now, t_last;
  char                s[256];

  t_last = t_now = time (NULL);
  ctime_r (&t_now, s);
  if (log_level > 10)
    syslog (LOG_WARNING, " resolve_loop : starting at %s", s);

  for (;;) {
    sleep (DT_SLEEP);
    t_now = time (NULL);

    syslog (LOG_WARNING, "%s %ld - %ld", J_FUNCTION, t_last, t_now);
    if (t_last + DT_SLEEP > t_now)
      continue;

    syslog (LOG_WARNING, "%s OK !", J_FUNCTION);

    resolve_tab_update (TRUE);

    if (log_level > 10) {
      t_now = time (NULL);
      ctime_r (&t_now, s);
      syslog (LOG_WARNING, " resolve_loop : %s", s);
    }
    t_last = t_now;
  }
}

void
resolve_tab_update_thread ()
{
  pthread_t           tid;
  int                 r;

  if ((r = pthread_create (&tid, NULL, resolve_loop, (void *) NULL)) != 0) {
    syslog (LOG_WARNING, "Couldn't launch resolve_loop thread");
  }

}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
void
resolve_tab_save (filename)
     char               *filename;
{
#if 0
  char               *work_dir = cf_get_str (CF_WORKDIR);
#else
  char               *work_dir = J_WORKDIR;
#endif
  char                fname[256];
  int                 fd;

  if (filename == NULL)
    snprintf (fname, sizeof (fname), "%s/%s", work_dir, "j-resolvedata");
  else
    strlcpy (fname, filename, sizeof (fname));

  if ((fd = open (fname, O_WRONLY | O_CREAT | O_TRUNC, 00644)) >= 0) {
    if (write (fd, conn, sizeof (conn)) != sizeof (conn)) {
      syslog (LOG_WARNING, "%s : Error writint %s file : %s", J_FUNCTION,
              fname, strerror (errno));
    }
    close (fd);
  } else {
    syslog (LOG_WARNING, "%s : openning %s file : %s", J_FUNCTION, fname,
            strerror (errno));
  }
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
int
resolve_tab_read (filename)
     char               *filename;
{
#if 0
  char               *work_dir = cf_get_str (CF_WORKDIR);
#else
  char               *work_dir = J_WORKDIR;
#endif
  char                fname[256];
  int                 fd;
  int                 i, imax;
  time_t              tmax;

#ifndef INT_MAX
#define INT_MAX    2147483647
#endif

  if (filename == NULL)
    snprintf (fname, sizeof (fname), "%s/%s", work_dir, "j-resolvedata");
  else
    strlcpy (fname, filename, sizeof (fname));

  if ((fd = open (fname, O_RDONLY)) >= 0) {
    if (read (fd, conn, sizeof (conn)) != sizeof (conn)) {
      syslog (LOG_WARNING, "%s : Error reading %s file : %s", J_FUNCTION,
              fname, strerror (errno));
    }
    close (fd);
  } else {
    syslog (LOG_WARNING, "%s : openning %s file : %s", J_FUNCTION, fname,
            strerror (errno));
  }

  for (i = 0, tmax = 0, imax = 0; i < DIM_CONN_DATA; i++) {
    if (tmax == 0 || conn[i].date > tmax) {
      tmax = conn[i].date;
      imax = i;
    }
  }
  if (imax > 0)
    i_conn = imax + 1;
  else
    i_conn = 0;
  i_conn %= DIM_CONN_DATA;

  return 0;
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
static int
resolve_compare_by_value (a, b)
     const void         *a;
     const void         *b;
{
  resolve_rec        *ta = (resolve_rec *) a;
  resolve_rec        *tb = (resolve_rec *) b;

  struct in_addr      aa, ab;

  if (tb->connect[4] != ta->connect[4])
    return ta->connect[4] - tb->connect[4];
  if (tb->connect[3] != ta->connect[3])
    return ta->connect[3] - tb->connect[3];
  if (tb->connect[2] != ta->connect[2])
    return ta->connect[2] - tb->connect[2];
  if (tb->connect[1] != ta->connect[1])
    return ta->connect[1] - tb->connect[1];
  if (tb->connect[0] != ta->connect[0])
    return ta->connect[0] - tb->connect[0];

  if (j_inet_pton (tb->host, &ab.s_addr) && j_inet_pton (ta->host, &aa.s_addr)) {
    if (htonl (aa.s_addr) > htonl (ab.s_addr))
      return 1;
    if (htonl (aa.s_addr) < htonl (ab.s_addr))
      return -1;
    return 0;
  }

  return strcmp (ta->host, tb->host);
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
static char        *
str_resolve (r)
     int                 r;
{
  static char        *names[] = { "OK", "FAIL", "FORGED", NULL };
  int                 i;

  for (i = 0; names[i] != NULL; i++) {
    if (i == r)
      return names[i];
  }
  return "";
}


/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
void
resolve_tab_print (what, verbose, histogram, resolve)
     int                 what;
     int                 verbose;
     int                 histogram;
     int                 resolve;
{
  int                 i, j;
  int                 v;

  char                nodename[128];

  time_t              tmin = 0, tmax = 0;
  long                hh, mm, ss;

  resolve_rec         pglob, pperc;
  STATS               st_conn[5], st_host[5];

  int                 last, actual, nhosts[5], nh;

  time_t              now;
  char                s[256];
  char               *pchar;

#if 0
  printf ("** %s : what      : %d\n", J_FUNCTION, what);
  printf ("** %s : verbose   : %d\n", J_FUNCTION, verbose);
  printf ("** %s : histogram : %d\n", J_FUNCTION, histogram);
  printf ("** %s : resolve   : %d\n", J_FUNCTION, resolve);
#endif

  printf ("%-30s : %s\n\n", "Version", PACKAGE);

#if 0
  now = time (NULL);
#else
  now = 0;
  for (i = 0; i < DIM_CONN_DATA; i++) {
    if (now < conn[i].date)
      now = conn[i].date;
  }
#endif
  ctime_r (&now, s);

  if ((pchar = strchr (s, '\n')) != NULL)
    *pchar = '\0';
  printf ("*** RESOLVE TABLE at %s\n", s);

  qsort (synth, i_synth, sizeof (resolve_rec), resolve_compare_by_value);

  tmin = tmax = conn[0].date;
  for (v = 0, i = 0; i < DIM_CONN_DATA; i++) {
    if (conn[i].date != 0) {
      if (tmin == 0)
        tmin = conn[i].date;
      if (conn[i].date < tmin)
        tmin = conn[i].date;
      if (conn[i].date > tmax)
        tmax = conn[i].date;
      v++;
    }
  }

  ss = tmax - tmin;
  hh = ss / 3600;
  ss -= hh * 3600;
  mm = ss / 60;
  ss -= mm * 60;

  printf ("*** CONNECTIONS HISTORY : %02ld:%02ld:%02ld (%d/%d entries)\n\n",
          hh, mm, ss, v, DIM_CONN_DATA);

  printf ("    Data displayed are in %s mode\n\n",
          (histogram ? "histogram" : "cumulative"));

  /**
   **
   ** GLOBAL SUMMARY    
   **
   **/
  for (j = 0; j < 5; j++) {
    kstats_reset (&st_conn[j]);
    kstats_reset (&st_host[j]);
  }
  memset (&pglob, 0, sizeof (pglob));
  memset (nhosts, 0, sizeof (nhosts));
  nh = 0;
  for (i = 0; i < i_synth; i++) {
    nh++;
    if (synth[i].connect[4] > 0) {
      last = 0;
      for (j = 0; j < 5; j++) {
        actual = synth[i].connect[j] - (histogram ? last : 0);
        if (actual > 0)
          nhosts[j]++;
        pglob.connect[j] += actual;
        last = synth[i].connect[j];
        if (actual > 0) {
          kstats_update (&st_conn[j], (double) synth[i].connect[j]);
          if (synth[i].connect[j] - synth[i].granted[j] > 0)
            kstats_update (&st_host[j], 1.);
          else
            kstats_update (&st_host[j], 0.);
        }
      }
      last = 0;
      for (j = 0; j < 5; j++) {
        actual = synth[i].granted[j] - (histogram ? last : 0);
        pglob.granted[j] += actual;
        last = synth[i].granted[j];
      }
    }
  }

  printf (" SUMMARY       |");
  printf ("    06H    12H    18H    24H    48H |\n");
  strset (s, '-', 52);
  printf (" %s\n", s);
  printf (" CONNECT...... |");
  for (j = 0; j < 5; j++)
    printf (" %6d", pglob.connect[j]);
  printf (" | connections\n");
  printf (" ACCEPT....... |");
  for (j = 0; j < 5; j++)
    printf (" %6d", pglob.granted[j]);
  printf (" | connections\n");
  printf (" REJECT....... |");
  for (j = 0; j < 5; j++)
    printf (" %6d", pglob.connect[j] - pglob.granted[j]);
  printf (" | connections\n");
  printf (" REJECT RATIO. |");
  for (j = 0; j < 5; j++) {
    double              v = 0;

    if (pglob.connect[j] > 0)
      v =
        100 * ((double) (pglob.connect[j] - pglob.granted[j])) /
        pglob.connect[j];
    printf (" %6.2f", v);
  }
  printf (" | %% connections\n");

  printf (" MEAN......... |");
  for (j = 0; j < 5; j++)
    printf (" %6.1f", kmean (&st_conn[j]));
  printf (" | connections/gateway\n");
  printf (" STD DEV...... |");
  for (j = 0; j < 5; j++)
    printf (" %6.1f", kstddev (&st_conn[j]));
  printf (" | connections/gateway\n");

  printf (" GATEWAYS..... |");
  for (j = 0; j < 5; j++)
    printf (" %6d", nhosts[j]);
  printf (" | Total = %d addresses\n", nh);

  printf (" REJECT....... |");
  for (j = 0; j < 5; j++)
    printf (" %6d", (int) (kmean (&st_host[j]) * nhosts[j]));
  printf (" | gateways\n");

  printf (" REJECT RATIO. |");
  for (j = 0; j < 5; j++) {
    printf (" %6.2f", 100. * kmean (&st_host[j]));
  }
  printf (" | %% gateways\n");

  memset (&pperc, 0, sizeof (pperc));
  memset (nhosts, 0, sizeof (nhosts));
  nh = 0;
  for (i = i_synth; i > 0; i--) {
    i--;
    if (synth[i].resolve != RESOLVE_FAIL)
      continue;
    nh++;
    if (synth[i].connect[4] > 0) {
      last = 0;
      for (j = 0; j < 5; j++) {
        if (pperc.connect[j] > 0.90 * pglob.connect[j])
          continue;
        actual = synth[i].connect[j] - (histogram ? last : 0);
        if (actual > 0)
          nhosts[j]++;
        pperc.connect[j] += actual;
        last = synth[i].connect[j];
      }
      last = 0;
      for (j = 0; j < 5; j++) {
        actual = synth[i].granted[j] - (histogram ? last : 0);
        pperc.granted[j] += actual;
        last = synth[i].granted[j];
      }
    }
  }
  printf (" GATEWAYS..... |");
  for (j = 0; j < 5; j++)
    printf (" %6d", nhosts[j]);
  printf (" | 90 %% connections\n\n");


  /**
   **
   ** RESOLVE_FAIL SUMMARY     
   **
   **/
  for (j = 0; j < 5; j++) {
    kstats_reset (&st_conn[j]);
    kstats_reset (&st_host[j]);
  }
  memset (&pglob, 0, sizeof (pglob));
  memset (nhosts, 0, sizeof (nhosts));
  nh = 0;
  for (i = 0; i < i_synth; i++) {
    if (synth[i].resolve != RESOLVE_FAIL)
      continue;
    nh++;
    if (synth[i].connect[4] > 0) {
      last = 0;
      for (j = 0; j < 5; j++) {
        actual = synth[i].connect[j] - (histogram ? last : 0);
        if (actual > 0)
          nhosts[j]++;
        pglob.connect[j] += actual;
        last = synth[i].connect[j];
        if (actual > 0) {
          kstats_update (&st_conn[j], (double) synth[i].connect[j]);
          if (synth[i].connect[j] - synth[i].granted[j] > 0)
            kstats_update (&st_host[j], 1.);
          else
            kstats_update (&st_host[j], 0.);
        }
      }
      last = 0;
      for (j = 0; j < 5; j++) {
        actual = synth[i].granted[j] - (histogram ? last : 0);
        pglob.granted[j] += actual;
        last = synth[i].granted[j];
      }
    }
  }

  printf (" RESOLVE FAIL  |");
  printf ("    06H    12H    18H    24H    48H |\n");
  strset (s, '-', 52);
  printf (" %s\n", s);
  printf (" CONNECT...... |");
  for (j = 0; j < 5; j++)
    printf (" %6d", pglob.connect[j]);
  printf (" | connections\n");
  printf (" ACCEPT....... |");
  for (j = 0; j < 5; j++)
    printf (" %6d", pglob.granted[j]);
  printf (" | connections\n");
  printf (" REJECT....... |");
  for (j = 0; j < 5; j++)
    printf (" %6d", pglob.connect[j] - pglob.granted[j]);
  printf (" | connections\n");
  printf (" REJECT RATIO. |");
  for (j = 0; j < 5; j++) {
    double              v = 0;

    if (pglob.connect[j] > 0)
      v =
        100 * ((double) (pglob.connect[j] - pglob.granted[j])) /
        pglob.connect[j];
    printf (" %6.2f", v);
  }
  printf (" | %% connections\n");

  printf (" MEAN......... |");
  for (j = 0; j < 5; j++)
    printf (" %6.1f", kmean (&st_conn[j]));
  printf (" | connections/gateway\n");
  printf (" STD DEV...... |");
  for (j = 0; j < 5; j++)
    printf (" %6.1f", kstddev (&st_conn[j]));
  printf (" | connections/gateway\n");

  printf (" GATEWAYS..... |");
  for (j = 0; j < 5; j++)
    printf (" %6d", nhosts[j]);
  printf (" | Total = %d addresses\n", nh);

  printf (" REJECT....... |");
  for (j = 0; j < 5; j++)
    printf (" %6d", (int) (kmean (&st_host[j]) * nhosts[j]));
  printf (" | gateways\n");

  printf (" REJECT RATIO. |");
  for (j = 0; j < 5; j++) {
    printf (" %6.2f", 100. * kmean (&st_host[j]));
  }
  printf (" | %% gateways\n");


  memset (&pperc, 0, sizeof (pperc));
  memset (nhosts, 0, sizeof (nhosts));
  nh = 0;
  for (i = i_synth; i > 0; i--) {
    i--;
    if (synth[i].resolve != RESOLVE_FAIL)
      continue;
    nh++;
    if (synth[i].connect[4] > 0) {
      last = 0;
      for (j = 0; j < 5; j++) {
        if (pperc.connect[j] > 0.90 * pglob.connect[j])
          continue;
        actual = synth[i].connect[j] - (histogram ? last : 0);
        if (actual > 0)
          nhosts[j]++;
        pperc.connect[j] += actual;
        last = synth[i].connect[j];
      }
      last = 0;
      for (j = 0; j < 5; j++) {
        actual = synth[i].granted[j] - (histogram ? last : 0);
        pperc.granted[j] += actual;
        last = synth[i].granted[j];
      }
    }
  }
  printf (" GATEWAYS..... |");
  for (j = 0; j < 5; j++)
    printf (" %6d", nhosts[j]);
  printf (" | 90 %% connections\n\n");


  /**
   **
   ** RESOLVE_FORGED SUMMARY     
   **
   **/
  for (j = 0; j < 5; j++) {
    kstats_reset (&st_conn[j]);
    kstats_reset (&st_host[j]);
  }
  memset (&pglob, 0, sizeof (pglob));
  memset (nhosts, 0, sizeof (nhosts));
  nh = 0;
  for (i = 0; i < i_synth; i++) {
    if (synth[i].resolve != RESOLVE_FORGED)
      continue;
    nh++;
    if (synth[i].connect[4] > 0) {
      last = 0;
      for (j = 0; j < 5; j++) {
        actual = synth[i].connect[j] - (histogram ? last : 0);
        if (actual > 0)
          nhosts[j]++;
        pglob.connect[j] += actual;
        last = synth[i].connect[j];
        if (actual > 0) {
          kstats_update (&st_conn[j], (double) synth[i].connect[j]);
          if (synth[i].connect[j] - synth[i].granted[j] > 0)
            kstats_update (&st_host[j], 1.);
          else
            kstats_update (&st_host[j], 0.);
        }
      }
      last = 0;
      for (j = 0; j < 5; j++) {
        actual = synth[i].granted[j] - (histogram ? last : 0);
        pglob.granted[j] += actual;
        last = synth[i].granted[j];
      }
    }
  }

  printf (" RESOLVE FORGED|");
  printf ("    06H    12H    18H    24H    48H |\n");
  strset (s, '-', 52);
  printf (" %s\n", s);
  printf (" CONNECT...... |");
  for (j = 0; j < 5; j++)
    printf (" %6d", pglob.connect[j]);
  printf (" | connections\n");
  printf (" ACCEPT....... |");
  for (j = 0; j < 5; j++)
    printf (" %6d", pglob.granted[j]);
  printf (" | connections\n");
  printf (" REJECT....... |");
  for (j = 0; j < 5; j++)
    printf (" %6d", pglob.connect[j] - pglob.granted[j]);
  printf (" | connections\n");
  printf (" REJECT RATIO. |");
  for (j = 0; j < 5; j++) {
    double              v = 0;

    if (pglob.connect[j] > 0)
      v =
        100 * ((double) (pglob.connect[j] - pglob.granted[j])) /
        pglob.connect[j];
    printf (" %6.2f", v);
  }
  printf (" | %% connections\n");

  printf (" MEAN......... |");
  for (j = 0; j < 5; j++)
    printf (" %6.1f", kmean (&st_conn[j]));
  printf (" | connections/gateway\n");
  printf (" STD DEV...... |");
  for (j = 0; j < 5; j++)
    printf (" %6.1f", kstddev (&st_conn[j]));
  printf (" | connections/gateway\n");

  printf (" GATEWAYS..... |");
  for (j = 0; j < 5; j++)
    printf (" %6d", nhosts[j]);
  printf (" | Total = %d addresses\n", nh);

  printf (" REJECT....... |");
  for (j = 0; j < 5; j++)
    printf (" %6d", (int) (kmean (&st_host[j]) * nhosts[j]));
  printf (" | gateways\n");

  printf (" REJECT RATIO. |");
  for (j = 0; j < 5; j++) {
    printf (" %6.2f", 100. * kmean (&st_host[j]));
  }
  printf (" | %% gateways\n");

  memset (&pperc, 0, sizeof (pperc));
  memset (nhosts, 0, sizeof (nhosts));
  nh = 0;
  for (i = i_synth; i > 0; i--) {
    i--;
    if (synth[i].resolve != RESOLVE_FORGED)
      continue;
    nh++;
    if (synth[i].connect[4] > 0) {
      last = 0;
      for (j = 0; j < 5; j++) {
        if (pperc.connect[j] > 0.90 * pglob.connect[j])
          continue;
        actual = synth[i].connect[j] - (histogram ? last : 0);
        if (actual > 0)
          nhosts[j]++;
        pperc.connect[j] += actual;
        last = synth[i].connect[j];
      }
      last = 0;
      for (j = 0; j < 5; j++) {
        actual = synth[i].granted[j] - (histogram ? last : 0);
        pperc.granted[j] += actual;
        last = synth[i].granted[j];
      }
    }
  }
  printf (" GATEWAYS..... |");
  for (j = 0; j < 5; j++)
    printf (" %6d", nhosts[j]);
  printf (" | 90 %% connections\n\n");

  /**
   **
   ** DETAIL    
   **
   **/
  if (what > 0) {
    printf ("                             |");
    printf ("         CONNECT          |");
    printf ("         GRANTED          |\n");

    printf (" HOST ADDRESS     RESOLV REJ |");
    printf ("  06H  12H  18H  24H  48H |");
    printf ("  06H  12H  18H  24H  48H |\n");

    strset (s, '-', 83);
    printf (" %s\n", s);

    for (i = 0; i < i_synth; i++) {
      if (synth[i].connect[3] > 0) {
        memset (nodename, 0, sizeof (nodename));
        if (resolve)
          j_gethostbyipaddr (synth[i].host, nodename, sizeof (nodename));
        printf (" %-16s %-6s  %c  |", synth[i].host,
                str_resolve (synth[i].resolve),
                synth[i].connect[3] > synth[i].granted[3] ? 'X' : '.');
        last = 0;
        for (j = 0; j < 5; j++) {
          actual = (synth[i].connect[j] - (histogram ? last : 0));
          printf (" %4d", actual);
          last = synth[i].connect[j];
        }
        printf (" |");
        last = 0;
        for (j = 0; j < 5; j++) {
          actual = (synth[i].granted[j] - (histogram ? last : 0));
          printf (" %4d", actual);
          last = synth[i].granted[j];
        }
        printf (" | ");
        printf (" %s\n", nodename);
      }
    }
    printf ("\n");
  }

  qsort (synth, i_synth, sizeof (resolve_rec), resolve_compare);
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
void
resolve_tab_log ()
{

  if (log_level >= 10)
    syslog (LOG_INFO, "*** RESOLVE TABLE");
#if JOE == 1
  for (i = 0; i < i_synth; i++) {
    if (log_level >= 10 && synth[i].nb > 5)
      syslog (LOG_INFO, " CONN RESOLVE : %-16s %5d",
              synth[i].host, synth[i].nb);
  }
#endif
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
static void
set_threshold ()
{
#if FIXED_ACCEPT == 1
  return;
#else
  _S06 = cf_get_int (CF_RESOLVE_ACCEPT_06H);
  _S12 = cf_get_int (CF_RESOLVE_ACCEPT_12H);
  _S18 = cf_get_int (CF_RESOLVE_ACCEPT_18H);
  _S24 = cf_get_int (CF_RESOLVE_ACCEPT_24H);
#endif
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
static void
update_resolve_rec (t, date, now, grant)
     resolve_rec        *t;
     time_t              date;
     time_t              now;
     int                 grant;
{

  if (t == NULL)
    return;

  if (grant) {
    if (date + _06H > now)
      t->granted[0]++;
    if (date + _12H > now)
      t->granted[1]++;
    if (date + _18H > now)
      t->granted[2]++;
    if (date + _24H > now)
      t->granted[3]++;
    if (date + _48H > now)
      t->granted[4]++;
    if (date + _72H > now)
      t->granted[5]++;
  }
  if (date + _06H > now)
    t->connect[0]++;
  if (date + _12H > now)
    t->connect[1]++;
  if (date + _18H > now)
    t->connect[2]++;
  if (date + _24H > now)
    t->connect[3]++;
  if (date + _48H > now)
    t->connect[4]++;
  if (date + _72H > now)
    t->connect[5]++;
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
static int
check_grant (r)
     resolve_rec        *r;
{
  if (r == NULL)
    return TRUE;

  return (r->granted[0] < _S06 &&
          r->granted[1] < _S12 &&
          r->granted[2] < _S18 && r->granted[3] < _S24);
}
