/*
 *
 * j-chkmail - filtre de messagerie pour sendmail - MILTER
 *
 * Copyright (c) 2001, 2002 Ecole des Mines de Paris
 *
 *  Auteur     : Jose Marcio Martins da Cruz
 *               martins@paris.ensmp.fr
 *
 *  Historique :
 *  Creation     : janvier 2002
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <j-sys.h>

#include "j-chkmail.h"

#undef  DEBUG
#define DEBUG 0



/*****************************************************************************
 *                                                                           * 
 *                                                                           *
 *****************************************************************************/
#define NETWORK_LEN          256

typedef struct {
  char            k[NETWORK_LEN];
  char            v[NETWORK_LEN];

  char            network[NETWORK_LEN];
  char            netmask[NETWORK_LEN];

  struct in_addr  net;
  struct in_addr  mask;
  int             class;
} NETWORK_REC;

static j_table  htbl;

static pthread_mutex_t st_mutex = PTHREAD_MUTEX_INITIALIZER;

static int      use_db = FALSE;

/*****************************************************************************
 *                                                                           * 
 *                                                                           *
 *****************************************************************************/
static int
network_comp (pa, pb)
     const void     *pa;
     const void     *pb;
{
  NETWORK_REC    *a = (NETWORK_REC *) pa;
  NETWORK_REC    *b = (NETWORK_REC *) pb;

  return strcasecmp (a->k, b->k);
}

/*****************************************************************************
 *                                                                           * 
 *                                                                           *
 *****************************************************************************/
void
dump_networks_table ()
{
  NETWORK_REC     p;

  printf ("*** Known networks table : \n");
  printf ("           network          netmask          class\n");
  if (j_table_get_first (&htbl, &p) == 0) {

    do {
      char            net[256];
      char            mask[256];

      memset (net, 0, sizeof (net));
      memset (mask, 0, sizeof (mask));

      if (j_inet_ntop ((void *) &p.net, net, sizeof (net)) &&
          j_inet_ntop ((void *) &p.mask, mask, sizeof (mask))) {
        printf ("-> NETWORK %-16s %-16s %-8s [%02X]\n", net, mask, p.v, p.class);
      } else
        printf ("-> NETWORK : %08X %08X %-50s %s\n",
                p.net.s_addr, p.mask.s_addr, p.k, p.v);

    } while (j_table_get_next (&htbl, &p) == 0);
  }
}

/*****************************************************************************
 *                                                                           * 
 *                                                                           *
 *****************************************************************************/
static long
mask_len2str (l, s)
     long            l;
     char           *s;
{
  long            t = 0;
  int             nz = 32 - l;

  while (l-- > 0)
    t = (t << 1) | 1;
  while (nz-- > 0)
    t = (t << 1);
  t = htonl (t);

  if (j_inet_ntop (&t, s, 256) == NULL) {
    strcpy (s, "255.255.255.0");
  }

  return t;
}

/*****************************************************************************
 *                                                                           * 
 *                                                                           *
 *****************************************************************************/
static int
add_network_rec (k, v)
     char           *k;
     char           *v;
{
  NETWORK_REC     r;
  int             n;
  char            s[256], t[256];
  char           *p;

  memset (&r, 0, sizeof (r));
  if (k == NULL || strlen (k) == 0)
    return 0;
  if (v == NULL)
    v = "";

  k += strspn (k, " \t\n\r");

  if ((n = strcspn (k, " \t\n\r")) == 0)
    return 0;

  strlcpy (s, k, n + 1);

  if ((p = strchr (s, '/')) != NULL) {
    /* format : n.n.n.n/m */
    *p++ = '\0';
    n = strspn (p, "1234567890");
    strlcpy (t, p, n + 1);
#if HAVE_STRTOL
    n = strtol (t, (char **) NULL, 10);
    if (errno == ERANGE || errno == EINVAL || n == 0 || n > 32)
      p = "255.255.255.0";
#else
    n = atoi (t);
    if (n <= 0 | n > 32)
      p = "255.255.255.0";
#endif
    else {
      mask_len2str (n, t);
      p = t;
    }

  } else {
    /* format : n.n.n.n m.m.m.m */
    p = k + n;
    if (*p)
      p += strspn (p, " \t\n\r");
    if (*p) {
      n = strcspn (p, " \t\n\r");
      strlcpy (t, p, n + 1);
      p = t;
    } else
      p = "255.255.255.0";
  }

  if (j_inet_pton (s, &r.net) <= 0)
    return 0;

  if (j_inet_pton (p, &r.mask) <= 0)
    return 0;

  r.net.s_addr &= r.mask.s_addr;

  if (strcasecmp (v, "LOCAL") == 0)
    r.class = N_LOCAL;
  if (strcasecmp (v, "DOMAIN") == 0)
    r.class = N_DOMAIN;
  if (strcasecmp (v, "FRIEND") == 0)
    r.class = N_FRIEND;


  strlcpy (r.k, s, sizeof (r.k));
  strlcpy (r.v, v, sizeof (r.v));

  return j_table_add (&htbl, &r);
}

/*****************************************************************************
 *                                                                           * 
 *                                                                           *
 *****************************************************************************/
int
load_networks_table (fname)
     char           *fname;
{
  int             res = 0;
  static int      htbl_ok = FALSE;

  pthread_mutex_lock (&st_mutex);

  if (htbl_ok == FALSE) {
    memset (&htbl, 0, sizeof (htbl));
    res = j_table_init (&htbl, sizeof (NETWORK_REC), 256, network_comp);
    if (res == 0)
      htbl_ok = TRUE;
  }
  if (res == 0)
    res = j_table_clear (&htbl);

  add_network_rec ("127.0.0.1 255.255.255.255", "LOCAL");

  if (res == 0) {
    if (use_db) {
      void           *dbh;

      if ((dbh = j_db_open (fname)) != NULL) {
        res = j_db_walk (dbh, add_network_rec);
        j_db_close (dbh);
      } else
        res = -1;
    } else {
      res = j_rd_text_file (fname, RD_TWO_COLUMN, RD_REVERSE, add_network_rec);
    }
  }

  if (res == 0)
    res = j_table_sort (&htbl);

  pthread_mutex_unlock (&st_mutex);

#if 0
  dump_networks_table ();
#endif

  return res;
}

/*****************************************************************************
 *                                                                           * 
 *                                                                           *
 *****************************************************************************/
int
check_host_class (host)
     char           *host;
{
  NETWORK_REC     p;
  struct in_addr  addr;

  int             ret = N_UNKNOWN;

  if (host == NULL)
    return N_UNKNOWN;

  memset (&p, 0, sizeof (p));

  if (j_inet_pton (host, &addr) <= 0)
    return N_UNKNOWN;

  pthread_mutex_lock (&st_mutex);
  if (j_table_get_first (&htbl, &p) == 0) {
    do {
      if (log_level >= 20)
        syslog (LOG_WARNING, "A=[%08X] M=[%08X] N=[%08X]",
                addr.s_addr, p.mask.s_addr, p.net.s_addr);
      if ((addr.s_addr & p.mask.s_addr) == p.net.s_addr) {
        if (log_level > 10)
          syslog (LOG_WARNING, "  ip=[%-16s], typenet=[%02X]", host, p.class);
        ret = p.class;
        break;
      }
    } while (j_table_get_next (&htbl, &p) == 0);
  }
  pthread_mutex_unlock (&st_mutex);

  return ret;
}
