/*
 *
 * j-chkmail - filtre de messagerie pour sendmail - MILTER
 *
 * Copyright (c) 2001, 2002 Ecole des Mines de Paris
 *
 *  Auteur     : Jose Marcio Martins da Cruz
 *               martins@paris.ensmp.fr
 *
 *  Historique :
 *  Creation     : janvier 2002
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <j-sys.h>

#include "j-chkmail.h"

#define DEBUG 0

void            usage (void);
int             j_survey ();

static pid_t    pid_parent = 0;
static pid_t    pid_filter = 0;
static pid_t    pid_avserv = 0;

static bool     avserv_enabled = TRUE;

static int      create_pid_file (void);
static void     remove_pid_file (void);

static int      DONE = FALSE;


static time_t   dt_cleanup_spool = 21600;
static time_t   quarantine_max_age = 172800;

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */

typedef struct {
  char           *name;
  int             log;
} table2log;

static table2log tablog[] = { {"host_access", FALSE},
{"user_access", FALSE},
{"users", FALSE},
{"networks", FALSE},
{"classw", FALSE},
{"regex", FALSE},
{NULL, FALSE}
};


int
main (argc, argv)
     int             argc;
     char           *argv[];
{
  int             c;

  const char     *args = "p:i:u:d:hvc:l:t:";

  char            logbuf[1024];
  char            s[256];

  j_output = J_OUT_ALL;

  opt = (OPT_REC *) malloc (sizeof (OPT_REC));
  memset (opt, 0, sizeof (OPT_REC));

  log_facility = LOG_LOCAL5;

  openlog ("j-chkmail", LOG_PID | LOG_NOWAIT | LOG_NDELAY, log_facility);

  init_default_file_extensions ();

  /* throttle_init (10000); */
  resolve_tab_init ();

  /* Process command line options 
     args
     p : socket
     inet:2000@localhost
     local:/var/sock
     i : 2000  (inet)
     u : /var/sock
     d : inet domain
     h : help
     c : configuration file
     l : log level
     v : version / compile time options
     version / configuration file options
     filter options
   */
  while ((c = getopt (argc, argv, args)) != -1) {
    switch (c) {
      case 'h':                /* OK */
        opt->arg_h = TRUE;
        usage ();
        exit (0);
        break;
      case 'v':
        opt->arg_v++;
        j_output = J_STDOUT;
        break;

        /*  */
      case 'c':
        if (optarg == NULL || *optarg == '\0') {
          (void) fprintf (stderr, "Erreur %s\n", optarg);
          exit (EX_USAGE);
        }
        if (opt->arg_c != NULL) {
          j_print_msg (j_output, LOG_ERR, "Only one c option, please");
          exit (1);
        }
        if ((opt->arg_c = strdup (optarg)) == NULL) {
          j_print_msg_sys (j_output, LOG_ERR,
                           "FATAL ERROR - memory allocation opt->arg_c");
          exit (1);
        }
        conf_file = opt->arg_c;
        break;

        /* */
        /* definition socket */
      case 'p':
        if (optarg == NULL || *optarg == '\0') {
          (void) fprintf (stderr, "Illegal conn: %s\n", optarg);
          exit (EX_USAGE);
        }
        if (opt->arg_p != NULL) {
          j_print_msg (j_output, LOG_ERR, "Only one p option, please");
          exit (1);
        }
        if ((opt->arg_p = strdup (optarg)) == NULL) {
          j_print_msg_sys (j_output, LOG_ERR,
                           "FATAL ERROR - memory allocation opt->arg_p");
          exit (1);
        }
        break;
      case 'u':
        if (optarg == NULL || *optarg == '\0') {
          (void) fprintf (stderr, "Erreur %s\n", optarg);
          exit (EX_USAGE);
        }
        if (opt->arg_u != NULL) {
          j_print_msg (j_output, LOG_ERR, "Only one u option, please");
          exit (1);
        }
        sprintf (s, "local:%s", optarg);
        if ((opt->arg_u = strdup (s)) == NULL) {
          j_print_msg_sys (j_output, LOG_ERR,
                           "FATAL ERROR - memory allocation opt->arg_u");
          exit (1);
        }
        break;
      case 'i':
        if (optarg == NULL || *optarg == '\0') {
          (void) fprintf (stderr, "Erreur %s\n", optarg);
          exit (EX_USAGE);
        }
        if (opt->arg_i != NULL) {
          j_print_msg (j_output, LOG_ERR, "Only one i option, please");
          exit (1);
        }
        sprintf (s, "inet:%s@localhost", optarg);
        if ((opt->arg_i = strdup (s)) == NULL) {
          j_print_msg_sys (j_output, LOG_ERR,
                           "FATAL ERROR - memory allocation opt->arg_i");
          exit (1);
        }
        break;

        /* */
      case 'd':
        if (optarg == NULL || *optarg == '\0') {
          (void) fprintf (stderr, "Illegal domain : %s\n", optarg);
          exit (EX_USAGE);
        }
        strcpy (domain, optarg);
        if (opt->arg_d != NULL) {
          j_print_msg (j_output, LOG_ERR, "Only one d option, please");
          exit (1);
        }
        if ((opt->arg_d = strdup (optarg)) == NULL) {
          j_print_msg_sys (j_output, LOG_ERR,
                           "FATAL ERROR - memory allocation opt->arg_d");
          exit (1);
        }
        break;

        /* */
      case 'l':
        if (optarg == NULL || *optarg == '\0') {
          (void) fprintf (stderr, "Erreur %s\n", optarg);
          exit (EX_USAGE);
        }
        if (opt->arg_l != NULL) {
          j_print_msg (j_output, LOG_ERR, "Only one l option, please");
          exit (1);
        }
        if ((opt->arg_l = strdup (optarg)) == NULL) {
          j_print_msg_sys (j_output, LOG_ERR,
                           "FATAL ERROR - memory allocation opt->arg_l");
          exit (1);
        }
        break;
      case 't':
        if (optarg == NULL || *optarg == '\0') {
          (void) fprintf (stderr, "Erreur %s\n", optarg);
          exit (EX_USAGE);
        } else {
          table2log      *p = tablog;

          while (p->name != NULL) {
            if (strcasecmp (p->name, optarg) == 0) {
              p->log = TRUE;
              opt->arg_t = TRUE;
              break;
            }
            p++;
          }
          if (p->name == NULL) {
            (void) fprintf (stderr, "Unknown table : %s\n", optarg);
            exit (EX_USAGE);
          }
        }
        break;
      default:
        (void) fprintf (stderr, "Error reading command line options : %c\n", c);
        exit (1);
    }
  }

  if (!opt->arg_t && !opt->arg_v) {
    snprintf (logbuf, sizeof (logbuf), "Starting %s", PACKAGE);
    j_print_msg (j_output, LOG_INFO, logbuf);
  }

  if (opt->arg_c != NULL)
    conf_file = opt->arg_c;

  configure (conf_file, 0);

  if (opt->arg_v) {
    snprintf (logbuf, sizeof (logbuf), "Checking domain : %s", domain ? domain : "--");
    j_print_msg (j_output, LOG_INFO, logbuf);
  }

  if (opt->arg_v) {
    dump_j_conf ();
    closelog ();
    exit (0);
  }

  {
    table2log      *t = tablog;
    int             log = FALSE;

    printf ("  %s\n", PACKAGE);
    printf ("  Compiled on %s %s\n", __DATE__, __TIME__);
    while (t->name != NULL) {
      if (t->log == TRUE) {
        log = TRUE;
        if (!strcasecmp ("host_access", t->name))
          dump_host_access_table ();
        if (!strcasecmp ("user_access", t->name))
          dump_user_access_table ();
        if (!strcasecmp ("classw", t->name))
          dump_classw_table ();
        if (!strcasecmp ("regex", t->name))
          dump_regex_table ();
        if (!strcasecmp ("networks", t->name))
          dump_networks_table ();
        if (!strcasecmp ("users", t->name))
          dump_local_users_table ();
      }
      t++;
    }
    if (log) {
      closelog ();
      exit (0);
    }
  }

  syslog (LOG_WARNING, "... Joe's j-chkmail OK !");

  return j_survey ();
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */

int             pipe_avserv[2];
int             pipe_filter[2];

static int      need_load_conf = FALSE;

static int      DT_ALARM = DT_SIGALRM;

static int      sighup_group = FALSE;

static int      last_cleanup = 0;


static          RETSIGTYPE
j_father_sig_handler (signo)
     int             signo;
{
  time_t          now = time (NULL);
  int             msg;

  signal (signo, j_father_sig_handler);
  if (log_level >= 20)
    syslog (LOG_DEBUG, "*** Received SIGNAL %d : %s", signo, ctime (&now));

  switch (signo) {
    case SIGTERM:
      msg = MSG_TERM;
      if (pid_avserv > 0)
        send_msg_channel (pipe_avserv, msg, CHAN_FATHER);
      if (pid_filter > 0)
        send_msg_channel (pipe_filter, msg, CHAN_FATHER);
      if (log_level >= 20)
        syslog (LOG_DEBUG, "%s : MSG SEND : %d", J_FUNCTION, msg);
      DONE = TRUE;
      break;
    case SIGUSR1:
      msg = MSG_DUMP;
      if (pid_avserv > 0)
        send_msg_channel (pipe_avserv, msg, CHAN_FATHER);
      if (pid_filter > 0)
        send_msg_channel (pipe_filter, msg, CHAN_FATHER);
      if (log_level >= 20)
        syslog (LOG_DEBUG, "%s : MSG SEND : %d", J_FUNCTION, msg);
      break;
    case SIGUSR2:
      reset_state ();
      msg = MSG_RESET;
      if (pid_avserv > 0)
        send_msg_channel (pipe_avserv, msg, CHAN_FATHER);
      if (pid_filter > 0)
        send_msg_channel (pipe_filter, msg, CHAN_FATHER);
      if (log_level >= 20)
        syslog (LOG_DEBUG, "%s : MSG SEND : %d", J_FUNCTION, msg);
      break;
    case SIGHUP:
      if (!sighup_group)
        need_load_conf = TRUE;
      sighup_group = FALSE;
      break;
    case SIGALRM:
#if 0
      msg = MSG_OK;
      if (pid_avserv > 0)
        send_msg_channel (pipe_avserv, msg, CHAN_FATHER);
      if (pid_filter > 0)
        send_msg_channel (pipe_filter, msg, CHAN_FATHER);
      if (log_level >= 20)
        syslog (LOG_DEBUG, "%s : MSG SEND : %d", J_FUNCTION, msg);
#endif
      break;
    default:
      syslog (LOG_WARNING, "%s : Undefined behavior for signal %d !",
              J_FUNCTION, signo);
      break;
  }
  alarm (DT_ALARM);
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
int
j_set_uid_gid (user, group)
     char           *user;
     char           *group;
{
  struct passwd  *pw;
  struct group   *gr;
  int             res = 0;

  uid_t           uid = getuid ();
  gid_t           gid = getgid ();

  if (uid != 0) {
    syslog (LOG_WARNING, "%s : Only root can set user and group", J_FUNCTION);
    return 0;
  }

  if ((gr = getgrnam (group)) != NULL) {
    syslog (LOG_DEBUG, "%s : GID DE %s : %d", J_FUNCTION, group, gr->gr_gid);
    if (gid == gr->gr_gid) {
      syslog (LOG_WARNING, "%s : j-chkmail is already running as group %s",
              J_FUNCTION, group);
    } else {
      if (uid == 0 && setregid (gr->gr_gid, gr->gr_gid) < 0) {
        syslog (LOG_ERR, "%s : Can't set process gid = %d : %s",
                J_FUNCTION, gr->gr_gid, strerror (errno));
        return 1;
      }
    }
  } else {
    syslog (LOG_ERR, "%s : Error getgrnam %s", J_FUNCTION, strerror (errno));
    return 1;
  }

  if ((pw = getpwnam (user)) != NULL) {
    syslog (LOG_DEBUG, "%s : UID DE %s : %d", J_FUNCTION, user, pw->pw_uid);
    if (uid == pw->pw_uid) {
      syslog (LOG_WARNING, "%s : j-chkmail is already running as user %s",
              J_FUNCTION, user);
    } else {
      if (uid == 0 && setreuid (pw->pw_uid, pw->pw_uid) < 0) {
        syslog (LOG_ERR, "%s : Can't set process uid = %d : %s",
                J_FUNCTION, pw->pw_uid, strerror (errno));
        return 1;
      }
    }
  } else {
    syslog (LOG_ERR, "%s : Error getpwnam %s", J_FUNCTION, strerror (errno));
    return 1;
  }

  return res;
}

/* ************************************
 *                                    * 
 *                                    *
 ************************************ */
int
daemon_init ()
{
  char           *user, *group;
  int             fd;

  printf (" Let's daemonize j-chkmail...\n");
  switch (fork ()) {
    case 0:
      printf (" j-chkmail daemonized !\n");
      break;
    case -1:
      perror (" Error daemonizing j-chkmail ");
      exit (1);
      break;
    default:
      exit (0);
  }

#if 0
  if (setsid () < 0) {
    syslog (LOG_ERR, "%s : Can't set process session leader : %s",
            J_FUNCTION, strerror (errno));
  }
#else
  if (setpgid (0, 0) < 0) {
    syslog (LOG_ERR, "%s : Can't set process group leader : %s",
            J_FUNCTION, strerror (errno));
  }
#endif

  chdir ("/");

  signal (SIGHUP, SIG_IGN);
  switch (fork ()) {
    case 0:
      break;
    case -1:
      perror (" Error daemonizing j-chkmail ");
      exit (1);
      break;
    default:
      exit (0);
  }

  umask (0000);

  if ((fd = open ("/dev/null", O_RDONLY, 0)) < 0) {
    syslog (LOG_ERR, "%s : Can't open /dev/null read-only : %s", J_FUNCTION,
            strerror (errno));
  }
  if (dup2 (fd, STDIN_FILENO) < 0) {
    syslog (LOG_ERR, "%s : Can't redirect stdin : %s", J_FUNCTION, strerror (errno));
  }
  close (fd);
  if ((fd = open ("/dev/null", O_WRONLY, 0)) < 0) {
    syslog (LOG_ERR, "%s : Can't open /dev/null write-only : %s",
            J_FUNCTION, strerror (errno));
  }
  if (dup2 (fd, STDOUT_FILENO) < 0) {
    syslog (LOG_ERR, "%s : Can't redirect stdout : %s", J_FUNCTION, strerror (errno));
  }
  if (dup2 (fd, STDERR_FILENO) < 0) {
    syslog (LOG_ERR, "%s : Can't redirect stderr : %s", J_FUNCTION, strerror (errno));
  }
  close (fd);

  if ((user = cf_get_str (CF_USER)) == NULL || strlen (user) == 0)
    user = RUN_AS_USER;
  if ((group = cf_get_str (CF_GROUP)) == NULL || strlen (group) == 0)
    group = RUN_AS_GROUP;
  j_set_uid_gid (user, group);

  closelog ();
  openlog ("j-chkmail", LOG_PID | LOG_NOWAIT | LOG_NDELAY, log_facility);

#if HAVE_THR_SETCONCURRENCY
  {
#define  CONCURRENCY_LEVEL  10
    int             n = thr_getconcurrency ();

    if (log_level > 15)
      syslog (LOG_DEBUG, "thr_getconcurrency = %d", thr_getconcurrency ());
    if (n < CONCURRENCY_LEVEL) {
      int             err = 0;

      if ((err = thr_setconcurrency (CONCURRENCY_LEVEL)) != 0) {
        syslog (LOG_ERR, "%s : th_setconcurrency error %d %s", J_FUNCTION,
                err, strerror (errno));
      }
    }
    if (log_level > 15)
      syslog (LOG_DEBUG, "thr_getconcurrency = %d", thr_getconcurrency ());
  }
#endif

  setup_file_descriptors ();

  return 0;
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
int
j_survey ()
{
  pid_t           res_filter = 0;
  pid_t           res_avserv = 0;

  time_t          now = time (NULL);

  daemon_init ();

  init_proc_state ();

  pid_parent = getpid ();
  create_pid_file ();

  atexit (remove_pid_file);

  dt_cleanup_spool = cf_get_int (CF_CLEANUP_INTERVAL);;
  quarantine_max_age = cf_get_int (CF_QUARANTINE_MAX_AGE);

  alarm (2 * DT_ALARM);

  signal (SIGTERM, j_father_sig_handler);
  signal (SIGUSR1, j_father_sig_handler);
  signal (SIGUSR2, j_father_sig_handler);
  signal (SIGHUP, j_father_sig_handler);
  signal (SIGALRM, j_father_sig_handler);
  signal (SIGPIPE, SIG_IGN);

  avserv_enabled = (cf_get_int (CF_AV_ACTION) != OPT_OK);
  if (avserv_enabled) {
    char           *fname = cf_get_str (CF_AV_PATH);
    int             res = 0;

    for (;;) {
      int             res = access (fname, X_OK);

      if (res == 0)
        break;
      if (errno == EINTR)
        continue;
    }
    if (res != 0) {
      syslog (LOG_ERR, "%s : External scanner access check : %s",
              J_FUNCTION, strerror (errno));
      avserv_enabled = FALSE;
    }
  }

  while (!DONE) {

    if (log_level >= 20)
      syslog (LOG_DEBUG, "%s : while(!DONE) ...", J_FUNCTION);

    if (pid_filter == 0) {
      int             flag;

      close (pipe_filter[0]);
      close (pipe_filter[1]);
      if (open_channel (pipe_filter) < 0) {
        syslog (LOG_ERR, "%s : pipe(pipe_filter) : %s", J_FUNCTION, strerror (errno));
        exit (1);
      }

      if ((pid_filter = fork ()) == -1) {
        syslog (LOG_ERR, "%s : Forking j-chkmail : %s", J_FUNCTION, strerror (errno));
        exit (1);
      }
      if (pid_filter == 0) {
        close (pipe_filter[1]);

        if ((flag = fcntl (pipe_filter[0], F_GETFL, 0)) < 0) {
          syslog (LOG_ERR, "%s : can't get pipe status : %s", J_FUNCTION,
                  strerror (errno));
        } else {
          if (fcntl (pipe_filter[0], F_SETFL, flag | O_NDELAY) < 0) {
            syslog (LOG_ERR, "%s : fcntl pipe_filter[0] : %s", J_FUNCTION,
                    strerror (errno));
          }
        }
        /* JOE - XXX - throttle_init(10000); */
        return j_chkmail ();
      }

      if (log_level >= 15)
        syslog (LOG_DEBUG, "%s : pid_filter : %d", J_FUNCTION, pid_filter);

      close (pipe_filter[0]);

      if ((flag = fcntl (pipe_filter[1], F_GETFL, 0)) < 0) {
        syslog (LOG_ERR, "%s : can't get pipe status : %s", J_FUNCTION,
                strerror (errno));
      } else {
        if (fcntl (pipe_filter[1], F_SETFL, flag | O_NDELAY) < 0) {
          syslog (LOG_ERR, "%s : fcntl pipe_filter[1] : %s", J_FUNCTION,
                  strerror (errno));
        }
      }

    }

    if (avserv_enabled == TRUE) {
      if (pid_avserv == 0) {
        int             flag;

        close (pipe_avserv[0]);
        close (pipe_avserv[1]);
        if (open_channel (pipe_avserv) < 0) {
          syslog (LOG_ERR, "%s : pipe(pipe_avserv) : %s", J_FUNCTION, strerror (errno));
          exit (1);
        }
        if ((pid_avserv = fork ()) == -1) {
          syslog (LOG_ERR, "%s : Forking avserv : %s", J_FUNCTION, strerror (errno));
          exit (1);
        }
        if (pid_avserv == 0) {
          close (pipe_avserv[1]);

          if ((flag = fcntl (pipe_avserv[0], F_GETFL, 0)) < 0) {
            syslog (LOG_ERR, "%s : can't get pipe status : %s", J_FUNCTION,
                    strerror (errno));
          } else {
            if (fcntl (pipe_avserv[0], F_SETFL, flag | O_NDELAY) < 0) {
              syslog (LOG_ERR, "%s : fcntl pipe_avserv[1] : %s", J_FUNCTION,
                      strerror (errno));
            }
          }

          return j_avserver ();
        }

        if (log_level >= 15)
          syslog (LOG_DEBUG, "%s : pid_avserv : %d", J_FUNCTION, pid_avserv);

        close (pipe_avserv[0]);

        if ((flag = fcntl (pipe_avserv[1], F_GETFL, 0)) < 0) {
          syslog (LOG_ERR, "%s : can't get pipe status : %s", J_FUNCTION,
                  strerror (errno));
        } else {
          if (fcntl (pipe_avserv[1], F_SETFL, flag | O_NDELAY) < 0) {
            syslog (LOG_ERR, "%s : fcntl pipe_avserv[1] : %s", J_FUNCTION,
                    strerror (errno));
          }
        }

      }
    }

    /* While not working, I'll wait here for the next signal... */
    sleep (DT_ALARM);

    now = time (NULL);

    /* send life signal */
    {
      int             msg = MSG_OK;

      if (pid_avserv > 0)
        send_msg_channel (pipe_avserv, msg, CHAN_FATHER);
      if (pid_filter > 0)
        send_msg_channel (pipe_filter, msg, CHAN_FATHER);
      if (log_level >= 20)
        syslog (LOG_DEBUG, "%s : MSG SEND : %d", J_FUNCTION, msg);
    }

    /* Well ! My boss want to reconfigure */
    if (need_load_conf) {
      syslog (LOG_WARNING, "LETS RECONFIGURE...");

      configure (conf_file, 0);

      setup_file_descriptors ();

      avserv_enabled = (cf_get_int (CF_AV_ACTION) != OPT_OK);
      if (avserv_enabled) {
        char           *fname = cf_get_str (CF_AV_PATH);
        int             res = 0;

        for (;;) {
          int             res = access (fname, X_OK);

          if (res == 0)
            break;
          if (errno == EINTR)
            continue;
        }
        if (res != 0) {
          syslog (LOG_ERR, "%s : External scanner access check : %s",
                  J_FUNCTION, strerror (errno));
          avserv_enabled = FALSE;
        }
      }

      dt_cleanup_spool = cf_get_int (CF_CLEANUP_INTERVAL);;
      quarantine_max_age = cf_get_int (CF_QUARANTINE_MAX_AGE);

      sighup_group = TRUE;
      kill (0, SIGHUP);
      alarm (DT_ALARM);
      need_load_conf = FALSE;
    }

    /* Is j-chkmail filter running ??? */
    if (pid_filter > 0) {
      res_filter = waitpid (pid_filter, NULL, WNOHANG);
      if (log_level >= 20)
        syslog (LOG_DEBUG, "%s : PID FILTER : %5d - RES %5d\n", J_FUNCTION,
                pid_filter, res_filter);

      if (res_filter == -1) {
        syslog (LOG_ERR, "%s : waitpid(pid_filter) : %s", J_FUNCTION, strerror (errno));
        exit (1);
      }
      /* does child died ? */
      if (res_filter == pid_filter) {
        remove_milter_unix_sock ();
        pid_filter = 0;
      }
    }

    /* Is av server running ??? */
    if (pid_avserv > 0) {
      res_avserv = waitpid (pid_avserv, NULL, WNOHANG);
      if (log_level >= 20)
        syslog (LOG_DEBUG, "%s : PID AVSERV : %5d - RES %5d\n", J_FUNCTION,
                pid_avserv, res_avserv);

      if (res_avserv == -1) {
        syslog (LOG_ERR, "%s : waitpid(pid_avserv) : %s", J_FUNCTION, strerror (errno));
        exit (1);
      }
      /* does child died ? */
      if (res_avserv == pid_avserv) {
        pid_avserv = 0;
      }
    }

    /* cleans quarantine spool dir */
    if ((dt_cleanup_spool > 0) && (now > last_cleanup + dt_cleanup_spool)) {
      char           *spooldir = cf_get_str (CF_SPOOLDIR);

      cleanup_spool (spooldir, quarantine_max_age);
      last_cleanup = time (NULL);
    }
  }

  sleep (2);
  remove_pid_file ();

  exit (0);
}


/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
void
usage ()
{
  printf ("Usage : j-chkmail options\n");
  printf ("  %s\n", PACKAGE);
  printf ("  Compiled on %s %s\n", __DATE__, __TIME__);
  printf ("        -p  : socket\n");
  printf ("              inet:2000@localhost\n");
  printf ("              local:/var/sock\n");
  printf ("        -i  : 2000  (AF_INET)\n");
  printf ("        -u  : /var/sock (AF_UNIX)\n");
  printf ("        -d  : inet domain\n");
  printf ("        -h  : help\n");
  printf ("        -c  : configuration file\n");
  printf ("        -l  : log level\n");
  printf ("        -v  : version / runtime configuration\n");
  printf ("        -vv : version / compile time configuration\n");
  printf ("        -t tablename, where tablename choosen between : \n");
  {
    table2log      *p = tablog;

    printf ("               ");
    while (p->name != NULL) {
      printf ("%s %s ", p == tablog ? "" : "|", p->name);
      p++;
    }
    printf ("\n");
  }
  printf ("\n%s - Copyright Ecole des Mines de Paris - (C) 2002\n\n", PACKAGE);
}


/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */

static int
create_pid_file (void)
{
  FILE           *fpid;
  char            logbuf[1024];
  char           *pid_file;

  if ((pid_file = cf_get_str (CF_PID_FILE)) == NULL || strlen (pid_file) == 0) {
    j_print_msg (j_output, LOG_ERR, "pid_file : NULL pointer");
    return 1;
  }

  if (access (pid_file, F_OK) == 0) {
    snprintf (logbuf, sizeof (logbuf),
              "PID_FILE %s exists. Is there another j-chkmail running ?", pid_file);
    syslog (LOG_ERR, logbuf);
    fprintf (stderr, "%s\n", logbuf);
    exit (1);
  }

  if ((fpid = fopen (pid_file, "w")) != NULL) {
    fprintf (fpid, "%d\n", (int) getpid ());

    fclose (fpid);
    if (chmod (pid_file, S_IRUSR | S_IRGRP | S_IROTH) != 0) {
      syslog (LOG_ERR, "%s : error changing pid file mode %s",
              J_FUNCTION, strerror (errno));
    }
    return 0;
  }
  snprintf (logbuf, sizeof (logbuf), "%s : Error opening %s file", J_FUNCTION,
            pid_file);
  j_print_msg_sys (j_output, LOG_ERR, logbuf);
  return 1;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
static void
remove_pid_file (void)
{
  char           *pid_file;
  pid_t           pid = getpid ();

  if (log_level >= 20)
    syslog (LOG_DEBUG, "%s : remove_pid_file : %d %d", J_FUNCTION, pid, pid_parent);

  if (pid <= 0 || pid != pid_parent)
    return;

  if ((pid_file = cf_get_str (CF_PID_FILE)) == NULL || strlen (pid_file) == 0) {
    j_print_msg (j_output, LOG_ERR, "pid_file : NULL pointer");
    return;
  }

  remove (pid_file);
}
