/*
 *
 * j-chkmail - filtre de messagerie pour sendmail - MILTER
 *
 * Copyright (c) 2001, 2002 Ecole des Mines de Paris
 *
 *  Auteur     : Jose Marcio Martins da Cruz
 *               martins@paris.ensmp.fr
 *
 *  Historique :
 *  Creation     : janvier 2002
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <j-sys.h>

#include "j-chkmail.h"

#undef  DEBUG
#define DEBUG 0

#define REGEX_MAX_LEN         256

#define REGCOMP_FLAGS         (REG_ICASE | REG_NEWLINE | REG_EXTENDED)

/*****************************************************************************
 *                                                                           * 
 *                                                                           *
 *****************************************************************************/

typedef struct {
  char            action[32];
  int             where;
  char            regex[REGEX_MAX_LEN];
  int             score;
  regex_t         re;
  int             re_ok;
  int             count;
} REGEX_REC;

static j_table  htbl;

static pthread_mutex_t st_mutex = PTHREAD_MUTEX_INITIALIZER;

static int      use_db = FALSE;

static bool     save_found_regex (unsigned long, int, int, char *);

/*****************************************************************************
 *                                                                           * 
 *                                                                           *
 *****************************************************************************/
void
dump_regex_table ()
{
  REGEX_REC       p;

  printf ("*** Regular Expressions lookup table : \n");
  if (j_table_get_first (&htbl, &p) == 0) {
    printf (" ** WHERE        : SCORE - Regular Expression\n");
    do {
      printf (" -> %-12s : %5d - /%s/\n", p.action, p.score, p.regex);
    } while (j_table_get_next (&htbl, &p) == 0);
  }
}

/*****************************************************************************
 *                                                                           * 
 *                                                                           *
 *****************************************************************************/
static int
add_regex_rec (k, v)
     char           *k;
     char           *v;
{
  REGEX_REC       r;
  int             i, j;

  memset (&r, 0, sizeof (r));
  if (k == NULL || strlen (k) == 0)
    return 1;
  if (v == NULL || strlen (v) == 0)
    return 1;

  r.where = MAIL_ANYWHERE;
  if (strcasecmp (v, "SUBJECT") == 0)
    r.where = MAIL_SUBJECT;
  if (strcasecmp (v, "BODY") == 0)
    r.where = MAIL_BODY;
  if (strcasecmp (v, "HEADERS") == 0)
    r.where = MAIL_HEADERS;
  if (strcasecmp (v, "ANYWHERE") == 0)
    r.where = MAIL_ANYWHERE;

  r.score = 1;

  if ((i = strspn (k, "0123456789")) == (j = strcspn (k, " \t"))) {
    char            s[32];

    if (i > 0) {
      strlcpy (s, k, sizeof (s));
#if HAVE_STRTOL
      r.score = strtol (k, (char **) NULL, 10);
      if (errno == ERANGE || errno == EINVAL)
        r.score = 1;
#else
      r.score = atoi (k);
#endif
      k += strspn (k, "01234567890");
      k += strspn (k, " \t");
    }
  }

  strlcpy (r.regex, k, sizeof (r.regex));

  r.re_ok = regcomp (&r.re, r.regex, REGCOMP_FLAGS);
  if (r.re_ok != 0) {
    char            s[256];

    regerror (r.re_ok, &r.re, s, sizeof (s));
    syslog (LOG_WARNING, "%s : %s", J_FUNCTION, s);
  }

  strlcpy (r.action, v, sizeof (r.action));

  return j_table_add (&htbl, &r);
}

/*****************************************************************************
 *                                                                           * 
 *                                                                           *
 *****************************************************************************/
static void
clear_compiled_regex ()
{
  REGEX_REC       p;

  if (j_table_get_first (&htbl, &p) == 0) {
    do {
      regfree (&p.re);
    } while (j_table_get_next (&htbl, &p) == 0);
  }
}

/*****************************************************************************
 *                                                                           * 
 *                                                                           *
 *****************************************************************************/
int
load_regex_table (fname)
     char           *fname;
{
  int             res = 0;
  static int      htbl_ok = FALSE;

  pthread_mutex_lock (&st_mutex);

  if (htbl_ok == FALSE) {
    memset (&htbl, 0, sizeof (htbl));
    res = j_table_init (&htbl, sizeof (REGEX_REC), 256, NULL);
    if (res == 0)
      htbl_ok = TRUE;
  }
  if (res == 0) {
    clear_compiled_regex ();
    res = j_table_clear (&htbl);
  }

  if (res == 0) {
    if (use_db) {
      void           *dbh;

      if ((dbh = j_db_open (fname)) != NULL) {
        res = j_db_walk (dbh, add_regex_rec);
        j_db_close (dbh);
      } else
        res = -1;
    } else {
      res = j_rd_text_file (fname, RD_TWO_COLUMN, RD_REVERSE, add_regex_rec);
    }
  }

  if (res == 0)
    res = j_table_sort (&htbl);

  pthread_mutex_unlock (&st_mutex);

#if 0
  dump_regex_table ();
#endif

  return res;
}

/*****************************************************************************
 *                                                                           * 
 *                                                                           *
 *****************************************************************************/
int
check_regex (conn_id, msg, where)
     unsigned long   conn_id;
     char           *msg;
     int             where;
{
  REGEX_REC       p;
  int             result = 0;
  regmatch_t      rm;

  if (msg == NULL || strlen (msg) == 0)
    return result;

  pthread_mutex_lock (&st_mutex);
  if (j_table_get_first (&htbl, &p) == 0) {
    do {
      if (p.where == MAIL_ANYWHERE || p.where == where) {
        char           *ptr = msg;
        int             nb = 0;

        if (p.re_ok != 0) {
          p.re_ok = regcomp (&p.re, p.regex, REGCOMP_FLAGS);
          if (p.re_ok != 0) {
            char            s[256];

            regerror (p.re_ok, &p.re, s, sizeof (s));
            syslog (LOG_WARNING, "%s : %s", J_FUNCTION, s);
            break;
          }
        }
        while (!regexec (&p.re, ptr, (size_t) 1, &rm, 0)) {
          ptr += rm.rm_eo;
          if (p.score >= 0)
            result += p.score;
          else
            result++;
          nb++;
        }
        if ((nb > 0) && (result > 0))
          (void) save_found_regex (conn_id, nb, result, p.regex);
      }
    } while (j_table_get_next (&htbl, &p) == 0);
  }
  pthread_mutex_unlock (&st_mutex);

  return result;
}

/*****************************************************************************
 *                                                                           * 
 *                                                                           *
 *****************************************************************************/
static int      fd = -1;
static int      nberr = 0;

#ifdef MAX_ERR
#undef MAX_ERR
#endif

#define MAX_ERR    16

static          bool
save_found_regex (id, nb, score, expr)
     unsigned long   id;
     int             nb;
     int             score;
     char           *expr;
{
  char           *regex_log = cf_get_str (CF_REGEX_LOG_FILE);
  char            sout[256];

  if (cf_get_int (CF_DUMP_FOUND_REGEX) == OPT_NO)
    return TRUE;

  if (nberr > MAX_ERR)
    return FALSE;

  if (fd < 0) {
    if ((regex_log == NULL) || (strlen (regex_log) == 0)) {
      syslog (LOG_ERR, "%s : CF_REGEX_LOG file : bad filename", J_FUNCTION);
      nberr++;
      return FALSE;
    }

    fd = open (regex_log, O_WRONLY | O_APPEND | O_CREAT, 0644);
    if (fd < 0) {
      syslog (LOG_ERR, "%s : Error opening %s : %s", J_FUNCTION, regex_log,
              strerror (errno));
      nberr++;
      return FALSE;
    }
  }

  snprintf (sout, sizeof (sout), "%10ld %08lX %3d %3d %s\n", id, id, nb, score, expr);
  if (write (fd, sout, strlen (sout)) != strlen (sout)) {
    syslog (LOG_ERR, "%s : Error writing on REGEX_LOG file : %s",
            J_FUNCTION, strerror (errno));
    fd = -1;
    nberr++;
    return FALSE;
  }

  nberr = 0;
  return TRUE;
}
