/*
 *
 * j-chkmail - filtre de messagerie pour sendmail - MILTER
 *
 * Copyright (c) 2001, 2002 Ecole des Mines de Paris
 *
 *  Auteur     : Jose Marcio Martins da Cruz
 *               martins@paris.ensmp.fr
 *
 *  Historique :
 *  Creation     : janvier 2002
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <j-sys.h>

#include "j-inet.h"


/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
int
j_gethostbyipaddr (ip, name, len)
     char           *ip;
     char           *name;
     int             len;
{
  struct hostent *hp;
  struct in_addr  addr;
  int             res = 0;


  if (ip == NULL || len == 0 || name == NULL)
    return res;

  *name = '\0';

  if (inet_pton (AF_INET, ip, &addr)) {
#if HAVE_GETIPNODEBYADDR
    int             error_num;

    hp = getipnodebyaddr (&addr, sizeof (addr), AF_INET, &error_num);
#else
    hp = gethostbyaddr (&addr, sizeof (addr), AF_INET);
#endif
    if (hp != NULL)
      snprintf (name, len, "%s", hp->h_name);
    res = (hp != NULL);
#if HAVE_GETIPNODEBYADDR
    freehostent (hp);
#endif
    return res;
  }
  return res;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
int
j_gethostbyname (name, ip, len)
     char           *name;
     char           *ip;
     int             len;
{
  struct hostent *hp;
  int             res = 0;

#if HAVE_GETIPNODEBYNAME
  int             error_num = 0;
#endif

  if (ip == NULL || len == 0 || name == NULL)
    return res;

  *ip = '\0';

#if HAVE_GETIPNODEBYNAME
  hp = getipnodebyname (name, AF_INET, AI_DEFAULT, &error_num);
#else
  hp = gethostbyname (name);
#endif

  if (hp != NULL)
    (void) j_inet_ntop (hp->h_addr_list[0], ip, len);

  res = (hp != NULL);
#if HAVE_GETIPNODEBYNAME
  freehostent (hp);
#endif
  return res;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
char           *
j_inet_ntop (addr, cp, size)
     void           *addr;
     char           *cp;
     size_t          size;
{
#if HAVE_INET_NTOP
  return (char *) inet_ntop (AF_INET, addr, cp, size);
#else
#if HAVE_INET_NTOA
  char           *p;

  if ((p = inet_ntoa (addr)) != NULL) {
    snprintf (cp, size, "%s", p);
    return cp;
  } else
    return NULL;
#else
  return NULL;
#endif
#endif
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
int
j_inet_pton (cp, addr)
     char           *cp;
     void           *addr;
{
#if HAVE_INET_PTON
  return inet_pton (AF_INET, cp, addr);
#else
#ifdef HAVE_INET_ATON
  return inet_aton (cp, addr) == 0 ? 0 : 1;
#else
  return 0;
#endif
#endif
}


/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
int
ip_strcmp (a, b)
     char           *a;
     char           *b;
{
  struct in_addr  aa, ab;

  if (j_inet_pton (b, &ab.s_addr) && j_inet_pton (a, &aa.s_addr)) {
    if (htonl (aa.s_addr) > htonl (ab.s_addr))
      return 1;
    if (htonl (aa.s_addr) < htonl (ab.s_addr))
      return -1;
    return 0;
  }

  return strcasecmp (a, b);
}


/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */

#if HAVE_POLL
# define J_POLL_RD_FLAGS (POLLIN | POLLPRI)
# define J_POLL_WR_FLAGS (POLLOUT)

#endif             /* HAVE_POLL */


int
j_fd_ready (sd, fdmode, to)
     int             sd;
     bool            fdmode;
     long            to;
{
#if HAVE_POLL
  struct pollfd   pfd;
  int             r;
  time_t          now = time (NULL);

  int             flags;

  flags = (fdmode == J_SOCK_READ ? J_POLL_RD_FLAGS : J_POLL_WR_FLAGS);

  for (;;) {
    pfd.fd = sd;
    pfd.events = flags;
    pfd.revents = 0;
    r = poll (&pfd, 1, to);
    if (r == 0)
      return J_SOCK_TIMEOUT;
    if (r < 0) {
      if (errno == EINTR)
        continue;
      return J_SOCK_ERROR;
    }
    if ((pfd.revents & (POLLERR | POLLHUP | POLLNVAL)) != 0)
      return J_SOCK_ERROR;
    if ((pfd.revents & flags) != 0)
      return J_SOCK_READY;
    break;
    if (time (NULL) > (now + to / 1000))
      return J_SOCK_TIMEOUT;
  }
  syslog (LOG_ERR,
          "ERROR=poll:mi_rd_socket_ready, socket=%d, r=%d, errno=%d revents=%d",
          sd, r, errno, pfd.revents);
  return J_SOCK_ERROR;
#else              /* HAVE_POLL */
  fd_set          rdfs, excfs;
  int             r;
  struct timeval  timeout;

  for (;;) {
    timeout.tv_usec = 1000 * (to % 1000);
    timeout.tv_sec = to / 1000;
    FD_ZERO (&rdfs);
    FD_SET ((unsigned int) sd, &rdfs);
    FD_ZERO (&excfs);
    FD_SET ((unsigned int) sd, &excfs);
    if (fdmode == J_SOCK_READ)
      r = select (sd + 1, &rdfs, NULL, &excfs, timeout);
    else
      r = select (sd + 1, NULL, &rdfs, &excfs, timeout);
    if (r == 0)
      return J_SOCK_TIMEOUT;
    if (r < 0) {
      if (errno == EINTR)
        continue;
      return J_SOCK_ERROR;
    }
    if (FD_ISSET (sd, &excfs))
      return J_SOCK_ERROR;
    if (FD_ISSET (sd, &rdfs))
      return J_SOCK_READY;
  }
  return J_SOCK_ERROR;
#endif             /* HAVE_POLL */
}
