/*
 *
 * j-chkmail - filtre de messagerie pour sendmail - MILTER
 *
 * Copyright (c) 2001, 2002 Ecole des Mines de Paris
 *
 *  Auteur     : Jose Marcio Martins da Cruz
 *               martins@paris.ensmp.fr
 *
 *  Historique :
 *  Creation     : janvier 2002
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <j-sys.h>

#if HAVE_SEARCH_H
#include <search.h>
#endif

#include "j-chkmail.h"

#include "j-filter.h"


/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
static unsigned long HISTORY_ENTRIES = 0x8000L;

#define    SZ_IP           32

struct HistRaw_T {
  uint32_t        signature;
  time_t          conn_id;
  char            ip[SZ_IP];

#if HAVE_HRTIME_T
  hrtime_t        t_open;
  hrtime_t        t_close;
#else
  time_t          t_open;
  time_t          t_close;
#endif
  long            delay;

  short           result;
  short           ip_class;
  short           throttle;
  short           host_access;
  short           resolve_res;
  short           resolve_grant;

  short           nb_rcpt;
  short           nb_files;
  short           nb_xfiles;
  short           nb_virus;
  short           nb_policy;
  short           nb_msgs;

  unsigned long   nb_bytes;

  short           rej_regex;
  short           rej_throttle;
  short           rej_resolve;
  short           rej_rcpt;
  short           rej_luser;

  short           dummy[15];
};


struct HistRes_T {
  char            ip[SZ_IP];

  time_t          ti;
  time_t          tf;

  int             nb_conn;
  int             nb_msgs;
  unsigned long   nb_bytes;
  int             nb_rcpt;
  int             nb_files;
  int             nb_xfiles;
  int             nb_virus;
  int             nb_policy;
  int             nb_reject;

  int             resolve_res;
  int             resolve_grant;

  int             throttle_max;
  int             ip_class;
  int             host_access;

  int             rej_resolve;
  int             rej_resolve_failed;
  int             rej_resolve_forged;
  int             rej_throttle;
  int             rej_rcpt;
  int             rej_regex;
  int             rej_luser;

  STATS           st_delay;
  long            delay_max;
  long            delay_min;
};


static void     ctx2histraw (HistRaw_T *, mlfiPriv *);
static void     histraw2histres (HistRes_T *, HistRaw_T *);

struct History_T {
  size_t          nb;
  long            dim;
  size_t          step;
  time_t          ptr;
  HistRes_T      *p;
  HistRes_T       glob;
};

static History_T history = HISTORY_T_INIT;

static void     res_history_clear (History_T *);

static bool     res_history_resize (History_T *);

static bool     res_history_full (History_T *);

static HistRes_T *res_history_lookup (History_T *, char *);

static void     res_history_add_entry (History_T *, HistRaw_T *);

static int      histres_cmp (const void *, const void *);

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
struct RawData_T {
  int             fd;
  long            ptr;
  pthread_mutex_t st_mutex;
};

typedef struct RawData_T RawData_T;

static RawData_T hfile = { -1, 0, PTHREAD_MUTEX_INITIALIZER };



/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
bool
raw_history_open (ronly)
     bool            ronly;
{
  char           *work_dir = cf_get_str (CF_WORKDIR);
  char            fname[256];
  uint32_t        history_entries = cf_get_int (CF_HISTORY_ENTRIES);

  if (history_entries > 0)
    HISTORY_ENTRIES = history_entries * 1024;

  if (work_dir == NULL)
    work_dir = J_WORKDIR;

  snprintf (fname, sizeof (fname), "%s/%s", work_dir, "j-history");

  pthread_mutex_lock (&hfile.st_mutex);
  if (hfile.fd < 0) {
    HistRaw_T       h;
    size_t          ind;
    time_t          idmax;
    ssize_t         r;

    mode_t          mode;
    int             oflag;

    if (ronly) {
      mode = (S_IRUSR | S_IRGRP | S_IROTH);
      oflag = O_RDONLY;
    } else {
      mode = (S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
      oflag = (O_RDWR | O_CREAT);
    }

    hfile.fd = open (fname, oflag, mode);
    if (hfile.fd < 0) {
      syslog (LOG_ERR, "%s : open error on history file : %s",
              J_FUNCTION, strerror (errno));
      pthread_mutex_unlock (&hfile.st_mutex);
      return FALSE;
    }
    ind = 0;
    idmax = 0;
    while ((r = read (hfile.fd, &h, sizeof (h))) == sizeof (h)) {
      if (h.conn_id > idmax) {
        idmax = h.conn_id;
        hfile.ptr = ind;
      }
      ind++;
    }
    if (r < 0)
      syslog (LOG_ERR, "%s : read error on history file : %s",
              J_FUNCTION, strerror (errno));
  }
  pthread_mutex_unlock (&hfile.st_mutex);

  return TRUE;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
void
raw_history_close ()
{
  if (hfile.fd >= 0)
    close (hfile.fd);
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
bool
raw_history_add_entry (ctx)
     SMFICTX        *ctx;
{
  struct mlfiPriv *priv = MLFIPRIV;
  HistRaw_T       history;

  if (priv == NULL)
    return TRUE;

  if (hfile.fd < 0) {
    (void) raw_history_open (FALSE);
  }

  pthread_mutex_lock (&hfile.st_mutex);
  if (hfile.fd >= 0) {
    ctx2histraw (&history, priv);

    if (lseek (hfile.fd, hfile.ptr * sizeof (history), SEEK_SET) == (off_t) - 1)
      syslog (LOG_INFO, "%08lX %s : lseek error on history file : %s",
              priv->conn_id, J_FUNCTION, strerror (errno));

    if (write (hfile.fd, &history, sizeof (history)) < 0)
      syslog (LOG_INFO, "%08lX %s : write error on history file : %s",
              priv->conn_id, J_FUNCTION, strerror (errno));

    hfile.ptr++;
    hfile.ptr = hfile.ptr % HISTORY_ENTRIES;
  }

  pthread_mutex_unlock (&hfile.st_mutex);

  return TRUE;
}


/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */

void
res_history_update (hst, ip, tf, dt)
     History_T      *hst;
     char           *ip;
     time_t          tf;
     time_t          dt;
{
  int             fd;
  time_t          ti = 0;
  HistRaw_T       buf;
  long            p = 0, i;
  off_t           ptr;
  int             nb;

  if (hst == NULL)
    hst = &history;

  if (hfile.fd < 0) {
    (void) raw_history_open (FALSE);
  }

  fd = hfile.fd;

  if (tf <= (time_t) 0)
    tf = time (NULL);

  ti = tf - dt;

#if 0
  printf (" ti tf dt : %ld %ld %ld\n", ti, tf, dt);
#endif

  pthread_mutex_lock (&hfile.st_mutex);

#if 0
  printf ("hst->ptr = %ld\n", hst->ptr);
#endif

  if (hst->ptr <= 0) {
    time_t          vi, vf;
    long            pi, pf;

    vi = vf = (time_t) 0;
    pi = pf = -1;

    p = -1;
    for (;;) {
      p++;
      ptr = p * sizeof (buf);

      if (lseek (fd, ptr, SEEK_SET) == (off_t) - 1) {
        syslog (LOG_ERR, "%s : lseek error : %s", J_FUNCTION, strerror (errno));
        break;
      }

      if (read (fd, &buf, sizeof (buf)) != sizeof (buf))
        break;

      if (buf.signature != SIGNATURE)
        continue;

      if ((buf.conn_id < ti) || (buf.conn_id > tf))
        continue;

      if ((vi == 0) || (buf.conn_id < vi)) {
        vi = buf.conn_id;
        pi = p;
      }

      if ((vf == 0) || (buf.conn_id > vf)) {
        vf = buf.conn_id;
        pf = p;
      }
    }

    if (pi >= 0)
      hst->ptr = pi;
  }

  if (hst->ptr < 0) {
    pthread_mutex_unlock (&hfile.st_mutex);
    return;
  }

  res_history_clear (hst);

#if 0
  printf ("history ptr = %ld \n", hst->ptr);
#endif

  p = hst->ptr;
  hst->ptr = -1;
  nb = 0;

  for (i = 0; i < HISTORY_ENTRIES; i++) {
    size_t          sz;

    ptr = ((i + p) % HISTORY_ENTRIES) * sizeof (buf);
    if (lseek (fd, ptr, SEEK_SET) == (off_t) - 1) {
      syslog (LOG_ERR, "%s : lseek error : %s", J_FUNCTION, strerror (errno));
      break;
    }

    if ((sz = read (fd, &buf, sizeof (buf))) != sizeof (buf))
      break;

    if (buf.signature != SIGNATURE)
      continue;

    if (buf.conn_id < ti)
      continue;

    if (buf.conn_id > tf)
      break;

    if (hst->ptr == -1)
      hst->ptr = p;

    if ((ip != NULL) && (strcmp (ip, buf.ip) != 0))
      continue;

    res_history_add_entry (hst, &buf);
  }
  qsort (hst->p, hst->nb, sizeof (HistRes_T), histres_cmp);

  pthread_mutex_unlock (&hfile.st_mutex);

#if 0
  printf (" IP %s : %ld entries (i = %ld)\n", ip ? ip : "(NULL)", hst->nb, i);
  printf ("History Results Updated : %ld entries \n", hst->nb);
#endif
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
void
res_history_print (hst, ip, name, verbose, hostnames)
     History_T      *hst;
     char           *ip;
     char           *name;
     bool            verbose;
     bool            hostnames;
{
  int             i;
  char            nodename[128];
  HistRes_T      *p = NULL;

  if (hst == NULL)
    hst = &history;

  if ((hst->p == NULL) || (hst->nb == 0))
    return;

  if (verbose || (ip != NULL)) {
    for (i = 0; i < hst->nb; i++) {
      char            sout[256], *s;

      p = &hst->p[i];

      if ((ip != NULL) && (strcasecmp (ip, p->ip) != 0))
        continue;

      if ((name == NULL) || hostnames) {
        s = nodename;
        *s = '\0';
        j_gethostbyipaddr (p->ip, nodename, sizeof (nodename));
      } else
        s = name;

      printf ("*** %-20s : %s\n", p->ip, s);
      printf (" Net Class        : %s\n", NET_CLASS (p->ip_class));

      printf (" DNS resolve      : %s\n", RESOLVE_VAL (p->resolve_res));
      strcpy (sout, ctime (&p->ti));
      if ((s = strchr (sout, '\n')) != NULL)
        *s = '\0';
      printf (" First Connection : %s \n", sout);

      strcpy (sout, ctime (&p->tf));
      if ((s = strchr (sout, '\n')) != NULL)
        *s = '\0';
      printf (" Last Connection  : %s \n", sout);
      printf (" Connections      : %7d\n", p->nb_conn);
      printf (" Throttle Max     : %7d / 10 min\n", p->throttle_max);
      printf (" Duration (sec)   : %7.3f %7.3f %7.3f %7.3f (min mean max std-dev)\n",
              ((double) p->delay_min) / 1000,
              kmean (&p->st_delay) / 1000,
              ((double) p->delay_max) / 1000, kstddev (&p->st_delay) / 1000);
      if ((p->nb_conn > 0) && (kmean (&p->st_delay) > 0))
        printf (" Mean Throuput    : %7.3f KBytes/sec\n",
                (1000. * p->nb_bytes) / (1024 * p->nb_conn * kmean (&p->st_delay)));

      printf ("Counts\n");
      printf (" Messages         : %7d\n", p->nb_msgs);
      printf (" Reject           : %7d\n", p->nb_reject);
      printf (" Volume           : %7ld KBytes\n", p->nb_bytes / 1024);
      printf (" Recipients       : %7d\n", p->nb_rcpt);
      printf (" Yield            : %7.2f rcpt/connection\n",
              ((double) p->nb_rcpt) / ((double) p->nb_conn));
      printf (" Files            : %7d\n", p->nb_files);
      printf (" X-Files          : %7d\n", p->nb_xfiles);
      printf (" Virus            : %7d\n", p->nb_virus);
      printf (" User Filter      : %7d\n", p->nb_policy);

      printf ("Reject\n");
      printf (" DNS resolve      : %7d\n", p->rej_resolve);
      printf (" Throttle reject  : %7d\n", p->rej_throttle);
      printf (" Content reject   : %7d\n", p->rej_regex);
      printf (" Rcpt reject      : %7d\n", p->rej_rcpt);
      printf (" Intranet User    : %7d\n", p->rej_luser);
      printf ("\n");
    }
  }

  if (ip == NULL) {
    char            sout[256], *s;

    p = &hst->glob;

    printf ("*** TOTAL\n");

    strcpy (sout, ctime (&p->ti));
    if ((s = strchr (sout, '\n')) != NULL)
      *s = '\0';
    printf (" First Connection : %s \n", sout);

    strcpy (sout, ctime (&p->tf));
    if ((s = strchr (sout, '\n')) != NULL)
      *s = '\0';
    printf (" Last Connection  : %s \n", sout);
    printf (" Connections      : %7d\n", p->nb_conn);
    printf (" Gateways         : %7d\n", hst->nb);
    printf (" Throttle Max     : %7d / 10 min (for a single gateway)\n", p->throttle_max);
    printf (" Duration (sec)   : %7.3f %7.3f %7.3f %7.3f (min mean max std-dev)\n",
            ((double) p->delay_min) / 1000,
            kmean (&p->st_delay) / 1000,
            ((double) p->delay_max) / 1000, kstddev (&p->st_delay) / 1000);
    if ((p->nb_conn > 0) && (kmean (&p->st_delay) > 0))
      printf (" Mean Throuput    : %7.3f KBytes/sec\n",
              (1000. * p->nb_bytes) / (1024 * p->nb_conn * kmean (&p->st_delay)));

    printf ("Counts\n");
    printf (" Messages         : %7d\n", p->nb_msgs);
    printf (" Reject           : %7d\n", p->nb_reject);
    printf (" Volume           : %7ld KBytes\n", p->nb_bytes / 1000);
    printf (" Recipients       : %7d\n", p->nb_rcpt);
    printf (" Yield            : %7.2f rcpt/connection\n",
            ((double) p->nb_rcpt) / ((double) p->nb_conn));
    printf (" Files            : %7d\n", p->nb_files);
    printf (" X-Files          : %7d\n", p->nb_xfiles);
    printf (" Virus            : %7d\n", p->nb_virus);
    printf (" User Filter      : %7d\n", p->nb_policy);

    printf ("Reject\n");
    printf (" DNS resolve      : %7d\n", p->rej_resolve);
    printf ("   FAIL           : %7d\n", p->rej_resolve_failed);
    printf ("   FORGED         : %7d\n", p->rej_resolve_forged);
    printf (" Throttle reject  : %7d\n", p->rej_throttle);
    printf (" Content reject   : %7d\n", p->rej_regex);
    printf (" Rcpt reject      : %7d\n", p->rej_rcpt);
    printf (" Intranet User    : %7d\n", p->rej_luser);
    printf ("\n");
  }
}


/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
static void
ctx2histraw (dst, src)
     HistRaw_T      *dst;
     mlfiPriv       *src;
{
  if ((dst == NULL) || (src == NULL))
    return;

  memset (dst, 0, sizeof (*dst));

  dst->signature = SIGNATURE;

  dst->conn_id = src->conn_id;
  snprintf (dst->ip, sizeof (dst->ip), "%s", src->peer_addr);

#if HAVE_GETHRTIME
  dst->delay = (long) ((src->t_close - src->t_open) / 1000000);
#else
  dst->delay = (long) ((src->t_close - src->t_open) * 1000);
#endif
  dst->t_open = src->t_open;
  dst->t_close = src->t_close;

  dst->throttle = src->throttle;
  dst->rej_throttle = src->rej_throttle;
  dst->rej_resolve = src->rej_resolve;
  dst->rej_rcpt = src->rej_rcpt;

  dst->resolve_res = src->resolve_res;
  dst->resolve_grant = (src->resolve_grant ? 1 : 0);

  dst->rej_regex = src->rej_regex;
  dst->nb_bytes = src->nb_bytes;

  dst->t_open = src->t_open;
  dst->t_close = src->t_close;

  dst->result = src->result;
  dst->ip_class = src->ip_class;
  dst->nb_rcpt = src->nb_rcpt;
  dst->nb_files = src->nb_files;
  dst->nb_xfiles = src->nb_xfiles;
  dst->nb_virus = src->nb_virus;
  dst->nb_policy = src->nb_policy;
  dst->host_access = src->host_access;
  dst->nb_msgs = src->nb_msgs;
  dst->rej_luser = src->rej_luser;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
static void
histraw2histres (dst, src)
     HistRes_T      *dst;
     HistRaw_T      *src;
{

  if ((dst == NULL) || (src == NULL))
    return;

  if (strlen (dst->ip) == 0)
    snprintf (dst->ip, sizeof (dst->ip), "%s", src->ip);

  if ((dst->ti == 0) || (src->conn_id < dst->ti))
    dst->ti = src->conn_id;
  if ((dst->tf == 0) || (src->conn_id > dst->tf))
    dst->tf = src->conn_id;
  if ((dst->throttle_max == 0) || (src->throttle > dst->throttle_max))
    dst->throttle_max = src->throttle;

  dst->nb_conn++;
  dst->nb_msgs += src->nb_msgs;
  dst->nb_bytes += src->nb_bytes;
  dst->nb_rcpt += src->nb_rcpt;

  dst->nb_files += src->nb_files;
  dst->nb_xfiles += src->nb_xfiles;
  dst->nb_virus += src->nb_virus;
  dst->nb_policy += src->nb_policy;
  if (src->result != SMFIS_CONTINUE)
    dst->nb_reject++;

  if (dst->resolve_res != RESOLVE_OK) {
    if (dst->resolve_res != src->resolve_res)
      dst->resolve_res = RESOLVE_NULL;
  } else
    dst->resolve_res = src->resolve_res;
  dst->resolve_grant += src->resolve_grant;

  dst->ip_class = src->ip_class;
  dst->host_access = src->host_access;

  dst->rej_resolve += src->rej_resolve;
  if (src->rej_resolve) {
    switch (src->resolve_res) {
      case RESOLVE_FAIL:
        dst->rej_resolve_failed++;
        break;
      case RESOLVE_FORGED:
        dst->rej_resolve_forged++;
        break;
    }
  }
  dst->rej_throttle += src->rej_throttle;
  dst->rej_regex += src->rej_regex;
  dst->rej_rcpt += src->rej_rcpt;
  dst->rej_luser += src->rej_luser;

  kstats_update (&dst->st_delay, (double) src->delay);
  if ((dst->delay_min == 0) || (src->delay < dst->delay_min))
    dst->delay_min = src->delay;
  if ((dst->delay_max == 0) || (src->delay > dst->delay_max))
    dst->delay_max = src->delay;
}


/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */

static void
res_history_clear (c)
     History_T      *c;
{
  if (c == NULL)
    return;
  c->nb = 0;
  memset (&c->glob, 0, sizeof (c->glob));
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */

static          bool
res_history_resize (c)
     History_T      *c;
{
  HistRes_T      *t;

  if (c == NULL)
    return FALSE;
  if (c->p == NULL) {
    c->nb = 0;
    c->dim = 0;
    if (c->step == 0)
      c->step = 32;
  }
  t = (HistRes_T *) realloc (c->p, (c->dim + c->step) * sizeof (HistRes_T));
  if (t == NULL) {
    syslog (LOG_ERR, "%s : malloc error : %s", J_FUNCTION, strerror (errno));
    return FALSE;               /* ??? JOE */
  }
  c->p = t;
  c->dim += c->step;

  return TRUE;
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */

static          bool
res_history_full (c)
     History_T      *c;
{
  if (c == NULL)
    return TRUE;
  if (c->p == NULL)
    return TRUE;
  return ((c->dim - c->nb) < 1 ? TRUE : FALSE);
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */

static int
histres_cmp (pa, pb)
     const void     *pa;
     const void     *pb;
{
  HistRes_T      *ha = (HistRes_T *) pa;
  HistRes_T      *hb = (HistRes_T *) pb;

  return ip_strcmp (ha->ip, hb->ip);
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */

static HistRes_T *
res_history_lookup (c, ip)
     History_T      *c;
     char           *ip;
{
  HistRes_T      *p;

  p = (HistRes_T *) bsearch (ip, c->p, c->nb, sizeof (HistRes_T), histres_cmp);

  return p;
}


/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */

static void
res_history_add_entry (c, h)
     History_T      *c;
     HistRaw_T      *h;
{
  HistRes_T      *ptr;

  if (res_history_full (c))
    res_history_resize (c);

  ptr = lfind (h->ip, c->p, &c->nb, sizeof (HistRes_T), histres_cmp);

  if (ptr != NULL) {
    histraw2histres (ptr, h);
  } else {
    HistRes_T       buf;

    memset (&buf, 0, sizeof (buf));
    histraw2histres (&buf, h);
    c->p[c->nb] = buf;
    c->nb++;
  }

  histraw2histres (&c->glob, h);

#if 0
  printf ("- %-20s %6ld : %3ld %3ld\n", h->ip, h->delay, c->nb, c->dim);
#endif
}
