/*
 *
 * j-chkmail - filtre de messagerie pour sendmail - MILTER
 *
 * Copyright (c) 2001, 2002 Ecole des Mines de Paris
 *
 *  Auteur     : Jose Marcio Martins da Cruz
 *               martins@paris.ensmp.frif
 *
 *  Historique :
 *  Creation     : janvier 2002
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <j-sys.h>

#include "j-chkmail.h"

#include "j-filter.h"


#define  DO_FILTER                    0

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */

void            strlower (char *s);

static sfsistat mlfi_connect (SMFICTX *, char *, _SOCK_ADDR *);
static sfsistat mlfi_envfrom (SMFICTX *, char **);
static sfsistat mlfi_envto (SMFICTX *, char **);
static sfsistat mlfi_header (SMFICTX *, char *, char *);
static sfsistat mlfi_eoh (SMFICTX *);
static sfsistat mlfi_body (SMFICTX *, u_char *, size_t);
static sfsistat mlfi_eom (SMFICTX *);
static sfsistat mlfi_close (SMFICTX *);
static sfsistat mlfi_abort (SMFICTX *);
static sfsistat mlfi_cleanup (SMFICTX *, bool);

static bool     spool_file_create (SMFICTX *);
static bool     spool_file_write (SMFICTX *, char *, size_t);
static bool     spool_file_close (SMFICTX *);
static bool     spool_file_forget (SMFICTX *);

static sfsistat check_dns_resolve (SMFICTX *);
static sfsistat check_throttle (SMFICTX *);
static sfsistat check_msg_contents (SMFICTX *);

#if 0
static sfsistat check_html_tags (SMFICTX *);
#endif

bool            check_intranet_user (char *, char *, char *);
int             check_ip (char *);
int             check_email_in_domain (char *, char *);

void            rcpt_list_free (rcpt_rec **);
int             rcpt_list_add (rcpt_rec **, char *);
int             count_rcpt (rcpt_rec *);

int
rcpt_count (rcpt_rec * r)
{
  return count_rcpt (r);
}

static char     *get_boundary_tag_value (char *, char *);

static void     warn_action (SMFICTX *, struct mlfiPriv *, attachment *,
                             char *, char *, char *);

static void     read_error_msg (char *buf, int, attachment *, char *,
                                char *, char *, char *, struct mlfiPriv *);

static bool     new_conn_id(CONNID_T *);


/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */

sfsistat
mlfi_connect (ctx, hostname, hostaddr)
     SMFICTX        *ctx;
     char           *hostname;
     _SOCK_ADDR     *hostaddr;
{
  struct mlfiPriv *priv;
  char            ip[64];
  char           *ident;
  time_t          conn_id = time (NULL);
  int             res = SMFIS_CONTINUE;
  bool            grant = TRUE;
  int             host_access = HOST_ACCESS_DEFAULT;

  struct sockaddr_in *sin = (struct sockaddr_in *) hostaddr;

  char           *resolve;
  int             res_resolve;

  int             ip_class, fd_check_res;

  CONNID_T        id;

  memset (&id, 0, sizeof (id));
  new_conn_id (&id);

  stats_inc (STAT_CONNECT, 1);

  ident = smfi_getsymval (ctx, "_");
  if ((ident == NULL) || (strlen (ident) == 0))
    ident = "???";

  if (log_level >= 9) {
    if (strlen(id.id) > 0)
      syslog (LOG_INFO, "%s Connect from %s", id.id, ident);
    else
      syslog (LOG_INFO, "%08lX Connect from %s", conn_id, ident);
  }

  if (hostname == NULL) {
    syslog (LOG_ERR, "mlfi_connect : hostname NULL : %s", strerror (errno));
    res = SMFIS_TEMPFAIL;
    return SMFIS_TEMPFAIL;
  }

  if (strlen (hostname) == 0) {
    j_syslog (conn_id, MSG_NO_PEER_HOSTNAME, ident, NULL, NULL, NULL, NULL, NULL);
    res = SMFIS_TEMPFAIL;
    return SMFIS_TEMPFAIL;
  }

  if (hostaddr == NULL) {
    syslog (LOG_ERR, "mlfi_connect : hostaddr NULL : %s", strerror (errno));
    res = SMFIS_TEMPFAIL;
    return SMFIS_TEMPFAIL;
  }

  if (!j_inet_ntop (&sin->sin_addr, ip, sizeof (ip))) {
    syslog (LOG_ERR, "%08lX mlfi_connect : inet_ntop : %s", conn_id, strerror (errno));
    res = SMFIS_TEMPFAIL;
    return SMFIS_TEMPFAIL;
  }

  /* Let's udpate Throttle computation data */
  throttle_add_host_entry (ip, conn_id);

  ip_class = check_ip (ip);

  /* Let's check resources level */
  fd_check_res = check_file_descriptors ();

  if (fd_check_res != 0) {
    switch (fd_check_res) {
      case FD_LEVEL_OK:
        break;
      case FD_LEVEL_SHORT:
        if (!IS_LOCAL (ip_class) && !IS_DOMAIN (ip_class))
          res = SMFIS_TEMPFAIL;
        break;
      case FD_LEVEL_HI:
        res = SMFIS_TEMPFAIL;
        break;
      default:
        break;
    }

    /* shall remark that smfi_setreply doesn't work at mlfi_connect step */
    if (res == SMFIS_TEMPFAIL) {
      if (smfi_setreply (ctx,
                         "450",
                         "4.5.1", "I'm too busy. Try again later !") == MI_FAILURE)
        syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
      return res;
    }
    if (res == SMFIS_REJECT) {
      if (smfi_setreply (ctx,
                         "550",
                         "5.5.1", "I'm too busy. Try again later !") == MI_FAILURE)
        syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
      return res;
    }
  }


  /* Let's check and update resolve_table */
  resolve = smfi_getsymval (ctx, "{client_resolve}");

  res_resolve = RESOLVE_NULL;
  if (resolve != NULL) {
    if (strcasecmp (resolve, "OK") == 0)
      res_resolve = RESOLVE_OK;
    else if (strcasecmp (resolve, "FAIL") == 0)
      res_resolve = RESOLVE_FAIL;
    else if (strcasecmp (resolve, "FORGED") == 0)
      res_resolve = RESOLVE_FORGED;
    else if (strcasecmp (resolve, "TEMPFAIL") == 0)
      res_resolve = RESOLVE_TEMPFAIL;
    else if (log_level > 9)
      syslog (LOG_ERR, "%s {client_resolve} returned %s", J_FUNCTION, resolve);
  }

  host_access = check_host_access (ip, NULL);
  if ((res_resolve == RESOLVE_OK) || (res_resolve == RESOLVE_NULL)) {
    grant = TRUE;
  } else {
    if (host_access == HOST_ACCESS_OK)
      grant = TRUE;
    else
      grant = resolve_tab_add_entry (ip, conn_id, res_resolve, TRUE);
    if (log_level >= 12)
      syslog (LOG_INFO, "%s - %-15s resolve result : %-6s - %d", J_FUNCTION,
              ip, resolve != NULL ? resolve : "NULL", grant);
  } 

  if (res_resolve == RESOLVE_NULL) {
    if ((log_level >= 9) &&
        ((cf_get_int (CF_RESOLVE_FAIL) != OPT_OK) ||
         (cf_get_int (CF_RESOLVE_FORGED) != OPT_OK))) {
      syslog (LOG_ERR, "%s {client_resolve} returns NULL : %s",
              J_FUNCTION, resolve != NULL ? resolve : "NULL");
    }
  }

  /* Let's check Throttle */
  if (log_level > 10)
    syslog (LOG_INFO, "%08lX TROTTLE : %s %d", conn_id, ip, throttle_check_host (ip));


  /* allocate some private memory */
  priv = malloc (sizeof *priv);
  if (priv == NULL) {
    /* can't accept this message right now */
    syslog (LOG_ERR, "%s : malloc priv : %s", J_FUNCTION, strerror (errno));
    res = SMFIS_TEMPFAIL;
    return SMFIS_TEMPFAIL;
  }

  memset (priv, '\0', sizeof *priv);
  priv->conn_id = conn_id;
  priv->resolve_res = res_resolve;
  priv->resolve_grant = grant;
  priv->ip_class = ip_class;
  priv->host_access = host_access;
  priv->id = id;

#if HAVE_GETHRTIME
  priv->t_open = gethrtime ();
#else
  priv->t_open = time (NULL);
#endif

  if ((priv->ident = strdup (ident)) == NULL) {
    free (priv);
    syslog (LOG_ERR, "%08lX mlfi_connect : malloc mlfi_ident : %s",
            priv->conn_id, strerror (errno));
    res = SMFIS_TEMPFAIL;
    return SMFIS_TEMPFAIL;
  }

  if ((priv->peer_addr = strdup (ip)) == NULL) {
    free (priv->ident);
    free (priv);
    syslog (LOG_ERR, "%08lX mlfi_connect : strdup ip : %s",
            priv->conn_id, strerror (errno));
    res = SMFIS_TEMPFAIL;
    return SMFIS_TEMPFAIL;
  }

  if ((priv->peer_name = strdup (hostname)) == NULL) {
    free (priv->ident);
    free (priv->peer_addr);
    free (priv);
    syslog (LOG_ERR, "%08lX mlfi_connect : strdup hostname : %s",
            priv->conn_id, strerror (errno));
    res = SMFIS_TEMPFAIL;
    return SMFIS_TEMPFAIL;
  }

  /* save the private data */
  if (res != SMFIS_CONTINUE) {
    if (priv != NULL) {
      if (priv->ident != NULL)
        free (priv->ident);
      if (priv->peer_addr != NULL)
        free (priv->peer_addr);
      if (priv->peer_name != NULL)
        free (priv->peer_name);
      free (priv);
    }
    priv = NULL;
    return res;
  }

  smfi_setpriv (ctx, priv);

  /* return at connection call ???? */
#if CHECK_RESOLVE_AT_CONNECT == 1
  if (res == SMFIS_CONTINUE) {
    if ((res = check_dns_resolve (ctx)) != SMFIS_CONTINUE)
      return res;

    if ((res = check_throttle (ctx)) != SMFIS_CONTINUE)
      return res;
  }
#endif

  /* continue processing */
  return res;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */

sfsistat
mlfi_helo (ctx, helohost)
     SMFICTX        *ctx;
     char           *helohost;
{
  struct mlfiPriv *priv = MLFIPRIV;
  int             result = SMFIS_CONTINUE;

  if (priv == NULL)
    return SMFIS_TEMPFAIL;

  if (priv->helohost != NULL)
    free (priv->helohost);
  priv->helohost = NULL;

  if ((result = check_dns_resolve (ctx)) != SMFIS_CONTINUE)
    return result;

  if ((result = check_throttle (ctx)) != SMFIS_CONTINUE)
    return result;

  /* check HELO contents */
  if ((helohost != NULL) && (cf_get_int (CF_CHECK_HELO_CONTENT) == OPT_YES)) {
    if (log_level >= 15)
      syslog (LOG_DEBUG, "%s : check_helo", J_FUNCTION);
    if (check_regex (priv->conn_id, helohost, MAIL_HELO) == TRUE) {
      stats_inc (STAT_HELO_CONTENTS, 1);
      j_syslog (priv->conn_id, MSG_HELO_CONTENTS, priv->peer_addr,
                helohost, NULL, NULL, NULL, NULL);
      if (smfi_setreply (ctx, "550", "5.7.1", MSG_HELO_CONTENTS) == MI_FAILURE)
        syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
      result = SMFIS_REJECT;
    }
  }

  /* continue processing */
  return result;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */

sfsistat
mlfi_envfrom (ctx, envfrom)
     SMFICTX        *ctx;
     char          **envfrom;
{
  struct mlfiPriv *priv = MLFIPRIV;
  int             result = SMFIS_CONTINUE;

  if (priv == NULL)
    return SMFIS_TEMPFAIL;

  if (priv->env_from != NULL)
    free (priv->env_from);
  priv->env_from = NULL;

  if ((result = check_dns_resolve (ctx)) != SMFIS_CONTINUE)
    return result;

  if ((result = check_throttle (ctx)) != SMFIS_CONTINUE)
    return result;

  if (envfrom[0] == NULL) {
    syslog (LOG_WARNING, "%08lX mlfi_from : envfrom[0] null : %s",
            priv->conn_id, strerror (errno));
    return SMFIS_CONTINUE;
  }
  if (strlen (envfrom[0]) == 0) {
    syslog (LOG_WARNING, "%08lX mlfi_from : strlen(envfrom[0]) == 0: %s",
            priv->conn_id, strerror (errno));
    return SMFIS_CONTINUE;
  }

  if ((priv->env_from = strdup (envfrom[0])) == NULL) {
    syslog (LOG_ERR, "%08lX mlfi_from : strdup env_from : %s",
            priv->conn_id, strerror (errno));
    return SMFIS_TEMPFAIL;
  }

  /* open a file to store this message */
  if (!spool_file_create (ctx))
    return SMFIS_TEMPFAIL;

  /* continue processing */
  return SMFIS_CONTINUE;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */

sfsistat
mlfi_envto (ctx, envto)
     SMFICTX        *ctx;
     char          **envto;
{
  struct mlfiPriv *priv = MLFIPRIV;
  int             result = SMFIS_CONTINUE;
  char           *rcpt_host = NULL, *rcpt_addr = NULL;

  stats_inc (STAT_ENVTO, 1);

  if (priv == NULL)
    return SMFIS_TEMPFAIL;

  if ((result = check_dns_resolve (ctx)) != SMFIS_CONTINUE)
    return result;

  if ((result = check_throttle (ctx)) != SMFIS_CONTINUE)
    return result;

  if (priv->env_to != NULL)
    free (priv->env_to);
  priv->env_to = NULL;

  /* ??? */
  if ((envto[0] != NULL) &&
      (strlen (envto[0]) > 0) && ((priv->env_to = strdup (envto[0])) == NULL)) {
    syslog (LOG_ERR, "%08lX mlfi_envto : strdup mlfi_envto : %s",
            priv->conn_id, strerror (errno));
    return SMFIS_TEMPFAIL;
  }

  if (rcpt_list_add (&priv->env_rcpt, envto[0])) {
    syslog (LOG_WARNING,
            "%08lX mlfi_envto : can't add %s to rcpt_list : %s",
            priv->conn_id, envto[0], strerror (errno));
  }
  priv->env_nb_rcpt++;

  /*
   * Check local users
   */
  if (result == SMFIS_CONTINUE) {
    if (IS_UNKNOWN (priv->ip_class)
        && (cf_get_int (CF_CHECK_LOCAL_USERS) == OPT_YES)) {
      rcpt_host = smfi_getsymval (ctx, "{rcpt_host}");
      if (rcpt_host == NULL || strlen (rcpt_host) == 0)
        rcpt_host = "???";
      rcpt_addr = smfi_getsymval (ctx, "{rcpt_addr}");
      if (rcpt_addr == NULL || strlen (rcpt_addr) == 0)
        rcpt_addr = "???";

      if (check_intranet_user (rcpt_host, rcpt_addr, envto[0])) {
        char            s[1024];

        stats_inc (STAT_LUSERS, 1);
        priv->rej_luser++;
        sprintf (s, MSG_LOCAL_USER, rcpt_addr);
        if (smfi_setreply (ctx, "550", "5.7.1", s) == MI_FAILURE)
          syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
        j_syslog (priv->conn_id,
                  MSG_INTRANET_USER,
                  priv->peer_addr,
                  priv->peer_name, priv->env_from, priv->env_to, priv->hdr_from, "");
        result = SMFIS_REJECT;
      }
    }
  }

  /* continue processing */
  return result;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
static void
clean_header_tag (out, in)
     char           *out;
     char           *in;
{
  char           *p = in, *q = out;
  int             n;

  if (out == NULL || in == NULL)
    return;
  p += strspn (p, " \t");
  n = strcspn (p, ": \t");
  strncpy (q, p, n);
  q[n] = 0;
}

sfsistat
mlfi_header (ctx, headerf, headerv)
     SMFICTX        *ctx;
     char           *headerf;
     char           *headerv;
{
  struct mlfiPriv *priv = MLFIPRIV;
  int             result = SMFIS_CONTINUE;

#define    HFSZ  1024

  char            hf[HFSZ];

  if (priv == NULL)
    return SMFIS_TEMPFAIL;

  if (headerf == NULL) {
    if (log_level >= 12)
      syslog (LOG_INFO, "%08lX mlfi_header : headerf NULL", priv->conn_id);
    return SMFIS_CONTINUE;
  }
  if (strlen (headerf) == 0) {
    if (log_level >= 12)
      syslog (LOG_INFO, "%08lX mlfi_header : headerf empty", priv->conn_id);
    return SMFIS_CONTINUE;
  }
  if (headerv == NULL) {
    if (log_level >= 12)
      syslog (LOG_INFO, "%08lX mlfi_header : headerv NULL", priv->conn_id);
    return SMFIS_CONTINUE;
  }
  if (strlen (headerv) == 0) {
    if (log_level >= 12)
      syslog (LOG_INFO, "%08lX mlfi_header : %s headerv empty", priv->conn_id, headerf);
    return SMFIS_CONTINUE;
  }

  /* write the header to the spool file */
  {
    char            buf[2048];

    snprintf (buf, sizeof (buf), "%s: %s%s", headerf, headerv, CRLF);
    spool_file_write (ctx, buf, strlen (buf));
  }

  memset (hf, 0, sizeof (hf));
  clean_header_tag (hf, headerf);

  if (string_has_tag (hf, "<html>")) {
    syslog (LOG_WARNING, "%08lX : HTML inside header %s:%s",
            priv->conn_id, headerf, headerv);
  }
  if (string_has_tag (hf, "<script>")) {
    syslog (LOG_WARNING, "%08lX : SCRIPT inside header %s:%s",
            priv->conn_id, headerf, headerv);
  }

  priv->hdr_has_headers = TRUE;

  if (strcasecmp (hf, "x-mailer") == 0) {
    if (priv->hdr_mailer)
      free (priv->hdr_mailer);
    priv->hdr_mailer = NULL;
    if ((priv->hdr_mailer = strdup (headerv)) == NULL) {
      syslog (LOG_ERR, "%08lX mlfi_header : strdup hdr_mailer : %s",
              priv->conn_id, strerror (errno));
      return SMFIS_TEMPFAIL;
    }
  }

  if (strcasecmp (hf, "from") == 0) {
    if (priv->hdr_from)
      free (priv->hdr_from);
    priv->hdr_from = NULL;
    if ((priv->hdr_from = strdup (headerv)) == NULL) {
      syslog (LOG_ERR, "%08lX mlfi_header : strdup hdr_from : %s",
              priv->conn_id, strerror (errno));
      return SMFIS_TEMPFAIL;
    }
    priv->hdr_has_from = TRUE;
  }

  if (strcasecmp (hf, "to") == 0) {
    priv->hdr_has_to = TRUE;
  }

  if (strcasecmp (hf, "bcc") == 0) {
    priv->hdr_has_bcc = TRUE;
  }

  if (strcasecmp (hf, "cc") == 0) {
    priv->hdr_has_cc = TRUE;
  }

  if (strcasecmp (hf, "subject") == 0) {
    priv->hdr_has_subject = TRUE;
    if (priv->hdr_subject == NULL) {
      if ((priv->hdr_subject = strdup (headerv)) == NULL) {
        syslog (LOG_ERR, "%08lX mlfi_header : strdup hdr_subject : %s",
                priv->conn_id, strerror (errno));
        return SMFIS_TEMPFAIL;
      }
    } else {
      /* XXX more than one subject ??? */
    }
  }

  if (strcasecmp (hf, "mime-version") == 0) {
    priv->hdr_has_mime_version = TRUE;
    if (priv->hdr_mime_version != NULL) {
      free (priv->hdr_mime_version);
      priv->hdr_mime_version = NULL;
    }
    if ((priv->hdr_mime_version = strdup (headerv)) == NULL) {
      syslog (LOG_ERR, "%08lX mlfi_header : strdup mime_version : %s",
              priv->conn_id, strerror (errno));
      return SMFIS_TEMPFAIL;
    }
  }

  if (strcasecmp (hf, "content-type") == 0) {
    char           *buf;
    size_t          sz;

    priv->hdr_has_content_type = TRUE;

    sz = strlen (hf) + strlen (headerv) + 8;
    if ((buf = (char *) malloc (sz)) != NULL) {
      memset (buf, 0, sz);
      snprintf (buf, sz, "%s: %s\n", hf, headerv);
      priv->body_res_scan = scan_block (priv->conn_id,
                                        priv->body_chunk,
                                        SZ_CHUNK,
                                        buf,
                                        strlen (buf),
                                        &priv->body_scan_state,
                                        &priv->tcontent, &priv->lcontent);
      free (buf);
    }

    /* NEED BETTER CHECKING */
    priv->hdr_boundary = get_boundary_tag_value (priv->hdr_boundary, headerv);

    if (log_level > 15)
      syslog(LOG_DEBUG,"%s : %s", J_FUNCTION, priv->hdr_boundary);
  }

  if (strcasecmp (hf, "content-disposition") == 0) {
    char           *buf;
    size_t          sz;

    sz = strlen (hf) + strlen (headerv) + 8;
    if ((buf = (char *) malloc (sz)) != NULL) {
      memset (buf, 0, sz);
      snprintf (buf, sz, "%s: %s\n", hf, headerv);
      priv->body_res_scan = scan_block (priv->conn_id,
                                        priv->body_chunk,
                                        SZ_CHUNK,
                                        buf,
                                        strlen (buf),
                                        &priv->body_scan_state,
                                        &priv->tcontent, &priv->lcontent);
      free (buf);
    }
  }

  if (strcasecmp (hf, "content-transfer-encoding") == 0) {
    priv->hdr_content_encoding = MIME_ENCODE_OTHER;
    if (strcasecmp (headerv, "7bit") == 0)
      priv->hdr_content_encoding = MIME_ENCODE_7BIT;
    else if (strcasecmp (headerv, "8bit") == 0)
      priv->hdr_content_encoding = MIME_ENCODE_8BIT;
    else if (strcasecmp (headerv, "binary") == 0)
      priv->hdr_content_encoding = MIME_ENCODE_BINARY;
    else if (strcasecmp (headerv, "base64") == 0)
      priv->hdr_content_encoding = MIME_ENCODE_BASE64;
    else if (strcasecmp (headerv, "quoted-printable") == 0)
      priv->hdr_content_encoding = MIME_ENCODE_QUOTED_PRINTABLE;
  }

  if ((strstr (hf, "x-loop") != NULL) ||
      (strstr (hf, "list-") != NULL) ||
      (strstr (hf, "x-list") != NULL) ||
      (strstr (hf, "x-lsv") != NULL) || (strstr (hf, "x-lsv") != NULL)) {
    priv->hdr_is_list = TRUE;
  }

  if ((strcasecmp (hf, "precedence") == 0) &&
      (!strcmp (headerv, "list") || !strcmp (headerv, "bulk"))) {
    priv->hdr_is_list = TRUE;
  }

  /* Check Subject contents */
  if (strcasecmp (hf, "subject") == 0) {
    if (cf_get_int (CF_CHECK_SUBJECT_CONTENT) == OPT_YES) {
      if (log_level >= 15)
        syslog (LOG_DEBUG, "%s : check_subject", J_FUNCTION);
      if (check_regex (priv->conn_id, headerv, MAIL_SUBJECT) == TRUE) {
        stats_inc (STAT_SUBJECT_CONTENTS, 1);
        j_syslog (priv->conn_id, MSG_SUBJECT_CONTENTS, priv->peer_addr,
                  NULL, NULL, NULL, NULL, NULL);
        if (smfi_setreply (ctx, "550", "5.7.1", MSG_SUBJECT_CONTENTS) == MI_FAILURE)
          syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
        result = SMFIS_REJECT;
        return result;
      }
    }
  }

  /* check header contents */
  if (cf_get_int (CF_CHECK_HEADERS_CONTENT) == OPT_YES) {
    if (log_level >= 15)
      syslog (LOG_DEBUG, "%s : check_header", J_FUNCTION);
    if (check_regex (priv->conn_id, headerv, MAIL_HEADERS) == TRUE) {
      stats_inc (STAT_HEADERS_CONTENTS, 1);
      j_syslog (priv->conn_id, MSG_HEADERS_CONTENTS, priv->peer_addr,
                NULL, NULL, NULL, NULL, NULL);
      if (smfi_setreply (ctx, "550", "5.7.1", MSG_HEADERS_CONTENTS) == MI_FAILURE)
        syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
      result = SMFIS_REJECT;
      return result;
    }
  }

  return SMFIS_CONTINUE;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */

sfsistat
mlfi_eoh (ctx)
     SMFICTX        *ctx;
{
  struct mlfiPriv *priv = MLFIPRIV;
  int             result = SMFIS_CONTINUE;

  int             nb_rcpt;
  int             rcpt_throttle;

  if (priv == NULL)
    return SMFIS_TEMPFAIL;

  /* write the header to the spool file */
  (void) spool_file_write (ctx, CRLF, strlen (CRLF));

#if 0
  nb_rcpt = count_rcpt (priv->env_rcpt);
#else
  nb_rcpt = priv->env_nb_rcpt;
#endif

  throttle_add_rcpt_entry (priv->peer_addr, nb_rcpt, priv->conn_id);
  priv->nb_rcpt += nb_rcpt;

  rcpt_throttle = throttle_check_rcpt (priv->peer_addr);

  if (priv->peer_addr == NULL) {
    syslog (LOG_ERR, "%s : peer_addr is NULL ???", J_FUNCTION);
    return SMFIS_TEMPFAIL;
  }

  if (priv->hdr_content_encoding != MIME_ENCODE_NONE) {
    char           *s = MSG_ENCODED_BODY;

    switch (priv->hdr_content_encoding) {
      case MIME_ENCODE_7BIT:
        break;
      case MIME_ENCODE_8BIT:
        break;
      case MIME_ENCODE_BINARY:
        s = MSG_BODY_ENCODED_BINARY;
        if (log_level > 10)
          syslog (LOG_INFO, "HEADER ENCODE : B64 %d", priv->hdr_content_encoding);
        if (cf_get_int (CF_ENCODING_BINARY) == OPT_REJECT) {
          stats_inc (STAT_BINARY, 1);
          j_syslog (priv->conn_id, s,
                    priv->peer_addr, NULL, priv->env_from, NULL, priv->hdr_from, NULL);
          if (smfi_setreply (ctx, "550", "5.7.1", s) == MI_FAILURE)
            syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
          result = SMFIS_REJECT;
        }
#if INC_STATS_ENCODE == 1
        else {
          stats_inc (STAT_BINARY, 1);
          if (log_level > 10)
            j_syslog (priv->conn_id, s,
                      priv->peer_addr, NULL,
                      priv->env_from, NULL, priv->hdr_from, NULL);
        }
#endif
        break;
      case MIME_ENCODE_BASE64:
        s = MSG_BODY_ENCODED_BASE64;
        if (log_level > 10)
          syslog (LOG_INFO, "HEADER ENCODE : B64 %d", priv->hdr_content_encoding);
        if (cf_get_int (CF_ENCODING_BASE64) == OPT_REJECT) {
          stats_inc (STAT_BASE64, 1);
          j_syslog (priv->conn_id, s,
                    priv->peer_addr, NULL, priv->env_from, NULL, priv->hdr_from, NULL);
          if (smfi_setreply (ctx, "550", "5.7.1", s) == MI_FAILURE)
            syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
          result = SMFIS_REJECT;
        }
#if INC_STATS_ENCODE == 1
        else {
          stats_inc (STAT_BASE64, 1);
          if (log - level > 10)
            j_syslog (priv->conn_id, s,
                      priv->peer_addr, NULL,
                      priv->env_from, NULL, priv->hdr_from, NULL);
        }
#endif
        break;
      case MIME_ENCODE_QUOTED_PRINTABLE:
        s = MSG_BODY_ENCODED_QP;
        if (log_level > 10)
          syslog (LOG_INFO, "HEADER ENCODE : QP  %d", priv->hdr_content_encoding);
        if (cf_get_int (CF_ENCODING_QUOTED_PRINTABLE) == OPT_REJECT) {
          stats_inc (STAT_QUOTED_PRINTABLE, 1);
          j_syslog (priv->conn_id, s,
                    priv->peer_addr, NULL, priv->env_from, NULL, priv->hdr_from, NULL);
          if (smfi_setreply (ctx, "550", "5.7.1", s) == MI_FAILURE)
            syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
          result = SMFIS_REJECT;
        }
#if INC_STATS_ENCODE == 1
        else {
          stats_inc (STAT_QUOTED_PRINTABLE, 1);
          if (log_level > 10)
            j_syslog (priv->conn_id, s,
                      priv->peer_addr, NULL,
                      priv->env_from, NULL, priv->hdr_from, NULL);
        }
#endif
        break;
      default:
        if (log_level > 10)
          syslog (LOG_INFO, "HEADER ENCODE : OTHER  %d", priv->hdr_content_encoding);
        break;
    }
  }

  /*
   * Check # of recipients
   */
  if (result == SMFIS_CONTINUE && cf_get_int (CF_CHECK_NB_RCPT) != OPT_NO) {
    int             ip_class = priv->ip_class;

    if (IS_UNKNOWN (ip_class) && log_level > 10) {
      char            s[256];

      snprintf (s, sizeof (s), "--- INFO %-16s : NB_RCPT :%d",
                priv->peer_addr, nb_rcpt);
      j_syslog (priv->conn_id, s, NULL, priv->peer_name, NULL, NULL, NULL, NULL);
    }

    if (IS_LOCAL (ip_class) && nb_rcpt > cf_get_int (CF_MAX_RCPT_FROM_LOCAL)) {
      result = SMFIS_REJECT;
    }

    if (IS_DOMAIN (ip_class) && nb_rcpt > cf_get_int (CF_MAX_RCPT_FROM_DOMAIN)) {
      result = SMFIS_REJECT;
    }

    if (IS_FRIEND (ip_class) && nb_rcpt > cf_get_int (CF_MAX_RCPT_FROM_FRIEND)) {
      result = SMFIS_REJECT;
    }

    if (IS_UNKNOWN (ip_class) && (nb_rcpt > cf_get_int (CF_MAX_RCPT_FROM_OUTSIDE))) {
      result = SMFIS_REJECT;
    }

    if (result != SMFIS_CONTINUE) {
      char            s[1024];

      stats_inc (STAT_MAX_RCPT, 1);
      priv->rej_rcpt++;
      sprintf (s, MSG_TOO_MUCH_RCPT);
      if (smfi_setreply (ctx, "550", "5.7.1", s) == MI_FAILURE)
        syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
      j_syslog (priv->conn_id,
                MSG_TOO_MUCH_RCPT,
                priv->peer_addr,
                priv->peer_name, priv->env_from, NULL, priv->hdr_from, NULL);
    }
  }

  /*
   * Check recipient throttle
   */
  if ((result == SMFIS_CONTINUE) &&
      ((cf_get_int (CF_CHECK_THROTTLE) == OPT_YES) ||
       (cf_get_int (CF_CHECK_THROTTLE_RCPT) == OPT_YES))) {
    int             ip_class = priv->ip_class;

    if (log_level >= 15)
      syslog (LOG_INFO, " RCPT_THROTTLE : %s %d %d",
              priv->peer_addr, rcpt_throttle, nb_rcpt);

    if (IS_UNKNOWN (ip_class) && log_level > 10) {
      char            s[256];

      snprintf (s, sizeof (s), "--- INFO %-16s : NB_RCPT :%d",
                priv->peer_addr, nb_rcpt);
      j_syslog (priv->conn_id, s, NULL, priv->peer_name, NULL, NULL, NULL, NULL);
    }


    if (IS_LOCAL (ip_class) && rcpt_throttle > cf_get_int (CF_RCPT_THROTTLE_FROM_LOCAL)) {
      result = SMFIS_TEMPFAIL;
    }

    if (IS_DOMAIN (ip_class) &&
        rcpt_throttle > cf_get_int (CF_RCPT_THROTTLE_FROM_DOMAIN)) {
      result = SMFIS_TEMPFAIL;
    }

    if (IS_FRIEND (ip_class) &&
        rcpt_throttle > cf_get_int (CF_RCPT_THROTTLE_FROM_FRIEND)) {
      result = SMFIS_REJECT;
    }

    if (IS_UNKNOWN (ip_class) &&
        (rcpt_throttle > cf_get_int (CF_RCPT_THROTTLE_FROM_OUTSIDE))) {
      result = SMFIS_REJECT;
    }

    if (result != SMFIS_CONTINUE) {
      char            s[1024];

      stats_inc (STAT_MAX_RCPT, 1);
      priv->rej_rcpt_throttle++;

      sprintf (s, MSG_RCPT_THROTTLE_EXCEEDED);
      switch (result) {
        case SMFIS_REJECT:
          if (smfi_setreply (ctx, "550", "5.7.1", s) == MI_FAILURE)
            syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
          break;
        case SMFIS_TEMPFAIL:
          if (smfi_setreply (ctx, "450", "4.7.1", s) == MI_FAILURE)
            syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
          break;
      }
      j_syslog (priv->conn_id,
                MSG_RCPT_THROTTLE_EXCEEDED,
                priv->peer_addr,
                priv->peer_name, priv->env_from, NULL, priv->hdr_from, NULL);
    }
  }


  /* headers existence checking */
  if (result == SMFIS_CONTINUE && !priv->hdr_has_headers) {
    if ((log_level >= 9 && cf_get_int (CF_NO_HEADERS) != OPT_OK)
        || log_level > 10) {
      j_syslog (priv->conn_id, MSG_NO_HEADERS, priv->peer_addr,
                priv->peer_name, priv->env_from, priv->env_to, NULL, NULL);
    }

    if (cf_get_int (CF_NO_HEADERS) == OPT_REJECT) {
      stats_inc (STAT_NO_HEADERS, 1);
      if (smfi_setreply (ctx, "550", "5.7.1", MSG_NO_HEADERS) == MI_FAILURE)
        syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
      result = SMFIS_REJECT;
    }
  }

  /* Subject header existence checking */
  if (result == SMFIS_CONTINUE && !priv->hdr_has_subject) {
    if ((log_level >= 9 && cf_get_int (CF_NO_SUBJECT_HEADER) != OPT_OK)
        || log_level > 10) {
      j_syslog (priv->conn_id, MSG_NO_SUBJECT_HEADER,
                priv->peer_addr, priv->peer_name, priv->env_from,
                priv->env_to, NULL, NULL);
    }

    if (cf_get_int (CF_NO_SUBJECT_HEADER) != OPT_OK) {
      stats_inc (STAT_NO_SUBJECT_HEADER, 1);
      if (smfi_setreply (ctx, "550", "5.7.1", MSG_NO_SUBJECT_HEADER) == MI_FAILURE)
        syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
      result = SMFIS_REJECT;
    }
  }

  /* TO, CC or BCC existence checking */
  if (result == SMFIS_CONTINUE &&
      !(priv->hdr_has_to || priv->hdr_has_cc || priv->hdr_has_bcc)) {
    if ((log_level >= 9 && cf_get_int (CF_NO_TO_HEADERS) != OPT_OK)
        || log_level > 10) {
      j_syslog (priv->conn_id, MSG_NO_RCPT_HEADER, priv->peer_addr,
                priv->peer_name, priv->env_from, priv->env_to,
                priv->hdr_from, priv->hdr_mailer);
    }

    if (cf_get_int (CF_NO_TO_HEADERS) == OPT_REJECT) {
      stats_inc (STAT_NO_TO_HEADERS, 1);
      if (smfi_setreply (ctx, "550", "5.7.1", MSG_NO_RCPT_HEADER) == MI_FAILURE)
        syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
      result = SMFIS_REJECT;
    }
  }

  /* valid From header field and From envelope field checking */
  if (result == SMFIS_CONTINUE) {
    if (priv->hdr_from == NULL ||
        strlen (priv->hdr_from) == 0 ||
        !strcmp (priv->hdr_from, "<>") ||
        priv->env_from == NULL ||
        strlen (priv->env_from) == 0 || !strcmp (priv->env_from, "<>")) {

      if ((log_level >= 9 && cf_get_int (CF_NO_FROM_HEADERS) != OPT_OK)
          || log_level > 10) {
        j_syslog (priv->conn_id, MSG_NO_FROM_HEADER,
                  priv->peer_addr, priv->peer_name, priv->env_from,
                  priv->env_to, priv->hdr_from, priv->hdr_mailer);
      }
      if (cf_get_int (CF_NO_FROM_HEADERS) == OPT_REJECT) {
        stats_inc (STAT_NO_FROM_HEADERS, 1);
        if (smfi_setreply (ctx, "550", "5.7.1", MSG_NO_FROM_HEADER) == MI_FAILURE)
          syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
        result = SMFIS_REJECT;
      }
    }
  }

  /* X-Mailer header field... */
  if (result == SMFIS_CONTINUE) {
    if ((priv->hdr_mailer != NULL) && strstr (priv->hdr_mailer, "Outlook")) {
      int             ip_class = priv->ip_class;
      char           *msg = NULL;

      if ((cf_get_int (CF_OUTLOOK_LOCAL) == OPT_REJECT) && IS_LOCAL (ip_class))
        msg = MSG_OUTLOOK;
      if ((cf_get_int (CF_OUTLOOK_DOMAIN) == OPT_REJECT)
          && IS_DOMAIN (ip_class))
        msg = MSG_OUTLOOK;
      if ((cf_get_int (CF_OUTLOOK_FRIEND) == OPT_REJECT)
          && ((ip_class & N_FRIEND) != 0))
        msg = MSG_OUTLOOK;
      if (cf_get_int (CF_OUTLOOK) == OPT_REJECT)
        msg = MSG_OUTLOOK;

      if (msg != NULL) {
        j_syslog (priv->conn_id,
                  msg,
                  priv->peer_addr,
                  priv->peer_name,
                  priv->env_from, priv->env_to, priv->hdr_from, priv->hdr_mailer);
      }

      if (msg != NULL) {
        stats_inc (STAT_OUTLOOK, 1);
        if (smfi_setreply (ctx, "550", "5.7.1", msg) == MI_FAILURE)
          syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
        result = SMFIS_REJECT;
      }
    }
  }


  /* NEED MORE CHECKING */
  /* domain address from outside domain checking */
  if (result == SMFIS_CONTINUE) {
    if (strlen (domain) != 0 &&
        ((0 && priv->env_from && check_email_in_domain (domain, priv->env_from))
         || (priv->hdr_from && check_email_in_domain (domain, priv->hdr_from)))) {
      if (!check_ip (priv->peer_addr)) {
        char            msg[128];

        sprintf (msg, MSG_ADDRESS_OUTSIDE_DOMAIN,
                 domain, domain, priv->hdr_is_list ? "LIST" : "");
#if DO_FILTER == 1
        if (smfi_setreply (ctx, "550", "5.7.1", msg) == MI_FAILURE)
          syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
        if (!priv->hdr_is_list)
          result = SMFIS_REJECT;
#endif
        j_syslog (priv->conn_id,
                  msg,
                  priv->peer_addr,
                  priv->peer_name,
                  priv->env_from, priv->env_to, priv->hdr_from, priv->hdr_mailer);
      }
    }
  }

  /* continue processing */
  return result;
}


/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */

sfsistat
mlfi_body (ctx, bodyp, bodylen)
     SMFICTX        *ctx;
     u_char         *bodyp;
     size_t          bodylen;
{
  struct mlfiPriv *priv = MLFIPRIV;

  stats_inc (STAT_BYTES, bodylen);

  if (priv == NULL)
    return SMFIS_TEMPFAIL;

  /* output body block to spool file */
  if (!spool_file_write (ctx, (char *) bodyp, bodylen)) {
    (void) spool_file_forget (ctx);
    syslog (LOG_WARNING, "%08lX - mlfi_body - spool file write error : %s",
            priv->conn_id, strerror (errno));
  }

  priv->body_nb++;
  priv->msg_size += bodylen;
  priv->nb_bytes += bodylen;

  /*
   *
   */
  if ((bodyp != NULL) && (bodylen > 0)) {
    if (priv->body_res_scan == 0) {
      priv->body_res_scan = scan_block (priv->conn_id,
                                        priv->body_chunk,
                                        SZ_CHUNK,
                                        (char *) bodyp,
                                        bodylen,
                                        &priv->body_scan_state,
                                        &priv->tcontent, &priv->lcontent);
      if (priv->body_res_scan != 0) {
        syslog (LOG_WARNING, "%08lX - scan_chunk res = %d",
                priv->conn_id, priv->body_res_scan);
        j_syslog (priv->conn_id,
                  MSG_BINARY_MESSAGE,
                  priv->peer_addr,
                  priv->peer_name, priv->env_from, priv->env_to, priv->hdr_from, NULL);
        return SMFIS_DISCARD;
      }
    }
  }

  /* continue processing */
  return SMFIS_CONTINUE;
}


/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */


sfsistat
mlfi_eom (ctx)
     SMFICTX        *ctx;
{
  struct mlfiPriv *priv = MLFIPRIV;
  char            host[512];
  char            hbuf[1024];

  int             result = SMFIS_CONTINUE;
  attachment     *ahead = NULL, *p;
  int             nb_files = 0;
  int             nb_exefiles = 0;
  char           *log_msg = NULL;

  char            log_buf_msg[1024];

  stats_inc (STAT_MSGS, 1);

  if (priv == NULL)
    return SMFIS_TEMPFAIL;

  if (!spool_file_close (ctx));

  priv->nb_msgs++;

  if (priv->tcontent.field_type != CT_NONE) {
    save_content_field (&priv->tcontent, &priv->lcontent);
  }

  /* 
     ** Let's check message contents against regular expressions
   */
  if (result == SMFIS_CONTINUE) {
    result = check_msg_contents (ctx);
  }

  /*
     **
     ** Let's check attached files
     **
   */
  if (result == SMFIS_CONTINUE && priv->lcontent) {
    nb_files = extract_attachments (priv->lcontent, &ahead);

    priv->nb_files += nb_files;
    stats_inc (STAT_FILES, nb_files);

    if (nb_files > 0) {
      p = ahead;
      while (p) {
        if (p->exefile == 1)
          nb_exefiles++;
        p = p->next;
      }

      priv->nb_xfiles += nb_exefiles;
      stats_inc (STAT_XFILES, nb_exefiles);

      if (log_level >= 20)
        syslog (LOG_DEBUG, "  NB_BODY  : %d", priv->body_nb);
    }
  }

  /* 
     **
     ** Is there any XFILE ?
     **
   */
  if (result == SMFIS_CONTINUE && cf_get_int (CF_XFILES) != OPT_OK && (nb_exefiles > 0)) {
    /* remplacer corps du message */
    /* message de log */
    /* ajoute l'expediteur dans la liste des destinataires */

    log_msg = MSG_SHORT_XFILE;

    if (cf_get_int (CF_XFILE_SAVE_MSG) == OPT_YES)
      priv->save_msg = TRUE;

    switch (cf_get_int (CF_XFILES)) {
      case OPT_REJECT:
        smfi_setreply (ctx, "550", "5.6.0", MSG_XFILE);
        result = SMFIS_REJECT;
        break;                  /* end of OPT_REJECT */

      case OPT_WARN:
        warn_action (ctx, priv, ahead, NULL, MSG_XFILE, "XFILE");
        result = SMFIS_CONTINUE;
        break;

      case OPT_DISCARD:
        result = SMFIS_DISCARD;
        break;                  /* end of OPT_DISCARD */

      default:
        break;
    }
    j_log_files (priv->conn_id, ahead);
  }

  /*
     **
     ** Let's do HTML/SCRIPT check
     **
   */
  if (result == SMFIS_CONTINUE) {

  }

  /*
     **
     ** Let's do antivirus checking or external scanner
     **
   */
  if (result == SMFIS_CONTINUE && cf_get_int (CF_AV_ACTION) != OPT_OK
      && (nb_files > 0 || cf_get_int (CF_AV_SCOPE) == OPT_ALL)) {
    char            answer[2048];
    int             avres;

    syslog (LOG_WARNING, "%s : Entering AV_CHECK", J_FUNCTION);
    memset (answer, 0, sizeof (answer));
    if (priv->fname != NULL) {
      avres = check_virus (answer, priv->fname);

      if (log_level >= 15 || avres != AVRES_OK)
        syslog (LOG_DEBUG, "CHECK : A = %s (%d): Q = %s", answer, avres, priv->fname);

      /* VIRUS FOUND */
      if (avres == AVRES_VIRUS) {
        priv->nb_virus++;
        stats_inc (STAT_VIRUS, 1);

        snprintf (log_buf_msg, sizeof (log_buf_msg), "%s : %s",
                  MSG_SHORT_VIRUS, answer);
        log_msg = log_buf_msg;

        if (cf_get_int (CF_AV_SAVE_MSG) == OPT_YES)
          priv->save_msg = TRUE;

        switch (cf_get_int (CF_AV_ACTION)) {
          case OPT_REJECT:
            smfi_setreply (ctx, "550", "5.6.0", MSG_VIRUS);
            result = SMFIS_REJECT;
            break;              /* end of OPT_REJECT */

          case OPT_WARN:
            warn_action (ctx, priv, ahead, answer, MSG_VIRUS, "VIRUS");
            result = SMFIS_CONTINUE;
            break;

          case OPT_DISCARD:
            result = SMFIS_DISCARD;
            break;              /* end of OPT_DISCARD */

          default:
            break;
        }
      }
      /* End of VIRUS FOUND */

      /* POLICY ... */
      if (avres == AVRES_POLICY) {
        priv->nb_policy++;
        stats_inc (STAT_POLICY, 1);

        snprintf (log_buf_msg, sizeof (log_buf_msg), "%s", MSG_SHORT_POLICY);
        log_msg = log_buf_msg;

        if (cf_get_int (CF_AV_SAVE_MSG) == OPT_YES)
          priv->save_msg = TRUE;

        switch (cf_get_int (CF_AV_ACTION)) {
          case OPT_REJECT:
            smfi_setreply (ctx, "550", "5.6.0", MSG_SHORT_POLICY);
            result = SMFIS_REJECT;
            break;              /* end of OPT_REJECT */

          case OPT_WARN:
            warn_action (ctx, priv, ahead, NULL, answer, "POLICY");
            result = SMFIS_CONTINUE;
            break;              /* end of OPT_WARN */

          case OPT_DISCARD:
            result = SMFIS_DISCARD;
            break;              /* end of OPT_DISCARD */

          default:
            break;
        }
      }
      /* End of POLICY FOUND */
    }
  }

  /*
     **
     ** Let's log what we found ...
     **
   */
  if ((nb_files > 0) &&
      ((cf_get_int (CF_LOG_ATTACHMENTS) == OPT_YES) ||
       (log_msg != NULL) || (log_level > 10))) {
    if (log_msg == NULL)
      log_msg = "*** ATTACHMENTS : ";
    j_syslog (priv->conn_id,
              log_msg,
              priv->peer_addr,
              priv->peer_name,
              priv->env_from, priv->env_to, priv->hdr_from, priv->hdr_mailer);
    p = ahead;
    while (p) {
      char           *serror;

      switch (p->exefile) {
        case 0:
          serror = "CLEAN";
          break;
        case 1:
          serror = "SUSPECT ???";
          break;
        default:
          serror = "UNCHECKED";
      }
      syslog (LOG_INFO, "  **** (%-11s) : %-10s %-30s %s\n",
              serror,
              p->disposition ? p->disposition : "",
              p->mimetype ? p->mimetype : "", p->name ? p->name : "");
      p = p->next;
    }
    if (log_level >= 20) {
      syslog (LOG_DEBUG, "  HAS_EXE : %s", nb_exefiles > 0 ? "OUI" : "NON");
    }
  }

  /*
     **
     ** Everything is done ! Let's clean the house !
     **
   */
  if (ahead != NULL)
    free_attachment_list (ahead);
  ahead = NULL;

  if (cf_get_int (CF_PRESENCE) == OPT_SHOW) {
    char           *p = NULL;

    strcpy (host, "");
    /* add a header to the message announcing our presence */
    switch (cf_get_int (CF_J_HOSTNAME)) {
      case OPT_SYSTEM:
        break;
      case OPT_SENDMAIL:
        p = smfi_getsymval (ctx, "j");
        break;
      default:
        p = cf_get_str (CF_J_HOSTNAME);
        break;
    }
    if (p != NULL && strlen (p) > 0)
      snprintf (host, sizeof (host), "%s", p);
    if (strlen (host) == 0) {
      if (gethostname (host, sizeof (host)) < 0)
        snprintf (host, sizeof (host), "localhost");
    }

    snprintf (hbuf, sizeof hbuf,
              "at %s by Joe's j-chkmail (\"http://j-chkmail.ensmp.fr\")!", host);
    smfi_addheader (ctx, "X-Miltered", hbuf);
  }

  /* return */
  (void) mlfi_cleanup (ctx, FALSE);
  return result;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */

sfsistat
mlfi_close (ctx)
     SMFICTX        *ctx;
{
  struct mlfiPriv *priv = MLFIPRIV;

  stats_inc (STAT_CLOSE, 1);

  if (priv == NULL)
    return SMFIS_TEMPFAIL;

#if HAVE_GETHRTIME
  priv->t_close = gethrtime ();
#else
  priv->t_close = time (NULL);
#endif
  (void) raw_history_add_entry (ctx);

  mlfi_cleanup (ctx, TRUE);
  return SMFIS_ACCEPT;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */

sfsistat
mlfi_abort (ctx)
     SMFICTX        *ctx;
{
  struct mlfiPriv *priv = MLFIPRIV;

  stats_inc (STAT_ABORT, 1);

  if (priv == NULL)
    return SMFIS_TEMPFAIL;

  priv->abort++;

  return mlfi_cleanup (ctx, FALSE);
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */

sfsistat
mlfi_cleanup (ctx, ok)
     SMFICTX        *ctx;
     bool            ok;
{
  sfsistat        rstat = SMFIS_CONTINUE;
  struct mlfiPriv *priv = MLFIPRIV;

  if (priv == NULL)
    return rstat;

  /* release private memory */
  if (priv->env_from != NULL) {
    free (priv->env_from);
    priv->env_from = NULL;
  }

  if (priv->env_to != NULL) {
    free (priv->env_to);
    priv->env_to = NULL;
  }

  rcpt_list_free (&priv->env_rcpt);
  priv->env_nb_rcpt = 0;

  if (priv->hdr_mailer != NULL) {
    free (priv->hdr_mailer);
    priv->hdr_mailer = NULL;
  }

  if (priv->hdr_from != NULL) {
    free (priv->hdr_from);
    priv->hdr_from = NULL;
  }

  if (priv->hdr_to != NULL) {
    free (priv->hdr_to);
    priv->hdr_to = NULL;
  }

  if (priv->hdr_mime_version != NULL) {
    free (priv->hdr_mime_version);
    priv->hdr_mime_version = NULL;
  }

  if (priv->hdr_boundary != NULL)
    free (priv->hdr_boundary);
  priv->hdr_boundary = NULL;

  priv->hdr_content_encoding = MIME_ENCODE_NONE;

  priv->hdr_has_to = FALSE;
  priv->hdr_has_cc = FALSE;
  priv->hdr_has_bcc = FALSE;
  priv->hdr_has_from = FALSE;
  priv->hdr_has_subject = FALSE;
  priv->hdr_has_mime_version = FALSE;
  priv->hdr_has_content_type = FALSE;
  priv->hdr_is_list = FALSE;
  priv->hdr_has_headers = FALSE;

  *priv->body_chunk = '\0';
  memset (priv->body_chunk, 0, sizeof (priv->body_chunk));
  priv->body_res_scan = 0;
  priv->body_nb = 0;
  priv->body_scan_state = 0;

  free_content_field_list (priv->lcontent);
  priv->lcontent = NULL;

  free_content_field_rec (&priv->tcontent);
  memset (&priv->tcontent, 0, sizeof (priv->tcontent));

  priv->msg_size = 0;
  priv->msg_regex_matches = 0;

  if (!spool_file_forget (ctx));

  if (priv->hdr_subject != NULL) {
    free (priv->hdr_subject);
    priv->hdr_subject = NULL;
  }

  if (ok) {
    if (priv->peer_addr != NULL)
      free (priv->peer_addr);
    priv->peer_addr = NULL;

    if (priv->peer_name != NULL)
      free (priv->peer_name);
    priv->peer_name = NULL;

    if (priv->ident != NULL) {
      free (priv->ident);
      priv->ident = NULL;
    }

    if (priv->helohost != NULL) {
      free (priv->helohost);
      priv->helohost = NULL;
    }

    free (priv);
    smfi_setpriv (ctx, NULL);
  }

  /* return status */
  return rstat;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
static          bool
spool_file_create (ctx)
     SMFICTX        *ctx;
{
  struct mlfiPriv *priv = MLFIPRIV;
  char           *p;
  char            fname[256];

  int             fd;

  if (priv->fp != NULL)
    return TRUE;

  if (priv->fname != NULL)
    syslog (LOG_WARNING, "%s : fp NULL, but fname not NULL", J_FUNCTION);

  if ((p = cf_get_str (CF_SPOOLDIR)) == NULL)
    p = J_SPOOLDIR;
  snprintf (fname, sizeof (fname), "%s/%08lX.XXXXXX", p, priv->conn_id);
  priv->fname = strdup (fname);
  if (priv->fname == NULL) {
    syslog (LOG_ERR, "%s : can't strdup(fname) %s : %s", J_FUNCTION,
            fname, strerror (errno));
    return FALSE;
  }

  if ((fd = mkstemp (priv->fname)) < 0 || (priv->fp = fdopen (fd, "w+")) == NULL) {
    syslog (LOG_ERR, "%s : can't create spool file %s : %s", J_FUNCTION,
            fname, strerror (errno));
    if (fd >= 0)
      (void) close (fd);
    free (priv->fname);
    priv->fname = NULL;
    return FALSE;
  } else
    priv->fp_open = TRUE;

  /* Let's add a fake From line */
  if (priv->fp != NULL) {
    char            s[256];
    char            t[256];

#if HAVE_CTIME_R
#if _POSIX_PTHREAD_SEMANTICS == 1
    ctime_r ((time_t *) & priv->conn_id, t);
#else
    ctime_r ((time_t *) & priv->conn_id, t, sizeof (t));
#endif
#elif HAVE_CTIME
    strlcpy (t, ctime (&priv->conn_id), sizeof (t));
#else
    strcpy (t, "-");
#endif
    if ((p = strchr (t, '\r')) != NULL)
      *p = '\0';
    if ((p = strchr (t, '\n')) != NULL)
      *p = '\0';

    if (priv->env_from != NULL)
      snprintf (s, sizeof (s) - 3, "From %s %s", priv->env_from, t);
    else
      snprintf (s, sizeof (s) - 3, "From %s %s", "<unknown>", t);
    strcat (s, CRLF);
    if (fwrite (s, 1, strlen (s), priv->fp) != strlen (s)) {
      syslog (LOG_WARNING, "Can't write fake From line : %s", strerror (errno));
    }
    fflush (priv->fp);
  }

  return TRUE;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
static          bool
spool_file_write (ctx, buf, size)
     SMFICTX        *ctx;
     char           *buf;
     size_t          size;
{
  struct mlfiPriv *priv = MLFIPRIV;

  if (priv->fp == NULL)
    return TRUE;

  if (fwrite (buf, size, 1, priv->fp) <= 0) {
    syslog (LOG_WARNING, "%s : Error writing spool file %s : %s",
            J_FUNCTION, priv->fname, strerror (errno));
    return FALSE;
  }

  fflush (priv->fp);

  return TRUE;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
static          bool
spool_file_close (ctx)
     SMFICTX        *ctx;
{
  struct mlfiPriv *priv = MLFIPRIV;

  if (priv->fp == NULL)
    return TRUE;

  if (fclose (priv->fp) != 0) {
    priv->fp = NULL;
    syslog (LOG_ERR, "%s : error closing spool file : %s",
            J_FUNCTION, strerror (errno));
    return FALSE;
  }
  priv->fp = NULL;
  priv->fp_open = FALSE;

  return TRUE;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
static          bool
spool_file_forget (ctx)
     SMFICTX        *ctx;
{
  struct mlfiPriv *priv = MLFIPRIV;

  (void) spool_file_close (ctx);

  if (priv->fname != NULL) {
    if (priv->save_msg == TRUE) {
      syslog (LOG_NOTICE, "%08lX : quarantine file %s", priv->conn_id, priv->fname);

    } else
      unlink (priv->fname);
  }

  priv->save_msg = FALSE;

  if (priv->fname != NULL)
    free (priv->fname);
  priv->fname = NULL;

  return TRUE;
}


/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
static          sfsistat
check_dns_resolve (ctx)
     SMFICTX        *ctx;
{
  struct mlfiPriv *priv = MLFIPRIV;
  int             result = SMFIS_CONTINUE;

  char            s[256];
  char            access_msg[HOST_ACCESS_MSG_LEN];
  char           *msg = NULL;
  int             grant;

  if (priv->resolve_res == RESOLVE_NULL || priv->resolve_res == RESOLVE_OK)
    return result;

  /* Let's check mail gateway against access table */
  strcpy (access_msg, "");
  priv->host_access = check_host_access (priv->peer_addr, access_msg);
  switch (priv->host_access) {
    case HOST_ACCESS_OK:
      return SMFIS_CONTINUE;
      break;

    case HOST_ACCESS_MSG:
      if (strlen (access_msg) > 0)
        msg = access_msg;
      switch (priv->resolve_res) {
        case RESOLVE_FAIL:
          stats_inc (STAT_RESOLVE_FAIL, 1);
          if (msg == NULL)
            msg = MSG_RESOLVE_FAIL;
          break;
        case RESOLVE_FORGED:
          stats_inc (STAT_RESOLVE_FORGED, 1);
          if (msg == NULL)
            msg = MSG_RESOLVE_FORGED;
          break;
        default:
          stats_inc (STAT_RESOLVE_FAIL, 1);
          if (msg == NULL)
            msg = MSG_RESOLVE_FAIL;
      }
      if (smfi_setreply (ctx, "550", "5.7.1", msg) == MI_FAILURE)
        syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");

      if (log_level >= 9) {
        snprintf (s, sizeof (s), MSG_SHORT_RESOLVE, "MSG", priv->ident);
        j_syslog (priv->conn_id, s, priv->ident, NULL, NULL, NULL, NULL, NULL);
      }
      priv->rej_resolve = 1;
      return SMFIS_REJECT;
      break;

    case HOST_ACCESS_REJECT:
      switch (priv->resolve_res) {
        case RESOLVE_FAIL:
          stats_inc (STAT_RESOLVE_FAIL, 1);
          if (msg == NULL)
            msg = MSG_RESOLVE_FAIL;
          break;
        case RESOLVE_FORGED:
          stats_inc (STAT_RESOLVE_FORGED, 1);
          if (msg == NULL)
            msg = MSG_RESOLVE_FORGED;
          break;
        default:
          stats_inc (STAT_RESOLVE_FAIL, 1);
          if (msg == NULL)
            msg = MSG_RESOLVE_FAIL;
      }
      if (smfi_setreply (ctx, "550", "5.7.1", msg) == MI_FAILURE)
        syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
      if (log_level >= 9) {
        snprintf (s, sizeof (s), MSG_SHORT_RESOLVE, "REJECT", priv->ident);
        j_syslog (priv->conn_id, s, priv->ident, NULL, NULL, NULL, NULL, NULL);
      }
      priv->rej_resolve = 1;
      return SMFIS_REJECT;
      break;
  }

  /* mail gateway isn't at access table */
  /* Let's check against resolve-tab... */
  if ((grant = priv->resolve_grant) != FALSE) {
    /* update stats */
    return SMFIS_CONTINUE;
  }

  /* resolve_tab doesn' grant access */
  switch (priv->resolve_res) {
      /* DNS lookup FAIL */
    case RESOLVE_FAIL:
      if ((log_level >= 9 && cf_get_int (CF_RESOLVE_FAIL) != OPT_OK)
          || log_level > 10) {
        snprintf (s, sizeof (s), MSG_SHORT_RESOLVE, "FAIL", priv->ident);
        j_syslog (priv->conn_id, s, priv->ident, NULL, NULL, NULL, NULL, NULL);
      }

      /* JOE - check j-access */
      if (msg == NULL)
        msg = MSG_RESOLVE_FAIL;
      switch (cf_get_int (CF_RESOLVE_FAIL) != OPT_OK) {
        case OPT_REJECT:
          if (smfi_setreply (ctx, "550", "5.7.1", msg) == MI_FAILURE)
            syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
          result = SMFIS_REJECT;
          break;
        case OPT_TEMPFAIL:
          if (smfi_setreply (ctx, "450", "4.7.1", msg) == MI_FAILURE)
            syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
          result = SMFIS_TEMPFAIL;
          break;
        default:
          break;
      }
      if (result != SMFIS_CONTINUE) {
        stats_inc (STAT_RESOLVE_FAIL, 1);
      }
      break;

      /* forged hostname */
    case RESOLVE_FORGED:
      if ((log_level >= 9 && cf_get_int (CF_RESOLVE_FORGED) != OPT_OK)
          || log_level > 10) {
        snprintf (s, sizeof (s), MSG_SHORT_RESOLVE, "FORGED", priv->ident);
        j_syslog (priv->conn_id, s, priv->ident, NULL, NULL, NULL, NULL, NULL);
      }

      if (msg == NULL)
        msg = MSG_RESOLVE_FORGED;
      switch (cf_get_int (CF_RESOLVE_FORGED) != OPT_OK) {
        case OPT_REJECT:
          if (smfi_setreply (ctx, "550", "5.7.1", msg) == MI_FAILURE)
            syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
          result = SMFIS_REJECT;
          break;
        case OPT_TEMPFAIL:
          if (smfi_setreply (ctx, "450", "4.7.1", msg) == MI_FAILURE)
            syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
          result = SMFIS_TEMPFAIL;
          break;
        default:
          break;
      }
      if (result != SMFIS_CONTINUE)
        stats_inc (STAT_RESOLVE_FORGED, 1);
      break;
  }
  if (result != SMFIS_CONTINUE)
    priv->rej_resolve = 1;

  return result;
}


/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
static          sfsistat
check_throttle (ctx)
     SMFICTX        *ctx;
{
  struct mlfiPriv *priv = MLFIPRIV;
  int             result = SMFIS_CONTINUE;

  if ((cf_get_int (CF_CHECK_THROTTLE) == OPT_YES)
      || (cf_get_int (CF_CHECK_THROTTLE_CONN) == OPT_YES)) {
    int             ip_class = priv->ip_class;
    int             n;
    char            s[256];
    int             THROTTLE_COEF = 1;

    /* Let's setup Throttle coefficient */
    switch (priv->resolve_res) {
      case RESOLVE_NULL:
        THROTTLE_COEF = 1;
        break;
      case RESOLVE_OK:
        THROTTLE_COEF = 1;
        break;
      case RESOLVE_FAIL:
        THROTTLE_COEF = 2;
        break;
      case RESOLVE_FORGED:
        THROTTLE_COEF = 2;
        break;
      case RESOLVE_TEMPFAIL:
        THROTTLE_COEF = 2;
        break;
      default:
        THROTTLE_COEF = 1;
    }

    priv->throttle = throttle_check_host (priv->peer_addr);
    n = THROTTLE_COEF * priv->throttle;

    if (log_level > 10)
      syslog (LOG_INFO, "%08lX - Throttle : %-20s %4d ip_class=[%02X]",
              priv->conn_id, priv->peer_addr, n, ip_class);

    if (IS_LOCAL (ip_class) && (n > cf_get_int (CF_CONN_THROTTLE_FROM_LOCAL))) {

      stats_inc (STAT_THROTTLE, 1);
      switch (cf_get_int (CF_RES_THROTTLE_FROM_LOCAL)) {
        case OPT_REJECT:
          result = SMFIS_REJECT;
          break;
        case OPT_TEMPFAIL:
          result = SMFIS_TEMPFAIL;
          break;
        default:
          result = SMFIS_CONTINUE;
      }
    }

    if (IS_DOMAIN (ip_class) && (n > cf_get_int (CF_CONN_THROTTLE_FROM_DOMAIN))) {
      stats_inc (STAT_THROTTLE, 1);
      switch (cf_get_int (CF_RES_THROTTLE_FROM_DOMAIN)) {
        case OPT_REJECT:
          result = SMFIS_REJECT;
          break;
        case OPT_TEMPFAIL:
          result = SMFIS_TEMPFAIL;
          break;
        default:
          result = SMFIS_CONTINUE;
      }
    }

    if (IS_FRIEND (ip_class) && (n > cf_get_int (CF_CONN_THROTTLE_FROM_FRIEND))) {
      stats_inc (STAT_THROTTLE, 1);
      switch (cf_get_int (CF_RES_THROTTLE_FROM_FRIEND)) {
        case OPT_REJECT:
          result = SMFIS_REJECT;
          break;
        case OPT_TEMPFAIL:
          result = SMFIS_TEMPFAIL;
          break;
        default:
          result = SMFIS_CONTINUE;
      }
    }

    if (IS_UNKNOWN (ip_class) && (n > cf_get_int (CF_CONN_THROTTLE_FROM_OUTSIDE))) {
      stats_inc (STAT_THROTTLE, 1);
      switch (cf_get_int (CF_RES_THROTTLE_FROM_OUTSIDE)) {
        case OPT_REJECT:
          result = SMFIS_REJECT;
          break;
        case OPT_TEMPFAIL:
          result = SMFIS_TEMPFAIL;
          break;
        default:
          result = SMFIS_CONTINUE;
      }
    }

    if (result != SMFIS_CONTINUE) {
      switch (result) {
        case SMFIS_REJECT:
          if (smfi_setreply (ctx, "550", "5.7.1", MSG_THROTTLE_EXCEEDED) == MI_FAILURE)
            syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
          break;
        case SMFIS_TEMPFAIL:
          if (smfi_setreply (ctx, "450", "4.7.1", MSG_THROTTLE_EXCEEDED) == MI_FAILURE)
            syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
          break;
      }
      snprintf (s, sizeof (s), MSG_SHORT_THROTTLE, priv->peer_addr, n, ip_class);
      j_syslog (priv->conn_id, s, NULL, NULL, NULL, NULL, NULL, NULL);
    }
  }
  if (result != SMFIS_CONTINUE)
    priv->rej_throttle = 1;

  return result;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
int
check_ip (char *ip)
{
  return check_host_class (ip);
}


/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
static          sfsistat
check_msg_contents (ctx)
     SMFICTX        *ctx;
{
  struct mlfiPriv *priv = MLFIPRIV;
  int             result = SMFIS_CONTINUE;

  int             ip_class = priv->ip_class;

  size_t          maxsize;

  int             res = 0;

  if (cf_get_int (CF_CHECK_BODY_CONTENT) == OPT_NO)
    return result;

  switch (cf_get_int (CF_CONTENT_CHECK_ORIGIN)) {
    case OPT_ALL:
      break;
    case OPT_UNKNOWN:
      if (IS_KNOWN (ip_class))
        return result;
      break;
    default:
      break;
  }

  maxsize = cf_get_int (CF_CONTENT_CHECK_SIZE);

#if 0
  res = check_body_contents(priv->conn_id, 
                            priv->fname, 
                            &priv->msg_regex_matches, 
                            maxsize,
                            priv->hdr_content_encoding,
                            priv->hdr_boundary);
#else
  switch (priv->hdr_content_encoding) {
    case MIME_ENCODE_BASE64:
      res = scan_msg_file_regex_base64 (priv->conn_id,
                                        priv->fname, &priv->msg_regex_matches, maxsize);
      break;
    default:
      res = scan_msg_file_regex (priv->conn_id,
                                 priv->fname, &priv->msg_regex_matches, maxsize);
      break;
  }
#endif

  if (res != 0) {
    /* do some syslog ... */
  }

  if (priv->msg_regex_matches >= cf_get_int (CF_CONTENT_CHECK_SCORE)) {
    char            logbuf[256];

    snprintf (logbuf, sizeof (logbuf), "%s : score %d",
              "*** SCAN_BODY_REGEX", priv->msg_regex_matches);
    if (log_level >= 9)
      j_syslog (priv->conn_id, logbuf, priv->peer_addr,
                priv->peer_name, priv->env_from, priv->env_to, priv->hdr_from, NULL);

    stats_inc (STAT_BODY_CONTENTS, 1);
    if (smfi_setreply (ctx, "550", "5.7.1", MSG_BODY_CONTENTS) == MI_FAILURE)
      syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
    result = SMFIS_REJECT;
  } else {
    if (log_level > 10 && priv->msg_regex_matches > 0) {
      char            logbuf[256];

      snprintf (logbuf, sizeof (logbuf), "%s : score %d",
                "*** SCAN_BODY_REGEX", priv->msg_regex_matches);
      j_syslog (priv->conn_id, "*** SCAN_BODY_REGEX", priv->peer_addr,
                priv->peer_name, priv->env_from, priv->env_to, priv->hdr_from, NULL);
      stats_inc (STAT_BODY_CONTENTS, 1);
    }
  }

  if (result != SMFIS_CONTINUE)
    priv->rej_regex++;

  return result;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
#if 0
static          sfsistat
check_html_tags (ctx)
     SMFICTX        *ctx;
{
  struct mlfiPriv *priv = MLFIPRIV;
  int             result = SMFIS_CONTINUE;

  int             ip_class = priv->ip_class;

  size_t          maxsize;

  int             res = 0;

  if (cf_get_int (CF_CHECK_BODY_CONTENT) == OPT_NO)
    return result;

  switch (cf_get_int (CF_CONTENT_CHECK_ORIGIN)) {
    case OPT_ALL:
      break;
    case OPT_UNKNOWN:
      if (IS_KNOWN (ip_class))
        return result;
      break;
    default:
      break;
  }

  maxsize = cf_get_int (CF_CONTENT_CHECK_SIZE);

  switch (priv->hdr_content_encoding) {
    case MIME_ENCODE_BASE64:
#if DO_CHECK_BODY_BASE64 == 1
      res = scan_msg_file_regex_base64 (priv->conn_id,
                                        priv->fname, &priv->msg_regex_matches, maxsize);
#endif
      break;
    default:
      res = scan_msg_file_regex (priv->conn_id,
                                 priv->fname, &priv->msg_regex_matches, maxsize);
      break;
  }

  if (res != 0) {
    /* do some syslog ... */
  }

  if (priv->msg_regex_matches >= cf_get_int (CF_CONTENT_CHECK_SCORE)) {
    char            logbuf[256];

    snprintf (logbuf, sizeof (logbuf), "%s : score %d",
              "*** SCAN_BODY_REGEX", priv->msg_regex_matches);
    if (log_level >= 9)
      j_syslog (priv->conn_id, logbuf, priv->peer_addr,
                priv->peer_name, priv->env_from, priv->env_to, priv->hdr_from, NULL);

    stats_inc (STAT_BODY_CONTENTS, 1);
#if DO_REJECT_BODY_CONTENTS == 1
    if (smfi_setreply (ctx, "550", "5.7.1", MSG_BODY_CONTENTS) == MI_FAILURE)
      syslog (LOG_WARNING, "smfi_setreply returned SMFI_FAILURE");
    result = SMFIS_REJECT;
#endif
  } else {
    if (log_level > 10 && priv->msg_regex_matches > 0) {
      char            logbuf[256];

      snprintf (logbuf, sizeof (logbuf), "%s : score %d",
                "*** SCAN_BODY_REGEX", priv->msg_regex_matches);
      j_syslog (priv->conn_id, "*** SCAN_BODY_REGEX", priv->peer_addr,
                priv->peer_name, priv->env_from, priv->env_to, priv->hdr_from, NULL);
      stats_inc (STAT_BODY_CONTENTS, 1);
    }
  }

  return result;
}

#endif

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
bool
check_intranet_user (rcpt_host, rcpt_addr, envto)
     char           *rcpt_host;
     char           *rcpt_addr;
     char           *envto;
{
  char           *host = NULL;
  char           *user = NULL;
  bool            host_local = FALSE;
  bool            user_local = FALSE;
  char           *p;

  if ((envto != NULL) && check_local_user (envto))
    return TRUE;

  if ((rcpt_host == NULL) || (strlen (rcpt_host) == 0))
    return FALSE;
  if ((rcpt_addr == NULL) || (strlen (rcpt_addr) == 0))
    return FALSE;

  /* First of all, let's check if rcpt_host is in sendmail class w */
  if ((host = strdup (rcpt_host)) == NULL) {
    syslog (LOG_ERR, "Error strdup(rcpt_host) %s", strerror (errno));
    return FALSE;
  }

  if (strcmp (host, "???") != 0)
    host_local = check_classw (host);
  else
    host_local = TRUE;

  if (!host_local) {
    int             len = strlen (host);

    if (host[len - 1] == '.')
      host[len - 1] = '\0';
    host_local = check_classw (host);
  }

  free (host);

  if (!host_local)
    return FALSE;

  /* Yes, it is ! Let's check if rcpt_addr is an intranet address */
  if ((user = strdup (rcpt_addr)) == NULL) {
    syslog (LOG_ERR, "Error strdup(rcpt_addr) %s", strerror (errno));
    return FALSE;
  }

  p = strchr (user, '@');
  if (p != NULL)
    *p = '\0';

  user_local = check_local_user (user);
  free (user);

  return user_local;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
int
check_email_in_domain (domain, email)
     char           *domain;
     char           *email;
{
  regex_t         re;
  char            sre[256];
  int             result = 0;

  if (email == NULL || domain == NULL || strlen (domain) == 0)
    return result;

  snprintf (sre, sizeof sre, "@[a-z0-9.-]*%s", domain);

  if (!regcomp (&re, sre, REG_ICASE)) {
    result = !regexec (&re, email, (size_t) 0, NULL, REG_NOTBOL | REG_NOTEOL);
    regfree (&re);
  }
  return result;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
static void
warn_action (ctx, priv, ahead, answer, why, tag)
     SMFICTX        *ctx;
     struct mlfiPriv *priv;
     attachment     *ahead;
     char           *answer;
     char           *why;
     char           *tag;
{
  char            msg[0x8000];

  read_error_msg (msg, sizeof (msg), ahead, answer, priv->env_from, why, tag, priv);
  if (smfi_replacebody (ctx, (u_char *) msg, strlen (msg)) != 0)
    syslog (LOG_ERR, "Erreur replacing Body Message : %s", strerror (errno));

  smfi_chgheader (ctx, "Content-type", 1, NULL);

  if (cf_get_int (CF_J_SENDER) != OPT_SENDER) {
    char           *sender = cf_get_str (CF_J_SENDER);

    if ((sender != NULL) && (strlen (sender) > 0))
      smfi_chgheader (ctx, "From", 1, sender);
  }

  if (cf_get_int (CF_J_SUBJECT) != OPT_SUBJECT) {
    char           *subject = cf_get_str (CF_J_SUBJECT);

    if ((subject != NULL) && (strlen (subject) > 0))
      smfi_chgheader (ctx, "Subject", 1, subject);
  }

  if ((cf_get_int (CF_WARN_SENDER) == OPT_YES ||
       cf_get_int (CF_WARN_RCPT) == OPT_YES) && (priv->env_from || priv->hdr_from)) {
    char           *p = NULL;

    if (priv->env_from != NULL && (strlen (priv->env_from) > 0)
        && strcmp (priv->env_from, "<>"))
      p = priv->env_from;
    if (p == NULL) {
      if (priv->hdr_from && (strlen (priv->hdr_from) > 0)
          && strcmp (priv->hdr_from, "<>"))
        p = priv->hdr_from;
    }
    if (p != NULL && strcmp (p, "<>") != 0)
      smfi_addrcpt (ctx, p);
  }

  if (cf_get_int (CF_WARN_RCPT) == OPT_NO) {
    rcpt_rec       *p = priv->env_rcpt;

    while (p != NULL) {
      if (p->rcpt != NULL)
        smfi_delrcpt (ctx, p->rcpt);
      p = p->next;
    }
  }
}



/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */

static void
read_error_msg (buf, sz, ahead, answer, from, why, tag, priv)
     char           *buf;
     int             sz;
     attachment     *ahead;
     char           *answer;
     char           *from;
     char           *why;
     char           *tag;
     struct mlfiPriv *priv;
{
  FILE           *fin;
  char            line[2048];
  char            s1[2048];
  char           *error_msg_file = cf_get_str (CF_ERROR_MSG_FILE);

  *buf = '\0';
  if ((fin = fopen (error_msg_file, "r")) != NULL) {
    regex_t         re;
    regmatch_t      pm;
    bool            compile_ok;
    int             state = 0;

    char            tag_begin[256];
    char            tag_end[256];

    snprintf (tag_begin, sizeof (tag_begin), "<%s>", tag);
    snprintf (tag_end, sizeof (tag_end), "</%s>", tag);
    compile_ok = !regcomp (&re, "_+[A-Z]+_+", REG_ICASE | REG_EXTENDED);

    while ((strlen (buf) < sz - 1)
           && (fgets (line, sizeof (line), fin) != NULL)) {

      char           *p = line + strlen (line) - 1;

      if (*p == '\n')
        *p = '\0';
      strcat (line, CRLF);

      if (state == 0) {
        p = line + strspn (line, " \t");
        if (*p == '#')
          continue;

        if (strstr (line, tag_begin) == NULL)
          continue;
        state = 1;
        continue;
      } else {
        if (strstr (line, tag_end) != NULL)
          break;
      }

      if (compile_ok && !regexec (&re, line, 1, &pm, 0)) {
        char           *p = s1;

        /*int  len = pm.rm_eo - pm.rm_so; */
        int             j;

        for (j = pm.rm_so; j < pm.rm_eo; j++) {
          if (line[j] != '_')
            *p++ = line[j];
        }
        *p = '\0';
        if (strcasecmp (s1, "MSGID") == 0) {
          char            filename[256];

          if (priv->fname == NULL)
            continue;
          /* delete directory part from filename */
          if (j_basename (filename, priv->fname) == filename)
            sprintf (line, "  **** MSG ID : %s", filename);
          else
            sprintf (line, "  **** MSG ID : %s", priv->fname);
          strcat (buf, line);
          strcat (buf, CRLF);
          continue;
        }
        if (strcasecmp (s1, "FROM") == 0) {
          from = priv->env_from ? priv->env_from : "???";
          sprintf (line, "  **** From : %s", from);
          strcat (buf, line);
          strcat (buf, CRLF);
          continue;
        }
        if (strcasecmp (s1, "SUBJECT") == 0) {
          char           *p;

          priv->hdr_has_subject = TRUE;
          p = priv->hdr_subject ? priv->hdr_subject : "-- NULL --";
          sprintf (line, "  **** Subject : %s", p);
          strcat (buf, line);
          strcat (buf, CRLF);
          continue;
        }
        if (strcasecmp (s1, "WHY") == 0) {
          sprintf (line, "%s", why);
          strcat (buf, line);
          strcat (buf, CRLF);
          continue;
        }
        if (strcasecmp (s1, "VIRUS") == 0) {
          sprintf (line, "  VIRUS FOUND : %s", answer ? answer : "NONE");
          strcat (buf, line);
          strcat (buf, CRLF);
          continue;
        }
        if (strcasecmp (s1, "TO") == 0) {
          continue;
        }
        if (strcasecmp (s1, "ATTACHMENT") == 0 && ahead != NULL) {
          attachment     *p = ahead;
          int             nb = 0;

          if (p == NULL) {
            sprintf (line, "      There isn't attached files !!!");
            strcat (buf, line);
            strcat (buf, CRLF);
            continue;
          }
          while (p != NULL) {
            char           *serror;

            switch (p->exefile) {
              case 0:
                serror = "CLEAN";
                break;
              case 1:
                nb++;
                serror = "X-FILE ???";
                break;
              default:
                serror = "UNCHECKED";
            }
            sprintf (line, "  **** (%-11s) : %s", serror, p->name);
            strcat (buf, line);
            strcat (buf, CRLF);
            if (p->mimetype != NULL) {
              sprintf (line, "        TYPE         : %s", p->mimetype);
              strcat (buf, line);
              strcat (buf, CRLF);
            }
            p = p->next;
          }
          if (nb > 0) {
            strcat (buf, CRLF);
            sprintf (line, "  **** SUSPECT FILES : %d", nb);
            strcat (buf, line);
            strcat (buf, CRLF);
          }
          continue;
        }
      } else
        strcat (buf, line);
    }
    fclose (fin);
    regfree (&re);
    if (state == 0) {
      syslog (LOG_ERR, "read_error_msg : msg for tag %s not found", tag);
    }
  } else {
    syslog (LOG_ERR, "Error opening %s file : %s", error_msg_file, strerror (errno));
  }

  /* Let's add a copyright footer */
  {
    char            s[128];
    char           *msg = "j-chkmail - (c) Ecole des Mines de Paris 2002";

    strset (s, '-', 64);
    strcat (buf, s);
    strcat (buf, CRLF);
    center_string (s, msg, 64);
    strcat (buf, s);
    strcat (buf, CRLF CRLF);
  }

}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
char *
get_boundary_tag_value (old, in)
     char           *old;
     char           *in;
{
  regex_t         re;

  /* let's check if the headear contains a boundary declaration */
  if (regcomp (&re, "multipart.*boundary=", REG_ICASE) == 0) {
    regmatch_t      pm;
    char           *p = in;

    if (!regexec (&re, p, 1, &pm, 0)) {
      /* does the message already contains a boundary declaration ? */
      if (old != NULL) {
        /* yes ! Strange - shall log it, and replace the old one */
        free(old);
        old = NULL;
      }

      p += pm.rm_eo;

      /* Let's allocate and clean the tag */
      if ((old = strdup(p)) != NULL) {
        bool quote = FALSE;
        char *ptr = p;
        char *s = old;

        s = old;
        for (*s = '\0'; *ptr != '\0'; ptr++) {
          if (*ptr == '"') {
            if (quote)
              break;
            quote = TRUE;
            continue;
          }
          if (strcspn("\n\r\t", ptr) == 0)
            break;
          if (*ptr == ' ') {
            if (!quote)
              break;
            *s++ = *ptr;
            continue;
          }
          *s++ = *ptr;
        }
        *s = '\0';
      } else {
        syslog (LOG_ERR, "--- %s : malloc hdr_boundary : %s",
                J_FUNCTION, strerror (errno));
      }
    }
  }
  regfree (&re);

  return (old);
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
static pthread_mutex_t st_mutex = PTHREAD_MUTEX_INITIALIZER;
static CONNID_T sid = { {0, 0} };

static          bool
new_conn_id (id)
     CONNID_T       *id;
{
  time_t          now;

  if (id == NULL)
    return FALSE;

  pthread_mutex_lock (&st_mutex);
  now = time (NULL);

  if (now != sid.id[0]) {
    sid.t[0] = now;
    sid.t[1] = 0;
  } else
    sid.t[1]++;

  snprintf(sid.id, sizeof(sid.id), "%08lX.%03lX", (long ) sid.t[0], (long ) sid.t[1]);

  *id = sid;

  if (log_level > 10)
    syslog(LOG_INFO, "%s : new id = %s", J_FUNCTION, id->id);

  pthread_mutex_unlock (&st_mutex);

  return TRUE;
}


/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */

void
rcpt_list_free (head)
     rcpt_rec      **head;
{
  rcpt_rec       *p;

  while (*head != NULL) {
    p = (*head)->next;
    free ((*head)->rcpt);
    free (*head);
    *head = p;
  }
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */

int
rcpt_list_add (head, rcpt)
     rcpt_rec      **head;
     char           *rcpt;
{
  rcpt_rec       *p, *q;

  if (rcpt == NULL || strlen (rcpt) == 0)
    return 1;

  if ((p = malloc (sizeof (rcpt_rec))) == NULL)
    return 1;

  memset (p, 0, sizeof (*p));
  if ((p->rcpt = strdup (rcpt)) == NULL) {
    free (p);
    return 1;
  }

  if (*head == NULL) {
    *head = p;
    return 0;
  }
  q = *head;
  while (q->next != NULL)
    q = q->next;
  q->next = p;
  return 0;
}

/* ****************************************************************************
 *                                                                            *
 *                                                                            *
 **************************************************************************** */
#if 0
int
rcpt_list_del (head, rcpt)
     rcpt_rec      **head;
     char           *rcpt;
{
  rcpt_rec       *p, *q;

  if (rcpt == NULL || strlen (rcpt) == 0)
    return 0;
  if (*head == NULL)
    return 0;

  return 0;
}
#endif

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
int
count_rcpt (rcpt)
     rcpt_rec       *rcpt;
{
  int             n = 0;
  rcpt_rec       *p = rcpt;

  while (p != NULL) {
    n++;
    p = p->next;
  }
  return n;
}

/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */

struct smfiDesc smfilter = {
  PACKAGE,                      /* filter name */
  SMFI_VERSION,                 /* version code -- do not change */
  SMFIF_ADDHDRS | SMFIF_CHGHDRS | SMFIF_CHGBODY | SMFIF_ADDRCPT | SMFIF_DELRCPT,
  /* flags */
  mlfi_connect,                 /* connection info filter */
  mlfi_helo,                    /* SMTP HELO command filter */
  mlfi_envfrom,                 /* envelope sender filter */
  mlfi_envto,                   /* envelope recipient filter */
  mlfi_header,                  /* header filter */
  mlfi_eoh,                     /* end of header */
  mlfi_body,                    /* body block filter */
  mlfi_eom,                     /* end of message */
  mlfi_abort,                   /* message aborted */
  mlfi_close                    /* connection cleanup */
};


/* ****************************************************************************
 *                                                                            * 
 *                                                                            *
 **************************************************************************** */
int
j_chkmail ()
{
  int             r;
  char           *dir;

  dir = cf_get_str (CF_WORKDIR);
  if (strlen (dir) > 0) {
    if (*dir != '/')
      syslog (LOG_WARNING,
              "j_chkmail : warning : WORKDIR doesn't begins with a / : %s", dir);
    if (chdir (dir) != 0)
      syslog (LOG_ERR,
              "j_chkmail : error changing to dir %s : %s", dir, strerror (errno));
  }

  signal (SIGTERM, SIG_DFL);
  signal (SIGHUP, SIG_DFL);
  signal (SIGUSR1, j_filter_sig_handler);
  signal (SIGUSR2, j_filter_sig_handler);
  signal (SIGALRM, j_filter_sig_handler);
  alarm (2 * DT_SIGALRM);

  j_output = J_SYSLOG;

  init_proc_state ();

  (void) throttle_init (10000);
  throttle_read_table (NULL);
  resolve_tab_read (NULL);

#if 0
  throttle_update_thread ();
  resolve_tab_update_thread ();
#endif

  launch_periodic_tasks_thread ();

  if (strlen (sm_sock) > 0) {
    (void) smfi_setconn (sm_sock);
  } else {
    syslog (LOG_ERR, "FATAL ERROR Don't know how to communicate with sendmail");
    exit (1);
  }

  if (smfi_register (smfilter) == MI_FAILURE) {
    fprintf (stderr, "smfi_register failed\n");
    exit (EX_UNAVAILABLE);
  }
#if HAVE_SMFI_SETBACKLOG
  smfi_setbacklog (24);
#endif

  (void) raw_history_open (FALSE);

  r = smfi_main ();
  syslog (LOG_ERR, "Joe's j-chkmail exiting smfi_main() = %d : ", r);
  if (r == 0)
    remove_milter_unix_sock ();

  raw_history_close ();

  log_stats (cf_get_int (CF_LOG_COUNTERS), cf_get_int (CF_DUMP_COUNTERS));

  save_state ();
  throttle_save_table (NULL);
  resolve_tab_save (NULL);

  return 0;
}
